import java.io.*;
import javax.microedition.lcdui.*;

public class WallRectangle extends LevelObject implements Collidable
{
    public int mode;
    public int bump_power;
    
    public int color;
    
    public int[] draw_vtx;
    
    public boolean is_rotated;
    public int[] center;
    public int[] axis_x, axis_y;
    public int[] axis_size;
    
    private int m_bumper;
    
    /**
     * Updates the level object.
     * 
     * @param time_secs
     */
    public void update( int time_secs )
    {
        if ( mode == MODE_BUMPER )
        {
            m_bumper -= FP.mul( g_bumper_speed, time_secs );
            if ( m_bumper < g_bumper_min )
                m_bumper = g_bumper_min;
        }
    }

    /**
     * Renders the level object.
     */
    public void render( Graphics g, int top_x, int top_y, int scale_x, int scale_y )
    {
        int x1 = FP.toInt( FP.mul( (draw_vtx[0]-top_x), scale_x ) );
        int y1 = FP.toInt( FP.mul( (draw_vtx[1]-top_y), scale_y ) );
        int x3 = FP.toInt( FP.mul( (draw_vtx[4]-top_x), scale_x ) );
        int y3 = FP.toInt( FP.mul( (draw_vtx[5]-top_y), scale_y ) );
        
        g.setColor( color );

        int x2 = FP.toInt( FP.mul( (draw_vtx[2]-top_x), scale_x ) );
        int y2 = FP.toInt( FP.mul( (draw_vtx[3]-top_y), scale_y ) );
        int x4 = FP.toInt( FP.mul( (draw_vtx[6]-top_x), scale_x ) );
        int y4 = FP.toInt( FP.mul( (draw_vtx[7]-top_y), scale_y ) );

        g.fillTriangle( x1, y1, x2, y2, x3, y3 );
        g.fillTriangle( x1, y1, x3, y3, x4, y4 );
        
        // Bumping?
        if ( mode == MODE_BUMPER )
        {
            int e1 = axis_size[0] + m_bumper;
            int e2 = axis_size[1] + m_bumper;
            
            int cx = center[0];
            int cy = center[1];
            int ax_x = FP.mul(axis_x[0],e1);
            int ax_y = FP.mul(axis_x[1],e1);
            int ay_x = FP.mul(axis_y[0],e2);
            int ay_y = FP.mul(axis_y[1],e2);
            
            x1 = FP.toInt( FP.mul( (cx+ax_x+ay_x-top_x), scale_x ) );
            y1 = FP.toInt( FP.mul( (cy+ax_y+ay_y-top_y), scale_y ) );
            x2 = FP.toInt( FP.mul( (cx-ax_x+ay_x-top_x), scale_x ) );
            y2 = FP.toInt( FP.mul( (cy-ax_y+ay_y-top_y), scale_y ) );
            x3 = FP.toInt( FP.mul( (cx-ax_x-ay_x-top_x), scale_x ) );
            y3 = FP.toInt( FP.mul( (cy-ax_y-ay_y-top_y), scale_y ) );
            x4 = FP.toInt( FP.mul( (cx+ax_x-ay_x-top_x), scale_x ) );
            y4 = FP.toInt( FP.mul( (cy+ax_y-ay_y-top_y), scale_y ) );
            
            g.drawLine( x1,y1, x2,y2 );
            g.drawLine( x2,y2, x3,y3 );
            g.drawLine( x3,y3, x4,y4 );
            g.drawLine( x4,y4, x1,y1 );
        }
    }
    
    /**
     * Loads the object from the stream.
     * 
     * @param in The data stream.
     * @throws IOException
     */
    public void load(DataInputStream in) throws IOException
    {
        // Read the position
        int cx = in.readInt();
        int cy = in.readInt();
        center = Vec2.create( cx, cy );
        
        // Mode
        mode = in.readByte();
        
        // Bump power
        bump_power = in.readInt();
        
        // Color
        color = in.readInt();
        
        // Is rotated?
        is_rotated = in.readBoolean();
        
        // Draw vertices
        draw_vtx = new int[ 4 * 2 ];
        for ( int i = 0; i < draw_vtx.length; ++i )
        {
            draw_vtx[i] = in.readInt();
        }
        
        // Axis x
        int axx = in.readInt();
        int axy = in.readInt();
        axis_x = Vec2.create(axx, axy);
        
        // Axis y
        int ayx = in.readInt();
        int ayy = in.readInt();
        axis_y = Vec2.create(ayx, ayy);
        
        // Axis sizes
        int sx = in.readInt();
        int sy = in.readInt();
        axis_size = Vec2.create(sx, sy);
    }
    
    /**
     * Resolves a possible verlet collision.
     * 
     * @param v The verlet.
     */
    public void resolveCollision( Verlet v )
    {
        if (mode == MODE_NO_COLLISIONS)
            return;
        if (!bounding_box.contains(v.pos[0], v.pos[1]))
            return;
        
        Vec2.sub(g_temp_vec, v.pos, center);
        
/*        System.out.println( "Pt: " + Vec2.toString(v.pos) );
        System.out.println( "Center: " + Vec2.toString(center) );
        System.out.println( "AxisX: " + Vec2.toString(axis_x) );
        System.out.println( "AxisY: " + Vec2.toString(axis_y) );
        System.out.println( "Size: " + Vec2.toString(axis_size) );*/
        
        // Do each axis
        int distx = 0;
        int disty = 0;
        for (int i = 0; i < 2; ++i)
        {
            int[] axis = (i==0) ? axis_x : axis_y;
            
            // Check how far we are in the axis
            int dist = Vec2.dot(g_temp_vec, axis);
            
//            System.out.println("Dist: " + FP.toString(dist));
            
            // Distance checks
            int size = axis_size[i];
            if (dist > size || dist < -size)
                return;
            
            // Get the distance
            if (i == 0)
                distx = dist;
            else
                disty = dist;
        }
        
//        System.out.println( "Result: " + Vec2.toString(g_temp_vec2) );
        
        // Move to the edge on the smaller axis fix (closest to the edge)
        int[] normal;
        boolean inv_normal;
        int[] tangent;
        if ((axis_size[0]-Math.abs(distx)) < (axis_size[1]-Math.abs(disty)))
        {
            inv_normal = (distx < 0);
            distx = (inv_normal) ? -axis_size[0] : axis_size[0];
            normal = axis_x;
            tangent = axis_y;
        }
        else
        {
            inv_normal = (disty < 0);
            disty = (inv_normal) ? -axis_size[1] : axis_size[1];
            normal = axis_y;
            tangent = axis_x;
        }
        
        // Is this a bumper?
        if (mode == MODE_BUMPER)
        {
            int mul = inv_normal ? -1 : 1;
            level.star.bump(v, mul*normal[0], mul*normal[1], bump_power);
            
            m_bumper = g_bumper_max;
        }
        
        // Kill?
        if (mode == MODE_DEATH)
        {
            level.star.kill();
            return;
        }
        
        // Start result at center of box; make steps from there
        Vec2.addScaled(g_temp_vec, center, axis_x, distx);
        Vec2.addScaled(v.pos, g_temp_vec, axis_y, disty);
        
        // Apply the friction
        v.applyFriction(tangent, g_friction_amount);
    }
    
    private static final int g_friction_amount = FP.toFP( 1.0f/*0.95f*/ );
    
    private static final int g_bumper_min = FP.toFP( 0.1f );
    private static final int g_bumper_max = FP.toFP( 0.4f );
    private static final int g_bumper_speed = FP.toFP( 2.f );
    
    private static int[] g_temp_vec = new int[ 2 ];
    private static int[] g_temp_vec2 = new int[ 2 ];
}
