package Spline;

Spline::boot_Spline();

sub linelen($$$$) {
	my ($x1, $y1, $x2, $y2) = @_;

	return sqrt(($x2-$x1)**2 + ($y2-$y1)**2);
}

sub linepoint($$$$$) {
	my ($n, $x1, $y1, $x2, $y2) = @_;

	return (($x2 - $x1) * $n + $x1, ($y2 - $y1) * $n + $y1);
}

sub quadraticpoint($$$$$$$) {
	my ($n, $x1, $y1, $x2, $y2, $x3, $y3) = @_;

	# Get point at $n on line (x1,y1)-(x2,y2)
	my($p1x, $p1y) = linepoint($n, $x1, $y1, $x2, $y2);

	# Get point at $n on line (x2,y2)-(x3,y3);
	my($p2x, $p2y) = linepoint($n, $x2, $y2, $x3, $y3);

	# Get point at $n on line (p1x, p1y)-(p2x, p2y);
	return linepoint($n, $p1x, $p1y, $p2x, $p2y);
}

sub cubicpoint($$$$$$$$$) {
	my ($n, $x1, $y1, $x2, $y2, $x3, $y3, $x4, $y4) = @_;

	my ($x, $y);

	$x = $x1 * ((1-$n) ** 3) +     3* $x2 * $n * ((1-$n) ** 2) +     3*$x3 * ($n ** 2) * (1-$n) +     $x4 * ($n ** 3);
	$y = $y1 * ((1-$n) ** 3) +     3* $y2 * $n * ((1-$n) ** 2) +     3*$y3 * ($n ** 2) * (1-$n) +     $y4 * ($n ** 3);

	return ($x, $y);
}

sub parseargs($) {
	my ($argstring) = @_;
	my @args;
	my @output;
	my $n = 0;

	@args = split(/([,-])/, $argstring);

	while($n < @args) {
		if($args[$n] eq '-') {
			push @output, -$args[$n+1];
			$n+=2;
		} elsif($args[$n] eq ',' || $args[$n] =~ /^$/) {
			$n+=1;
		} else {
			push @output, $args[$n];
			$n+=1;
		}
	}
	return @output;
	
}

# Parses SVG and turns it into MoveTo and Bezier cubic curve commands
sub parsesvg($) {
	my ($spline) = @_;
	my ($lastx, $lasty);

	my $points = [];

	my @pointstrings = split(/([A-Za-z])/, $spline);

	my $n = 0;

	while($n<@pointstrings) {
		if($pointstrings[$n] =~ /^$/) { $n++; next; }

		my %point = {};
		my $type = $pointstrings[$n];
		my @args = parseargs($pointstrings[$n+1]);

		# print "Type: $type\n";
		# print "Args: " . join(",", @args) . "\n";

		if($type eq 'M') { # abs. moveto
			$point{"type"} = 1;
			$point{"x"} = $args[0];
			$point{"y"} = $args[1];

			$lastx = $point{"x"};
			$lasty = $point{"y"};
			$lastcurvex = $point{"x"};
			$lastcurvey = $point{"y"};
			$startx = $point{"x"};
			$starty = $point{"y"};
		} elsif($type eq 'v') { # rel. lineto vert
			$point{"type"} = 3;
			$point{"x"} = $lastx;
			$point{"y"} = $args[0]+$lasty;

			$lastx = $point{"x"};
			$lasty = $point{"y"};
			$lastcurvex = $point{"x"};
			$lastcurvey = $point{"y"};
		} elsif($type eq 'V') { # abs. lineto vert
			$point{"type"} = 3;
			$point{"x"} = $lastx;
			$point{"y"} = $args[0];

			$lastx = $point{"x"};
			$lasty = $point{"y"};
			$lastcurvex = $point{"x"};
			$lastcurvey = $point{"y"};
		} elsif($type eq 'l') { # rel. lineto
			$point{"type"} = 3;
			$point{"x"} = $args[0]+$lastx;
			$point{"y"} = $args[1]+$lasty;

			$lastx = $point{"x"};
			$lasty = $point{"y"};
			$lastcurvex = $point{"x"};
			$lastcurvey = $point{"y"};
		} elsif($type eq 'L') { # abs. lineto
			$point{"type"} = 3;
			$point{"x"} = $args[0];
			$point{"y"} = $args[1];

			$lastx = $point{"x"};
			$lasty = $point{"y"};
			$lastcurvex = $point{"x"};
			$lastcurvey = $point{"y"};
		} elsif($type eq 'h') { # rel. lineto horiz
			$point{"type"} = 3;
			$point{"x"} = $lastx + $args[0];
			$point{"y"} = $lasty;

			$lastx = $point{"x"};
			$lasty = $point{"y"};
			$lastcurvex = $point{"x"};
			$lastcurvey = $point{"y"};
		} elsif($type eq 'H') { # abs. lineto horiz
			$point{"type"} = 3;
			$point{"x"} = $args[0];
			$point{"y"} = $lasty;

			$lastx = $point{"x"};
			$lasty = $point{"y"};
			$lastcurvex = $point{"x"};
			$lastcurvey = $point{"y"};
		} elsif($type eq 'c') { # rel. curve
			$point{"type"} = 2;
			$point{"x1"} = $args[0] + $lastx;
			$point{"y1"} = $args[1] + $lasty;
			$point{"x2"} = $args[2] + $lastx;
			$point{"y2"} = $args[3] + $lasty;
			$point{"x"} = $args[4] + $lastx;
			$point{"y"} = $args[5] + $lasty;

			$lastx = $point{"x"};
			$lasty = $point{"y"};
			$lastcurvex = $point{"x2"};
			$lastcurvey = $point{"y2"};
		} elsif($type eq 'C') { # abs. curve
			$point{"type"} = 2;
			$point{"x1"} = $args[0];
			$point{"y1"} = $args[1];
			$point{"x2"} = $args[2];
			$point{"y2"} = $args[3];
			$point{"x"} = $args[4];
			$point{"y"} = $args[5];

			$lastx = $point{"x"};
			$lasty = $point{"y"};
			$lastcurvex = $point{"x2"};
			$lastcurvey = $point{"y2"};
		} elsif($type eq 'S') { # abs smooth curve
			$point{"type"} = 2;
			$point{"x2"} = $args[0];
			$point{"y2"} = $args[1];
			$point{"x"} = $args[2];
			$point{"y"} = $args[3];
			# get x1, y1
			$point{"x1"} = -($lastcurvex - $lastx) + $lastx; # Last x2 reflected in current (x,y)
			$point{"y1"} = -($lastcurvey - $lasty) + $lasty; # Last y2 reflected in current (x,y)

			$lastx = $args[2];
			$lasty = $args[3];
			$lastcurvex = $point{"x2"};
			$lastcurvey = $point{"y2"};
		} elsif($type eq 's') { # rel. smooth curve
			$point{"type"} = 2;
			$point{"x2"} = $args[0] + $lastx;
			$point{"y2"} = $args[1] + $lasty;
			$point{"x"} = $args[2] + $lastx;
			$point{"y"} = $args[3] + $lasty;
			# get x1, y1
			$point{"x1"} = -($lastcurvex - $lastx) + $lastx; # Last x2 reflected in current (x,y)
			$point{"y1"} = -($lastcurvey - $lasty) + $lasty; # Last y2 reflected in current (x,y)

			$lastx = $args[2];
			$lasty = $args[3];
			$lastcurvex = $point{"x2"};
			$lastcurvey = $point{"y2"};
		} elsif($type eq 'z' || $type eq 'Z') {
			$point{"type"} = 3;
			$point{"x"} = $startx;
			$point{"y"} = $starty;
				
			$lastx = $args[2];
			$lasty = $args[3];
			$lastcurvex = $point{"x2"};
			$lastcurvey = $point{"y2"};
		} else {
			print "UNKNOWN TYPE $type\n";
		}

		$n+=2;

		print join(",", %point). "\n";

		push @{$points}, \%point;
	}

	return $points;
}

# Override for SetParams
sub SetParams($$) {
	my ($self, $params) = @_;

	# First, parse SVG string
	my @lines;

	foreach $pathstr (@{$params->{"paths"}}) {
		my $pathpart;

		my $spline = parsesvg($pathstr);
		my $splinepart;

		my ($curx, $cury);

		my $i;

		# $spline is ref to array of hash references, each with type = 1 (moveto) or type = 2 (bezier cubic curve) or type = 3 (lineto)

		foreach $splinepart (@{$spline}) {
			$pathpart = {};

			if($splinepart->{"type"} == 1) {
				# MoveTo
				# print "M $splinepart->{'x'} $splinepart->{'y'}\n";

				$pathpart->{"type"} = 1;
				$pathpart->{"x"} = $splinepart->{"x"};
				$pathpart->{"y"} = $splinepart->{"y"};

				push @lines, $pathpart;

			}
			elsif($splinepart->{"type"} == 2) {
				# CurveTo
				# print "C $splinepart->{'x'} $splinepart->{'y'} $splinepart->{'x1'} $splinepart->{'y1'} $splinepart->{'x2'} $splinepart->{'y2'}\n";

				my $parts = linelen($curx,$cury,$splinepart->{"x"}, $splinepart->{"y"}) * 10 + 1;
				print "Parts $parts\n";

				for($i=0;$i<=$parts;$i++) {
					my ($x, $y) = cubicpoint($i / $parts,
											 $curx, $cury, 
											 $splinepart->{"x1"}, $splinepart->{"y1"},
											 $splinepart->{"x2"}, $splinepart->{"y2"},
											 $splinepart->{"x"}, $splinepart->{"y"} );


					$pathpart = { type => 2, x => $x, y => $y };

					push @lines, $pathpart;
				}
			}
			elsif($splinepart->{"type"} == 3) {
				# LineTo
				# print "L $splinepart->{'x'} $splinepart->{'y'}\n";

				my $parts = linelen($curx,$cury,$splinepart->{"x"}, $splinepart->{"y"}) * 10 + 1;
				print "Parts $parts\n";

				for($i=0; $i<= $parts; $i++) {
					my ($x, $y) = linepoint($i / $parts,
											$curx, $cury,
											$splinepart->{"x"},$splinepart->{"y"});

					$pathpart = { type => 2, x => $x, y => $y };

					push @lines, $pathpart;

				}
			}
				

			($curx, $cury) = ($splinepart->{"x"}, $splinepart->{"y"});
		}
	}

	undef $params->{"paths"};

	$self->_SetParams( { path => \@lines, trails => $params->{"trails"}, alpha => $params->{"alpha"}, x => $params->{"x"}, y => $params->{"y"}, size => $params->{"size"} } );
}

1;
