//############################################################################
// Definet ###################################################################
//############################################################################

// Ruutu
#define	SCREEN_W	800
#define	SCREEN_H	600
#define	SCREEN_B	32

// Aika
#define TIMER_FPS	60
#define TIMER_MS	(Uint32)(1000.0f / (float)TIMER_FPS)

//#define FLAG_FULLSCREEN
#define SDL_GFX_FLAGS	(SDL_HWSURFACE | SDL_OPENGL | SDL_HWSURFACE)
#ifdef FLAG_FULLSCREEN
#define SDL_GFX_FLAGS	(SDL_GFX_FLAGS | SDL_FULLSCREEN)
#endif
//#define REMOVE_STDLIB

//############################################################################
// Includet ##################################################################
//############################################################################

// Posix
#include <math.h>

// Windows
#ifdef FLAG_WINDOWS
#include "windows.h"
#endif

// 3rd party
#include "SDL.h"
#include "GL/gl.h"
#include "GL/glu.h"

#ifndef M_PI
#define M_PI 3.14159265358979323846f
#endif

//############################################################################
// Timer #####################################################################
//############################################################################

Uint32 var_timer;

Uint32 timer_func(Uint32 interval)
{
  var_timer++;
  return interval;
}

//############################################################################
// Helper funcs ##############################################################
//############################################################################

int modulo(int val, int base)
{
  if(val < 0)
    val += base;
  else if(val >= base)
    val -= base;
  return val;
}

void memclr(void *op, size_t bytes)
{
  Uint32 *dst = (Uint32*)op;
  for(; bytes; bytes -= 4, dst++)
    *dst = 0;
}

/*
 * Kahden pisteen etäisyys ilman neliöjuurta.
 */
float dist_sqr(float *a, float *b)
{
  float dx = a[0] - b[0],
	dy = a[1] - b[1],
	dz = a[2] - b[2];
  return dx * dx + dy * dy + dz * dz;
}

/*
 * Vaihtaa kaksi flotaria keskenään.
 */
void xchgf(float *a, float *b)
{
  float c = *a;
  *a = *b;
  *b = c;
}

//############################################################################
// Kamera ####################################################################
//############################################################################

void cameraGL(float fov)
{
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(fov * (float)SCREEN_H / (float)SCREEN_W,
      (float)SCREEN_W / (float)SCREEN_H, 1.0, 5000.0);
  glMatrixMode(GL_MODELVIEW);
}

//############################################################################
// Interpolation #############################################################
//############################################################################

/*
 * All interpolation functions here.
 */

void interpolate_linear_i16d(double* target, Sint16* from, Sint16* to,
    float pos, int count)
{
  do
  {
    target[count] = ((float)(to[count]) - from[count]) * pos +
      (float)(from[count]);
  } while(count--);
}

void interpolate_linear_ff(float* target, float* from, float* to, float pos,
    int count)
{
  do
  {
    target[count] = (to[count] - from[count]) * pos + from[count];
  } while(count--);
}

void interpolate_linear_i16f(float* target, Sint16* from, Sint16* to,
    float pos, int count)
{
  do
  {
    target[count] = ((float)(to[count] - from[count])) * pos +
      (float)(from[count]);
  } while(count--);
}

float interpolate_sine(float base, float var, float currtime, float speed)
{
  return base + var * sin((float)currtime * speed);
}

float interpolate_count_se(int freeze, int start, int end)
{
  float pos = (float)(freeze - start) / (float)(end - start);
  float pos2 = 1.0f - fabs(0.5f - pos) * 2.0f;
  pos2 *= pos2 * 0.5f;
  if(pos < 0.5f)
    return pos2;
  return 1.0f - pos2;
}

float interpolate_count_lt(int freeze, int start, int life)
{
  return (float)(freeze - start) / (float)life;
}

//############################################################################
// Random ####################################################################
//############################################################################

/*
 * Random number generator: x[n + 1] = a * x[n] mod m
 * Made according to source by Glenn C Rhoads.
 * http://remus.rutgers.edu/~rhoads/Code/code.html
 */

static Uint32 var_rand_seed = 93186752, var_rand_a = 1588635695,
	      var_rand_q = 2, var_rand_r = 1117695901;

// Initialize random number generator
void rand_seed(int op)
{
  var_rand_seed = op;
}

// Random natural number [0, param[
int rand_int(int op)
{
  var_rand_seed = var_rand_a * (var_rand_seed % var_rand_q) -
    var_rand_r * (var_rand_seed / var_rand_q);
  return var_rand_seed % op;
}

// Random real number [0, number]
float rand_float(float op)
{
  return (float)(rand_int(32768)) / 32767.0f * op;
}

// Random real number [-number, number]
float rand_float_neg(float op)
{
  return (float)(rand_int(65534) - 32767) / 32767.0f * op;
}

//############################################################################
// Arrayt ####################################################################
//############################################################################

// Paljonko haluamme tilaa
#define ARRAY_HEAP		10000
#define SHAPEUNIT_STRIDE	20

/*
 * Struktuurit muodoille.
 */
typedef struct
{
  float tex[2];
  Uint32 col;
  float ver[3];
} ShapeUnit;

typedef struct
{
  ShapeUnit data[ARRAY_HEAP];
  Uint32 index[ARRAY_HEAP];
  int ins_vertex, ins_index;
} ShapeDB;

/*
 * Clear a shape database.
 */
void shapedb_clear(ShapeDB *op)
{
  op->ins_vertex = op->ins_index = 0;
}

/*
 * Functions to select the array pointers in databases.
 */
void shapedb_draw_elements(ShapeDB *op, int t, int element)
{
  // Specify the interleaevd array

  glInterleavedArrays(GL_T2F_C4UB_V3F, 0, op->data);

  // Tekstuuri päälle / pois
  if(t >= 0)
  {
    glEnable(GL_TEXTURE_2D);
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
    glDisableClientState(GL_COLOR_ARRAY);
    glBindTexture(GL_TEXTURE_2D, t);
  }
  else
  {
    glDisable(GL_TEXTURE_2D);
  }

  // Remove later
#ifndef REMOVE_STDLIB
  if((op->ins_index >= ARRAY_HEAP) || (op->ins_vertex >= ARRAY_HEAP))
  {
    printf("ERROR: too large object.\n");
    exit(0);
  }
#endif

  // Draw the elements
  glDrawElements(element, op->ins_index, GL_UNSIGNED_INT, op->index);
}

/*
 * Add a vertex and a color to a shape.
 */
void shapedb_add_cv(ShapeDB *op, float x, float y, float z, Uint32 c)
{
  // Shift the arrays
  ShapeUnit *dst = op->data + (op->ins_vertex++);
  // Set
  dst->ver[0] = x;
  dst->ver[1] = y;
  dst->ver[2] = z;
  dst->col = c;
}

/*
 * Add a vertex and a texcoord to a shape.
 */
void shapedb_add_tv(ShapeDB *op, float x, float y, float z, float t,
    float u)
{
  // Shift the arrays
  ShapeUnit *dst = op->data + (op->ins_vertex++);
  // Set
  dst->ver[0] = x;
  dst->ver[1] = y;
  dst->ver[2] = z;
  dst->tex[0] = t;
  dst->tex[1] = u;
}

/*
 * Add a vertex that is an interpolation between two other vertices.
 */
void shapedb_interpolate_v(ShapeDB *op, int from, int to, float pos,
    Uint32 col)
{
  ShapeUnit *dst = op->data + (op->ins_vertex++);
  // Interpoloi
  interpolate_linear_ff(dst->ver, op->data[from].ver, op->data[to].ver, pos,
      3);
  // Insertoi väri
  dst->col = col;
}

/*
 * Insert an index to the array.
 */
void shapedb_add_index(ShapeDB *op, Uint32 idx)
{
  op->index[op->ins_index++] = idx;
}

/*
 * Return a certain vertex.
 */
float* shapedb_get_vertex(ShapeDB *op, int idx)
{
  return op->data[idx].ver;
}

/*
 * Tulosta shapedb.
 */
#ifndef REMOVE_STDLIB
void shapedb_print(ShapeDB *op)
{
  int i;
  for(i = 0; i < op->ins_vertex; i++)
  {
    ShapeUnit *dst = &(op->data[i]);
    printf("<%f, %f, %f> [%f, %f]: %x\n", dst->ver[0], dst->ver[1], dst->ver[2],
	dst->tex[0], dst->tex[1], dst->col);
  }
  for(i = 0; i < op->ins_index; i++)
    printf("%i, ", op->index[i]);
  printf("\r\r\n");
}
#endif

//############################################################################
// Kontrolli #################################################################
//############################################################################

// Kontrolli
#define CIRCLES_START	0
#define PILLAR_START	3000
#define WINGS_START	3600
#define WINGS_END	4600
#define PILLAR_END	4900
#define GITS_START	5300
#define GITS_END	10000
#define FOLD_START	10000
#define TIME_END	15000

// Katsomissuunnat
typedef struct
{
  unsigned time;
  Sint16 data[6];
} LookAtUnit;

LookAtUnit data_look_at_buf[50] =
{
  // Circles
  { CIRCLES_START, { 10, 30, 0, 50, 30, 0 } },
  { CIRCLES_START + 300, { 5, 40, 20, -10, 30, -40 } },
  { CIRCLES_START + 550, { -5, 80, 10, -50, 50, 10 } },
  { CIRCLES_START + 800, { 5, 140, -20, 10, 40, 40 } },
  { CIRCLES_START + 1200, { -45, 160, -9, 40, 60, 5 } },
  { CIRCLES_START + 1500, { 5, 180, -9, -30, 10, -10 } },
  { CIRCLES_START + 1700, { 5, 180, -30, -10, 20, -20 } },
  { CIRCLES_START + 2200, { 10, 190, 40, 0, 20, 0 } },
  { CIRCLES_START + 2400, { -5, 190, 30, 15, 20, -10 } },
  { PILLAR_START, { -40, 150, 190, 0, 50, 0 } },
  { WINGS_START, { -60, 140, 220, 0, 80, 0 } },
  { WINGS_END, { -120, 90, 350, 0, 130, 0 } },
  { PILLAR_END, { -80, 150, 110, 0, 50, 0 } },
  { GITS_START, { -80, 130, 110, 0, 200, 0 } },
  // GitS
  { GITS_START, { 1000, 750, 7000, 1000, 750, -1000 } },
  { GITS_START + 50, { 1000, 750, 6000, 1000, 750, -1000 } },
  { GITS_START + 180, { 1000, 750, 1500, 1000, 750, -1000 } },
  { GITS_START + 1000, { 1500, 750, 1500, 1500, 750, -1000 } },
  // Fold
  { FOLD_START, { 0, 50, 350, 0, 90, 0 }, },
  { TIME_END, { 0, 50, 350, 0, 90, 0 } }
  /*{ CIRCLES_START, { 0, 180, 20, 0, 0, 0 } },
  { TIME_END, { 0, 180, 20, 0, 0, 0 } }*/
};

//############################################################################
// Tyhjennysvärit ############################################################
//############################################################################

#define FLASH_COUNT	1
#define FLASH_DURATION	100

float var_clearcolordb[3 * 4] =
{
  0.0f, 0.0f, 0.0f, 0.0f,
  0.3f, 0.2f, 0.0f, 0.0f,
  0.0f, 0.0f, 0.0f, 0.0f
};

float var_flashcolor[4] =
{
  0.0f, 1.0f, 1.0f, 1.0f
};

int var_flash[FLASH_COUNT] = { WINGS_END };

/*
 * Tyhjennä GL-ruutu.
 */
void clearGL(int op)
{
  float *col = var_clearcolordb + op * 4;

  // Tsekataan väläys
  int i = 0, freezetime = var_timer;
  do
  {
    int flash = var_flash[i];
    if((freezetime >= flash) && (freezetime < FLASH_DURATION + flash))
    {
      float intcol[4],
	    pos = interpolate_count_lt(freezetime, flash, FLASH_DURATION);
      interpolate_linear_ff(intcol, var_flashcolor, col, pos, 3);
      glClearColor(intcol[0], intcol[1], intcol[2], intcol[3]);
      glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
      return;
    }
  } while(i--);

  glClearColor(col[0], col[1], col[2], col[3]);
  glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
}

//############################################################################
// LookAt ####################################################################
//############################################################################

void look_at(void)
{
  LookAtUnit *to = &data_look_at_buf[1], *from = &data_look_at_buf[0];
  double data[6];

  while(!((to->time > var_timer) && (var_timer >= from->time)))
  {
    to++;
    from++;
  }

  // Laske arvot
  interpolate_linear_i16d(data, from->data, to->data,
      interpolate_count_se(var_timer, from->time, to->time), 5);

  // Katso
  gluLookAt(data[0], data[1], data[2], data[3], data[4], data[5],
      0.0, 1.0, 0.0);
}

//############################################################################
// Spline ####################################################################
//############################################################################

#define BEZIER_HISTORY	5

typedef struct
{
  Sint16 pos[BEZIER_HISTORY * 3];
} BezierControl;

typedef struct
{
  int d[3], u[3];
} BezierLine;

ShapeDB shape_bezier;

/*
 * Anna piste kolmen pisteen bezier-käyrältä, t:[0,1]
 */
void qbezier(float *dst, float *a, float *b, float *c, float t)
{
  float m = 1.0f - t;
  float m2 = m * m,
	t2 = t * t;

  int i = 2;
  do
  {
    dst[i] = m2 * a[i] + 2.0f * t * m * b[i] + t2 * c[i];
  } while(i--);
}

/*
 * Piirrä bspline.
 */
void efu_qbezier(BezierControl *bc, BezierLine *bl, int *ba, int freezetime)
{
  int from = 0,
      to = 1;
  float d[3 * 3], u[3 * 3];

  while(1)
  {
    if((freezetime >= ba[from]) && (freezetime < ba[to]))
      break;
    from++;
    to++;
  }

  float pos = interpolate_count_se(freezetime, ba[from], ba[to]);

  from *= 3;
  to *= 3;
  int i = 2;
  do
  {
    Sint16 *dl = bc[bl->d[i]].pos,
	   *ul = bc[bl->u[i]].pos;
    interpolate_linear_i16f(d + i * 3, dl + from, dl + to, pos, 2);
    interpolate_linear_i16f(u + i * 3, ul + from, ul + to, pos, 2);
  } while(i--);

  shapedb_clear(&shape_bezier);
  i = 11;
  pos = 0.0f;
  do
  {
    //printf("Pos = %f\n", pos);

    /*if((d[6] == u[6]) && (d[7] == u[7]) && (d[8] == u[8]))
      printf("Moi!\n");*/
    float tgt[3];
    qbezier(tgt, d + 0, d + 3, d + 6, pos);
    shapedb_add_cv(&shape_bezier, tgt[0], tgt[1], tgt[2], 0xF0FFFFFF);
    qbezier(tgt, u + 0, u + 3, u + 6, pos);
    shapedb_add_cv(&shape_bezier, tgt[0], tgt[1], tgt[2], 0xF0FFFFFF);
    shapedb_add_index(&shape_bezier, shape_bezier.ins_vertex - 2);
    shapedb_add_index(&shape_bezier, shape_bezier.ins_vertex - 1);
    pos += 0.1f;
  } while(--i);

  /*shapedb_clear(&shape_bezier3);
  shapedb_add_cv(&shape_bezier3, d[0], d[1], d[2], 0x00FF0000);
  shapedb_add_cv(&shape_bezier3, d[3], d[4], d[5], 0x00FF0000);
  shapedb_add_cv(&shape_bezier3, d[6], d[7], d[8], 0x00FF0000);
  shapedb_add_index(&shape_bezier3, 0);
  shapedb_add_index(&shape_bezier3, 1);
  shapedb_add_index(&shape_bezier3, 2);
  shapedb_draw_elements(&shape_bezier3, -1, GL_LINE_STRIP);

  shapedb_clear(&shape_bezier3);
  shapedb_add_cv(&shape_bezier3, u[0], u[1], u[2], 0x00FF0000);
  shapedb_add_cv(&shape_bezier3, u[3], u[4], u[5], 0x00FF0000);
  shapedb_add_cv(&shape_bezier3, u[6], u[7], u[8], 0x00FF0000);
  shapedb_add_index(&shape_bezier3, 0);
  shapedb_add_index(&shape_bezier3, 1);
  shapedb_add_index(&shape_bezier3, 2);
  shapedb_draw_elements(&shape_bezier3, -1, GL_LINE_STRIP);*/

  /*shapedb_add_index(&shape_bezier, 0);
  shapedb_add_index(&shape_bezier, 2);
  shapedb_add_index(&shape_bezier, 2);
  shapedb_add_index(&shape_bezier, 4);
  shapedb_add_index(&shape_bezier, 1);
  shapedb_add_index(&shape_bezier, 3);
  shapedb_add_index(&shape_bezier, 3);
  shapedb_add_index(&shape_bezier, 5);*/

  // Piirrä
  shapedb_draw_elements(&shape_bezier, -1, GL_TRIANGLE_STRIP);
}

//############################################################################
// Siivet ####################################################################
//############################################################################

/*
 * Siipien pisteet.
 */
BezierControl var_wingc[9] = 
{
  { // 0
    { 0, 0, 0,
      0, 0, 0,
      0, 0, 0,
      0, 0, 0,
      0, 0, 0
    }
  },
  { // 1
    {
      0, 50, 0,
      5, 50, -10,
      10, 50, -10,
      15, 50, -5,
      50, 80, -0
    }
  },
  { // 2
    {
      0, 100, 0,
      10, 100, 0,
      15, 110, 0,
      25, 115, 0,
      75, 140, 0
    }
  },
  { // 3
    {
      0, 70, 0,
      30, 80, 10,
      40, 90, 10,
      60, 120, 5,
      160, 180, 0
    }
  },
  { // 4
    {
      0, 20, 0,
      45, 30, 15,
      75, 40, 15,
      130, 180, 5,
      220, 220, 0
    }
  },
  { // 5
    {
      0, 60, 0,
      55, 70, 10,
      70, 80, 10,
      90, 110, 5,
      230, 200, 0
    }
  },
  { // 6
    {
      0, 90, 0,
      35, 90, 0,
      40, 125, -5,
      50, 105, 0,
      90, 110, 0
    }
  },
  { // 7
    {
      0, 40, 0,
      30, 55, -10,
      35, 50, -10,
      40, 50, -5,
      60, 70, 0      
    }
  },
  { // 8
    {
      0, 0, 0,
      10, 0, 0,
      10, 0, 0,
      10, 0, 0,
      10, 0, 0
    }
  }
};

/*
 * Siipien bezier-käyrät.
 */
BezierLine var_wingl[2] =
{
  {
    { 0, 1, 2 },
    { 8, 7, 6 }
  },
  {
    { 2, 3, 4 },
    { 6, 5, 4 }
  }
};

/*
 * Siipien ajoitus.
 */
int var_wingt[BEZIER_HISTORY] =
{
  0,
  100,
  300,
  500,
  800
};

/*
 * Piirrä siipi.
 */
void efu_wing(int freezetime, float x, float y, float z)
{
  rand_seed(0xdeadbeef);
  int i = 20;
  do
  {
    glPushMatrix();
    glScalef(x + rand_float_neg(1.0f) / 30.0f,
	y + rand_float_neg(1.0f) / 30.0f, z + rand_float_neg(1.0f) / 30.0f);
    efu_qbezier(var_wingc, var_wingl, var_wingt, freezetime);
    efu_qbezier(var_wingc, var_wingl + 1, var_wingt, freezetime);
    glPopMatrix();
  } while(--i);	
}

/*
 * Siipiefekti.
 */
void efu_wings(int freezetime)
{
  glPushMatrix();
  glRotatef(-20.0f, 0.0f, 1.0f, 0.0f);
  efu_wing(freezetime, 0.4f, 1.9f, 1.0f);
  efu_wing(freezetime, 1.0f, 1.2f, 1.0f);
  efu_wing(freezetime, 1.6f, 0.5f, 1.0f);
  efu_wing(freezetime, -0.4f, 1.9f, 1.0f);
  efu_wing(freezetime, -1.0f, 1.2f, 1.0f);
  efu_wing(freezetime, -1.6f, 0.5f, 1.0f);
  glPopMatrix();
}

//############################################################################
// Valopilari ################################################################
//############################################################################

#define LIGHTPILLAR_CNT		5
#define LIGHTPILLAR_WIDTH	5.0f
#define LIGHTPILLAR_HEIGHT	300.0f
#define LIGHTPILLAR_ROT		0.03
#define LIGHTPILLAR_SHARD_COMP	9
#define LIGHTPILLAR_SHARD_ARC	(M_PI / 1.8f)

// Valopilarien tyypit
typedef struct
{
  float height, width, radius;
  int lifetime, delay;
  Uint32 col;
} LightPillarParams;


// Valopilarien data
LightPillarParams pillar_draw = { 300.0f, 2.0f, 5.0f, 90, 5, 0xF0888888 };
LightPillarParams pillar_wings = { 400.0f, 16.0f, 100.0f, 300, 2, 0xFFC05050 };
ShapeDB shape_pillar_draw, shape_pillar_wings;

// Paskat muistista, generoidaan yksi joka framelle!
float var_pillar_shard[TIME_END * TIMER_MS];

/*
 * Generoi valopilarin shard.
 */
void generate_light_pillar_shard(ShapeDB *op, LightPillarParams *params)
{
  // Clear the shape
  shapedb_clear(op);

  // Generoi verteksit
  int i = LIGHTPILLAR_SHARD_COMP;
  do
  {
    // Common
    float percent = (float)i / (float)LIGHTPILLAR_SHARD_COMP;
    // Arc and height
    float ang = (LIGHTPILLAR_SHARD_ARC / 2.0f) - LIGHTPILLAR_SHARD_ARC *
	    ((float)i / (float)LIGHTPILLAR_SHARD_COMP),
	  mul = sin((float)(i + 1) / (float)(LIGHTPILLAR_SHARD_COMP + 2) *
	      M_PI),
	  ca = (cos(ang) - 1.0f),
	  sa = sin(ang);
    // Color
    float brightness_mul = sin(percent * M_PI);
    Uint32 col = 0;
    int j = 24;
    for(; j >= 0; j-= 8) 
    {
      col |= (Uint32)((float)((params->col & (0xFF << j)) >> j) *
	  brightness_mul) << j;
    }
    // Add
    shapedb_add_cv(op, ca, 0.0f, sa, col);
    shapedb_add_cv(op, ca, mul, sa, 0x00000000);
  } while(i--);

  // Generoi quadit
  i = LIGHTPILLAR_SHARD_COMP - 1;
  do
  {
    shapedb_add_index(op, i * 2 + 0);
    shapedb_add_index(op, i * 2 + 1);
    shapedb_add_index(op, i * 2 + 3);
    shapedb_add_index(op, i * 2 + 2);
  } while(i--);
}

/*
 * Piirtää yhden valopilarin ennalta annetuilla asetuksilla.
 */
void efu_light_pillar(float *pos, int stime, int etime, int freezetime,
    LightPillarParams *params, ShapeDB *shape)
{
  int i = stime;

  // Perutaanko suoraan?
  if(freezetime > etime + params->lifetime)
    return;

  // Piirretään
  for(i = stime; (i <= freezetime); i += params->delay)
    if((i + params->lifetime >= freezetime) && (i < etime))
    {
      float mul = interpolate_count_lt(freezetime, i, params->lifetime),
	    rad = mul * params->radius,
	    ang = var_pillar_shard[i],
	    dx = cos(ang) * rad,
	    dz = sin(ang) * rad;
      // Piirrä matriisitempuilla
      glPushMatrix();
      glTranslatef(dx + pos[0], pos[1], dz + pos[2]);
      glRotatef(ang / M_PI * -180.0f, 0.0f, 1.0f, 0.0f);
      glScalef(params->width, (1.0 - mul) * params->height, params->width);
      shapedb_draw_elements(shape, -1, GL_QUADS);
      glPopMatrix();
    }
}

//############################################################################
// Ympyrät ###################################################################
//############################################################################

#define CIRCLES_RADIALS		13
#define CIRCLES_SECTORS		48
#define CIRCLES_RADIUS		10.0f
#define CIRCLES_HEIGHT_BASE	60.0f
#define CIRCLES_HEIGHT_VAR	40.0f
#define CIRCLES_HEIGHT_SPEED	1.0f
#define CIRCLES_SPEED		20
#define CIRCLES_RUNNER_MAX	100
#define CIRCLES_OFF		0
#define CIRCLES_CCW		1
#define CIRCLES_CW		2
#define CIRCLES_IN		3
#define CIRCLES_OUT		4
#define CIRCLES_CFLOOR		0x995555FF
#define CIRCLES_CCEIL		0xFF00FFFF
#define CIRCLESRUNNER_RCHANGE	2
#define CIRCLESRUNNER_NCHANGE	40
#define CIRCLESRUNNER_DIVIDE	25

/*
 * Variables of circle
 */
char var_circles[CIRCLES_RADIALS * CIRCLES_SECTORS];
ShapeDB shape_circles;

typedef struct
{
  int radial, sector, dir, stime, etime;
} CirclesRunner;

CirclesRunner var_runner[CIRCLES_RUNNER_MAX];

/*
 * Generate one runner to draw the circle.
 */
void efu_circles_create_runner(unsigned radial, unsigned sector, unsigned dir,
    int elapsed)
{
  int i = CIRCLES_RUNNER_MAX;
  CirclesRunner *runner = var_runner;

  do
  {
    if(runner->dir == 0)
    {
      runner->radial = radial;
      runner->sector = sector;
      runner->dir = dir;
      runner->stime = elapsed;
      return;
    }
    runner++;
  } while(--i);
#ifndef REMOVE_STDLIB
  printf("No room!\n");
#endif
}

/*
 * True/false for the direction to be unblocked.
 */
int efu_circles_dir_ok(int idx)
{
  if((idx >= 0) && (idx < CIRCLES_SECTORS * CIRCLES_RADIALS))
    if(var_circles[idx] == 0)
      return 1;
  return 0;
}

/*
 * Calculate the index in the circle array of a given coordinate.
 */
int efu_circles_idx(int radial, int sector)
{
  if(radial < 0)
    return -1;
  else if(radial >= CIRCLES_RADIALS)
    return -1;
  return radial * CIRCLES_SECTORS + modulo(sector, CIRCLES_SECTORS);
}

/*
 * Try to expire a runner after it has stopped drawing.
 */
void efu_circles_expire_runner(CirclesRunner *op, int left, int freezetime)
{
  // If we're not in the end, let's quit
  if(left > CIRCLES_SPEED)
    return;
  // Otherwise try to expire
  efu_light_pillar(shapedb_get_vertex(&shape_circles,
	(op->radial * CIRCLES_SECTORS + op->sector) * 2), op->stime,
      op->etime, freezetime, &pillar_draw, &shape_pillar_draw);
}

/*
 * Draw a runner. Make some polygons while at it.
 */
void efu_circles_runner(CirclesRunner* op, int left, int elapsed,
    int freezetime)
{
  // Jos ei mitään niin ei tehdä
  if(op->dir == 0)
    return;
  // Jos enää pelkkä pilari
  else if(op->dir == -1)
  {
    efu_circles_expire_runner(op, left, freezetime);
    return;    
  }

  // Muutoin voidaan jatkaa
  int dir[4] =
  {
    dir[0] = efu_circles_idx(op->radial, op->sector - 1),
    dir[1] = efu_circles_idx(op->radial, op->sector + 1),
    dir[2] = efu_circles_idx(op->radial - 1, op->sector),
    dir[3] = efu_circles_idx(op->radial + 1, op->sector)
  };

  // Jos kohde ei ole ok, vaihda suuntaa
  if(efu_circles_dir_ok(dir[op->dir - 1]) == 0)
  {
    // Aseta päätös ja ajankohta
    op->dir = -1;
    op->etime = elapsed;
    // Anna uusi mahdollisuus vaihtaa suuntaa 5 kertaa
    int i = 6;
    do
    {
      int j = rand_int(4);
      if(efu_circles_dir_ok(dir[j]))
      {
	op->dir = j + 1;
	break;
      }
    } while(--i);
  }
  // Jos kaikki vieläkin pielessä
  if(op->dir <= 0)
  {
    efu_circles_expire_runner(op, left, freezetime);
    return;
  }

  // Valitaan ja tehdään neliö
  int curr = (op->radial * CIRCLES_SECTORS + op->sector) * 2,
      next = dir[op->dir - 1];
  // Kohde on käytetty
  var_circles[next] = 1;
  next *= 2;

  // Insertoi
  shapedb_add_index(&shape_circles, curr);
  shapedb_add_index(&shape_circles, curr + 1);
  // Haku
  if(left > CIRCLES_SPEED)
  {
    shapedb_add_index(&shape_circles, next + 1);
    shapedb_add_index(&shape_circles, next);
  }
  // Interpoloi
  else
  {
    float len = (float)left / (float)CIRCLES_SPEED;
    shapedb_add_index(&shape_circles, shape_circles.ins_vertex + 1);
    shapedb_add_index(&shape_circles, shape_circles.ins_vertex);
    shapedb_interpolate_v(&shape_circles, curr, next, len, CIRCLES_CFLOOR);
    shapedb_interpolate_v(&shape_circles, curr + 1, next + 1, len,
	CIRCLES_CCEIL);
    efu_light_pillar(shapedb_get_vertex(&shape_circles,
	  shape_circles.ins_vertex - 2), op->stime, TIME_END, freezetime,
	&pillar_draw, &shape_pillar_draw);
  }

  // Jakaudu
  if(!rand_int(CIRCLESRUNNER_DIVIDE))
    efu_circles_create_runner(op->radial, op->sector, rand_int(4) + 1,
	elapsed);

  // Jatkamme matkaa
  switch(op->dir)
  {
    case CIRCLES_CCW:
      op->sector -= 2;
    case CIRCLES_CW:
      op->sector++;
      op->sector = modulo(op->sector, CIRCLES_SECTORS);
      break;
    case CIRCLES_IN:
      op->radial -= 2;
    case CIRCLES_OUT:
      op->radial++;
      break;
  }

  // Vaihda suuntaa
  if(!rand_int(CIRCLESRUNNER_NCHANGE) ||
      ((op->dir >= CIRCLES_IN) && rand_int(2)))
    op->dir = rand_int(4) + 1;
}

/*
 * Ympyräefun ylihommeli.
 */
void efu_circles(Uint32 freezetime)
{
  int i, j, k;

  // Tyhjennys ja random
  clearGL(0);
  rand_seed(0x01);

  // Clear the shape
  shapedb_clear(&shape_circles);

  // Generate the vertices
  for(i = 0; i < CIRCLES_RADIALS; i++)
    for(j = 0; j < CIRCLES_SECTORS; j++)
    {
      float ang = (float)j / (float)CIRCLES_SECTORS * M_PI * 2;
      float rad = (i + 1) * CIRCLES_RADIUS;
      // Korkeus on monimutk.
      float height = interpolate_sine(CIRCLES_HEIGHT_BASE / 2.0f,
	CIRCLES_HEIGHT_VAR / 2.0f, (float)j + (float)freezetime / 100.0f,
	CIRCLES_HEIGHT_SPEED) + interpolate_sine(CIRCLES_HEIGHT_BASE / 2.0f,
	  CIRCLES_HEIGHT_VAR / 2.0f,
	  (float)j * 1.5f + (float)freezetime / 15.0f, -CIRCLES_HEIGHT_SPEED);
      // Add
      shapedb_add_cv(&shape_circles, cos(ang) * rad, 0.0f, sin(ang) * rad,
	  CIRCLES_CFLOOR);
      shapedb_add_cv(&shape_circles, cos(ang) * rad, height, sin(ang) * rad,
	  CIRCLES_CCEIL);
    }

  // Piirretään
  memclr(var_circles, CIRCLES_RADIALS * CIRCLES_SECTORS);
  var_circles[0] = 1;
  memclr(var_runner, CIRCLES_RUNNER_MAX * sizeof(CirclesRunner));
  efu_circles_create_runner(CIRCLES_RADIALS - 1, 0, CIRCLES_CCW, CIRCLES_START);
  for(i = freezetime, j = CIRCLES_START; i > 0;
      i -= CIRCLES_SPEED, j += CIRCLES_SPEED)
    for(k = 0; k < CIRCLES_RUNNER_MAX; k++)
      efu_circles_runner(&var_runner[k], i, j, freezetime);

  /*float pos[3] = { 0, 0, 0 };
  ShapeDB shp;
  LightPillarParams prm = { 400.0f, 40.0f, 25.0f, 100, 100, 0x00FFFFFF };
  generate_light_pillar_shard(&shp, &prm);
  efu_light_pillar(pos, 0, 200, &prm, &shp);*/

  shapedb_draw_elements(&shape_circles, -1, GL_QUADS);

  }

//############################################################################
// LineTextures ##############################################################
//############################################################################

#define LINETEXTURE_H		512
#define LINETEXTURE_W		32
#define LINETEXTURE_LINES	8
#define LINETEXTURE_CHANGE	20
#define LINECIRCLE_COMPLEXITY	32
#define LINECIRCLE_THICKNESS	0.025f

// Define the textures.
Uint32 linetexture1[LINETEXTURE_W * LINETEXTURE_H],
       linetexture2[LINETEXTURE_W * LINETEXTURE_H],
       linetexture3[LINETEXTURE_W * LINETEXTURE_H];
int linetexture1_bind, linetexture2_bind, linetexture3_bind;
ShapeDB shape_line_circle, shape_line_line;

/*
 * Generate the texture.
 */
void generate_line_texture(Uint32 *op, int *bind, Uint32 col)
{
  int i = LINETEXTURE_LINES;
  do
  {
    int cw = rand_int(LINETEXTURE_W - 2 + 1),
	h = LINETEXTURE_H;
    Uint32 *bmp = op;
    do
    {
      // Vaihdetaanko kolumnia?
      if(!rand_int(LINETEXTURE_CHANGE))
      {
	int ncw = rand_int(LINETEXTURE_W - 2) + 1;
	int j = ((ncw - cw) > 0) ? 1 : -1;
	for(; cw != ncw; cw += j)
	  bmp[cw] = col;
	cw = ncw;
      }
      // Kolumniin väri
      bmp[cw] = col;
      // Eteenpäin
      bmp += LINETEXTURE_W;
    } while(--h);
  } while(--i);

  glEnable(GL_TEXTURE_2D);
  glGenTextures(1, bind);
  glBindTexture(GL_TEXTURE_2D, *bind);
  gluBuild2DMipmaps(GL_TEXTURE_2D, 4, LINETEXTURE_W, LINETEXTURE_H, GL_RGBA,
      GL_UNSIGNED_BYTE, op);
}

/*
 * Generate line circle.
 */
void generate_line_circle()
{
  int i = LINECIRCLE_COMPLEXITY;
  do
  {
    float percent = (float)i / (float)LINECIRCLE_COMPLEXITY,
	  ang = percent * M_PI * 2,
	  ca = cos(ang),
	  sa = sin(ang); 
    shapedb_add_tv(&shape_line_circle, ca, -LINECIRCLE_THICKNESS, sa,
	0.0f, percent);
    shapedb_add_tv(&shape_line_circle, ca, LINECIRCLE_THICKNESS, sa,
	1.0f, percent);
    shapedb_add_index(&shape_line_circle, i * 2);
    shapedb_add_index(&shape_line_circle, i * 2 + 1);
  } while(i--);
}

/*
 * Generate line with a line texture from point a to point b.
 */
float efu_line_line(ShapeDB *shp, float start, float end, float other1,
    float other2, float radius, float texpos, float texwrap, int axis)
{
  float pos[3],
	ang = rand_float(M_PI * 2.0f),
	ca = cos(ang) * radius,
	sa = sin(ang) * radius,
	texend = fabs(end - start) / texwrap + texpos;
  int ax1 = (axis + 1) % 3,
      ax2 = (axis + 2) % 3,
      idx = shp->ins_vertex;

  // Indeksit
  shapedb_add_index(shp, idx);
  shapedb_add_index(shp, idx + 1);
  shapedb_add_index(shp, idx + 2);
  shapedb_add_index(shp, idx + 3);
  // Data
  pos[axis] = start;
  pos[ax1] = other1 + ca;
  pos[ax2] = other2 + sa;
  shapedb_add_tv(shp, pos[0], pos[1], pos[2], 0.0f, texpos);
  pos[ax1] = other1 - ca;
  pos[ax2] = other2 - sa;
  shapedb_add_tv(shp, pos[0], pos[1], pos[2], 1.0f, texpos);
  pos[axis] = end;
  shapedb_add_tv(shp, pos[0], pos[1], pos[2], 1.0f, texend);
  pos[ax1] = other1 + ca;
  pos[ax2] = other2 + sa;
  shapedb_add_tv(shp, pos[0], pos[1], pos[2], 0.0f, texend);

  return fmod(texend, 1.0);
}

/*
 * Draw line circle
 */
void efu_line_circle(float *pos, int texture, float radius, float xr,
    float yr, float zr)
{
  glPushMatrix();
  glTranslatef(pos[0], pos[1], pos[2]);
  glRotatef(xr, 1.0f, 0.0f, 0.0f);
  glRotatef(yr, 0.0f, 1.0f, 0.0f);
  glRotatef(zr, 0.0f, 0.0f, 1.0f);
  glScalef(radius, radius * 10.0f, radius);
  shapedb_draw_elements(&shape_line_circle, texture, GL_TRIANGLE_STRIP);
  glPopMatrix();
}

//############################################################################
// GitS -efu #################################################################
//############################################################################

#define GITS_BOUND_X	7000.0f
#define GITS_BOUND_YZ	1500.0f
#define GITS_GRIDSPACE	130
#define GITS_GRIDCNT_X	((int)(GITS_BOUND_X / (float)GITS_GRIDSPACE))
#define GITS_GRIDCNT_YZ	((int)(GITS_BOUND_YZ / (float)GITS_GRIDSPACE))
#define GITS_GRIDLINE_W	5.0f
#define GITS_FOGDENSITY	0.0005f
#define GITS_SPHERES	30
#define GITS_LINKSPER	5
#define GITS_PATH_RAND	300.0f
#define GITS_PATH_BASE	100.0f
#define GITS_TEXWRAP	1000.0f
#define GITS_PATH_MAX	(GITS_PATH_RAND + GITS_PATH_BASE)
#define GITS_PATH_LEN	1000
#define GITS_SPHERERAD	125.0f

//Spheretyyppi
typedef struct
{
  float pos[3];
  int connections[GITS_LINKSPER], conncnt;
} GitsSphere;

// Polku
typedef struct
{
  float pos[3 * GITS_PATH_LEN], flen;
  int len;
} GitsPath;

// Taulukko sphereille
GitsSphere var_gits_spheres[GITS_SPHERES];
GitsPath var_gits_paths[GITS_SPHERES * GITS_LINKSPER];
ShapeDB shape_gits_paths, shape_gits_grid;
int var_gits_path_start[TIME_END];
int var_gits_pathcount;

/*
 * Varmistaa ettei jossakin spheressä ole yhteyttä johonkin toiseen.
 */
int sphere_has_no_conn(GitsSphere *op, int conn)
{
  int i = GITS_LINKSPER - 1;
  do
  {
    if(op->connections[i] == conn)
      return 0;
  } while(i--);
  return 1;
}

/*
 * Generoi polku kahden pisteen välille.
 */
void generate_gits_path(GitsSphere *src, GitsSphere *dst, GitsPath *pth)
{
  float *pos = pth->pos, texpos = 0.0f;
  pth->len = 1;
  pth->flen = 0.0f;
  pos[0] = src->pos[0] + rand_float(GITS_SPHERERAD / 2.0f);
  pos[1] = src->pos[1] + rand_float(GITS_SPHERERAD / 2.0f);
  pos[2] = src->pos[2] + rand_float(GITS_SPHERERAD / 2.0f);
  do
  {
    int crd = rand_int(3);
    float diff = dst->pos[crd] - pos[crd];

    if(diff != 0.0f)
    {
      // Jos ollaan liian kaukana niin mennään randomilla
      if(sqrt(dist_sqr(pos, dst->pos)) >
	  sqrt(GITS_PATH_MAX * GITS_PATH_MAX * 3))
	diff = (rand_float(GITS_PATH_RAND) + GITS_PATH_BASE) *
	  ((diff >= 0) ? 1.0f : -1.0f);
      // Mennään eteenpäin
      int ax1 = (crd + 1) % 3,
	  ax2 = (crd + 2) % 3;
      pos[ax1 + 3] = pos[ax1];
      pos[ax2 + 3] = pos[ax2];
      pos[crd + 3] = pos[crd] + diff;
      // Lisää useampi linja, saadaan paksuutta!
      int i = 7;
      do
      {
	efu_line_line(&shape_gits_paths, pos[crd], pos[crd + 3],
	    pos[ax1] + rand_float_neg(GITS_GRIDLINE_W * 2.0f),
	    pos[ax2] + rand_float_neg(GITS_GRIDLINE_W * 2.0f),
	    GITS_GRIDLINE_W, texpos, GITS_TEXWRAP, crd);
      } while(--i);
      // Liike ok
      pos += 3;
      pth->flen += fabs(diff);
      pth->len++;
      // Jos saavuttu niin end!
      if(sqrt(dist_sqr(dst->pos, pos)) < GITS_SPHERERAD / 2.0f)
	break;
    }
  } while(1);
}

/*
 * Generoi yhteydet renkaitten välille.
 */
void generate_gits_connections()
{
  int i;

  // Arvo paikat
  i = GITS_SPHERES - 1;
  do
  {
    GitsSphere *sph = var_gits_spheres + i;
    sph->pos[0] = rand_float(GITS_BOUND_X);
    sph->pos[1] = rand_float(GITS_BOUND_YZ);
    sph->pos[2] = rand_float(GITS_BOUND_YZ);
    sph->conncnt = rand_int(GITS_LINKSPER) + 1;
    // Nollaa linkit
    int j = GITS_LINKSPER - 1;
    do
    {
      sph->connections[j] = -1;
    } while(j--);
  } while(i--);


  // Grid
  shapedb_clear(&shape_gits_grid);
  i = GITS_GRIDCNT_X;
  do
  {
    float xpos = (float)i / (float)GITS_GRIDCNT_X * GITS_BOUND_X;
    int j = GITS_GRIDCNT_YZ;
    do
    {
      float yzpos = (float)j / (float)GITS_GRIDCNT_YZ * GITS_BOUND_YZ,
	    boundmin = -1000.0f,
	    boundmax = GITS_BOUND_YZ + 1000.0f;
      // Vaihdetaan randomilla
      if(rand_int(2))
	xchgf(&boundmin, &boundmax);
      efu_line_line(&shape_gits_grid, boundmin, boundmax, yzpos, xpos,
	  GITS_GRIDLINE_W, rand_float(1.0f), GITS_TEXWRAP, 1);
      if(rand_int(2))
	xchgf(&boundmin, &boundmax);
      efu_line_line(&shape_gits_grid, boundmin, boundmax, xpos, yzpos,
	  GITS_GRIDLINE_W, rand_float(1.0f), GITS_TEXWRAP, 2);
      // Viimeisellä kierroksella piirretään myös loput viivat
      if(i == 0)
      {
	int k = GITS_GRIDCNT_YZ;
	do
	{
	  float yzpos2 = (float)k / (float)GITS_GRIDCNT_YZ * GITS_BOUND_YZ,
		boundmin = -1000.0f,
		boundmax = GITS_BOUND_X + 1000.0f;
	  if(rand_int(2))
	    xchgf(&boundmin, &boundmax);
	  efu_line_line(&shape_gits_grid, boundmin, boundmax, yzpos, yzpos2,
	      GITS_GRIDLINE_W, rand_float(1.0f), GITS_TEXWRAP, 0);
	} while(k--);
      }
    } while(j--);
  } while(i--);
  
  // Merkkaa yhteydet
  for(i = 0; i < GITS_SPHERES; i++)
  {
    GitsSphere *sph = var_gits_spheres + i;
    //printf("%i has connections: %i\n", i, sph->conncnt);
    // Hae sovittu määrä lähimpiä
    while(sph->conncnt > 0)
    {
      float closest = 10000000.0f;
      int j, closestidx = -1;
      for(j = i + 1; j < GITS_SPHERES; j++)
      {
	GitsSphere *tgt = var_gits_spheres + j;
	float newdist = dist_sqr(tgt->pos, sph->pos);
	if((newdist < closest) && (sphere_has_no_conn(sph, j)))
	{
	  closestidx = j;
	  closest = newdist;
	}
      }
      GitsSphere *tgt = var_gits_spheres + closestidx;
      // Jos ei löydy niin quit
      if(closestidx < 0)
	break;
      // Jos kohteessa ei tilaa niin quit
      if(tgt->conncnt <= 0)
	break;
      // Muutoin kaikki ok
      sph->connections[--(sph->conncnt)] = closestidx;
      tgt->conncnt--;
      //printf("Found connection: %i\n", closestidx);
    }
  }

  // Generoi linkit, erikseen koska tämä voisi olla cool jos se tehtäisiin
  // joka frame!
  shapedb_clear(&shape_gits_paths);
  var_gits_pathcount = 0;
  i = GITS_SPHERES - 1;
  do
  {
    GitsSphere *sph = var_gits_spheres + i;
    int j = GITS_LINKSPER - 1;
    do
    {
      int k = sph->connections[j];
      if(k >= 0)
      {
	GitsSphere *tgt = var_gits_spheres + k;
	generate_gits_path(sph, tgt, var_gits_paths + var_gits_pathcount);
	var_gits_pathcount++;
      }
    } while(j--);
  } while(i--);
}

/*
 * Piirrä liikkuva härpäke polulle.
 */
void efu_gits_path_haerpake(GitsPath *pth, int starttime, int currtime,
    float speed)
{
  float *pos = pth->pos;
  int lenleft = pth->len;

  // Jos ei riitä aika kummiskaan
  if(pth->flen < (currtime - starttime) * speed)
    return;

  while((starttime < currtime) && lenleft)
  {
    // Koska retangulaarisia, ok
    float dist[3] = { pos[3] - pos[0], pos[4] - pos[1], pos[5] - pos[2] },
	  use = fabs(dist[0] + dist[1] + dist[2]) / speed,
	  timeleft = currtime - starttime;
    // Jos aika riittää
    if(use >= timeleft)
    {
      use = timeleft / use;
      dist[0] = pos[0] + use * dist[0];
      dist[1] = pos[1] + use * dist[1];
      dist[2] = pos[2] + use * dist[2];
      rand_seed(0xdeadbeef);
      int j = 10;
      do
      {
	efu_line_circle(dist, linetexture1_bind, 10.0f, rand_float(360.0f),
	    rand_float(360.0f), rand_float(360.0f));
      } while(--j);
      break;
    }
    // Muutoin eteenpäin
    else
    {
      starttime += (int)use;
      pos += 3;
      lenleft--;
    }
  }
}

/*
 * Piirrä itse gits-efu.
 */
void efu_gits(int freezetime)
{
  // Tyhjennys ja random
  clearGL(1);
  rand_seed(0xdeadbeef);

  // Hyper draiv
  if((freezetime >= 50) && (freezetime <= 180))
    cameraGL((float)(130 - (freezetime - 50)) / 130.0f * 80.0f + 90.0f);

  // This effect has fog
  glFogfv(GL_FOG_COLOR, var_clearcolordb);
  glFogi(GL_FOG_MODE, GL_EXP2);
  glFogf(GL_FOG_DENSITY, GITS_FOGDENSITY);
  glEnable(GL_FOG);

  // Randomize a few circles
  glMatrixMode(GL_TEXTURE);
  glPushMatrix();
  glTranslatef(0.0f, 0.001f * freezetime, 0.0f);
  glMatrixMode(GL_MODELVIEW);

  int i = GITS_SPHERES - 1;
  do
  {
    GitsSphere *sph = var_gits_spheres + i;
    int j = 10;
    do
    {
      efu_line_circle(sph->pos, linetexture1_bind,
	  GITS_SPHERERAD + rand_float_neg(25.0f), rand_float(360.0f),
	  rand_float(360.0f), rand_float(360.0f));
    } while(--j);
  } while(i--);

  // Texture matrix again
  glMatrixMode(GL_TEXTURE);
  glTranslatef(0.0f, - 0.0005f * freezetime, 0.0f);

  // Grid and paths
  shapedb_draw_elements(&shape_gits_grid, linetexture2_bind, GL_QUADS);
  shapedb_draw_elements(&shape_gits_paths, linetexture3_bind, GL_QUADS);

  // Return to normal matrix mode
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);

  // Piirrä kaikki härpäkkeet
  for(i = 0; i < GITS_END - GITS_START; i+= 10)
  {
    efu_gits_path_haerpake(var_gits_paths + var_gits_path_start[i], i,
	freezetime, 5.0f);
  }
}

//############################################################################
// Main ######################################################################
//############################################################################

int main(void)
{
  // Lokaalit muuttujat
  int loop = 1;
  SDL_Event event;
  SDL_Surface *screen;

  /*
   * Init.
   */
  
  // SDL
  SDL_Init(SDL_INIT_TIMER | SDL_INIT_VIDEO | SDL_INIT_AUDIO);
  screen = SDL_SetVideoMode(SCREEN_W, SCREEN_H, SCREEN_B, SDL_GFX_FLAGS);
  SDL_SetTimer(TIMER_MS, timer_func);
  // OpenGL
  glViewport(0, 0, SCREEN_W, SCREEN_H);
  glClearDepth(1.0);
  //glDepthFunc(GL_LEQUAL);
  glEnable(GL_BLEND);
  glDisable(GL_DEPTH_TEST);
  glBlendFunc(GL_ONE_MINUS_SRC_ALPHA, GL_ONE);
  //glShadeModel(GL_SMOOTH);

  /*
   * Precalc.
   */
  
  // Tekstuurit
  generate_line_texture(linetexture1, &linetexture1_bind, 0x99A0C0FF);
  generate_line_texture(linetexture2, &linetexture2_bind, 0x44FF4444);
  generate_line_texture(linetexture3, &linetexture3_bind, 0x882222FF);

  // Muodot
  generate_light_pillar_shard(&shape_pillar_draw, &pillar_draw);
  generate_light_pillar_shard(&shape_pillar_wings, &pillar_wings);
  generate_line_circle();
  generate_gits_connections();

  // Asiat joita tarvii joka framelle
  int i = TIME_END * TIMER_MS - 1;
  do
  {
    var_pillar_shard[i] = rand_float(M_PI * 2.0f);
    var_gits_path_start[i] = rand_int(var_gits_pathcount);
  } while(i--);

  // Skip
  var_timer = 0;

  // Loop
  while(loop)
  {
    cameraGL(90.0f);
    // Olemme modelviewissä
    glLoadIdentity();
    // Katsellaan
    look_at();
    // Valitaan efu
    if(var_timer < GITS_START)
    {
      int freezetime = var_timer - CIRCLES_START;
      // Piirrä renkaat
      efu_circles(freezetime);
      // Jos lisäehto täyttyy, piirrä myös valopilari
      if(var_timer > PILLAR_START)
      {
	float pos[3] = { 0, 0, 0 };
	efu_light_pillar(&pos[0], PILLAR_START, PILLAR_END, freezetime,
	    &pillar_wings, &shape_pillar_wings);
      }
      if((var_timer > WINGS_START) && (var_timer <= WINGS_END))
      {
	efu_wings(var_timer - WINGS_START);
      }
    }
    else if(var_timer < FOLD_START)
      efu_gits(var_timer - GITS_START);
    // Muutoin ollaan lopussa
    else if(var_timer < TIME_END)
      efu_wings(var_timer - FOLD_START);
    else
      loop = 0;
    // Ruudulle
    SDL_GL_SwapBuffers();
    // Pois jos nappia painettu
    if((SDL_PollEvent(&event)) && (event.type == SDL_KEYDOWN))
      switch(event.key.keysym.sym)
      {
	case SDLK_ESCAPE:
	  loop = 0;
	  break;
	default:
	  var_timer += 300;
	  break;
      }
  }

  // Pois
  SDL_Quit();
  return 0;
}

//############################################################################
// Loppu #####################################################################
//############################################################################
