#include <stdio.h>
#include <stdlib.h>

#include "syn_types.h"

#include "connect.h"
#include "sequencer.h"
#include "constant.h"
#include "oscillator.h"
#include "envelope.h"
#include "filter.h"
#include "filter2.h"
#include "mixer.h"
#include "delay.h"
#include "multiplier.h"
#include "adder.h"
#include "overdrive.h"

#include "song.h"

void process()
{
    int i;

    for (i=0; i<n_sequencers; i++){
	process_sequencer(sequencers[i]);
    }
    for (i=0; i<n_envelopes; i++){
	process_envelope(envelopes[i]);
    }
    for (i=0; i<n_oscillators; i++){
	process_oscillator(oscillators[i]);
    }
    for (i=0; i<n_filters; i++){
	process_filter(filters[i]);
    }
    for (i=0; i<n_mixers; i++){
	process_mixer(mixers[i]);
    }
    for (i=0; i<n_multipliers; i++){
	process_multiplier(multipliers[i]);
    }
    for (i=0; i<n_adders; i++){
	process_adder(adders[i]);
    }
    for (i=0; i<n_delays; i++){
	process_delay(delays[i]);
    }    
    for (i=0; i<n_overdrives; i++){
	process_overdrive(overdrives[i]);
    }    
}

// bassokone

void bassmachine(Mixer *mix, int port)
{
    Sequencer *seq1 = new_sequencer(1, 0, 1, basspatterns, bassorder);

    Oscillator *osc1 = new_oscillator(SAW, 1.0f, 0.5f);
    Oscillator *osc2 = new_oscillator(SAW, 2.005f, 1.0f);
    Filter *fil1 = new_filter(LP, 0.12f, 0.30f);
    Envelope *env1 = new_envelope(0.01f, 0.5f, 0.1f, 0.5f);
    Envelope *env2 = new_envelope(0.02f, 1.0f, 0.0f, 0.1f);
    Adder *add1 = new_adder(2);

    // oscillators & envelopes
    conn(seq1, 0, osc1, 0);
    conn(env2, 0, osc1, 1);
    conn(seq1, 0, osc2, 0);
    conn(env2, 0, osc2, 1);

    conn(seq1, 1, env1, 0);
    constant(1.0f, env1, 1);
    conn(seq1, 1, env2, 0);
    constant(1.0f, env2, 1);

    // add together & feed to filter
    conn(osc1, 0, add1, 0);
    conn(osc2, 0, add1, 1);
    conn(add1, 0, fil1, 0);
    conn(env1, 0, fil1, 1);
    constant(1.0f, fil1, 2);

    // to mix
    conn(fil1, 0, mix, port);
}

void stringmachine(Mixer *mix, int port) // uses 4 output ports!
{
    int i;

    Sequencer *seq1 = new_sequencer(4, 1, 8, stringpatterns, stringorder);

    for (i=0; i<4; i++){

	Oscillator *osc1 = new_oscillator(SAW, 1.0f, 1.0f);
	Oscillator *osc2 = new_oscillator(SAW, 1.01f, 1.0f);
	Envelope *env1 = new_envelope(0.9f, 0.1f, 1.0f, 1.2f);
	Envelope *env2 = new_envelope(1.0f, 0.8f, 0.8, 1.8f);
	Filter *fil1 = new_filter(HP, 0.08f, 0.60f);
	Filter *fil2 = new_filter(LP, 0.14f, 0.60f);
	Adder *add1 = new_adder(2);
	Delay *del1 = new_delay(1.0f, TRUE);
	Adder *add2 = new_adder(2);
	Adder *add3 = new_adder(2);
	Oscillator *lfo = new_oscillator(TRIANGLE, 1.0f + i*0.02f, 1.0f);
	
	//constant(1.0f, env1, ENV_IN_AMP);
	conn(seq1, 8, env1, ENV_IN_AMP);
	conn(seq1, i*2 + 1, env1, ENV_IN_TRIG);
	constant(1.0f, env2, ENV_IN_AMP);
	conn(seq1, i*2 + 1, env2, ENV_IN_TRIG);
	
	// oscillators
	conn(seq1, i*2, osc1, OSC_MOD_FREQ);
	conn(env1, ENV_OUT_SIGNAL, osc1, OSC_MOD_AMP);
	conn(osc1, OSC_OUT_SIGNAL, add1, 0);
	
	conn(seq1, i*2, osc2, OSC_MOD_FREQ);
	conn(env1, ENV_OUT_SIGNAL, osc2, OSC_MOD_AMP);
	conn(osc2, OSC_OUT_SIGNAL, add1, 1);
	
	// filters
	conn(add1, OSC_OUT_SIGNAL, fil1, 0);
	constant(1.0f, fil1, 1);
	conn(env2, ENV_OUT_SIGNAL, fil2, 1);
	conn(fil1, 0, fil2, 0);
	
	// flanger
	conn(fil2, 0, del1, 0);
	constant(0.2f, lfo, OSC_MOD_FREQ);
	constant(0.010f, lfo, OSC_MOD_AMP);
	
	conn(lfo, 0, add2, 0);
	constant(0.015f, add2, 1);
	
	conn(add2, 0, del1, 1);
	
	// mix dry & wet
	conn(fil2, 0, add3, 0);
	conn(del1, 0, add3, 1);
	
	conn(add3, 0, mix, port + i*3);
    }    
}

void dingmachine(Mixer *mix, int port) // uses 3 output ports
{
    int i;
    Sequencer *seq1 = new_sequencer(3, 0, 4, dingpatterns, dingorder);

    for (i=0; i<3; i++){
	Oscillator *osc1 = new_oscillator(SAW, 1.0f, 1.0f);
	Oscillator *osc2 = new_oscillator(SQUARE, 2.01f, 0.5f);
	Envelope *env1 = new_envelope(0.01f, 0.4f, 0.2f, 1.0f); // osc
	Envelope *env2 = new_envelope(0.05f, 1.5f, 0.3f, 1.0f); // fil
	Envelope *env3 = new_envelope(0.03f, 0.6f, 0.3f, 1.0f); // fil2
	Filter *fil1 = new_filter(HP, 0.35f, 0.3f);
	Filter *fil2 = new_filter(LP, 0.25f, 1.6f);
	Adder *add1 = new_adder(2);
	
	conn(seq1, 0 + i*2, osc1, OSC_MOD_FREQ);
	conn(seq1, 0 + i*2, osc2, OSC_MOD_FREQ);
	
	conn(osc1, OSC_OUT_SIGNAL, add1, 0);
	conn(osc2, OSC_OUT_SIGNAL, add1, 1);
	
	conn(seq1, 1 + i*2, env1, ENV_IN_TRIG);
	conn(seq1, 1 + i*2, env2, ENV_IN_TRIG);
	conn(seq1, 1 + i*2, env3, ENV_IN_TRIG);
	constant(1.0f, env1, ENV_IN_AMP);
	constant(1.0f, env2, ENV_IN_AMP);
	constant(1.0f, env3, ENV_IN_AMP);
	
	conn(env1, 0, osc1, OSC_MOD_AMP);
	conn(env1, 0, osc2, OSC_MOD_AMP);
	conn(env2, ENV_OUT_SIGNAL, fil1, FILTER_MOD_CUTOFF);
	conn(env3, ENV_OUT_SIGNAL, fil2, FILTER_MOD_CUTOFF);
	conn(add1, 0, fil1, FILTER_IN_SIGNAL);
	
	constant(1.0f, fil2, FILTER_MOD_CUTOFF);
	conn(fil1, FILTER_OUT_SIGNAL, fil2, FILTER_IN_SIGNAL);
	
	conn(fil2, FILTER_OUT_SIGNAL, mix, port + i*3);
    }
}

void drummachine(Mixer *mix, int port)
{
    Sequencer *seq1 = new_sequencer(4, 0, 2, drumpatterns, drumorder);

    // bd
    Oscillator *osc1 = new_oscillator(TRIANGLE, 200.0f, 1.0f);
    Envelope *env1 = new_envelope(0.001f, 0.15f, 0.0f, 0.1f);
//    Envelope *env2 = new_envelope(0.001f, 0.03f, 0.2f, 0.3f);
    
    // sd
    Oscillator *osc2 = new_oscillator(NOISE, 1.0f, 1.0f);
    Filter *fil1 = new_filter(BP, 0.30f, 0.30f);
    Envelope *env3 = new_envelope(0.001f, 0.8f, 0.0f, 1.0f);
    Envelope *env4 = new_envelope(0.001f, 0.2f, 0.5f, 1.0f);

    //hh1
    Oscillator *osc3 = new_oscillator(NOISE, 1.0f, 1.0f);
    Filter *fil2 = new_filter(HP, 0.60f, 0.20f);
    Envelope *env5 = new_envelope(0.001f, 0.05f, 0.0f, 0.1f);

    //hh2
    Oscillator *osc4 = new_oscillator(NOISE, 1.0f, 1.0f);
    Filter *fil3 = new_filter(HP, 0.60f, 0.10f);
    Envelope *env6 = new_envelope(0.001f, 0.10f, 0.05f, 0.5f);

    // bd
    conn(seq1, 1, env1, ENV_IN_TRIG);
    constant(1.0f, env1, ENV_IN_AMP);
    conn(env1, ENV_OUT_SIGNAL, osc1, OSC_MOD_AMP);

    constant(1.0f, env1, ENV_IN_AMP);
    conn(env1, 0, osc1, OSC_MOD_FREQ);

    conn(osc1, 0, mix, port);

    // sd
    conn(seq1, 3, env3, ENV_IN_TRIG);
    conn(seq1, 3, env4, ENV_IN_TRIG);
    constant(1.0f, env4, ENV_IN_AMP);
    conn(env4, ENV_OUT_SIGNAL, env3, ENV_IN_AMP);

    constant(1.0f, osc2, OSC_MOD_FREQ);
    conn(env3, ENV_OUT_SIGNAL, osc2, OSC_MOD_AMP);

    conn(osc2, OSC_OUT_SIGNAL, fil1, 0);
    constant(1.0f, fil1, FILTER_MOD_CUTOFF);

    conn(fil1, 0, mix, port + 3);

    // hh1
    conn(seq1, 5, env5, ENV_IN_TRIG);
    constant(1.0f, env5, ENV_IN_AMP);

    constant(1.0f, osc3, OSC_MOD_FREQ);
    conn(env5, ENV_OUT_SIGNAL, osc3, OSC_MOD_AMP);

    constant(1.0f, fil2, 1);
    conn(osc3, 0, fil2, 0);
    
    conn(fil2, 0, mix, port + 6);

    // hh 2
    conn(seq1, 7, env6, ENV_IN_TRIG);
    constant(1.0f, env6, ENV_IN_AMP);

    constant(1.0f, osc4, OSC_MOD_FREQ);
    conn(env6, ENV_OUT_SIGNAL, osc4, OSC_MOD_AMP);

    constant(1.0f, fil3, 1);
    conn(osc4, 0, fil3, 0);
    
    conn(fil3, 0, mix, port + 9);

}

void resomachine(Mixer *mix, int port)
{
    Sequencer *seq1 = new_sequencer(1, 1, 2, resopatterns, resoorder);

    Oscillator *osc1 = new_oscillator(SAW, 1.0f, 1.0f);
    Envelope *env1 = new_envelope(0.01f, 0.01f, 1.0f, 0.01f);
    Filter *fil1 = new_filter(LP, 0.30f, 0.07f);
    Delay *del1 = new_delay(1.0f, FALSE);

    conn(seq1, 1, env1, ENV_IN_TRIG);
    constant(1.0f, env1, ENV_IN_AMP);

    conn(env1, 0, osc1, OSC_MOD_AMP);
    conn(seq1, 0, osc1, OSC_MOD_FREQ);

    conn(osc1, 0, fil1, 0);
    conn(seq1, 2, fil1, 1);

    conn(fil1, 0, del1, 0);
    constant(0.2f, del1, 1);

    conn(fil1, 0, mix, port);
    conn(del1, 0, mix, port + 3);
}

int main(void)
{
    FILE *f;
    int i;

//////////////////////////////////////////////////////////////////////

    Mixer *master_mix = new_mixer(16);

    Mixer *string_bus = new_mixer(4);
    Delay *string_del1 = new_delay(1.0f, FALSE);
    Delay *string_del2 = new_delay(1.0f, FALSE);

    bassmachine(master_mix, 0);
    constant(0.17f, master_mix, 1);
    constant(0.0f, master_mix, 2);

    resomachine(master_mix, 3);
    constant(0.08f, master_mix, 4);
    constant(0.8f, master_mix, 5);

    // 6
    constant(0.03f, master_mix, 7);
    constant(-1.0f, master_mix, 8);


    dingmachine(master_mix, 9);
    constant(0.25f, master_mix, 10);
    constant(-0.4f, master_mix, 11);
    constant(0.25f, master_mix, 13);
    constant(0.0f, master_mix, 14);
    constant(0.25f, master_mix, 16);
    constant(-0.2f, master_mix, 17);

    stringmachine(string_bus, 0);
    constant(0.1f, string_bus, 1);
    constant(-0.5f, string_bus, 2); 
    constant(0.13f, string_bus, 4);
    constant(0.4f, string_bus, 5);
    constant(0.1f, string_bus, 7);
    constant(-0.4f, string_bus, 8);
    constant(0.13f, string_bus, 10);
    constant(0.5f, string_bus, 11);

    // to master mix with stereo delay
    conn(string_bus, 0, string_del1, 0);
    conn(string_bus, 1, string_del2, 0);
    constant(0.3f, string_del1, DELAY_IN_LEN);
    constant(0.5f, string_del1, DELAY_IN_LEN);

    conn(string_bus, 0, master_mix, 18);
    constant(0.4f, master_mix, 19);
    constant(-1.0f, master_mix, 20);
    conn(string_bus, 0, master_mix, 21);
    constant(0.4f, master_mix, 22);
    constant(1.0f, master_mix, 23);
    conn(string_del1, 0, master_mix, 24);
    constant(0.15f, master_mix, 25);
    constant(1.0f, master_mix, 26);
    conn(string_del2, 0, master_mix, 27);
    constant(0.15f, master_mix, 28);
    constant(-1.0f, master_mix, 29);

    drummachine(master_mix, 30); // 4 ports
    constant(0.8f, master_mix, 31);
    constant(0.0f, master_mix, 32);
    constant(0.35f, master_mix, 34);
    constant(0.0f, master_mix, 35);
    constant(0.04f, master_mix, 37);
    constant(-0.7f, master_mix, 38);
    constant(0.04f, master_mix, 40);
    constant(-0.7f, master_mix, 41);



//////////////////////////////////////////////////////////////////////

    fprintf(stderr, "n_sequencers: %d\n", n_sequencers);
    fprintf(stderr, "n_envelopes: %d\n", n_envelopes);
    fprintf(stderr, "n_oscillators: %d\n", n_oscillators);
    fprintf(stderr, "n_filters: %d\n", n_filters);
    fprintf(stderr, "n_mixers: %d\n", n_mixers);
    fprintf(stderr, "n_multipliers: %d\n", n_multipliers);
    fprintf(stderr, "n_adders: %d\n", n_adders);
    fprintf(stderr, "n_delays: %d\n", n_delays);
    fprintf(stderr, "n_overdrives: %d\n", n_overdrives);

    f = fopen("foo", "wb");
    if (f == NULL){
	exit(1);
    }

    for (i=0; i<SRATE * 250; i++){
	process();
	// LSB 16b signed stereo vittu sovitaan että menee vaikka näin
	float fl = OUT(master_mix, 0);
	float fr = OUT(master_mix, 1);

	if (fl > 1.5f) fl = 1.5f; else if (fl < -1.5f) fl = -1.5f;
	if (fr > 1.5f) fr = 1.5f; else if (fr < -1.5f) fr = -1.5f;

 	signed short l = fl * 20000;
 	signed short r = fr * 20000;
/* 	unsigned char l_l = l % 255; */
/* 	unsigned char l_h = l >> 8; */
/* 	unsigned char r_l = r % 255; */
/* 	unsigned char r_h = r >> 8; */

/* 	fwrite(&l_l, 1, 1, f); */
/* 	fwrite(&l_h, 1, 1, f); */
/* 	fwrite(&r_l, 1, 1, f); */
/* 	fwrite(&r_h, 1, 1, f); */

	fwrite(&l, 1, 2, f);
	fwrite(&r, 1, 2, f);
    }

    return 0;
}
