#include "_types.hpp"
#include "_start.hpp"
#include "fg.hpp"

/****************************************************************************/

LandscapeTexture::LandscapeTexture()
{
  Tex = sSystem->AddTexture(xRes,yRes,sTF_DYNAMIC,0,1);
  heightMap = 0;
  CalcHeightmap();

  gridColor = new sInt[(gridX+1)*(gridY+1)];
  buffer = new sU32[xRes * yRes];
  sSetMem(buffer,0,xRes * yRes * 4);

  Angle = 0.0f;
  Camera.Init();

  frame = 0;
}

LandscapeTexture::~LandscapeTexture()
{
  sSystem->RemTexture(Tex);

  delete[] gridColor;
  delete[] heightMap;
  delete[] buffer;
}

sInt LandscapeTexture::Trace(const sVector &pos,const sVector &dir)
{
  static const sF32 heightScale = 32.0f;
  sF32 enterT = 0.0f, leaveT = 0.0f;

  if (sFAbs(dir.y) < 1e-6f)
  {
    if (pos.y >= 0.0f && pos.y < heightScale)
      leaveT = 256.0f;
  }
  else
  {
    const sF32 idY = 1.0f / dir.y;

    enterT = (0.0f - pos.y) * idY;
    leaveT = (heightScale - pos.y) * idY;

    if (enterT > leaveT)
      sSwap(enterT, leaveT);
  }

  if (enterT < leaveT && leaveT > 0.0f)
  {
    sF32 dsv = sFAbs(dir.x);
    if (sFAbs(dir.z) > dsv)
      dsv = sFAbs(dir.z);

    dsv = 65536.0f / dsv;

    const sInt dsx = dir.x * dsv;
    const sInt dsy = dir.y * dsv * 256 / heightScale;
    const sInt dsz = dir.z * dsv;

    sInt steps = (leaveT - enterT) * dsv + 0.999f;
    if (steps > 256) 
      steps = 256;

    sInt posx = (pos.x + enterT * dir.x) * 65536.0f;
    sInt posy = (pos.y + enterT * dir.y) * 65536.0f * 256 / heightScale;
    sInt posz = (pos.z + enterT * dir.z) * 65536.0f;

    sBool origSign;

    for (sInt i = 0; i < steps; i++)
    {
      sU32 heightXo = (posx & 0xffffff) >> 16;
      sU32 heightYo = ((posz & 0xffffff) >> 16) << 8;
      sU32 height1 = heightMap[heightXo + heightYo] << 16;
      
      sBool sign = height1 < posy;
      if (i == 0) // first step
        origSign = sign;
      else if (sign != origSign) // we passed the intersection!
      {
        sS32 col = height1;
        if (col < 0)
          col = 0;
        
        if (col > 0xffffff)
          col = 0xffffff;

        return col;
      }

      posx += dsx;
      posy += dsy;
      posz += dsz;
    }
  }

  return 0;
}

void LandscapeTexture::CalcGrid()
{
  sVector dir;
  sInt *grid = gridColor;

  sa = sFSin(Angle);
  ca = sFCos(Angle);

  for (sInt y = 0; y <= gridY; y++)
  {
    for (sInt x = 0; x <= gridX; x++)
    {
      sInt xo = sInt(x - gridX / 2);
      sInt yo = sInt(gridY / 2 - y);
      dir.Init((xo * ca - 128.0f * sa) * 640.0f / 512.0f, yo * 640.0f / 512.0f, xo * sa + 128.0f * ca,0.0f);

      *grid++ = Trace(Camera,dir);
    }
  }
}

void LandscapeTexture::TraceBlock(sU32 *dest,sInt xg,sInt yg,sInt t)
{
  sVector dir;

  dest += t & 3;
  dest += ((t/7) & 3) * xRes;

  for (sInt yb = 0; yb < gridStep; yb+=4)
  {
    for (sInt xb = 0; xb < gridStep; xb+=4)
    {
      sInt xo = sInt(xg + xb - xRes / 2);
      sInt yo = sInt(yRes / 2 - yg - yb);
      dir.Init((xo * ca - 128.0f * sa) * 640.0f / 512.0f, yo * 640.0f / 512.0f, xo * sa + 128.0f * ca,0.0f);

      dest[xb] = (Trace(Camera, dir) >> 16) * 0x10101;
    }

    dest += xRes * 4;
  }
}

void LandscapeTexture::DrawGrid(sInt t)
{
  const sInt *gridPtr = gridColor;
  const sInt tol = 1 << 12;

  for (sInt yg = 0; yg < yRes; yg += gridStep)
  {
    for (sInt xg = 0; xg < xRes; xg += gridStep)
    {
      // calc deltas
      sInt dLeft = (gridPtr[gridX + 1] - gridPtr[0]) / gridStep;
      sInt dRght = (gridPtr[gridX + 2] - gridPtr[1]) / gridStep;
      sU32 *dest = buffer + yg * xRes + xg;

      if (dLeft <= -tol || dLeft >= tol || dRght <= -tol || dRght >= tol)
        TraceBlock(dest, xg, yg, t+5*xg+47*yg);
      else
      {
        sInt vLeft = gridPtr[0];
        sInt vRght = gridPtr[1];

        // draw block
        for (sInt yb = 0; yb < gridStep; yb++)
        {
          sInt vX = vLeft;
          sInt dX = (vRght - vLeft) / gridStep;

          for (sInt xb = 0; xb < gridStep; xb++)
          {
            dest[xb] = (vX >> 16) * 0x10101;
            vX += dX;
          }

          vLeft += dLeft;
          vRght += dRght;
          dest += xRes;
        }
      }

      gridPtr++;
    }

    gridPtr++;
  }
}

void LandscapeTexture::Render()
{
  sBitmapLock lock;
  sInt y;

  CalcGrid();
  DrawGrid(frame++);

  if(sSystem->StreamTextureStart(Tex,0,lock))
  {
    for(y=0;y<yRes;y++)
      sCopyMem(lock.Data + y * lock.BPR,buffer + y * xRes,xRes * 4);
    sSystem->StreamTextureEnd();
  }
}

void LandscapeTexture::CalcHeightmap()
{
  int p,i,j,k,k2,p2;

  heightMap = new sU8[65536];

	// subd plasma
  heightMap[0]=128;
  for ( p=256; p>1; p=p2 )
  {
    p2=p>>1;
    k=p*8+20; k2=k>>1;
    for ( i=0; i<256; i+=p )
    {
      for ( j=0; j<256; j+=p )
      {
        int a,b,c,d;

        a=heightMap[(i<<8)+j];
        b=heightMap[(((i+p)&255)<<8)+j];
        c=heightMap[(i<<8)+((j+p)&255)];
        d=heightMap[(((i+p)&255)<<8)+((j+p)&255)];

        heightMap[(i<<8)+((j+p2)&255)]=sRange<sInt>(((a+c)>>1)+(sGetRnd(k)-k2),255,0);
        heightMap[(((i+p2)&255)<<8)+((j+p2)&255)]=sRange<sInt>(((a+b+c+d)>>2)+(sGetRnd(k)-k2),255,0);
        heightMap[(((i+p2)&255)<<8)+j]=sRange<sInt>(((a+b)>>1)+(sGetRnd(k)-k2),255,0);
      }
    }
  }

	// blaarrr
  for ( k=0; k<3; k++ )
  {
    for ( i=0; i<256*256; i+=256 )
    {
      for ( j=0; j<256; j++ )
      {
        heightMap[i+j]=(heightMap[((i+256)&0xFF00)+j]+heightMap[i+((j+1)&0xFF)]+heightMap[((i-256)&0xFF00)+j]+heightMap[i+((j-1)&0xFF)])>>2;
      }
    }
  }
}
