#include <stdio.h>
#include <stdlib.h>

#define SAMPSPERTICK 16384

int mixbuf[SAMPSPERTICK];

int done=0;

/****************************************************************************

                              SYSTEM-LEVEL STUFF

*****************************************************************************/

#ifdef X86LINUX

/*** some syscall macros so we won't need libc ***/

static inline void _exit(int ret)
{
   __asm__ volatile("int $0x80" : : "a"(1),"b"(ret));
}

static inline int write(int fd,const char*buf,int count)
{
   int ret;
   __asm__ volatile("int $0x80" : "=a"(ret) :
           "a"(4),"b"(fd),"c"(buf),"d"(count));
   return ret;
}

static inline int open(const char*fn,int flag,int mode)
{
   int ret;
   __asm__ volatile("int $0x80" : "=a"(ret) :
           "a"(5),"b"(fn),"c"(flag),"d"(mode));
   return ret;
}

static inline void close(int fd)
{
   __asm__ volatile("int $0x80" : : "a"(6),"b"(fd));
}

static inline void unlink(const char*name)
{
   __asm__ volatile("int $0x80" : : "a"(10),"b"(name));
}

static inline int ioctl(int fd,int req,void*arg)
{
   int ret;
   __asm__ volatile("int $0x80" : "=a"(ret) :
           "a"(54),"b"(fd),"c"(req),"d"(arg) : "memory");
   return ret;
}

#define O_RDONLY 0
#define O_WRONLY 1
#define O_RDWR   2

#endif

/*** if we're building on another unix, we need the standard posix stuff ***/

#if (defined(UNIX) && !defined(X86LINUX))

#  include <unistd.h>
#  include <fcntl.h>

# ifdef OSSDSP
#    include <sys/ioctl.h>
# endif

#endif


/****************************************************************************

                               SOUND OUTPUT DRIVER

*****************************************************************************/

/******************** Variables for sound output *************************/

static struct
{
#ifdef UNIX
   int outfile;
#else
   FILE*outfile;
#endif

   int outbufsamps,outbufbytes;

   int sd,dd0,dd1,wavebase;

}  snddrv;

/********* Sound driver: OSS-like /dev/dsp, stdout fallback *************/

#ifdef OSSDSP

#include <sys/soundcard.h>

#if (0)
void debugwrite(int v)
{
   char buf[9]; int i;
   buf[8]='\n';
   for(i=7;i>=0;i--)
   {
      buf[i]="0123456789ABCDEF"[v&15]; v>>=4;
   }
   
   write(2,buf,9);
}
#endif

void initwave()
{
   int v,res;

   /* default to U16_LE */
   snddrv.sd=snddrv.dd0=snddrv.dd1=1;

   snddrv.wavebase=32768;

   snddrv.outfile=open("/dev/dsp",O_WRONLY,0);

   if(snddrv.outfile<0) snddrv.outfile=1; else
   {
    ioctl(snddrv.outfile,SNDCTL_DSP_GETFMTS,&v);
    if(v&(AFMT_U16_LE|AFMT_U16_BE|AFMT_S16_LE|AFMT_S16_BE))
    {
       if(v&(AFMT_U16_LE|AFMT_S16_LE))
       {
          if(v&AFMT_U16_LE) v=AFMT_U16_LE; else v=AFMT_S16_LE;
       }
        else
       {
          snddrv.dd0=-1; snddrv.dd1=3;
 
          if(v&AFMT_U16_BE) v=AFMT_U16_BE; else v=AFMT_S16_BE; 
       }
    }
    else
    {
       snddrv.dd0=0;
    
       if(v&AFMT_U8) v=AFMT_U8; else v=AFMT_S8;
    }
   
    ioctl(snddrv.outfile,SNDCTL_DSP_SETFMT,&v);

    if(v&(AFMT_S8|AFMT_S16_LE|AFMT_S16_BE)) snddrv.wavebase=0;

    v=1;
    res=ioctl(snddrv.outfile,SNDCTL_DSP_STEREO,&v);
    if(!v) snddrv.sd<<=1;

    v=44100;
    res=ioctl(snddrv.outfile,SNDCTL_DSP_SPEED,&v);
    if(v<40000)
    {
       v=22050;
       res=ioctl(snddrv.outfile,SNDCTL_DSP_SPEED,&v);
       snddrv.sd *= (int)(48000/v);
    }
   }

   snddrv.outbufsamps=SAMPSPERTICK/snddrv.sd;
   snddrv.outbufbytes=snddrv.outbufsamps*(snddrv.dd0+snddrv.dd1);
}   

#endif

/*********************** Sound driver: stdout ****************************/

#ifndef OSSDSP

void initwave()
{
#ifdef UNIX
   snddrv.outfile=1;
#else
   snddrv.outfile=stdout;
#endif

   snddrv.outbufsamps=SAMPSPERTICK;
   snddrv.outbufbytes=SAMPSPERTICK*2;
   snddrv.sd=snddrv.dd0=snddrv.dd1=1;
   snddrv.wavebase=0;
}

#endif

/****************** Sound clipping + conversion + output ***************/

void sendwave()
{
   unsigned char outbuf[SAMPSPERTICK*4],*d=outbuf;
   int*s=mixbuf;
   
   int i=snddrv.outbufsamps;

   if(snddrv.dd0<0) d++;

   for(;i;i--)
   {
      signed int a=*s;
      s+=snddrv.sd;
      a>>=8;
      if(a<-32768)a=-32768;else
      if(a> 32767)a= 32767;
      a+=snddrv.wavebase;

      *d=a&255; d+=snddrv.dd0;
      *d=a>>8;  d+=snddrv.dd1;
   }

#ifdef UNIX
   write(snddrv.outfile,outbuf,snddrv.outbufbytes);
#else
   fwrite(outbuf,1,snddrv.outbufbytes,snddrv.outfile);
#endif
}


/************************************************************************

                           PRECALC SOUNDS

*************************************************************************/

int wavetable[512*1024];

int scalefreqs[128];

#define BASS0_PTR (wavetable)
#define BASS0_LGT (0x10000/SAMPSPERTICK)

#define STORM_PTR (BASS0_PTR+65536)
#define STORM_LGT (0x40000/SAMPSPERTICK)

#define CRASH_PTR (STORM_PTR+0x40000)
#define CRASH_LGT (0x4000/SAMPSPERTICK)

#define WNOISE_PTR (CRASH_PTR+0x8000)
#define WNOISE_LGT (0x4000/SAMPSPERTICK)

int myrand()
{
   static long int seed=0;

   seed *= 13;
   seed ^= 0x55555555;

   return seed;
}

void gen_additive(int*buf,int lgt,
                  int f0base,int f0range,
                  int f1base,int f1range)
{
   long int p=0,i,j;

   for(j=128;j;j--)
   {
      int dp,ddp,*d;
      int dp1;
      
      dp  = f0base + (myrand()%f0range);
      dp1 = f1base + (myrand()%f1range);
      ddp = (dp1-dp)/lgt;

      for(d=buf,i=lgt;i;i--)
      {
#ifdef UNIX
         *d+++=p>>16;
#else
         *d+++=((p>>16)&65535)-32768;
#endif
         p+=dp;
         dp+=ddp;
      }
   }
}

void gen_volumeslide(int*buf,int lgt,int vol,int dvol)
{
   for(;lgt;lgt--)
   {
      *buf=(*buf>>8)*(vol>>16);
      vol+=dvol;
      buf++;
   }
}

void synth_precalc()
{
   int i;

   /*** clear buffer ***/

   gen_volumeslide(wavetable,512*1024,0,0);

   /*** CRASH ***/

   for(i=64;i;i--)
   {
      gen_additive(CRASH_PTR+(myrand()%((myrand()&0x3fff)+256)),
                   0x3ff+(myrand()&0x7ff),
                   myrand()&0x07ffffff,127+(myrand()&0x001fffff),
                   myrand()&0x007fffff,127+(myrand()&0x0001ffff));
   }

   gen_volumeslide(CRASH_PTR,0x5000,0x800000,0);

   /*** BASSDRUM ***/

   gen_additive(BASS0_PTR,16384*4,
   		0x00600000/4,0x00001000/4,
   		0x00300000/4,0x00010000/4 );

   gen_volumeslide(BASS0_PTR,16384*4,256*65536,-1024/4);

   gen_additive(BASS0_PTR,2048*4,
   		0x00000000,0x80000000/4,
   		0x00000000,0x20000000/4 );

   gen_additive(BASS0_PTR,4096*4,
                0x02000000/4,0x00600000/4,
                0x00300000/4,0x00000100/4 );

   /*** STORM ***/

   gen_additive(STORM_PTR,0x40000,
                0x00800000,0x03000000,
                0x01000000,0x01000000 );

   gen_volumeslide(STORM_PTR,0x10000,0,256);

   gen_volumeslide(STORM_PTR+0x10000,0x30000,256*0x10000,-96);


   /*** WNOISE ***/
   
   for(i=8;i;i--)
   gen_additive(WNOISE_PTR,0x4000,
                0x00000000,0x40000000,
                0x00000000,0x40000000 );

   /*** calculate frequency table ***/
   
   {double f=256;int i=128;int*d=scalefreqs;

    for(;i;i--)
    {
       *d++=(int)f;
       f*=1.0594630943593;
    }
   }
}

/************************************************************************

                            PLAYER PARMS

*************************************************************************/

int*wtchans_ptr0[3]           = { BASS0_PTR,CRASH_PTR,STORM_PTR };
unsigned char wtchans_ctr0[3] = { BASS0_LGT,CRASH_LGT,STORM_LGT };

int*wtchans_ptr[3];
unsigned char wtchans_ctr[3] = { 0,0,0 };

int strchans_buf[3][16384];
int strchans_idx[3];
int strchans_wvlgt[3]; /* + deltas? */
unsigned char strchans_ctr[3] = { 0,0,0 };

/************************** sequencer data *****************************/

#define O_0    -24
#define O_1    -12
#define O_2      0
#define O_3     12
#define O_4     24
#define O_5     36 

#define O_BASE  36

#define N_C  (0+O_BASE)
#define N_Cp (1+O_BASE)
#define N_D  (2+O_BASE)
#define N_Dp (3+O_BASE)
#define N_E  (4+O_BASE)
#define N_F  (5+O_BASE)
#define N_Fp (6+O_BASE)
#define N_G  (7+O_BASE)
#define N_Gp (8+O_BASE)
#define N_A  (9+O_BASE)
#define N_Ap (10+O_BASE)
#define N_B  (11+O_BASE)

#define F_KEY 64

/* drumtrack encoding: 1=BASS 2=CRASH 4=STORM, negative=jump */

signed char nulltrack[] =
{
   0x00,-1
};

signed char drumtrack[] =
{
   0x05, 0x00, 0x00, 0x00,
   0x01, 0x00, 0x00, 0x00,
   0x01, 0x00, 0x00, 0x00,
   0x01, 0x00, 0x00, 0x00,

   0x05, 0x00, 0x02, 0x00,
   0x01, 0x00, 0x02, 0x00,
   0x01, 0x00, 0x02, 0x00,
   0x01, 0x00, 0x02, 0x02,

   0x07, 0x00, 0x02, 0x00,
   0x03, 0x00, 0x02, 0x00,
   0x03, 0x00, 0x00, 0x00,
   0x04, 0x00, 0x00, 0x01,

   0x07, 0x02, 0x02, 0x02,
   0x03, 0x02, 0x02, 0x02,
   0x07, 0x02, 0x02, 0x02,
   0x03, 0x02, 0x02, 0x03,

   -16
};

signed char chordtrack[] =
{
   N_C,            N_C,
   N_C,            N_C,
   N_C,            N_C,
   N_C,            N_C,

   N_C,            N_C,
   N_C+F_KEY,      N_C,
   N_C,            N_C,
   N_Dp+F_KEY,     N_D+F_KEY,

   N_C+F_KEY,      N_C,
   N_Dp+F_KEY,     N_C,
   N_D,            N_C,
   N_Dp+F_KEY,     N_D+F_KEY,

   N_C+F_KEY,      N_C,
   N_Ap+O_1+F_KEY, N_Ap,
   N_Ap+O_1,       N_Ap,
   N_C+F_KEY,      N_C,

   N_C+F_KEY,      N_C,
   N_F+F_KEY,      N_F,
   N_F,            N_F,
   N_C+F_KEY,      N_C,
   
   -16,
};

signed char melodytrack[] =
{
   N_C+O_3,  N_D+O_3, N_Dp+O_3, N_F+O_3,
   N_G+O_3,  N_C+O_4, N_G+O_3,  N_Dp+O_3,
   N_Ap+O_3, N_F+O_3, N_Dp+O_3, N_F+O_3,
   N_Ap+O_2, 0,       0,        0,

   N_Ap+O_3, N_F+O_3, N_Dp+O_3, N_F+O_3,
   N_Ap+O_2, N_F+O_3, N_Ap+O_3, N_F+O_3,
   N_C+O_4,  0,       N_G+O_3,  N_C+O_4,
   N_C+O_3,  0,       0,        0,
   
   N_C+O_3,  N_G+O_3, N_C+O_4,  N_G+O_3,
   N_C+O_3,  N_G+O_3, N_C+O_4,  N_Dp+O_4,
   N_D+O_4,  N_Ap+O_3,N_F+O_3,  N_Ap+O_3,
   N_F+O_3,  N_D+O_3, N_Ap+O_2, N_D+O_3,
   
   N_D+O_4,  N_Ap+O_3,N_F+O_3,  N_D+O_3,
   N_D+O_4,  N_Ap+O_3,N_F+O_3,  N_D+O_3,
   N_C+O_4,  0,       N_G+O_3,  N_C+O_4,
   N_C+O_3,  0,       0,        0,
   
   -64
};

signed char track_tempo[3] = { 1,         3,          1 };
signed char*track_ptr[3]   = { drumtrack, chordtrack, nulltrack };
signed char track_cmd[3];

/************************************************************************

                               SYNTHESIS
                               
*************************************************************************/

void stringpluck(int ch,int note,int dur,int vol)
{
   int i;

   for(i=0;i<16384;i++)
      strchans_buf[ch][i]=(vol*(*(WNOISE_PTR+i)))>>2;

   strchans_idx[ch]=0; 
   strchans_wvlgt[ch]=(2048*1024)/scalefreqs[note];
   strchans_ctr[ch]=dur;
}

int tc=0;

void synth_fill()
{
  static int tcc=1,adder=0;
  int i,j,*d;

    /* sequence the tick */

   if(tc&1) { tcc++; tc+=tcc&1; } 

   switch(tc)
   {
   case(3*32): track_ptr[2]=melodytrack; break;
   case(7*32): adder=5; break;
   case(11*32):adder=0; track_ptr[2]=nulltrack;
   	       track_ptr[0]=drumtrack+32;
   	       track_ptr[1]=chordtrack+16; break;
   case(12*32):track_ptr[2]=melodytrack; adder=-2; break;
   case(16*32):adder=3; break;
   case(18*32):track_ptr[0]=track_ptr[1]=nulltrack;  break;
   case(20*32-2): track_ptr[2]=nulltrack; break;
   case(20*32+8):done++; break;
   }

   for(i=0;i<3;i++)
   if(!(tc&track_tempo[i]))
   {
      int a;
   
      fetch:
        a=*(track_ptr[i]);
        if(a<0) { track_ptr[i]+=a; goto fetch; }
      track_ptr[i]++;

      track_cmd[i]=a;
   } else track_cmd[i]=-1;

   if(track_cmd[0]>=0)
   {
      int a=track_cmd[0];

      for(i=0;a;a>>=1,i++)
      if(a&1) { wtchans_ptr[i]=wtchans_ptr0[i];
                wtchans_ctr[i]=wtchans_ctr0[i];
              }
   }
   
   if((track_cmd[1]>=0) && (track_cmd[1]&64))
      stringpluck(0,adder+(track_cmd[1]&63)-12,16,4);

   if(track_cmd[2]>0)
      stringpluck(1+((tc&2)>>1),adder+track_cmd[2],8,6);

   tc++;

  /* echo out the previous tick */

  for(d=mixbuf,i=SAMPSPERTICK;i;i--) *d++>>=1;

  /* add wavetable channels */

  for(j=2;j>=0;j--)
  if(wtchans_ctr[j])
  {
     int*s=wtchans_ptr[j];
     for(d=mixbuf,i=SAMPSPERTICK;i;i--) *d+++=*s++;
     wtchans_ptr[j]=s;
     wtchans_ctr[j]--;
  }

  /* add string channels */

  for(j=2;j>=0;j--)
  if(strchans_ctr[j])
  {
     int idx=strchans_idx[j],a, pidxhi;
     int*b=strchans_buf[j];

     for(d=mixbuf,i=SAMPSPERTICK;i;i--)
     {
        *d+++=b[idx]+b[idx>>1];

        idx++;
        if(idx>=strchans_wvlgt[j])
        {
           int k=idx;
           int*d0=b,*d1=b+k-1;

           for(;k;k--)
           {
              *d0 = (*d0+*d1)>>1;
              d1=d0; d0++;
           }

           idx=0;
        }
     }

     strchans_idx[j]=idx;
     strchans_ctr[j]--;
  }

  /* add vocal channel: NOT ENOUGH TIME! */

  /* some final filtering */
  
  for(d=mixbuf,i=SAMPSPERTICK-1;i;i--)
  {
     *d=(d[0]+d[1]+8)>>1; d++;
  }
}

/************************************************************************

                              MAIN FUNCTION

*************************************************************************/

#define BANNER \
"\033[2J\033[H\n" /*7*/ \
"                      ___________\n" /* 34  39 */ \
"          ____________\\         |\n" /* 34  73 */ \
"          |           \\\\        |\n" /* 34  107*/ \
"          |                     |\n" /* 34  141  */ \
"   _______:_______________ _____:____  __________   _______________________\n" /* 76  217 */ \
"   \\               _     //          /\\         (   \\               _     /\n" /* 76  293 */ \
"   /              /_\\___(/          /  \\         \\  /              /_\\___(\n"  /* 75  368 */ \
"  /__________________\\  /____________/\\___________\\/__________________\\\n"     /* 72  540 */ \
"          :                     :\n" /* 34  574 */ \
"          |       //____________|   Trollenscheisse by Viznut/PWP\n" /* 57  531 */ \
"          |______/                  Softsynth compo, Altparty 2004\n" /* 67  598 */

#define BANNERLGT 610

#ifdef X86LINUX
void _start()
#else
int main()
#endif
{
#ifdef CLEANCRUNCH
  unlink("S");
#endif

#ifdef UNIX
  write(2,BANNER,BANNERLGT);
#else
  fprintf(stderr,BANNER);
#endif

  initwave();

  synth_precalc();

  {int vol=0;
  while(!done)
  {
     synth_fill();
     sendwave();
  }}

#ifdef X86LINUX
  _exit(0);
#else
  return 0;
#endif
}
