#include "types.h"
#include "cubemap.h"
#include <kernel.h>
#include "myassert.h"
#include "gif.h"
#include "dma.h"
#include "math.h"

#define NUMBER_OF_LAYERS 1
//#define NUMBER_OF_LAYERS 2

static int128 gifList[2*65536+6*2];//static to make sure that elf file does not grow in size??

void CubeMap::init(float camDistance, int newCubeTextureSize, int newEnvTextureSize, texture *t0, texture *t1, texture *t2, texture *t3, texture *t4, texture *t5)
{
  this->camDistance=camDistance;
  this->newCubeTextureSize=newCubeTextureSize;
  this->newEnvTextureSize=newEnvTextureSize;
  sourceTextures[0]=t0;
  sourceTextures[1]=t1;
  sourceTextures[2]=t2;
  sourceTextures[3]=t3;
  sourceTextures[4]=t4;
  sourceTextures[5]=t5;

  theMeshes[0].init(4,2);
  theMeshes[0].vlist[0]=fvec_(-1,-1,1);
  theMeshes[0].vlist[1]=fvec_(-1,1,1);
  theMeshes[0].vlist[2]=fvec_(1,-1,1);
  theMeshes[0].vlist[3]=fvec_(1,1,1);
  ((ivec*)theMeshes[0].vlist)[0].w=0;
  ((ivec*)theMeshes[0].vlist)[1].w=0x10000000;
  ((ivec*)theMeshes[0].vlist)[2].w=0x00001000;
  ((ivec*)theMeshes[0].vlist)[3].w=0x10001000;
  theMeshes[0].flist[0]=face(0,1,2,0x80808080);
  theMeshes[0].flist[1]=face(1,3,2,0x80808080);

  theMeshes[1].init(4,2);
  theMeshes[1].vlist[0]=fvec_(-1,1,-1);
  theMeshes[1].vlist[1]=fvec_(-1,-1,-1);
  theMeshes[1].vlist[2]=fvec_(1,1,-1);
  theMeshes[1].vlist[3]=fvec_(1,-1,-1);
  ((ivec*)theMeshes[1].vlist)[0].w=0;
  ((ivec*)theMeshes[1].vlist)[1].w=0x10000000;
  ((ivec*)theMeshes[1].vlist)[2].w=0x00001000;
  ((ivec*)theMeshes[1].vlist)[3].w=0x10001000;
  theMeshes[1].flist[0]=face(0,1,2,0x80808080);
  theMeshes[1].flist[1]=face(1,3,2,0x80808080);
  
  theMeshes[2].init(4,2);
  theMeshes[2].vlist[0]=fvec_(-1,-1,1);
  theMeshes[2].vlist[1]=fvec_(-1,-1,-1);
  theMeshes[2].vlist[2]=fvec_(-1,1,1);
  theMeshes[2].vlist[3]=fvec_(-1,1,-1);
  ((ivec*)theMeshes[2].vlist)[0].w=0;
  ((ivec*)theMeshes[2].vlist)[1].w=0x10000000;
  ((ivec*)theMeshes[2].vlist)[2].w=0x00001000;
  ((ivec*)theMeshes[2].vlist)[3].w=0x10001000;
  theMeshes[2].flist[0]=face(0,1,2,0x80808080);
  theMeshes[2].flist[1]=face(1,3,2,0x80808080);
  
  theMeshes[3].init(4,2);
  theMeshes[3].vlist[0]=fvec_(1,1,1);
  theMeshes[3].vlist[1]=fvec_(1,1,-1);
  theMeshes[3].vlist[2]=fvec_(1,-1,1);
  theMeshes[3].vlist[3]=fvec_(1,-1,-1);
  ((ivec*)theMeshes[3].vlist)[0].w=0;
  ((ivec*)theMeshes[3].vlist)[1].w=0x10000000;
  ((ivec*)theMeshes[3].vlist)[2].w=0x00001000;
  ((ivec*)theMeshes[3].vlist)[3].w=0x10001000;
  theMeshes[3].flist[0]=face(0,1,2,0x80808080);
  theMeshes[3].flist[1]=face(1,3,2,0x80808080);
  
  theMeshes[4].init(4,2);
  theMeshes[4].vlist[0]=fvec_(1,-1,1);
  theMeshes[4].vlist[1]=fvec_(1,-1,-1);
  theMeshes[4].vlist[2]=fvec_(-1,-1,1);
  theMeshes[4].vlist[3]=fvec_(-1,-1,-1);
  ((ivec*)theMeshes[4].vlist)[0].w=0;
  ((ivec*)theMeshes[4].vlist)[1].w=0x10000000;
  ((ivec*)theMeshes[4].vlist)[2].w=0x00001000;
  ((ivec*)theMeshes[4].vlist)[3].w=0x10001000;
  theMeshes[4].flist[0]=face(0,1,2,0x80808080);
  theMeshes[4].flist[1]=face(1,3,2,0x80808080);
  
  theMeshes[5].init(4,2);
  theMeshes[5].vlist[0]=fvec_(-1,1,1);
  theMeshes[5].vlist[1]=fvec_(-1,1,-1);
  theMeshes[5].vlist[2]=fvec_(1,1,1);
  theMeshes[5].vlist[3]=fvec_(1,1,-1);
  ((ivec*)theMeshes[5].vlist)[0].w=0;
  ((ivec*)theMeshes[5].vlist)[1].w=0x10000000;
  ((ivec*)theMeshes[5].vlist)[2].w=0x00001000;
  ((ivec*)theMeshes[5].vlist)[3].w=0x10001000;
  theMeshes[5].flist[0]=face(0,1,2,0x80808080);
  theMeshes[5].flist[1]=face(1,3,2,0x80808080);

  generateGifLists();

  envMap.init();
  envMap.width=256;
  envMap.height=256;
  envMap.addr=0;
}


static fvec refraction(fvec in, fvec n, float c)
{
  // in and n must point in the same direction
  myassert(dot(in,n)>=-0.001);

  // compute matrices that transforms the problem to local coordinates

  fmatrix m;
  fvec m1=normalized(n);
  fvec m2=normalized(cross(in,n));
  fvec m3=cross(m1,m2);
  fvec o=fvec_(0,0,0,0);
  setmatrixcolumnsf(&m,&m1,&m2,&m3,&o);
  fmatrix M=m;
  transposematrixf(&M);

  //transform to local coordinates
  fvec in2;
  transformf(&M,&in,&in2,1);
  myassert(in2.y<0.01 && in2.y>-0.01);

  // solve the problem in local coordinates
  float sinInAngle=in2.z/length(in2);
  float sinOutAngle=c*sinInAngle;
  fvec out2=fvec_(sqrt(1-sinOutAngle*sinOutAngle),0,sinOutAngle);

  //transform again
  fvec out;
  transformf(&m,&out2,&out,1);

  return out;
}


static fvec sphereIntersection(fvec p, fvec v, float sign, bool *hit)
{
  float A=dot(v,v);
  float B=2*dot(v,p);
  float C=dot(p,p)-1;
  
  float D=B*B-4*A*C;

  if(D<0)
    {
      *hit=false;
      return fvec_(0,0,0,0);
    }

  *hit=true;
  float t=(-B+sign*sqrt(D))/(2*A);

  return p+t*v;
}


fvec CubeMap::rayCast(int x, int y, fvec *refraction_)
{
  if(refraction_)*refraction_=fvec_(0,0,0,0);;
  fvec v=fvec_((x-128)/128.0,(y-128)/128.0,camDistance,0);
  fvec p=fvec_(0,0,-camDistance,0);

  bool hit;
  fvec nor=sphereIntersection(p,v,-1,&hit);
  if(hit)
    {
      fvec ref=v-(2*dot(v,nor))*nor;
      
      if(hit && refraction_)
	{
	  float factor=1.0f/1.6f;
	  fvec v2=refraction(v,-1.0f*nor,factor);
	  fvec nor2=sphereIntersection(nor,v2,1,&hit);
	  if(hit)
	    *refraction_=refraction(v2,nor2,1.0/factor);
	}
      return ref;
    }
  else
    return fvec_(0,0,0,0);
}

void CubeMap::getMappingCoordinates(fvec v, int &x, int &y, int &n)
{
  float U,V;

  if(v.z>=v.x && v.z>=-v.x && v.z>=v.y && v.z>=-v.y)   //UP
    {
      U=v.x/v.z;//?
      V=v.y/v.z;//?
      n=0;
    }
  else if(-v.z>=v.x && -v.z>=-v.x && -v.z>=v.y && -v.z>=-v.y) //DOWN
    {
      U=-v.x/v.z;
      V=v.y/v.z;
      n=1;
    }
  else if(v.y>=v.x && v.y>=-v.x && v.y>=v.z && v.y>=-v.z) //BACK
    {
      U=v.x/v.y;
      V=-v.z/v.y;
      n=5;
    }
  else if(-v.y>=v.x && -v.y>=-v.x && -v.y>=v.z && -v.y>=-v.z) //FRONT
    {
      U=v.x/v.y;
      V=v.z/v.y;
      n=4;
    }
  else if(v.x>=v.y && v.x>=-v.y && v.x>=v.z && v.x>=-v.z) //RIGHT
    {
      U=-v.y/v.x;
      V=-v.z/v.x;
      n=3;
    }
  else if(-v.x>=v.y && -v.x>=-v.y && -v.x>=v.z && -v.x>=-v.z) //LEFT
    {
      U=-v.y/v.x;
      V=v.z/v.x;
      n=2;
    }
  else myassert(0);

  x=(int)((U*0.5+0.5)*4096*0.5);
  y=(int)((V*0.5+0.5)*4096*0.5);
}


void CubeMap::deinit()
{
  newCubeTextureSize=0;
  newEnvTextureSize=0;
  for(int i=0;i<6;i++)sourceTextures[i]=0;
}


void CubeMap::renderSide(fmatrix *m, texture *t)
{
  t->setcurrent_vif2(0);
  for(int i=0;i<6;i++)
    {
      sourceTextures[i]->setSourceContext2_vif();
      theMeshes[i].draw_zclipRel(m,t,false);
    }
}


static void dump(void *v)
{
  if(v)
    {
      for(int i=0;i<16;i++)
	{
	  printf("%x : ",v);
	  printhexvector(((ivec*)v)[i]);
	}
    }
  else
    printf("nothing to dump.\n");
}


void CubeMap::generateGifLists()
{
  int128 *m=gifList;
		
  int b=0;

  for(int p=0;p<6;p++)
    {
      for(int r=0;r<NUMBER_OF_LAYERS;r++)
	{
	  gif_env;
	  
	  gifLists[p][r]=m;
	  
	  gif_begin(m);
	  
	  gif_tag(0x5353,4,1,PRIMpoint+PRIMtexture+PRIMuv+PRIMcontext2,1); //REGLIST mode
	  
	  for(int y=0;y<2*128;y++)
	    for(int x=0;x<2*128;x++)
	      {
		int u,v,n;

		fvec a;
		a=rayCast(x,y,0);
		
		if(!(a.x==0 && a.y==0 && a.z==0 && a.w==0))
		  {
		    getMappingCoordinates(a,u,v,n);
		    if(n==p)
		      {
			gVEC(wwd(hhw(x*16+32000,y*16+32000),0),wwd(hhw(u,v),0));
			b++;
		      }
		  }
	      }
	  if((GIF_i&1)==0)
	    {
	      gVEC(wwd(hhw(0+32000,0+32000),0),wwd(hhw(0,0),0));
	    }
	  m+=GIF_i;
	  
	  gifListsEnd[p][r]=m;
	  
	  if(GIF_i==1)
	    {
	      gifLists[p][r]=0;
	      gifListsEnd[p][r]=0;
	    }
	  
	  printf(" Number of pixels for map %i: %i\n",p,GIF_i-1);
	  printf(" b: %i\n",b);
	  
	  gif_endfinalregmode();
	}
    }
  myassert(b<=2*65536);
}

texture *CubeMap::getEnvMap(int addr)
{
  envMap.addr=addr;
  return &envMap;
}

static    fmatrix mb[6];

texture *CubeMap::computeEnvMap(fmatrix *m, int addr)
{
  texture tempTextures[6];
  
  for(int i=0;i<6;i++)
    {
      tempTextures[i].init();
      tempTextures[i].width=128;
      tempTextures[i].height=128;
      tempTextures[i].addr=addr+65536*i+4*65536;
    }
  envMap.addr=addr;
  
  envMap.setcurrent_vif2(0);
  for(int r=0;r<NUMBER_OF_LAYERS;r++)
    for(int i=0;i<6;i++)
      if(gifLists[i][r]!=gifListsEnd[i][r])
	{
	  
	  if(gifLists[i][r])
	    {
	      tempTextures[i].setSourceContext2_vif();
	      
	      {
		gif_env;
		
		dma01_beginp();
		vif_nop();
		vif_nop();
		vif_nop();
		vif_direct_begin();
		
		{
		  gif_tag(0xe,1,0,0,0);
		  
		  gPRMODECONT(1);
		  gCOLCLAMP(1);
		  gDTHE(0);
		  
		  gCLAMP_2(0);
		  
		  if(r)
		    {
		      gALPHA_2(0x8000000068);//additive
		    }
		  else
		    {
		      gALPHA_2(0x80000000a8);//alpha
		    }
		  
		  gALPHA_2(0x58);
		  
		  gTEST_2(0x20000);
		  gZBUF_2(0x100000000);
		  
		  gPRIM(PRIMpoint+PRIMtexture+PRIMuv+PRIMcontext2+PRIMalpha);
		  gPABE(0);
		  gRGBAQ(0x3F80000080808080);
		  gif_endfinal();
		}
		vif_direct_end();
		dma01_endp();
	      }
	      
	      {
		dma01_beginp();
		vif_nop();
		vif_nop();
		vif_nop();
		vif_directhl(gifListsEnd[i][r]-gifLists[i][r]);
		dma01_endp();
		dma01_ref(gifLists[i][r],gifListsEnd[i][r]);
	      }
	      dma01_endchain();
	    }
	}
  
  for(int i=0;i<6;i++)mb[i]=*m;
  
  rotatematrixxf(&(mb[1]),3.1415926);
  rotatematrixyf(&(mb[3]),-0.5*3.1415926);
  rotatematrixzf(&(mb[3]),0.5*3.1415926);
  rotatematrixyf(&(mb[2]),-0.5*3.1415926);
  rotatematrixzf(&(mb[2]),0.5*3.1415926);
  rotatematrixyf(&(mb[2]),3.1415926);
  rotatematrixxf(&(mb[4]),-0.5*3.1415926);
  rotatematrixzf(&(mb[4]),3.1415926);
  rotatematrixxf(&(mb[5]),0.5f*3.1415926f);
  
  renderSide(&(mb[0]),&tempTextures[0]);
  renderSide(&(mb[1]),&tempTextures[1]);
  renderSide(&(mb[2]),&tempTextures[2]);
  renderSide(&(mb[3]),&tempTextures[3]);
  renderSide(&(mb[4]),&tempTextures[4]);
  renderSide(&(mb[5]),&tempTextures[5]);
  
  rotatematrixxf(&(mb[0]),2.1f*3.1415926f);
  rotatematrixxf(&(mb[0]),2.2f*3.1415926f);
  rotatematrixxf(&(mb[0]),2.3f*3.1415926f);
  
  return &envMap;
}


void CubeMap::showTexture(texture *t, int x, int y)
{
  
  t->boxuv_vif(16*x,16*y,16*(x+2*128),16*(y+2*128),0,2*128*16*65537,0);
}
