///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMemory.h>
#include <shared/ThString.h>
#include <shared/ThStream.h>

#include <theresa/ThStream.h>

#if HAVE_STRING_H
	#include <string.h>
#endif

#if HAVE_STDLIB_H
	#include <stdlib.h>
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThStream constructors -------------------------------------------------------------------------

IThStream::~IThStream(void)
{
}

// IThStream static methods -----------------------------------------------------------------------

IThStream* IThStream::createInstance(unsigned int size)
{
	return new ThBlockStream(size);
}

IThStream* IThStream::createInstance(IThStream* inner)
{
	return new ThAggregateStream(inner);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThBlockStream constructors ---------------------------------------------------------------------

ThBlockStream::ThBlockStream(unsigned int size):
	m_size(0),
	m_locks(0),
	m_position(0)
{
	m_data.allocate(size);
}

ThBlockStream::~ThBlockStream(void)
{
	THASSERT(m_locks == 0, "Cannot release locked block stream.");
}

// ThBlockStream methods --------------------------------------------------------------------------

unsigned int ThBlockStream::read(void* data, unsigned int size)
{
	if (isEOF())
		return 0;

	if (m_position + size >= m_size)
		size = m_size - m_position - 1;

	memcpy(data, m_data + m_position, size);

	m_position += size;
	return size;
}

unsigned int ThBlockStream::write(const void* data, unsigned int size)
{
	if (m_position + size >= m_data.getCount())
	{
		m_data.resize(m_position + size);
		m_size = m_data.getCount();
	}

	memcpy(m_data + m_position, data, size);

	m_position += size;
	return size;
}

void* ThBlockStream::lock(void)
{
	m_locks++;

	return m_data;
}

void ThBlockStream::unlock(void)
{
	THASSERT(m_locks > 0, "Cannot unlock non-locked block stream.");

	m_locks--;
}

void ThBlockStream::allocate(unsigned int size)
{
	THASSERT(m_locks == 0, "Cannot allocate locked block stream.");

	m_data.allocate(size);
	m_size = m_data.getCount();

	m_position = 0;
}

void ThBlockStream::resize(unsigned int size)
{
	THASSERT(m_locks == 0, "Cannot resize locked block stream.");

	if (m_data.getCount() > size)
		m_size = size;

	m_data.resize(size);
}

void ThBlockStream::release(void)
{
	THASSERT(m_locks == 0, "Cannot release locked block stream.");

	m_data.release();
	m_size = 0;

	m_position = 0;
}

// ThBlockStream attributes -----------------------------------------------------------------------

bool ThBlockStream::isEOF(void) const
{
	if (m_position < m_size)
		return false;

	return true;
}

bool ThBlockStream::isReadable(void) const
{
	return true;
}

bool ThBlockStream::isWritable(void) const
{
	return true;
}

unsigned int ThBlockStream::getSize(void) const
{
	return m_size;
}

unsigned int ThBlockStream::getPosition(void) const
{
	return m_position;
}

bool ThBlockStream::setPosition(unsigned int position)
{
	m_position = position;
	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThAggregateStream constructors -----------------------------------------------------------------

ThAggregateStream::ThAggregateStream(IThStream* inner):
	m_inner(inner)
{
}

ThAggregateStream::~ThAggregateStream(void)
{
}

// ThAggregateStream methods ----------------------------------------------------------------------

unsigned int ThAggregateStream::read(void* data, unsigned int size)
{
	return m_inner->read(data, size);
}

unsigned int ThAggregateStream::write(const void* data, unsigned int size)
{
	return m_inner->write(data, size);
}

// ThAggregateStream attributes -------------------------------------------------------------------

bool ThAggregateStream::isEOF(void) const
{
	return m_inner->isEOF();
}

bool ThAggregateStream::isReadable(void) const
{
	return m_inner->isReadable();
}

bool ThAggregateStream::isWritable(void) const
{
	return m_inner->isWritable();
}

unsigned int ThAggregateStream::getSize(void) const
{
	return m_inner->getSize();
}

unsigned int ThAggregateStream::getPosition(void) const
{
	return m_inner->getPosition();
}

bool ThAggregateStream::setPosition(unsigned int position)
{
	return m_inner->setPosition(position);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
