///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMemory.h>
#include <shared/ThString.h>
#include <shared/ThError.h>
#include <shared/ThStream.h>
#include <shared/ThStorage.h>

#if HAVE_STDIO_H
	#include <stdio.h>
#endif

#include <theresa/ThStorage.h>

#if HAVE_ERRNO_H
	#include <errno.h>
#endif

#if HAVE_DIRECT_H
	#include <direct.h>
#endif

#if HAVE_UNISTD_H
	#include <unistd.h>
#endif

#if HAVE_SYS_TYPES_H
	#include <sys/types.h>
#endif

#if HAVE_SYS_STAT_H
	#include <sys/stat.h>
#endif

#if HAVE_STRING_H
	#include <string.h>
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////

ThSingleton<IThStorage> Storage;

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThStorage constructors ------------------------------------------------------------------------

IThStorage::~IThStorage(void)
{
}

// IThStorage static methods ----------------------------------------------------------------------

bool IThStorage::create(const char* fileName, bool persistent)
{
	if (Storage)
		return true;

	if (fileName)
	{
		// attempt to open storage file

		ThPtr<ThZipStorage> storage = new ThZipStorage();

		if (storage->open(fileName, persistent))
		{
			Storage.attach(storage.detach());
			return true;
		}
	}
	else
	{
		// attempt to create root storage

		ThPtr<ThRootStorage> storage = new ThRootStorage();

		if (storage->open())
		{
			Storage.attach(storage.detach());
			return true;
		}
	}
	
	return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThFileStream constructors ----------------------------------------------------------------------

ThFileStream::ThFileStream(FILE* file, const char* name):
	m_file(file),
	m_name(name)
{
}

ThFileStream::~ThFileStream(void)
{
	fclose(m_file);
	m_file = NULL;
}

// ThFileStream interface methods -----------------------------------------------------------------

unsigned int ThFileStream::read(void* data, unsigned int size)
{
	fseek(m_file, 0, SEEK_CUR);

	const unsigned int bytesRead = fread(data, 1, size, m_file);

	if (ferror(m_file))
	{
		Error->write("Storage", "Unable to read %u bytes from file '%s'.%s%s", size, m_name.getData(), THERESA_NEWLINE, strerror(errno));
		return 0;
	}

	return bytesRead;
}

unsigned int ThFileStream::write(const void* data, unsigned int size)
{
	fseek(m_file, 0, SEEK_CUR);

	const unsigned int bytesWritten = fwrite(data, 1, size, m_file);

	if (ferror(m_file))
	{
		Error->write("Storage", "Unable to write %u bytes to file '%s'.%s%s", size, m_name.getData(), THERESA_NEWLINE, strerror(errno));
		return 0;
	}

	return bytesWritten;
}

// ThFileStream interface attributes --------------------------------------------------------------

bool ThFileStream::isEOF(void) const
{
	return feof(m_file) != 0;
}

bool ThFileStream::isReadable(void) const
{
	return true;
}

bool ThFileStream::isWritable(void) const
{
	return true;
}

unsigned int ThFileStream::getSize(void) const
{
	const unsigned int position = ftell(m_file);

	// TODO: clean this up and add error handling!

	fseek(m_file, 0, SEEK_END);

	const unsigned int size = ftell(m_file);

	fseek(m_file, position, SEEK_SET);

	return size;
}

unsigned int ThFileStream::getPosition(void) const
{
	const unsigned int position = ftell(m_file);

	if (position == (unsigned int) -1)
	{
		Error->write("Storage", "Unable to retrieve position for file '%s'.%s%s", m_name.getData(), THERESA_NEWLINE, strerror(errno));
		return 0;
	}

	return position;
}

bool ThFileStream::setPosition(unsigned int position)
{
	if (fseek(m_file, position, SEEK_SET))
	{
		Error->write("Storage", "Unable to set position to %u in file '%s'.%s%s", position, m_name.getData(), THERESA_NEWLINE, strerror(errno));
		return false;
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThRootStorage constructors ---------------------------------------------------------------------

ThRootStorage::ThRootStorage(void)
{
}

ThRootStorage::~ThRootStorage(void)
{
	close();
}

// ThRootStorage methods --------------------------------------------------------------------------

bool ThRootStorage::open(void)
{
	close();

	return true;
}

void ThRootStorage::close(void)
{
	if (m_files.getCount())
	{
		Error->display("Storage", "File stream resource leak detected.");

		m_files.release();
	}
}

// ThRootStorage folder methods -------------------------------------------------------------------

bool ThRootStorage::createFolder(const char* name)
{
#if THERESA_VISUALSTUDIO
	if (mkdir(name))
#else
	if (mkdir(name, S_IRWXU | S_IRWXG | S_IRWXO))
#endif
	{
		Error->write("Storage", "Unable to create folder '%s'.", name);
		return false;
	}

	return true;
}

bool ThRootStorage::destroyFolder(const char* name)
{
	if (rmdir(name))
	{
		Error->write("Storage", "Unable to destroy folder '%s'.", name);
		return false;
	}

	return true;
}

// ThRootStorage file methods ---------------------------------------------------------------------

IThStream* ThRootStorage::createFile(const char* name)
{
	FILE* file = fopen(name, "wb+");
	if (!file)
	{
		Error->write("Storage", "Unable to create file '%s'.%s%s", name, THERESA_NEWLINE, strerror(errno));
		return NULL;
	}

	return new ThFileStream(file, name);
}

IThStream* ThRootStorage::openFile(const char* name)
{
	FILE* file = fopen(name, "rb+");
	if (!file)
	{
		Error->write("Storage", "Unable to open file '%s'.%s%s", name, THERESA_NEWLINE, strerror(errno));
		return NULL;
	}

	return new ThFileStream(file, name);
}

bool ThRootStorage::cacheFile(const char* name, ThByteBlock& data)
{
	data.release();
	
	ThPtr<IThStream> file = openFile(name);
	if (!file)
		return false;
		
	data.allocate(file->getSize());
	
	if (data.getSize())
	{
		if (!file->readItems(data.getData(), data.getCount()))
		{
			Error->write("Storage", "Unable to cache data from file '%s'.%s%s", name, THERESA_NEWLINE, strerror(errno));

			data.release();
			return false;
		}
	}
	
	return true;
}

bool ThRootStorage::destroyFile(const char* name)
{
	if (remove(name))
	{
		Error->write("Storage", "Unable to destroy file '%s'.%s%s", name, THERESA_NEWLINE, strerror(errno));
		return false;
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThZipStorage constructors ----------------------------------------------------------------------

ThZipStorage::ThZipStorage(void)
{
}

ThZipStorage::~ThZipStorage(void)
{
	close();
}

// ThZipStorage methods ---------------------------------------------------------------------------

bool ThZipStorage::open(const char* fileName, bool persistent)
{
	close();

	return false;
}

void ThZipStorage::close(void)
{
}

// ThZipStorage interface folder methods ----------------------------------------------------------

bool ThZipStorage::createFolder(const char* name)
{
	return false;
}

bool ThZipStorage::destroyFolder(const char* name)
{
	return false;
}

// ThZipStorage interface file methods ------------------------------------------------------------

IThStream* ThZipStorage::createFile(const char* name)
{
	return NULL;
}

IThStream* ThZipStorage::openFile(const char* name)
{
	return NULL;
}

bool ThZipStorage::cacheFile(const char* name, ThByteBlock& data)
{
	return false;
}

bool ThZipStorage::destroyFile(const char* name)
{
	return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
