///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMemory.h>
#include <shared/ThError.h>
#include <shared/ThServer.h>

#include <theresa/ThServer.h>

#if HAVE_STRING_H
	#include <string.h>
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////

ThSingleton<IThServer> Server;

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThServerObject attributes ----------------------------------------------------------------------

unsigned int ThServerObject::getID(void) const
{
	return m_id;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThServerObject constructors --------------------------------------------------------------------

ThServerObject::ThServerObject(unsigned int id)
{
	THASSERT(Server != NULL, "Cannot create server objects until the server object is created.");

	if (id == THID_INVALID)
		m_id = Server->createObjectID();
	else
		m_id = id;

	Server->registerObject(this);
}

ThServerObject::~ThServerObject(void)
{
}

// ThServerObject methods -------------------------------------------------------------------------

bool ThServerObject::receive(const IThMessage* message)
{
	return true;
}

// ThServerObject methods -------------------------------------------------------------------------

void ThServerObject::sendMessage(unsigned int message, unsigned int toID)
{
	THASSERT(Server != NULL, "Cannot send messages until the server object is created.");

	Server->sendMessage(message, toID, m_id);
}

void ThServerObject::sendMessage(unsigned int message, unsigned int toID, const void* data, unsigned int size)
{
	THASSERT(Server != NULL, "Cannot send messages until the server object is created.");

	Server->sendMessage(message, toID, m_id, data, size);
}

void ThServerObject::postMessage(unsigned int message, unsigned int toID)
{
	THASSERT(Server != NULL, "Cannot post messages until the server object is created.");

	Server->postMessage(message, toID, m_id);
}

void ThServerObject::postMessage(unsigned int message, unsigned int toID, const void* data, unsigned int size)
{
	THASSERT(Server != NULL, "Cannot post messages until the server object is created.");

	Server->postMessage(message, toID, m_id, data, size);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThClient constructors --------------------------------------------------------------------------

ThClient::ThClient(void):
	ThServerObject(THID_CLIENT)
{
}

ThClient::~ThClient(void)
{
}

// ThClient callbacks -----------------------------------------------------------------------------

bool ThClient::receive(const IThMessage* message)
{
	return ThServerObject::receive(message);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThMessage constructors ------------------------------------------------------------------------

IThMessage::~IThMessage(void)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThServer constructors -------------------------------------------------------------------------

IThServer::~IThServer(void)
{
}

// IThServer static methods -----------------------------------------------------------------------

bool IThServer::create(void)
{								
	if (Server)
		return true;

	ThPtr<ThServer> server = new ThServer();

	if (!server->open())
		return false;

	Server.attach(server.detach());
	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThMessage constructors -------------------------------------------------------------------------

ThMessage::ThMessage(unsigned int message, unsigned int toID, unsigned int fromID):
	m_message(message),
	m_toID(toID),
	m_fromID(fromID)
{
}

ThMessage::ThMessage(unsigned int message, unsigned int toID, unsigned int fromID, const void* data, unsigned int size):
	m_message(message),
	m_toID(toID),
	m_fromID(fromID)
{
	if (data && size)
	{
		m_data.allocate(size);
		memcpy(m_data, data, size);
	}
}

// ThMessage attributes ---------------------------------------------------------------------------

unsigned int ThMessage::getMessage(void) const
{
	return m_message;
}

unsigned int ThMessage::getToID(void) const
{
	return m_toID;
}

unsigned int ThMessage::getFromID(void) const
{
	return m_fromID;
}

const void* ThMessage::getData(void) const
{
	return m_data;
}

unsigned int ThMessage::getSize(void) const
{
	return m_data.getSize();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThServer constructors --------------------------------------------------------------------------

ThServer::ThServer(void)
{
	m_nextID = THID_CUSTOM_BASE;
}

ThServer::~ThServer(void)
{
	close();
}

// ThServer methods -------------------------------------------------------------------------------

bool ThServer::open(void)
{
	close();

	return true;
}

void ThServer::close(void)
{
	m_objects.release();
}

// ThServer interface methods ---------------------------------------------------------------------

void ThServer::sendMessage(unsigned int message, unsigned int toID, unsigned int fromID)
{
	ThMessage messageObject(message, toID, fromID);

	sendMessage(&messageObject);
}

void ThServer::sendMessage(unsigned int message, unsigned int toID, unsigned int fromID, const void* data, unsigned int size)
{
	ThMessage messageObject(message, toID, fromID, data, size);

	sendMessage(&messageObject);
}

void ThServer::postMessage(unsigned int message, unsigned int toID, unsigned int fromID)
{
	ThMessage* messageObject = new ThMessage(message, toID, fromID);
	
	m_messages.attachLast(messageObject);
}

void ThServer::postMessage(unsigned int message, unsigned int toID, unsigned int fromID, const void* data, unsigned int size)
{
	ThMessage* messageObject = new ThMessage(message, toID, fromID, data, size);
	
	m_messages.attachLast(messageObject);
}

// ThServer interface object methods --------------------------------------------------------------

unsigned int ThServer::createObjectID(void)
{
	return m_nextID++;
}

bool ThServer::registerObject(ThServerObject* object)
{
	if (findObject(object->getID()))
	{
		Error->display("Server", "Duplicate id:s detected during object registration.");
		return false;
	}

	m_objects.attachFirst(object);
	return true;
}

ThServerObject* ThServer::findObject(unsigned int id)
{
	// TODO: optimize!
	
	for (ThIterator<ThServerObject> object(m_objects);  object;  object.next())
	{
		if (object->getID() == id)
			return object;
	}

	return NULL;
}

// ThServer methods -------------------------------------------------------------------------------

void ThServer::sendMessage(const IThMessage* message)
{
	switch (message->getToID())
	{
		case THID_ANNOUNCE:
		{
			// send to all except sender

			if (message->getFromID() != THID_SERVER)
				receive(message);

			for (ThIterator<ThServerObject> object(m_objects);  object; )
			{
				if (object->getID() == message->getFromID())
					object.next();
				else
				{
					if (object->receive(message))
						object.next();
					else
						object.release();
				}
			}

			break;
		}

		case THID_SERVER:
		{
			// send to server

			receive(message);
			break;
		}

		default:
		{
			// send to receiver

			if (ThServerObject* object = findObject(message->getToID()))
			{
				if (!object->receive(message))
					delete object;
			}

			break;
		}
	}
}

void ThServer::receive(const IThMessage* message)
{
	switch (message->getMessage())
	{
		case THMSG_SERVER_UPDATE:
		{
			for (ThIterator<ThMessage> message(m_messages);  message;  message.release())
				sendMessage(message);

			break;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////
