///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef THERESA_THSCRIPT_H
#define THERESA_THSCRIPT_H
///////////////////////////////////////////////////////////////////////////////////////////////////

class ThSymbolTable : public IThSymbolTable
{
public:
// constructors
																		ThSymbolTable(void);
	virtual														~ThSymbolTable(void);
// methods
	virtual bool											registerSymbol(unsigned int id, const char* name);
	virtual const char*								translateSymbol(unsigned int id) const;
	virtual unsigned int							translateSymbol(const char* name) const;
private:
// classes
	class Symbol;
// data
	ThList<Symbol>										m_symbols;
};

//-------------------------------------------------------------------------------------------------

class ThSymbolTable::Symbol : public ThListItem<Symbol>
{
public:
// constructors
																		Symbol(unsigned int id, const char* name);
// data
	unsigned int											m_id;
	unsigned int											m_hash;
	ThString													m_name;
};

//-------------------------------------------------------------------------------------------------

class ThTokenizer : public IThTokenizer
{
public:
// constructors
																		ThTokenizer(void);
	virtual														~ThTokenizer(void);
// methods
	bool															open(IThSymbolTable* symbolTable);
	void															close(void);
// interface methods
	virtual bool											registerPattern(unsigned int list, unsigned int symbol, const char* data, const char* substitute = NULL, unsigned int branchList = THSYM_INVALID);
	virtual bool											tokenize(ThTokenList& target, unsigned int root, const char* source);
// interface attributes
	virtual const IThSymbolTable*			getSymbolTable(void) const;
private:
// classes
	class PatternNode;
	class Pattern;
	class PatternList;
// methods
	bool															compilePattern(Pattern* pattern, const char* data);
	const char*												compilePatternFlags(Pattern* pattern, const char* data);
	const char*												compilePatternNode(PatternNode* patternNode, const char* data);
	char															convertEscapeCharacter(char c);
	unsigned int											validatePattern(Pattern* pattern, const char* data);
	PatternList*											findPatternList(unsigned int symbol);
// data
	IThSymbolTable*										m_symbolTable;
	ThList<PatternList>								m_patternLists;
	Pattern*													m_pattern;
	unsigned int											m_line;
};

//-------------------------------------------------------------------------------------------------

class ThTokenizer::PatternNode : public ThListItem<PatternNode>
{
public:
// constructors
																		PatternNode(void);
// methods
	void															fillRange(unsigned int start, unsigned int length, bool value);
	bool															validate(unsigned char c) const;
// data
	bool															m_repeated;
	bool															m_fallback;
	bool															m_table[256];
};

//-------------------------------------------------------------------------------------------------

class ThTokenizer::Pattern : public ThListItem<Pattern>
{
public:
// constructors
																		Pattern(unsigned int symbol, const char* substitute, unsigned int branchList);
// data
	unsigned int											m_symbol;
	ThString													m_substitute;
	ThList<PatternNode>								m_nodes;
	bool															m_mergePrev;
	bool															m_lineBreak;
	bool															m_discardData;
	bool															m_discardToken;
	unsigned int											m_branchList;
};

//-------------------------------------------------------------------------------------------------

class ThTokenizer::PatternList : public ThList<Pattern>, public ThListItem<PatternList>
{
public:
// constructors
																		PatternList(unsigned int symbol);
// data
	unsigned int											m_symbol;
};

//-------------------------------------------------------------------------------------------------

class ThAnalyzer : public IThAnalyzer
{
public:
// constructors
																		ThAnalyzer(void);
	virtual														~ThAnalyzer(void);
// methods
	bool															open(IThSymbolTable* symbolTable);
	void															close(void);
// interface methods
	bool															registerTerminator(unsigned int symbol);
	bool															registerBalancedPair(unsigned int openingSymbol, unsigned int closingSymbol);
	bool															registerGrammar(unsigned int list, unsigned int symbol, const char* data, unsigned int level = 0);
	bool															analyze(ThTokenList& target, unsigned int root, const ThTokenList& source);
// interface attributes
	const IThSymbolTable*							getSymbolTable(void) const;
private:
// classes
	class GrammarNode;
	class Grammar;
	class GrammarList;
// methods
	bool															compileGrammarNodes(Grammar* grammar, const char* data);
	bool															compileGrammarList(GrammarList* grammarList, unsigned int level, ThList<ThToken>& target, ThConstIterator<ThToken>& source);
	bool															compileGrammar(Grammar* grammar, unsigned int list, ThList<ThToken>& target, ThConstIterator<ThToken>& source);
	bool															examineGrammar(Grammar* grammar, unsigned int symbol, const ThToken* source);
	GrammarList*											findGrammarList(unsigned int symbol);
// constants
	enum
	{
		SYM_ROOT = THSYM_SYSTEM_BASE,
		SYM_DISCARD,
		SYM_STATIC,
		SYM_ABSOLUTE,
		SYM_RECURSIVE,
		SYM_PRESCAN,
		SYM_REQUIRED,
		SYM_OPTIONAL,
	};
// data
	bool															m_error;
	ThBlock<unsigned int>							m_terminators;
	ThBlock<unsigned int>							m_balancedPairs;
	IThSymbolTable*										m_externalSymbols;
	ThList<GrammarList>								m_grammarLists;
// static data
	static ThPtr<IThTokenizer>				m_tokenizer;
	static ThPtr<IThSymbolTable>			m_internalSymbols;
};

//-------------------------------------------------------------------------------------------------

class ThAnalyzer::GrammarNode : public ThListItem<GrammarNode>
{
public:
// constructors
																		GrammarNode(unsigned int symbol, unsigned int type, bool required, bool optional);
// data
	unsigned int											m_symbol;
	unsigned int											m_type;
	bool															m_required;
	bool															m_optional;
};

//-------------------------------------------------------------------------------------------------

class ThAnalyzer::Grammar : public ThListItem<Grammar>
{
public:
// constructors
																		Grammar(unsigned int symbol);
// data
	unsigned int											m_symbol;
	unsigned int											m_level;
	ThList<GrammarNode>								m_nodes;
};

//-------------------------------------------------------------------------------------------------

class ThAnalyzer::GrammarList : public ThList<Grammar>, public ThListItem<GrammarList>
{
public:
// constructors
																		GrammarList(unsigned int symbol);
// data
	unsigned int											m_symbol;
	unsigned int											m_levels;
	unsigned int											m_first;
};

//-------------------------------------------------------------------------------------------------

class ThSyntax : public IThSyntax
{
public:
// constructors
																		ThSyntax(void);
																		~ThSyntax(void);
// methods
	bool															open(IThSymbolTable* symbolTable, const char* script);
	void															close(void);
// attributes
	IThSymbolTable*										getSymbolTable(void);
	IThTokenizer*											getTokenizer(void);
	IThAnalyzer*											getAnalyzer(void);
private:
// methods
	bool															evaluateCommand(ThToken* root);
	ThToken*													findToken(ThToken* token, unsigned int depth);
	unsigned int											translateSymbolToken(ThToken* symbolToken);
// constants
	enum
	{
		SYM_ROOT = THSYM_SYSTEM_BASE,
		SYM_COMMANDLIST,
		SYM_COMMAND,
		SYM_DISCARD,
		SYM_NEWLINE,
		SYM_SYMBOL,
		SYM_PATTERN,
		SYM_TERMINATOR,
		SYM_BALANCED_PAIR,
		SYM_GRAMMAR,
		SYM_STRING,
		SYM_INTEGER,
	};
// data
	IThSymbolTable*										m_externalSymbols;
	ThPtr<IThTokenizer>								m_externalTokenizer;
	ThPtr<IThAnalyzer>								m_externalAnalyzer;
	ThStringList											m_foundSymbols;
// static data
	static ThPtr<IThSymbolTable>			m_internalSymbols;
	static ThPtr<IThTokenizer>				m_internalTokenizer;
	static ThPtr<IThAnalyzer>					m_internalAnalyzer;
};

///////////////////////////////////////////////////////////////////////////////////////////////////
#endif /* THERESA_THSCRIPT_H */
///////////////////////////////////////////////////////////////////////////////////////////////////
