///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef THERESA_THENGINE_H
#define THERESA_THENGINE_H
///////////////////////////////////////////////////////////////////////////////////////////////////

class ThMesh;
class ThLight;
class ThMaterial;
class ThEngine;

///////////////////////////////////////////////////////////////////////////////////////////////////

class ThMesh : public IThMesh, public ThListItem<ThMesh>
{
public:
// constructors
																		ThMesh(const char* name, IThEngine* engine);
																		~ThMesh(void);
// interface methods
	void															render(void);
// interface attributes
	ThVertex*													getVertex(unsigned int index);
	unsigned int											getVertexCount(void) const;
	ThTriangle*												getTriangle(unsigned int index);
	unsigned int											getTriangleCount(void) const;
	ThGeometry*												getGeometry(unsigned int index);
	unsigned int											getGeometryCount(void) const;
	const char*												getName(void) const;
	unsigned int											getHash(void) const;
	ThMeshData&												getData(void);
private:
// data
	ThMeshData												m_data;
	ThString													m_name;
	unsigned int											m_hash;
	IThEngine* 												m_engine;
};

//-------------------------------------------------------------------------------------------------

class ThLight : public IThLight, public ThListItem<ThLight>
{
// friends
	friend class ThEngine;
public:
// constructors
																		ThLight(const char* name, IThEngine* engine);
																		~ThLight(void);
// interface methods
	bool															enable(void);
	void															disable(void);
// interface attributes
	bool															isEnabled(void) const;
	const ThVector4&									getAmbientColor(void) const;
	void															setAmbientColor(const ThVector4& color);
	const ThVector4&									getDiffuseColor(void) const;
	void															setDiffuseColor(const ThVector4& color);
	const ThVector4&									getSpecularColor(void) const;
	void															setSpecularColor(const ThVector4& color);
	const ThVector3&									getPosition(void) const;
	void															setPosition(const ThVector3& position);
	const ThVector3&									getDirection(void) const;
	void															setDirection(const ThVector3& direction);
	unsigned int											getType(void) const;
	void															setType(unsigned int type);
	const char*												getName(void) const;
	unsigned int											getHash(void) const;
	ThLightData&											getData(void);
private:
// data
	ThLightData												m_data;
	bool															m_enabled;
	unsigned int											m_index;
	ThString													m_name;
	unsigned int											m_hash;
	IThEngine* 												m_engine;
// static data
	static ThBlock<ThLightData>				m_current;
	static ThBlock<bool>							m_indices;
};

//-------------------------------------------------------------------------------------------------

class ThMaterial : public IThMaterial, public ThListItem<ThMaterial>
{
public:
// constructors
																		ThMaterial(const char* name, IThEngine* engine);
																		~ThMaterial(void);
// interface methods
	void															apply(void);
// interface attributes
	bool															isCulling(void);
	bool															isLighting(void);
	bool															isBlending(void);
	bool															isDepthTesting(void);
	bool															isDepthWriting(void);
	void															setCulling(bool enabled);
	void															setLighting(bool enabled);
	void															setBlending(bool enabled);
	unsigned int											getSrcBlend(void);
	void															setSrcBlend(unsigned int factor);
	unsigned int											getDstBlend(void);
	void															setDstBlend(unsigned int factor);
	void															setDepthTesting(bool enabled);
	void															setDepthWriting(bool enabled);
	unsigned int											getDepthFunc(void);
	void															setDepthFunc(unsigned int func);
	const ThVector4&									getAmbientColor(void);
	void															setAmbientColor(const ThVector4& color);
	const ThVector4&									getDiffuseColor(void);
	void															setDiffuseColor(const ThVector4& color);
	const ThVector4&									getSpecularColor(void);
	void															setSpecularColor(const ThVector4& color);
	unsigned int											getShadeModel(void);
	void															setShadeModel(unsigned int mode);
	unsigned int											getCulledFace(void);
	void															setCulledFace(unsigned int face);
	unsigned int											getCombineMode(void);
	void															setCombineMode(unsigned int mode);
	const ThVector4&									getCombineColor(void);
	void															setCombineColor(const ThVector4& color);
	bool															getCoordGen(unsigned int axis);
	void															setCoordGen(unsigned int axis, bool enabled);
	unsigned int											getCoordGenMode(unsigned int axis);
	void															setCoordGenMode(unsigned int axis, unsigned int mode);
	const ThVector4&									getCoordObjPlane(unsigned int axis);
	void															setCoordObjPlane(unsigned int axis, const ThVector4& plane);
	const ThVector4&									getCoordEyePlane(unsigned int axis);
	void															setCoordEyePlane(unsigned int axis, const ThVector4& plane);
	const char*												getTextureName(void);
	void															setTextureName(const char* name);
	const char*												getName(void) const;
	unsigned int											getHash(void) const;
	ThMaterialData&										getData(void);
private:
// methods
	unsigned int											axisToIndex(unsigned int axis) const;
	unsigned int											indexToAxis(unsigned int index) const;
// data
	ThMaterialData										m_data;
	ThString													m_name;
	unsigned int											m_hash;
	IThEngine* 												m_engine;
// static data
	static ThMaterialData							m_current;
};

//-------------------------------------------------------------------------------------------------

class ThRenderQueue : public IThRenderQueue
{
public:
// constructors
																		ThRenderQueue(ThEngine* engine);
																		~ThRenderQueue(void);
// methods
	void															prepare(void);
	void															render(void);
	void															reset(void);
// object methods
	void															addMesh(IThMesh* mesh, const ThMatrix4& transform);
	void															addGeometry(const ThGeometry* geometry, const ThMatrix4& transform);
	void															addTriangle(const ThTriangle* triangle, const ThMatrix4& transform);
	void															addLight(IThLight* light, const ThMatrix4& transform);
private:
// classes
	class Geometry;
	class Triangle;
	class Light;
// data
	ThList<Geometry>									m_geometries;
	ThList<Triangle>									m_triangles;
	ThList<Light>											m_lights;
	ThBlock<Triangle*>								m_sortBuffer;
	IThEngine*												m_engine;
};

//-------------------------------------------------------------------------------------------------

class ThRenderQueue::Geometry : public ThListItem<Geometry>
{
public:
// data
	const ThGeometry*									m_geometry;
	ThMatrix4													m_transform;
};

//-------------------------------------------------------------------------------------------------

class ThRenderQueue::Triangle : public ThListItem<Triangle>
{
public:
// data
	const ThTriangle*									m_triangle;
	ThMatrix4													m_transform;
};

//-------------------------------------------------------------------------------------------------

class ThRenderQueue::Light : public ThListItem<Light>
{
public:
// data
	IThLight*													m_light;
	ThMatrix4													m_transform;
};

//-------------------------------------------------------------------------------------------------

class ThEngine : public IThEngine
{
public:
// constructors
																		ThEngine(void);
																		~ThEngine(void);
// methods
	bool															open(IThDisplay* display);
	void															close(void);
// interface methods
	bool															loadResources(const char* fileName);
	bool															loadResources(IThStream* stream);
// object methods
	IThMesh*													createMesh(const char* name);
	IThMesh*													findMesh(const char* name);
	IThLight*													createLight(const char* name);
	IThLight*													findLight(const char* name);
	IThMaterial*											createMaterial(const char* name);
	IThMaterial*											findMaterial(const char* name);
	IThRenderQueue*										createRenderQueue(void);
// attributes
	IThDisplay*												getDisplay(void);
	ThEngineNode*											getRootNode(void);
private:
// classes
	class Geometry;
// methods
	bool															loadResources(Lib3dsFile* file);
	IThMesh*													findMesh(unsigned int hash);
	IThLight*													findLight(unsigned int hash);
	IThMaterial*											findMaterial(unsigned int hash);
// data
	IThDisplay*												m_display;
	ThList<ThMesh>										m_meshes;
	ThList<ThLight>										m_lights;
	ThList<ThMaterial>								m_materials;
	ThEngineNode											m_root;
};

//-------------------------------------------------------------------------------------------------

class ThEngine::Geometry : public ThListItem<Geometry>
{
public:
// data
	unsigned int											m_hash;
	unsigned int											m_count;
};

///////////////////////////////////////////////////////////////////////////////////////////////////
#endif /*THERESA_THENGINE_H*/
///////////////////////////////////////////////////////////////////////////////////////////////////
