///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMemory.h>
#include <shared/ThMath.h>
#include <shared/ThString.h>
#include <shared/ThError.h>
#include <shared/ThStream.h>
#include <shared/ThStorage.h>
#include <shared/ThServer.h>
#include <shared/ThContext.h>
#include <shared/ThDisplay.h>
#include <shared/ThEngine.h>

#include <lib3ds/io.h>
#include <lib3ds/file.h>
#include <lib3ds/material.h>
#include <lib3ds/mesh.h>

#include <theresa/ThEngine.h>

///////////////////////////////////////////////////////////////////////////////////////////////////

static inline void convertVector2(ThVector2& intVector, float extVector[])
{
	intVector.x = extVector[0];
	intVector.y = extVector[1];
}

static inline void convertVector3(ThVector3& intVector, float extVector[])
{
	intVector.x = extVector[0];
	intVector.y = extVector[1];
	intVector.z = extVector[2];
}

static inline void convertVector4(ThVector4& intVector, float extVector[])
{
	intVector.x = extVector[0];
	intVector.y = extVector[1];
	intVector.z = extVector[2];
	intVector.w = extVector[3];
}

static Lib3dsBool lib3dsIoError(void* self)
{
	IThStream* stream = reinterpret_cast<IThStream*>(self);

	return stream->isEOF();
}

static long lib3dsIoSeek(void* self, long offset, Lib3dsIoSeek origin)
{
	IThStream* stream = reinterpret_cast<IThStream*>(self);

	// TODO: verify arithmetic!
	if (origin == LIB3DS_SEEK_CUR)
	{
		offset += stream->getPosition();
		if ((unsigned int) offset >= stream->getSize())
			return -1;
	}
	else if (origin == LIB3DS_SEEK_END)
	{
		offset = stream->getSize() - offset - 1;
		if (offset < 0)
			return -1;
	}
		
	stream->setPosition(offset);
	
	return 0;
}

static long lib3dsIoTell(void* self)
{
	IThStream* stream = reinterpret_cast<IThStream*>(self);
	
	return stream->getPosition();
}

static int lib3dsIoRead(void* self, Lib3dsByte* buffer, int size)
{
	IThStream* stream = reinterpret_cast<IThStream*>(self);
	
	return stream->read(buffer, size);
}

static int lib3dsIoWrite(void* self, const Lib3dsByte* buffer, int size)
{
	IThStream* stream = reinterpret_cast<IThStream*>(self);
	
	return stream->write(buffer, size);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThLightData constructors -----------------------------------------------------------------------

ThLightData::ThLightData(void)
{
	reset();
}

// ThLightData methods ----------------------------------------------------------------------------

void ThLightData::reset(void)
{
	// NOTE: these must correspond to the default values as set by the engine.

	m_ambient.reset();
	m_diffuse.set(1.f, 1.f, 1.f, 0.f);
	m_specular.reset();
	m_position.set(0.f, 0.f, 1.f);
	m_direction.set(0.f, 0.f, 1.f);
	m_type = DIRECTIONAL;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThMaterialData constructors -------------------------------------------------------------------

ThMaterialData::ThMaterialData(void)
{
	reset();
}

// ThMaterialData methods -------------------------------------------------------------------------

void ThMaterialData::reset(void)
{
	// NOTE: these must correspond to OpenGL default values.
	// TODO: create new render state defaults!

	m_culling       = false;
	m_lighting      = false;
	m_blending      = false;
	m_depthTesting  = false;
	m_depthWriting  = false;
	m_shadeModel    = GL_SMOOTH;
	m_depthFunction = GL_LESS;
	m_srcFactor     = GL_ONE;
	m_dstFactor     = GL_ONE;
	m_culledFace    = GL_BACK;
	m_combineMode   = GL_MODULATE;
	m_coordGen[0]   = false;
	m_coordGen[1]   = false;
	m_coordGen[2]   = false;
	m_coordGen[3]   = false;
	m_genModes[0]   = GL_EYE_LINEAR;
	m_genModes[1]   = GL_EYE_LINEAR;
	m_genModes[2]   = GL_EYE_LINEAR;
	m_genModes[3]   = GL_EYE_LINEAR;

	m_objPlanes[0].set(1.f, 0.f, 0.f, 0.f);
	m_eyePlanes[0].set(1.f, 0.f, 0.f, 0.f);
	m_objPlanes[1].set(0.f, 1.f, 0.f, 0.f);
	m_eyePlanes[1].set(0.f, 1.f, 0.f, 0.f);
	m_objPlanes[2].reset();
	m_eyePlanes[2].reset();
	m_objPlanes[3].reset();
	m_eyePlanes[3].reset();

	m_ambientColor.reset();
	m_diffuseColor.reset();
	m_specularColor.reset();
	m_combineColor.reset();

	m_textureName.release();
	m_textureHash = 0;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThEngineNode constructors ----------------------------------------------------------------------

ThEngineNode::ThEngineNode(void)
{
	m_parent = NULL;
	
	m_invalidWorld = true;
	m_invalidBound = true;
	
	m_local.reset();
	m_world.reset();
	m_nodeBound.reset();
	m_treeBound.reset();
}

ThEngineNode::~ThEngineNode(void)
{
}

// ThEngineNode methods ---------------------------------------------------------------------------

void ThEngineNode::attachChild(ThEngineNode* child)
{
	child->m_parent = this;
	
	m_children.attachFirst(child);
}

// ThEngineNode attributes ------------------------------------------------------------------------

ThEngineNode* ThEngineNode::getParent(void)
{
	if (isAttached())
		return m_parent;
		
	return m_parent = NULL;
}

ThEngineNode* ThEngineNode::getFirstChild(void)
{
	return m_children.getFirst();
}

void ThEngineNode::setLocalTransform(const ThTransform& transform)
{
	invalidateWorld();

	m_local = transform;
}

const ThTransform& ThEngineNode::getLocalTransform(void) const
{
	return m_local;
}

const ThTransform& ThEngineNode::getWorldTransform(void)
{
	updateWorld();
	
	return m_world;
}

const ThSphere& ThEngineNode::getNodeBound(void)
{
	return m_nodeBound;
}

const ThSphere& ThEngineNode::getTreeBound(void)
{
	updateBound();
	
	return m_treeBound;
}

// ThEngineNode methods ---------------------------------------------------------------------------

void ThEngineNode::invalidateWorld(void)
{
	m_invalidWorld = true;
	
	if (m_parent)
		m_parent->invalidateWorld();
}

bool ThEngineNode::updateWorld(void)
{
	bool invalidWorld = m_invalidWorld;
	
	if (m_parent)
		invalidWorld = m_parent->updateWorld();
		
	if (invalidWorld)
	{
		if (m_parent)
			m_world.concatenate(m_parent->m_world);
		else
			m_world = m_local;
			
		m_invalidWorld = false;
	}
	
	return invalidWorld;
}

void ThEngineNode::updateBound(void)
{
	if (m_invalidBound)
	{
		m_treeBound = m_nodeBound;
		
		for (ThIterator<ThEngineNode> child(m_children);  child;  child.next())
		{
			if (child->m_invalidBound)
				child->updateBound();
				
			//m_treeBound.
		}
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThMesh constructors ---------------------------------------------------------------------------

IThMesh::~IThMesh(void)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThLight constructors --------------------------------------------------------------------------

IThLight::~IThLight(void)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThMaterial constructors -----------------------------------------------------------------------

IThMaterial::~IThMaterial(void)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThRenderQueue constructors --------------------------------------------------------------------

IThRenderQueue::~IThRenderQueue(void)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThEngine constructors -------------------------------------------------------------------------

IThEngine::~IThEngine(void)
{
}

// IThEngine static methods -----------------------------------------------------------------------

IThEngine* IThEngine::createInstance(IThDisplay* display)
{
	ThPtr<ThEngine> engine = new ThEngine();
	if (!engine->open(display))
		return false;
		
	return engine.detach();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThMesh constructors ----------------------------------------------------------------------------

ThMesh::ThMesh(const char* name, IThEngine* engine):
	m_name(name),
	m_engine(engine)
{
	m_hash = m_name.hashNoCase();
}

ThMesh::~ThMesh(void)
{
}

// ThMesh interface methods -----------------------------------------------------------------------

void ThMesh::render(void)
{
	for (unsigned int i = 0;  i < m_data.m_geometries.getCount();  i++)
	{
		ThGeometry& geometry = m_data.m_geometries[i];

		geometry.m_material->apply();

		if (geometry.m_material->getShadeModel() == GL_SMOOTH)
		{
			glPushClientAttrib(GL_CLIENT_VERTEX_ARRAY_BIT);
	
			glEnableClientState(GL_NORMAL_ARRAY);
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
			glEnableClientState(GL_VERTEX_ARRAY);
	
			glInterleavedArrays(GL_T2F_N3F_V3F, 0, m_data.m_vertices);
	
			glBegin(geometry.m_mode);
	
			for (unsigned int j = 0;  j < geometry.m_indices.getCount();  j++)
				glArrayElement(geometry.m_indices[j]);
	
			glEnd();
	
			glPopClientAttrib();
		}
		else
		{
			THASSERT(geometry.m_mode == GL_TRIANGLES, "Cannot render other primitives than triangles in flat mode.");

			glBegin(geometry.m_mode);
			
			for (unsigned int j = 0;  j < geometry.m_indices.getCount();  j++)
			{
				ThVertex& vertex = m_data.m_vertices[geometry.m_indices[j]];
				
				glTexCoord2fv(vertex.m_texCoord);
				glNormal3fv(geometry.m_triangles[j / 3]->m_normal);
				glVertex3fv(vertex.m_vertex);
			}
			
			glEnd();
		}
	}
}

// ThMesh interface attributes --------------------------------------------------------------------

ThVertex* ThMesh::getVertex(unsigned int index)
{
	if (index < m_data.m_vertices.getCount())
		return m_data.m_vertices + index;

	return NULL;
}

unsigned int ThMesh::getVertexCount(void) const
{
	return m_data.m_vertices.getCount();
}

ThTriangle* ThMesh::getTriangle(unsigned int index)
{
	if (index < m_data.m_triangles.getCount())
		return m_data.m_triangles + index;

	return NULL;
}

unsigned int ThMesh::getTriangleCount(void) const
{
	return m_data.m_triangles.getCount();
}

ThGeometry* ThMesh::getGeometry(unsigned int index)
{
	if (index < m_data.m_geometries.getCount())
		return m_data.m_geometries + index;

	return NULL;
}

unsigned int ThMesh::getGeometryCount(void) const
{
	return m_data.m_geometries.getCount();
}

const char* ThMesh::getName(void) const
{
	return m_name;
}

unsigned int ThMesh::getHash(void) const
{
	return m_hash;
}

ThMeshData& ThMesh::getData(void)
{
	return m_data;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThLight constructors ---------------------------------------------------------------------------

ThLight::ThLight(const char* name, IThEngine* engine):
	m_name(name),
	m_engine(engine)
{
	m_hash    = m_name.hashNoCase();
	m_index   = 0;
	m_enabled = false;

	if (!m_indices)
	{
		unsigned int count;

		glGetIntegerv(GL_MAX_LIGHTS, (GLint*) &count);

		m_indices.allocate(count);
		m_current.allocate(count);

		while (count--)
			m_indices[count] = false;
	}
}

ThLight::~ThLight(void)
{
	disable();
}

// ThLight interface methods ----------------------------------------------------------------------

bool ThLight::enable(void)
{
	if (m_enabled)
		return true;

	if (!m_indices[m_index])
		m_indices[m_index] = m_enabled = true;
	else
	{
		for (unsigned int i = 0;  i < m_indices.getCount();  i++)
		{
			if (!m_indices[i])
			{
				m_indices[m_index = i] = m_enabled = true;
				break;
			}
		}
	}

	if (!m_enabled)
		return false;

	glEnable(GL_LIGHT0 + m_index);

	// load complete state into index slot
	{
		if (m_current[m_index].m_ambient != m_data.m_ambient)
		{
			glLightfv(GL_LIGHT0 + m_index, GL_AMBIENT, m_data.m_ambient);
			m_current[m_index].m_ambient = m_data.m_ambient;
		}

		if (m_current[m_index].m_diffuse != m_data.m_diffuse)
		{
			glLightfv(GL_LIGHT0 + m_index, GL_DIFFUSE, m_data.m_diffuse);
			m_current[m_index].m_diffuse = m_data.m_diffuse;
		}

		if (m_current[m_index].m_specular != m_data.m_specular)
		{
			glLightfv(GL_LIGHT0 + m_index, GL_SPECULAR, m_data.m_specular);
			m_current[m_index].m_specular = m_data.m_specular;
		}

		m_current[m_index].m_type = m_data.m_type;

		// NOTE: the cached position and direction cannot be trusted, as they depend on the call-time modelview transform.

		if (m_data.m_type == ThLightData::DIRECTIONAL)
		{
			ThVector4 data(m_data.m_direction.x, m_data.m_direction.y, m_data.m_direction.z, 0.f);
			glLightfv(GL_LIGHT0 + m_index, GL_POSITION, data);
		}
		else
		{
			ThVector4 data(m_data.m_position.x, m_data.m_position.y, m_data.m_position.z, 1.f);
			glLightfv(GL_LIGHT0 + m_index, GL_POSITION, data);
		}
	}

	return true;
}

void ThLight::disable(void)
{
	if (!m_enabled)
		return;

	glDisable(GL_LIGHT0 + m_index);

	m_indices[m_index] = m_enabled = false;
}

// ThLight interface attributes -------------------------------------------------------------------

bool ThLight::isEnabled(void) const
{
	return m_enabled;
}

const ThVector4& ThLight::getAmbientColor(void) const
{
	return m_data.m_ambient;
}

void ThLight::setAmbientColor(const ThVector4& color)
{
	m_data.m_ambient = color;

	if (m_enabled)
	{
		if (m_current[m_index].m_ambient != color)
		{
			glLightfv(GL_LIGHT0 + m_index, GL_AMBIENT, color);
			m_current[m_index].m_ambient = color;
		}
	}
}

const ThVector4& ThLight::getDiffuseColor(void) const
{
	return m_data.m_diffuse;
}

void ThLight::setDiffuseColor(const ThVector4& color)
{
	m_data.m_diffuse = color;

	if (m_enabled)
	{
		if (m_current[m_index].m_diffuse != color)
		{
			glLightfv(GL_LIGHT0 + m_index, GL_DIFFUSE, color);
			m_current[m_index].m_diffuse = color;
		}
	}
}

const ThVector4& ThLight::getSpecularColor(void) const
{
	return m_data.m_specular;
}

void ThLight::setSpecularColor(const ThVector4& color)
{
	m_data.m_specular = color;

	if (m_enabled)
	{
		if (m_current[m_index].m_specular != color)
		{
			glLightfv(GL_LIGHT0 + m_index, GL_SPECULAR, color);
			m_current[m_index].m_specular = color;
		}
	}
}

const ThVector3& ThLight::getPosition(void) const
{
	return m_data.m_position;
}

void ThLight::setPosition(const ThVector3& position)
{
	m_data.m_position = position;

	// NOTE: the cached position cannot be trusted, as it depends on the call-time modelview transform.

	if (m_enabled && m_data.m_type == ThLightData::POSITIONAL)
	{
		ThVector4 data(position.x, position.y, position.z, 1.f);
		glLightfv(GL_LIGHT0 + m_index, GL_POSITION, data);
	}
}

const ThVector3& ThLight::getDirection(void) const
{
	return m_data.m_direction;
}

void ThLight::setDirection(const ThVector3& direction)
{
	m_data.m_direction = direction;

	// NOTE: the cached direction cannot be trusted, as it depends on the call-time modelview transform.

	if (m_enabled && m_data.m_type == ThLightData::DIRECTIONAL)
	{
		ThVector4 data(direction.x, direction.y, direction.z, 0.f);
		glLightfv(GL_LIGHT0 + m_index, GL_POSITION, data);
	}
}

unsigned int ThLight::getType(void) const
{
	return m_data.m_type;
}

void ThLight::setType(unsigned int type)
{
	m_data.m_type = type;

	if (m_enabled)
	{
		if (m_current[m_index].m_type != type)
		{
			if (type == ThLightData::DIRECTIONAL)
				setDirection(m_data.m_direction);
			else
				setPosition(m_data.m_position);

			m_current[m_index].m_type = type;
		}
	}
}

const char* ThLight::getName(void) const
{
	return m_name;
}

unsigned int ThLight::getHash(void) const
{
	return m_hash;
}

ThLightData& ThLight::getData(void)
{
	return m_data;
}

// ThLight static data ----------------------------------------------------------------------------

ThBlock<ThLightData>	ThLight::m_current;
ThBlock<bool>					ThLight::m_indices;

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThMaterial constructors ------------------------------------------------------------------------

ThMaterial::ThMaterial(const char* name, IThEngine* engine):
	m_name(name),
	m_engine(engine)
{
	m_hash = m_name.hashNoCase();
}

ThMaterial::~ThMaterial(void)
{
}

// ThMaterial interface methods -------------------------------------------------------------------

void ThMaterial::apply(void)
{
	unsigned int visibleFace = GL_FRONT;

	if (m_data.m_culling)
	{
		// find non-culled faces

		if (m_data.m_culledFace == GL_FRONT)
			visibleFace = GL_BACK;
		else if (m_data.m_culledFace == GL_BACK)
			visibleFace = GL_FRONT;
		else
		{
			THASSERT(false, "What is your one purpose in life? To explode of course!");

			// TODO: something intelligent!
		}

		// set culled faces mode
		if (m_data.m_culledFace != m_current.m_culledFace)
		{
			glCullFace(m_data.m_culledFace);
			m_current.m_culledFace = m_data.m_culledFace;
		}
	}

	// set culling
	if (m_data.m_culling != m_current.m_culling)
	{
		if (m_data.m_culling)
			glEnable(GL_CULL_FACE);
		else
			glDisable(GL_CULL_FACE);

		m_current.m_culling = m_data.m_culling;
	}

	// set lighting
	if (m_data.m_lighting != m_current.m_lighting)
	{
		if (m_data.m_lighting)
			glEnable(GL_LIGHTING);
		else
			glDisable(GL_LIGHTING);

		m_current.m_lighting = m_data.m_lighting;
	}

	// set blending
	if (m_data.m_blending != m_current.m_blending)
	{
		if (m_data.m_blending)
			glEnable(GL_BLEND);
		else
			glDisable(GL_BLEND);

		m_current.m_blending = m_data.m_blending;
	}

	if (m_data.m_blending)
	{
		// set blend factors
		if (m_data.m_srcFactor != m_current.m_srcFactor || m_data.m_dstFactor != m_current.m_dstFactor)
		{
			glBlendFunc(m_data.m_srcFactor, m_data.m_dstFactor);
			m_current.m_srcFactor = m_data.m_srcFactor;
			m_current.m_dstFactor = m_data.m_dstFactor;
		}
	}
	
	if (m_data.m_shadeModel != m_current.m_shadeModel)
	{
		glShadeModel(m_data.m_shadeModel);
		m_current.m_shadeModel = m_data.m_shadeModel;
	}

	// set depth buffer usage
	if ((m_data.m_depthTesting || m_data.m_depthWriting) != (m_current.m_depthTesting || m_current.m_depthWriting))
	{
		if (m_data.m_depthTesting || m_data.m_depthWriting)
			glEnable(GL_DEPTH_TEST);
		else
			glDisable(GL_DEPTH_TEST);

		m_current.m_depthTesting = m_data.m_depthTesting;
		m_current.m_depthWriting = m_data.m_depthWriting;
	}

	if (m_data.m_depthTesting || m_data.m_depthWriting)
	{
		// set depth buffer writing
		if (m_data.m_depthWriting != m_current.m_depthWriting)
		{
			glDepthMask(m_data.m_depthWriting ? GL_TRUE : GL_FALSE);
		}

		if (m_data.m_depthTesting)
		{
			// set depth buffer function
			if (m_data.m_depthFunction != m_current.m_depthFunction)
			{
				glDepthFunc(m_data.m_depthFunction);
				m_current.m_depthFunction = m_data.m_depthFunction;
			}
		}
		else if (m_data.m_depthWriting)
		{
			// NOTE: special case; depth buffer filling

			// set specific depth buffer function
			const unsigned int depthFunction = GL_ALWAYS;

			if (m_current.m_depthFunction != depthFunction)
			{
				glDepthFunc(depthFunction);
				m_current.m_depthFunction = depthFunction;
			}
		}
	}

	if (m_data.m_lighting)
	{
		// set ambient material color
		if (m_data.m_ambientColor != m_current.m_ambientColor)
		{
			glMaterialfv(visibleFace, GL_AMBIENT, m_data.m_ambientColor);
			m_current.m_ambientColor = m_data.m_ambientColor;
		}

		// set diffuse material color
		if (m_data.m_diffuseColor != m_current.m_diffuseColor)
		{
			glMaterialfv(visibleFace, GL_DIFFUSE, m_data.m_diffuseColor);
			m_current.m_diffuseColor = m_data.m_diffuseColor;
		}

		// set specular material color
		if (m_data.m_specularColor != m_current.m_specularColor)
		{
			glMaterialfv(visibleFace, GL_SPECULAR, m_data.m_specularColor);
			m_current.m_specularColor = m_data.m_specularColor;
		}
	}

	const unsigned int modes[] = { GL_TEXTURE_GEN_S, GL_TEXTURE_GEN_T, GL_TEXTURE_GEN_R, GL_TEXTURE_GEN_Q };

	// set texturing
	if (m_data.m_textureName)
	{
		// enable texturing

		if (!m_current.m_textureName)
			glEnable(GL_TEXTURE_2D);

		// make texture active
		{
			IThTexture* texture = m_engine->getDisplay()->findTexture(m_data.m_textureHash);
			if (!texture)
				Error->display("Engine", "Unable to find texture '%s' for material '%s'.", m_data.m_textureName.getData(), m_name.getData());

			texture->apply();
		}

		m_current.m_textureName = m_data.m_textureName;
		m_current.m_textureHash = m_data.m_textureHash;

		for (unsigned int i = 0;  i < 4;  i++)
		{
			// set texture coordinate generation
			if (m_data.m_coordGen[i] != m_current.m_coordGen[i])
			{
				if (m_data.m_coordGen[i])
					glEnable(modes[i]);
				else
					glDisable(modes[i]);

				m_current.m_coordGen[i] = m_data.m_coordGen[i];
			}

			if (m_data.m_coordGen[i])
			{
				// set coordinate generation mode
				if (m_data.m_genModes[i] != m_current.m_genModes[i])
				{
					glTexGeni(indexToAxis(i), GL_TEXTURE_GEN_MODE, m_data.m_genModes[i]);
					m_current.m_genModes[i] = m_data.m_genModes[i];
				}

				if (m_data.m_genModes[i] == GL_OBJECT_LINEAR)
				{
					// set generation object plane
					if (m_data.m_objPlanes[i] != m_current.m_objPlanes[i])
					{
						glTexGenfv(indexToAxis(i), GL_OBJECT_PLANE, m_data.m_objPlanes[i]);
						m_current.m_objPlanes[i] = m_data.m_objPlanes[i];
					}
				}
				else if (m_data.m_genModes[i] == GL_EYE_LINEAR)
				{
					// set generation eye plane
					if (m_data.m_eyePlanes[i] != m_current.m_eyePlanes[i])
					{
						glTexGenfv(indexToAxis(i), GL_EYE_PLANE, m_data.m_eyePlanes[i]);
						m_current.m_eyePlanes[i] = m_data.m_eyePlanes[i];
					}
				}
			}
		}

		// set texture combine mode
		if (m_data.m_combineMode != m_current.m_combineMode)
		{
			glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, m_data.m_combineMode);
			m_current.m_combineMode = m_data.m_combineMode;
		}

		// set texture environment color
		if (m_data.m_combineColor != m_current.m_combineColor)
		{
			glTexEnvfv(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, m_data.m_combineColor);
			m_current.m_combineColor = m_data.m_combineColor;
		}
	}
	else
	{
		// disable texturing
		if (m_current.m_textureName)
		{
			glDisable(GL_TEXTURE_2D);
			m_current.m_textureName = NULL;
			m_current.m_textureHash = 0;
		}
	}
}

// ThMaterial interface attributes ----------------------------------------------------------------

bool ThMaterial::isCulling(void)
{
	return m_data.m_culling;
}

bool ThMaterial::isLighting(void)
{
	return m_data.m_lighting;
}

bool ThMaterial::isBlending(void)
{
	return m_data.m_blending;
}

bool ThMaterial::isDepthTesting(void)
{
	return m_data.m_depthTesting;
}

bool ThMaterial::isDepthWriting(void)
{
	return m_data.m_depthWriting;
}

void ThMaterial::setCulling(bool enabled)
{
	m_data.m_culling = enabled;
}

void ThMaterial::setLighting(bool enabled)
{
	m_data.m_lighting = enabled;
}

void ThMaterial::setBlending(bool enabled)
{
	m_data.m_blending = enabled;
}

unsigned int ThMaterial::getSrcBlend(void)
{
	return m_data.m_srcFactor;
}

void ThMaterial::setSrcBlend(unsigned int factor)
{
	m_data.m_srcFactor = factor;
}

unsigned int ThMaterial::getDstBlend(void)
{
	return m_data.m_dstFactor;
}

void ThMaterial::setDstBlend(unsigned int factor)
{
	m_data.m_dstFactor = factor;
}

void ThMaterial::setDepthTesting(bool enabled)
{
	m_data.m_depthTesting = enabled;
}

void ThMaterial::setDepthWriting(bool enabled)
{
	m_data.m_depthWriting = enabled;
}

unsigned int ThMaterial::getDepthFunc(void)
{
	return m_data.m_depthFunction;
}

void ThMaterial::setDepthFunc(unsigned int func)
{
	m_data.m_depthFunction = func;
}

const ThVector4& ThMaterial::getAmbientColor(void)
{
	return m_data.m_ambientColor;
}

void ThMaterial::setAmbientColor(const ThVector4& color)
{
	m_data.m_ambientColor = color;
}

const ThVector4& ThMaterial::getDiffuseColor(void)
{
	return m_data.m_diffuseColor;
}

void ThMaterial::setDiffuseColor(const ThVector4& color)
{
	m_data.m_diffuseColor = color;
}

const ThVector4& ThMaterial::getSpecularColor(void)
{
	return m_data.m_specularColor;
}

void ThMaterial::setSpecularColor(const ThVector4& color)
{
	m_data.m_specularColor = color;
}

unsigned int ThMaterial::getShadeModel(void)
{
	return m_data.m_shadeModel;
}

void ThMaterial::setShadeModel(unsigned int mode)
{
	m_data.m_shadeModel = mode;
}

unsigned int ThMaterial::getCulledFace(void)
{
	return m_data.m_culledFace;
}

void ThMaterial::setCulledFace(unsigned int face)
{
	m_data.m_culledFace = face;
}

unsigned int ThMaterial::getCombineMode(void)
{
	return m_data.m_combineMode;
}

void ThMaterial::setCombineMode(unsigned int mode)
{
	m_data.m_combineMode = mode;
}

const ThVector4& ThMaterial::getCombineColor(void)
{
	return m_data.m_combineColor;
}

void ThMaterial::setCombineColor(const ThVector4& color)
{
	m_data.m_combineColor = color;
}

bool ThMaterial::getCoordGen(unsigned int axis)
{
	return m_data.m_coordGen[axisToIndex(axis)];
}

void ThMaterial::setCoordGen(unsigned int axis, bool enabled)
{
	m_data.m_coordGen[axisToIndex(axis)] = enabled;
}

unsigned int ThMaterial::getCoordGenMode(unsigned int axis)
{
	return m_data.m_genModes[axisToIndex(axis)];
}

void ThMaterial::setCoordGenMode(unsigned int axis, unsigned int mode)
{
	m_data.m_genModes[axisToIndex(axis)] = mode;
}

const ThVector4& ThMaterial::getCoordObjPlane(unsigned int axis)
{
	return m_data.m_objPlanes[axisToIndex(axis)];
}

void ThMaterial::setCoordObjPlane(unsigned int axis, const ThVector4& plane)
{
	m_data.m_objPlanes[axisToIndex(axis)] = plane;
}

const ThVector4& ThMaterial::getCoordEyePlane(unsigned int axis)
{
	return m_data.m_eyePlanes[axisToIndex(axis)];
}

void ThMaterial::setCoordEyePlane(unsigned int axis, const ThVector4& plane)
{
	m_data.m_eyePlanes[axisToIndex(axis)] = plane;
}

const char* ThMaterial::getTextureName(void)
{
	return m_data.m_textureName;
}

void ThMaterial::setTextureName(const char* name)
{
	m_data.m_textureName = name;
	m_data.m_textureHash = ThString::hashNoCase(name);
}

const char* ThMaterial::getName(void) const
{
	return m_name;
}

unsigned int ThMaterial::getHash(void) const
{
	return m_hash;
}

ThMaterialData& ThMaterial::getData(void)
{
	return m_data;
}

// ThMaterial methods -----------------------------------------------------------------------------

unsigned int ThMaterial::axisToIndex(unsigned int axis) const
{
	switch (axis)
	{
		case GL_S:
			return 0;
		case GL_T:
			return 1;
		case GL_R:
			return 2;
		case GL_Q:
			return 3;
		default:
			THASSERT(false, "Invalid axis parameter detected.");
			return 0;
	}
}

unsigned int ThMaterial::indexToAxis(unsigned int index) const
{
	THASSERT(index < 4, "Invalid index parameter detected");

	const unsigned int axes[] = { GL_S, GL_T, GL_R, GL_Q };

	return axes[index];
}

// ThMaterial static data -------------------------------------------------------------------------

ThMaterialData ThMaterial::m_current;


///////////////////////////////////////////////////////////////////////////////////////////////////

// ThRenderQueue constructors ---------------------------------------------------------------------

ThRenderQueue::ThRenderQueue(ThEngine* engine):
	m_engine(engine)
{
}

ThRenderQueue::~ThRenderQueue(void)
{
}

// ThRenderQueue methods --------------------------------------------------------------------------

void ThRenderQueue::prepare(void)
{
	if (m_sortBuffer.getCount() < m_triangles.getCount())
		m_sortBuffer.allocate(m_triangles.getCount());

	unsigned int index = 0;

	for (ThIterator<Triangle> item(m_triangles);  item;  item.next())
		m_sortBuffer[index++] = item;

	// TODO: sort triangles!

	// TODO: sort geometries!
}

void ThRenderQueue::render(void)
{
}

void ThRenderQueue::reset(void)
{
	m_geometries.release();
	m_triangles.release();
	m_lights.release();
}

// ThRenderQueue object methods -------------------------------------------------------------------

void ThRenderQueue::addMesh(IThMesh* mesh, const ThMatrix4& transform)
{
	// add geometries
	{
		ThBlock<ThGeometry>& geometries = mesh->getData().m_geometries;

		unsigned int index = geometries.getCount();

		while (index--)
		{
			ThGeometry& geometry = geometries[index];

			if (!geometry.m_material->isBlending())
				addGeometry(&geometry, transform);
		}
	}

	// add triangles
	{
		ThBlock<ThTriangle*>& triangles = mesh->getData().m_transparent;

		unsigned int index = triangles.getCount();

		while (index--)
			addTriangle(triangles[index], transform);
	}
}

void ThRenderQueue::addGeometry(const ThGeometry* geometry, const ThMatrix4& transform)
{
	Geometry* item = new Geometry();
	item->m_geometry  = geometry;
	item->m_transform = transform;

	m_geometries.attachFirst(item);
}

void ThRenderQueue::addTriangle(const ThTriangle* triangle, const ThMatrix4& transform)
{
	Triangle* item = new Triangle();
	item->m_triangle  = triangle;
	item->m_transform = transform;

	m_triangles.attachFirst(item);
}

void ThRenderQueue::addLight(IThLight* light, const ThMatrix4& transform)
{
	Light* item = new Light();
	item->m_light     = light;
	item->m_transform = transform;

	m_lights.attachFirst(item);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThEngine constructors --------------------------------------------------------------------------

ThEngine::ThEngine(void)
{
	m_display = NULL;
}

ThEngine::~ThEngine(void)
{
}

// ThEngine methods -------------------------------------------------------------------------------

bool ThEngine::open(IThDisplay* display)
{
	close();
	
	unsigned int lightCount;
	
	glGetIntegerv(GL_MAX_LIGHTS, (GLint*) &lightCount);
	
	for (unsigned int i = 0;  i < lightCount;  i++)
	{
		ThVector4 color;

		color.set(1.f, 1.f, 1.f, 0.f);
		glLightfv(GL_LIGHT0 + i, GL_DIFFUSE, color);

		color.reset();
		glLightfv(GL_LIGHT0 + i, GL_SPECULAR, color);
	}
	
	// TODO: initialize all render states to default values!
	glBlendFunc(GL_ONE, GL_ONE);

	m_display = display;

	return true;
}

void ThEngine::close(void)
{
	m_meshes.release();
	m_lights.release();
	m_materials.release();
}

// ThEngine interface methods ---------------------------------------------------------------------

bool ThEngine::loadResources(const char* fileName)
{
	ThPtr<IThStream> file = Storage->openFile(fileName);
	if (!file)
		return false;

	return loadResources(file);
}

bool ThEngine::loadResources(IThStream* stream)
{
	Lib3dsIo* io = lib3ds_io_new(stream, lib3dsIoError, lib3dsIoSeek, lib3dsIoTell, lib3dsIoRead, lib3dsIoWrite);
	if (!io)
		return false;

	bool result = false;
	
	Lib3dsFile* file = lib3ds_file_new();
	if (file)
	{
		if (lib3ds_file_read(file, io))
		{
			result = loadResources(file);
		}
	}
	
	lib3ds_file_free(file);
	file = NULL;
	
	lib3ds_io_free(io);
	io = NULL;
	
	return true;
}

// ThEngine object methods ------------------------------------------------------------------------

IThMesh* ThEngine::createMesh(const char* name)
{
	if (IThMesh* mesh = findMesh(name))
		return mesh;

	ThMesh* mesh = new ThMesh(name, this);

	m_meshes.attachFirst(mesh);
	return mesh;
}

IThMesh* ThEngine::findMesh(const char* name)
{
	const unsigned int hash = ThString::hashNoCase(name);

	return findMesh(hash);
}

IThLight* ThEngine::createLight(const char* name)
{
	if (IThLight* light = findLight(name))
		return light;

	ThLight* light = new ThLight(name, this);

	m_lights.attachFirst(light);
	return light;
}

IThLight* ThEngine::findLight(const char* name)
{
	const unsigned int hash = ThString::hashNoCase(name);

	return findLight(hash);
}

IThMaterial* ThEngine::createMaterial(const char* name)
{
	if (IThMaterial* material = findMaterial(name))
		return material;

	ThMaterial* material = new ThMaterial(name, this);

	m_materials.attachFirst(material);
	return material;
}

IThMaterial* ThEngine::findMaterial(const char* name)
{
	const unsigned int hash = ThString::hashNoCase(name);

	return findMaterial(hash);
}

IThRenderQueue* ThEngine::createRenderQueue(void)
{
	return new ThRenderQueue(this);
}

// ThEngine attributes ----------------------------------------------------------------------------

IThDisplay* ThEngine::getDisplay(void)
{
	return m_display;
}

ThEngineNode* ThEngine::getRootNode(void)
{
	return &m_root;
}

// ThEngine methods -------------------------------------------------------------------------------

bool ThEngine::loadResources(Lib3dsFile* file)
{
	// load materials

	Lib3dsMaterial* extMaterial = file->materials;
	while (extMaterial)
	{
		if (!ThString::equalsNoCase(extMaterial->name, "default") && !ThString::equalsNoCase(extMaterial->name, "_hole_"))
		{
			IThMaterial* material = createMaterial(extMaterial->name);
			if (!material)
			{
				Error->display("Engine", "Unable to create material '%s'.", extMaterial->name);
				return false;
			}

			ThMaterialData& intMaterial = material->getData();

			convertVector4(intMaterial.m_ambientColor, extMaterial->ambient);
			convertVector4(intMaterial.m_diffuseColor, extMaterial->diffuse);
			convertVector4(intMaterial.m_specularColor, extMaterial->specular);

			if (extMaterial->two_sided)
				intMaterial.m_culling = false;
			else
			{
				intMaterial.m_culling    = true;
				intMaterial.m_culledFace = GL_BACK;
			}
			
			if (extMaterial->shading == LIB3DS_GOURAUD)
				intMaterial.m_shadeModel = GL_SMOOTH;
			else
				intMaterial.m_shadeModel = GL_FLAT;
			
			intMaterial.m_lighting     = true;
			intMaterial.m_depthWriting = true;
			intMaterial.m_depthTesting = true;
		}

		extMaterial = extMaterial->next;
	}

	// load meshes

	Lib3dsMesh* extMesh = file->meshes;
	while (extMesh)
	{
		IThMesh* mesh = createMesh(extMesh->name);
		if (!mesh)
		{
			Error->display("Engine", "Unable to create mesh '%s'.", extMesh->name);
			return false;
		}

		ThMeshData& intMesh = mesh->getData();

		// load vertex data, calculate bounding sphere center
		{
			intMesh.m_vertices.allocate(extMesh->points);

			for (unsigned int i = 0;  i < intMesh.m_vertices.getCount();  i++)
			{
				ThVertex& vertex = intMesh.m_vertices[i];

				convertVector3(vertex.m_vertex, extMesh->pointL[i].pos);

				if (extMesh->texels)
					convertVector2(vertex.m_texCoord, extMesh->texelL[i]);
				else
					vertex.m_texCoord.reset();
				
				vertex.m_normal.reset();

				intMesh.m_sphere.m_center += vertex.m_vertex;
			}

			intMesh.m_sphere.m_center *= 1.f / intMesh.m_vertices.getCount();
		}

		ThList<Geometry> geometries;

		// load face data
		{
			intMesh.m_triangles.allocate(extMesh->faces);

			for (unsigned int i = 0;  i < intMesh.m_triangles.getCount();  i++)
			{
				Lib3dsFace& face     = extMesh->faceL[i];
				ThTriangle& triangle = intMesh.m_triangles[i];

				triangle.m_center.reset();

				convertVector3(triangle.m_normal, face.normal);
				triangle.m_normal.normalize();

				for (unsigned int j = 0;  j < 3;  j++)
				{
					const unsigned int index = face.points[j];

					triangle.m_indices[j] = index;
					intMesh.m_vertices[index].m_normal += triangle.m_normal;
					triangle.m_center += intMesh.m_vertices[index].m_vertex / 3.f;
				}

				triangle.m_material = findMaterial(face.material);
				if (!triangle.m_material)
				{
					Error->display("Engine", "Polygon in mesh '%s' uses undefined material '%s'.", mesh->getName(), face.material);
					return false;
				}

				const unsigned int hash = ThString::hashNoCase(face.material);

				ThIterator<Geometry> geometry;

				for (geometry = geometries.iterate();  geometry;  geometry.next())
				{
					if (geometry->m_hash == hash)
					{
						geometry->m_count++;
						break;
					}
				}

				if (!geometry)
				{
					geometry = new Geometry();
					geometry->m_hash  = hash;
					geometry->m_count = 1;

					geometries.attachFirst(geometry);
				}
			}
		}

		// normalize vertex normals, calculate bounding sphere radius
		{
			float maxDiff = 0.f;

			for (unsigned int i = 0;  i < intMesh.m_vertices.getCount();  i++)
			{
				ThVertex& vertex = intMesh.m_vertices[i];

				vertex.m_normal.normalize();

				const float curDiff = (intMesh.m_sphere.m_center - vertex.m_vertex).lengthSquared();
				if (curDiff > maxDiff)
					maxDiff = curDiff;
			}

			intMesh.m_sphere.m_radius = sqrtf(maxDiff);
		}

		// create geometry
		{
			intMesh.m_geometries.allocate(geometries.getCount());

			for (unsigned int i = 0;  i < geometries.getCount();  i++)
			{
				Geometry* geometry = geometries.getAt(i);

				intMesh.m_geometries[i].m_material = findMaterial(geometry->m_hash);
				intMesh.m_geometries[i].m_mode     = GL_TRIANGLES;
				intMesh.m_geometries[i].m_triangles.allocate(geometry->m_count);
				intMesh.m_geometries[i].m_indices.allocate(geometry->m_count * 3);

				unsigned int index = 0;

				for (unsigned int j = 0;  j < intMesh.m_triangles.getCount();  j++)
				{
					if (intMesh.m_triangles[j].m_material->getHash() == geometry->m_hash)
					{
						intMesh.m_geometries[i].m_triangles[index / 3] = intMesh.m_triangles + j;
						intMesh.m_geometries[i].m_indices[index++] = intMesh.m_triangles[j].m_indices[0];
						intMesh.m_geometries[i].m_indices[index++] = intMesh.m_triangles[j].m_indices[1];
						intMesh.m_geometries[i].m_indices[index++] = intMesh.m_triangles[j].m_indices[2];
					}
				}
			}
		}

		extMesh = extMesh->next;
	}

	return true;
}

IThMesh* ThEngine::findMesh(unsigned int hash)
{
	for (ThIterator<ThMesh> mesh(m_meshes);  mesh;  mesh.next())
	{
		if (mesh->getHash() == hash)
			return mesh;
	}

	return NULL;
}

IThLight* ThEngine::findLight(unsigned int hash)
{
	for (ThIterator<ThLight> light(m_lights);  light;  light.next())
	{
		if (light->getHash() == hash)
			return light;
	}

	return NULL;
}

IThMaterial* ThEngine::findMaterial(unsigned int hash)
{
	for (ThIterator<ThMaterial> material(m_materials);  material;  material.next())
	{
		if (material->getHash() == hash)
			return material;
	}

	return NULL;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
