///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMemory.h>
#include <shared/ThMath.h>
#include <shared/ThString.h>
#include <shared/ThError.h>
#include <shared/ThScript.h>
#include <shared/ThStorage.h>
#include <shared/ThServer.h>
#include <shared/ThDisplay.h>
#include <shared/ThEffect.h>

#include <theresa/ThEffect.h>

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThEffectType constructors ----------------------------------------------------------------------

ThEffectType::ThEffectType(const char* name):
	m_name(name)
{
	m_hash = m_name.hashNoCase();
}

ThEffectType::~ThEffectType(void)
{
}

// ThEffectType attribute methods -----------------------------------------------------------------

bool ThEffectType::registerAttribute(const char* name, unsigned int id)
{
	if (translateAttribute(name) != THATTR_INVALID)
		return false;
		
	m_attributes.attachFirst(new Attribute(name, id));
	return true;
}

unsigned int ThEffectType::translateAttribute(const char* name)
{
	const unsigned int hash = ThString::hashNoCase(name);
	
	for (ThIterator<Attribute> attribute(m_attributes);  attribute;  attribute.next())
	{
		if (attribute->m_hash == hash)
			return attribute->m_id;
	}
	
	return THATTR_INVALID;
}

// ThEffectType attributes ------------------------------------------------------------------------

const char* ThEffectType::getName(void) const
{
	return m_name;
}

unsigned int ThEffectType::getHash(void) const
{
	return m_hash;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThEffectType::Attribute constructors -----------------------------------------------------------

ThEffectType::Attribute::Attribute(const char* name, unsigned int id):
	m_name(name),
	m_id(id)
{
	m_hash = m_name.hashNoCase();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThEffectObject constructors --------------------------------------------------------------------

ThEffectObject::ThEffectObject(ThEffectType* type, const char* name):
	m_type(type),
	m_name(name)
{
	m_hash = m_name.hashNoCase();
	
	m_lifeTime = 0.f;
	m_timeLeft = 0.f;
	m_timePast = 0.f;
}

// ThEffectObject callbacks -----------------------------------------------------------------------

bool ThEffectObject::update(float deltaTime)
{
	m_timePast += deltaTime;

	if (m_lifeTime)
	{
		m_timeLeft -= deltaTime;
		if (m_timeLeft <= 0.f)
		{
			m_timeLeft = 0.f;
			return false;
		}
	}
	
	return true;
}

bool ThEffectObject::render(IThCanvas* target)
{
	return true;
}

bool ThEffectObject::notify(unsigned int attributeID, const char* value)
{
	return true;
}

// ThEffectObject attributes ----------------------------------------------------------------------

float ThEffectObject::getLifeTime(void) const
{
	return m_lifeTime;
}

void ThEffectObject::setLifeTime(float lifeTime)
{
	m_lifeTime = lifeTime;
	m_timeLeft = lifeTime;
}

float ThEffectObject::getTimeLeft(void) const
{
	return m_timeLeft;
}

float ThEffectObject::getTimePast(void) const
{
	return m_timePast;
}

const char* ThEffectObject::getName(void) const
{
	return m_name;
}

unsigned int ThEffectObject::getHash(void) const
{
	return m_hash;
}

ThEffectType* ThEffectObject::getType(void) const
{
	return m_type;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThEffect constructors -------------------------------------------------------------------------

IThEffect::~IThEffect(void)
{
}

// IThEffect static methods -----------------------------------------------------------------------

IThEffect* IThEffect::createInstance(IThDisplay* display)
{
	ThPtr<ThEffect> object = new ThEffect();
	
	if (!object->open(display))
		return false;
		
	return object.detach();
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThEffect constructors --------------------------------------------------------------------------

ThEffect::ThEffect(void):
	ThServerObject(THID_EFFECT)
{
	m_display = NULL;
	m_time    = 0.f;
}

ThEffect::~ThEffect(void)
{
	close();
}

// ThEffect methods -------------------------------------------------------------------------------

bool ThEffect::open(IThDisplay* display)
{
	close();

	m_display = display;
	return true;
}

void ThEffect::close(void)
{
	m_objects.release();
	m_types.release();
	m_syntax.release();
	m_symbols.release();
	m_timeEvents.release();
	m_syncEvents.release();

	m_display = NULL;
	m_time    = 0.f;
}

// ThEffect interface object methods --------------------------------------------------------------

bool ThEffect::registerType(ThEffectType* type)
{
	if (findType(type->getName()))
		return false;
	
	m_types.attachFirst(type);
	return true;
}

ThEffectObject* ThEffect::createObject(const char* objectName, const char* typeName, unsigned int layerID)
{
	if (findObject(objectName))
		return NULL;
	
	ThEffectType* type = findType(typeName);
	if (!type)
		return NULL;
	
	ThPtr<ThEffectObject> object = type->createObject(objectName);
	if (!object)
		return NULL;

	if (layerID != THLAYER_INVALID)
	{
		if (m_display->registerObject(object, layerID))
			return false;
	}
	
	m_objects.attachFirst(object);
	return object.detach();
}

// ThEffect interface event methods ---------------------------------------------------------------

void ThEffect::registerCreateEvent(float time, const char* objectName, const char* typeName)
{
	Event* event = new Event(Event::CREATE, time);
	event->m_objectName = objectName;
	event->m_typeName   = typeName;
	
	registerTimeEvent(event);
}

void ThEffect::registerCreateEvent(unsigned char sync, const char* objectName, const char* typeName)
{
	Event* event = new Event(Event::CREATE, sync);
	event->m_objectName = objectName;
	event->m_typeName   = typeName;
	
	m_syncEvents.attachFirst(event);
}

void ThEffect::registerDeleteEvent(float time, const char* objectName)
{
	Event* event = new Event(Event::DELETE, time);	
	event->m_objectName = objectName;
	
	registerTimeEvent(event);
}

void ThEffect::registerDeleteEvent(unsigned char sync, const char* objectName)
{
	Event* event = new Event(Event::DELETE, sync);
	event->m_objectName = objectName;
	
	m_syncEvents.attachFirst(event);
}

void ThEffect::registerNotifyEvent(float time, const char* objectName, const char* attributeName, const char* value)
{
	Event* event = new Event(Event::NOTIFY, time);
	event->m_objectName     = objectName;
	event->m_attributeName  = attributeName;
	event->m_attributeValue = value;

	registerTimeEvent(event);
}

void ThEffect::registerNotifyEvent(unsigned char sync, const char* objectName, const char* attributeName, const char* value)
{
	Event* event = new Event(Event::NOTIFY, sync);
	event->m_objectName     = objectName;
	event->m_attributeName  = attributeName;
	event->m_attributeValue = value;

	m_syncEvents.attachFirst(event);
}

void ThEffect::registerStopEvent(float time)
{
	Event* event = new Event(Event::STOP, time);
	
	registerTimeEvent(event);
}

void ThEffect::registerStopEvent(unsigned char sync)
{
	Event* event = new Event(Event::STOP, sync);
	
	m_syncEvents.attachFirst(event);
}

// ThEffect methods -------------------------------------------------------------------------------

void ThEffect::registerTimeEvent(Event* event)
{
	unsigned int index = 0;
	
	for (ThIterator<Event> item(m_timeEvents);  item;  item.next())
	{
		if (item->m_time > event->m_time)
			break;
			
		index++;
	}
	
	m_timeEvents.attachAt(event, index);
}

void ThEffect::triggerEvent(Event* event)
{
	switch (event->m_type)
	{
		case Event::STOP:
		{
			sendMessage(THMSG_REQUEST_EXIT, THID_ANNOUNCE);
			break;
		}

		case Event::CREATE:
		{
			if (!createObject(event->m_objectName, event->m_typeName))
			{
				Error->display("Effect", "Unable to create effect '%s' of type '%s'.", event->m_objectName.getData(), event->m_typeName.getData());
				sendMessage(THMSG_REQUEST_EXIT, THID_ANNOUNCE);
			}

			break;
		}

		case Event::DELETE:
		{
			ThPtr<ThEffectObject> object = findObject(event->m_objectName);
			if (!object)
			{
				Error->display("Effect", "Unable to find effect '%s' for deletion.", event->m_objectName.getData());
				sendMessage(THMSG_REQUEST_EXIT, THID_ANNOUNCE);
				break;
			}

			object.release();
			break;
		}

		case Event::NOTIFY:
		{
			ThEffectObject* object = findObject(event->m_objectName);
			if (!object)
			{
				Error->display("Effect", "Unable to find effect '%s' for notification.", event->m_objectName.getData());
				sendMessage(THMSG_REQUEST_EXIT, THID_ANNOUNCE);
				break;
			}

			const unsigned int attributeID = object->getType()->translateAttribute(event->m_attributeName);
			if (attributeID == THATTR_INVALID)
			{
				Error->display("Effect", "Unable to find attribute '%s' in effect '%s'.", event->m_attributeName.getData(), event->m_objectName.getData());
				sendMessage(THMSG_REQUEST_EXIT, THID_ANNOUNCE);
				break;
			}

			if (!object->notify(attributeID, event->m_attributeValue))
			{
				delete object;
				object = NULL;
			}

			break;
		}
	}
}

ThEffectType* ThEffect::findType(const char* name)
{
	const unsigned int hash = ThString::hashNoCase(name);
	
	for (ThIterator<ThEffectType> type(m_types);  type;  type.next())
	{
		if (type->getHash() == hash)
			return type;
	}
	
	return NULL;
}

ThEffectObject* ThEffect::findObject(const char* name)
{
	const unsigned int hash = ThString::hashNoCase(name);
	
	for (ThIterator<ThEffectObject> object(m_objects);  object;  object.next())
	{
		if (object->getHash() == hash)
			return object;
	}
	
	return NULL;
}

// ThEffect callbacks -----------------------------------------------------------------------------

bool ThEffect::receive(const IThMessage* message)
{
	switch (message->getMessage())
	{
		case THMSG_MUSIC_SYNCH:
		{
			const unsigned char sync = *reinterpret_cast<const unsigned char*>(message->getData());

			for (ThIterator<Event> event(m_syncEvents);  event;  event.next())
			{
				if (event->m_sync == sync)
					triggerEvent(event);
			}

			break;
		}

		case THMSG_LOCAL_UPDATE:
		{
			m_time += *reinterpret_cast<const float*>(message->getData());

			for (ThIterator<Event> event(m_timeEvents);  event;  event.release())
			{
				if (event->m_time > m_time)
					break;

				triggerEvent(event);
			}

			break;
		}
	}
	
	return ThServerObject::receive(message);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThEffect::Event --------------------------------------------------------------------------------

ThEffect::Event::Event(unsigned int type, float time):
	m_type(type),
	m_time(time)
{
}

ThEffect::Event::Event(unsigned int type, unsigned char sync):
	m_type(type),
	m_sync(sync)
{
}

///////////////////////////////////////////////////////////////////////////////////////////////////
