///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>
#include <shared/ThMemory.h>
#include <shared/ThString.h>
#include <shared/ThError.h>
#include <shared/ThServer.h>
#include <shared/ThContext.h>

#if THERESA_WGL
	#include <theresa/WGL/ThDialog.h>
	#include <theresa/WGL/ThContext.h>
#elif THERESA_AGL
	#include <theresa/AGL/ThContext.h>
#elif THERESA_GLX
	#include <theresa/GLX/ThContext.h>
#endif

#if HAVE_MATH_H
	#include <math.h>
#endif

#if HAVE_LIMITS_H
	#include <limits.h>
#endif

#if HAVE_STDLIB_H
	#include <stdlib.h>
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////

unsigned int getClosestPower(unsigned int value, unsigned int maximum)
{
	unsigned int result;

	if (value > maximum)
		return maximum;

	result = value;

	if (result & (result - 1))
	{
		// value is not power of two

		unsigned int i;

		for (i = 0;  result & ~1;  i++)
			result >>= 1;

		result = 1 << i;
	}

	// maximum is not power of two
	if ((result << 1) > maximum)
		return result;

	// adjust to closest power within range
	if ((value - result) > ((result << 1) - value))
		result <<= 1;

	return result;
}

unsigned int getPixelSize(unsigned int type, unsigned int format)
{
	unsigned int size;

	switch (type)
	{
		case GL_UNSIGNED_BYTE:
			size = sizeof(unsigned char);
			break;
			
		case GL_BYTE:
			size = sizeof(char);
			break;
			
		case GL_UNSIGNED_SHORT:
			size = sizeof(unsigned short);
			break;
			
		case GL_SHORT:
			size = sizeof(short);
			break;
			
		case GL_UNSIGNED_INT:
			size = sizeof(unsigned int);
			break;

		case GL_INT:
			size = sizeof(int);
			break;
			
		case GL_FLOAT:
			size = sizeof(float);
			break;

		default:
			return 0;
	}

	size *= getPixelComponentCount(type, format);

	return size;
}

unsigned int getPixelComponentCount(unsigned int type, unsigned int format)
{
	switch (format)
	{
		case GL_RGB:
#if THERESA_VISUALSTUDIO
		case GL_BGR_EXT:
#else
		case GL_BGR:
#endif
			return 3;
		 
		case GL_RGBA:
#if THERESA_VISUALSTUDIO
		case GL_BGRA_EXT:
#else
		case GL_BGRA:
#endif
			return 4;

		case GL_LUMINANCE_ALPHA:
			return 2;
	}

	return 1;
}

const ThResolution* findExactResolution(const ThList<ThResolution>& resolutions, unsigned int width, unsigned int height, unsigned int depth)
{
	for (ThConstIterator<ThResolution> resolution(resolutions);  resolution;  resolution.next())
	{
		if (resolution->m_width == width && resolution->m_height == height)
		{
			if (depth)
			{
				if (resolution->m_depth == depth)
					return resolution;
			}
			else
			{
				if (resolution->m_depth == THERESA_DEFAULT_DEPTH)
					return resolution;
			}
		}
	}

	return NULL;
}

const ThResolution* findClosestResolution(const ThList<ThResolution>& resolutions, unsigned int width, unsigned int height, unsigned int depth)
{
	unsigned int areaSize = width * height;
	unsigned int minDiff = UINT_MAX;

	const ThResolution* closest = NULL;

	for (ThConstIterator<ThResolution> resolution(resolutions);  resolution;  resolution.next())
	{
		if (depth && resolution->m_depth != depth)
			continue;

		unsigned int currDiff = abs(resolution->m_width * resolution->m_height - areaSize);

		if (currDiff < minDiff)
		{
			minDiff = currDiff;
			closest = resolution;
		}
	}

	return closest;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThResolution constructors ----------------------------------------------------------------------

ThResolution::ThResolution(void):
	m_width(0),
	m_height(0),
	m_depth(0),
	m_windowed(false)
{
}

ThResolution::ThResolution(unsigned int width, unsigned int height, unsigned int depth, bool windowed):
	m_width(width),
	m_height(height),
	m_depth(depth),
	m_windowed(windowed)
{
}

// ThResolution methods ---------------------------------------------------------------------------

void ThResolution::reset(void)
{
	m_width    = 0;
	m_height   = 0;
	m_depth    = 0;
	m_windowed = false;
}

// ThResolution attributes ------------------------------------------------------------------------

void ThResolution::set(unsigned int width, unsigned int height, unsigned int depth, bool windowed)
{
	m_width    = width;
	m_height   = height;
	m_depth    = depth;
	m_windowed = windowed;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThContextMode constructors ---------------------------------------------------------------------

ThContextMode::ThContextMode(void):
	m_depthBits(0),
	m_stencilBits(0),
	m_textureBits(0)
{
}

ThContextMode::ThContextMode(const ThResolution& resolution, unsigned int depthBits, unsigned int stencilBits, unsigned int textureBits):
	m_resolution(resolution),
	m_depthBits(depthBits),
	m_stencilBits(stencilBits),
	m_textureBits(textureBits)
{
}

// ThContextMode methods --------------------------------------------------------------------------

void ThContextMode::reset(void)
{
	m_resolution.reset();
	m_depthBits   = 0;
	m_stencilBits = 0;
	m_textureBits = 0;
}

// ThContextMode attributes -----------------------------------------------------------------------

void ThContextMode::set(const ThResolution& resolution, unsigned int depthBits, unsigned int stencilBits, unsigned int textureBits)
{
	m_resolution  = resolution;
	m_depthBits   = depthBits;
	m_stencilBits = stencilBits;
	m_textureBits = textureBits;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// IThContext constructors ------------------------------------------------------------------------

IThContext::~IThContext(void)
{
}

// IThContext static methods ----------------------------------------------------------------------

IThContext* IThContext::createInstance(const ThContextMode& mode, unsigned int listenerID)
{
	ThPtr<ThContext> context = new ThContext();

	if (!context->open(mode, listenerID))
		return false;

	return context.detach();
}

///////////////////////////////////////////////////////////////////////////////////////////////////
