///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#include <shared/ThCore.h>

#include <theresa/AGL/ThCore.h>

#if HAVE_STDIO_H
	#include <stdio.h>
#endif

#if HAVE_STDLIB_H
	#include <stdlib.h>
#endif

#if HAVE_STRING_H
	#include <string.h>
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////

const char* getPlatformName(void)
{
	static char platformName[THERESA_BUFFER_SIZE];
	static char versionName[THERESA_BUFFER_SIZE];
	
	strcpy(platformName, "Mac OS X");
	
	long int version;

	if (Gestalt(gestaltSystemVersion, &version) == noErr)
	{
		sprintf(versionName, " %li%li.%li.%li", (version & 0xf000) >> 12, (version & 0x0f00) >> 8, (version & 0xf0) >> 4, version & 0x0f);
		strcat(platformName, versionName);
	}

	return platformName;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
#ifdef THERESA_DEBUG_BUILD
///////////////////////////////////////////////////////////////////////////////////////////////////

//! \note This function may not use \b any system code.
void writeDebugMessage(const char* message)
{
	fprintf(stderr, "%s", message);
}

//! \note This function may not use \b any system code.
bool assertExpression(const char* expression, const char* message, const char* fileName, unsigned int line)
{
	static char buffer[THERESA_BUFFER_SIZE];

	if (snprintf(buffer, sizeof(buffer), "Assertion failed!%s%sExpression: %s%sFile: %s%sLine: %u%s%s%s", THERESA_NEWLINE, THERESA_NEWLINE, expression, THERESA_NEWLINE, fileName, THERESA_NEWLINE, line, THERESA_NEWLINE, THERESA_NEWLINE, message) < 0)
		buffer[sizeof(buffer) - 1] = '\0';
		
	writeDebugMessage(buffer);

	return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
#endif /* THERESA_DEBUG_BUILD */
///////////////////////////////////////////////////////////////////////////////////////////////////

// ThCriticalSection constructors -----------------------------------------------------------------

ThCriticalSection::ThCriticalSection(void)
{
	m_region = NULL;
}

ThCriticalSection::~ThCriticalSection(void)
{
	close();
}

// ThCriticalSection methods ----------------------------------------------------------------------

bool ThCriticalSection::open(void)
{
	if (MPCreateCriticalRegion(&m_region) != noErr)
		return false;
		
	return true;
}

void ThCriticalSection::close(void)
{
	if (m_region)
	{
		MPDeleteCriticalRegion(m_region);
		m_region = NULL;
	}
}

// ThCriticalSection interface methods ------------------------------------------------------------

void ThCriticalSection::begin(void)
{
	MPEnterCriticalRegion(m_region, kDurationForever);
}

void ThCriticalSection::end(void)
{
	MPExitCriticalRegion(m_region);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThMutex constructors ---------------------------------------------------------------------------

ThMutex::ThMutex(void)
{
	m_region = NULL;
}

ThMutex::~ThMutex(void)
{
	close();
}

// ThMutex methods --------------------------------------------------------------------------------

bool ThMutex::open(void)
{
	if (MPCreateCriticalRegion(&m_region) != noErr)
		return false;
		
	return true;
}

void ThMutex::close(void)
{
	if (m_region)
	{
		MPDeleteCriticalRegion(m_region);
		m_region = NULL;
	}
}

// ThMutex interface methods ----------------------------------------------------------------------

bool ThMutex::request(float timeout)
{
	Duration duration;
	
	if (timeout)
		duration = (int) (timeout * 1000.f) * kDurationMillisecond;
	else
		duration = kDurationForever;

	if (MPEnterCriticalRegion(m_region, duration) == noErr)
		return true;

	return false;
}

void ThMutex::release(void)
{
	MPExitCriticalRegion(m_region);
}

///////////////////////////////////////////////////////////////////////////////////////////////////
