///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef SHARED_THSCRIPT_H
#define SHARED_THSCRIPT_H
///////////////////////////////////////////////////////////////////////////////////////////////////

enum
{
	THSYM_INVALID,

	THSYM_SYSTEM_BASE = 1000,

	THSYM_CUSTOM_BASE = 2000,
};

///////////////////////////////////////////////////////////////////////////////////////////////////

typedef ThList<ThToken> ThTokenList;

//-------------------------------------------------------------------------------------------------

class ThToken : public ThListItem<ThToken>
{
public:
// constructors
																		ThToken(unsigned int symbol, unsigned int line);
// data
	unsigned int											m_symbol;
	unsigned int											m_line;
	ThString													m_data;
	ThTokenList												m_children;
};

///////////////////////////////////////////////////////////////////////////////////////////////////

/*! \class IThSymbolTable
 *	Symbol table interface. Represents a generic symbol table.
 */
/*! \fn bool IThSymbolTable::registerSymbol(unsigned int id, const char* name)
 *	Registers a new symbol in the table.
 *	\param id The numeric ID of the symbol.
 *	\param name The human-readable name of the symbol.
 *	\return \c true if symbol is unique, or \c false if \c id or \c name is duplicate.
 *	\remarks Zero is not a valid numeric ID.
 */
/*!	\fn const char* IThSymbolTable::translateSymbol(unsigned int id) const
 *	Translates a numeric ID to the corresponding human-readable name.
 *	\param id The ID of the requested symbol.
 *	\return The human-readable name of the requested symbol, or \c NULL if no such symbol exists.
 *	\remarks Zero is not a valid numeric ID.
 */
/*!	\fn unsigned int IThSymbolTable::translateSymbol(const char* name) const
 *	Translates a human-readable name to the corresponding numeric ID.
 *	\param name The human-readable name of the requested symbol.
 *	\return The numeric ID of the requested symbol, or zero if no such symbol exists.
 */
/*!	\fn IThSymbolTable* IThSymbolTable::createInstance(void)
 *	Creates and returns an instance of a symbol table.
 *	\return The newly created symbol table object.
 */
class IThSymbolTable
{
public:
// constructors
	virtual														~IThSymbolTable(void);
// methods
	virtual bool											registerSymbol(unsigned int id, const char* name) = 0;
	virtual const char*								translateSymbol(unsigned int id) const = 0;
	virtual unsigned int							translateSymbol(const char* name) const = 0;
// static methods
	static IThSymbolTable*						createInstance(void);
};

//-------------------------------------------------------------------------------------------------

class IThTokenizer
{
public:
// constructors
	virtual														~IThTokenizer(void);
// methods
	virtual bool											registerPattern(unsigned int list, unsigned int symbol, const char* data, const char* substitute = NULL, unsigned int branchList = THSYM_INVALID) = 0;
	virtual bool											tokenize(ThTokenList& target, unsigned int root, const char* source) = 0;
// attributes
	virtual const IThSymbolTable*			getSymbolTable(void) const = 0;
// static methods
	static IThTokenizer*							createInstance(IThSymbolTable* symbolTable);
};

//-------------------------------------------------------------------------------------------------

class IThAnalyzer
{
public:
// constructors
	virtual														~IThAnalyzer(void);
// methods
	virtual bool											registerTerminator(unsigned int symbol) = 0;
	virtual bool											registerBalancedPair(unsigned int openingSymbol, unsigned int closingSymbol) = 0;
	virtual bool											registerGrammar(unsigned int list, unsigned int symbol, const char* data, unsigned int level = 0) = 0;
	virtual bool											analyze(ThTokenList& target, unsigned int root, const ThTokenList& source) = 0;
// attributes
	virtual const IThSymbolTable*			getSymbolTable(void) const = 0;
// static methods
	static IThAnalyzer*								createInstance(IThSymbolTable* symbolTable);
};

//-------------------------------------------------------------------------------------------------

class IThSyntax
{
public:
// constructors
	virtual														~IThSyntax(void);
// attributes
	virtual IThSymbolTable*						getSymbolTable(void) = 0;
	virtual IThTokenizer*							getTokenizer(void) = 0;
	virtual IThAnalyzer*							getAnalyzer(void) = 0;
// static methods
	static IThSyntax*									createInstance(IThSymbolTable* symbolTable, const char* script);
};

///////////////////////////////////////////////////////////////////////////////////////////////////
#endif /* SHARED_THSCRIPT_H */
///////////////////////////////////////////////////////////////////////////////////////////////////
