///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#if HAVE_STDLIB_H
	#include <stdlib.h>
#endif

#if !THERESA_VISUALSTUDIO
	#define cosf(x) ((float)cos(x))
	#define sinf(x) ((float)sin(x))
	#define sqrtf(x) ((float)sqrt(x))
	#define acosf(x) ((float)acos(x))
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThRange constructors ---------------------------------------------------------------------------

template <typename T>
inline ThRange<T>::ThRange(void):
	m_min(0),
	m_max(0)
{
}

template <typename T>
inline ThRange<T>::ThRange(T min, T max):
	m_min(min),
	m_max(max)
{
}

// ThRange methods --------------------------------------------------------------------------------

template <typename T>
inline T ThRange<T>::generate(void) const
{
	return m_min + ((m_max - m_min) * (T) rand()) / (T) RAND_MAX;
}

template <typename T>
inline void ThRange<T>::reset(void)
{
	m_min = 0;
	m_max = 0;
}

// ThRange attributes -----------------------------------------------------------------------------

template <typename T>
inline void ThRange<T>::set(T min, T max)
{
	m_min = min;
	m_max = max;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThSpace constructors ---------------------------------------------------------------------------

template <typename T, unsigned int N>
inline ThSpace<T,N>::ThSpace(void)
{
	reset();
}

// ThSpace methods --------------------------------------------------------------------------------

template <typename T, unsigned int N>
inline void ThSpace<T,N>::generate(T* target) const
{
	for (unsigned int i = 0;  i < N;  i++)
		target[i] = m_min[i] + ((m_max[i] - m_min[i]) * (T) rand()) / (T) RAND_MAX;
}

template <typename T, unsigned int N>
inline void ThSpace<T,N>::reset(void)
{
	for (unsigned int i = 0;  i < N;  i++)
	{
		m_min[i] = 0;
		m_max[i] = 0;
	}
}

// ThSpace attributes -----------------------------------------------------------------------------

template <typename T, unsigned int N>
inline void ThSpace<T,N>::set(unsigned int index, T min, T max)
{
	THASSERT(index < N, "Invalid index when accessing space!");

	m_min[index] = min;
	m_max[index] = max;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThVector2 constructors -------------------------------------------------------------------------

inline ThVector2::ThVector2(void):
	x(0.f),
	y(0.f)
{
}

inline ThVector2::ThVector2(float sx, float sy):
	x(sx),
	y(sy)
{
}

// ThVector2 methods ------------------------------------------------------------------------------

inline float ThVector2::length(void) const
{
	return (float) sqrtf(float(x * x + y * y));
}

inline float ThVector2::lengthSquared(void) const
{
	return x * x + y * y;
}

inline float ThVector2::dotProduct(const ThVector2& vector) const
{
	return x * x + y * y;
}

inline void ThVector2::scaleTo(float len)
{
	const float scale = len / length();

	x *= scale;
	y *= scale;
}

inline void ThVector2::normalize(void)
{
	const float scale = 1.f / length();

	x *= scale;
	y *= scale;
}

inline void ThVector2::reset(void)
{
	x = 0.f;
	y = 0.f;
}

// ThVector2 operators ----------------------------------------------------------------------------

inline ThVector2::operator float* (void)
{
	return &(x);
}

inline ThVector2::operator const float* (void) const
{
	return &(x);
}

inline ThVector2 ThVector2::operator - (void) const
{
	return ThVector2(-x, -y);
}

inline ThVector2 ThVector2::operator + (float value) const
{
	return ThVector2(x + value, y + value);
}

inline ThVector2 ThVector2::operator - (float value) const
{
	return ThVector2(x - value, y - value);
}

inline ThVector2 ThVector2::operator * (float value) const
{
	return ThVector2(x * value, y * value);
}

inline ThVector2 ThVector2::operator / (float value) const
{
	return ThVector2(x / value, y / value);
}

inline ThVector2 ThVector2::operator += (float value)
{
	x += value;
	y += value;
	return *this;
}

inline ThVector2& ThVector2::operator -= (float value)
{
	x -= value;
	y -= value;
	return *this;
}

inline ThVector2& ThVector2::operator *= (float value)
{
	x *= value;
	y *= value;
	return *this;
}

inline ThVector2& ThVector2::operator /= (float value)
{
	x /= value;
	y /= value;
	return *this;
}

inline ThVector2 ThVector2::operator + (const ThVector2& vector) const
{
	return ThVector2(x + vector.x, y + vector.y);
}

inline ThVector2 ThVector2::operator - (const ThVector2& vector) const
{
	return ThVector2(x - vector.x, y - vector.y);
}

inline ThVector2 ThVector2::operator * (const ThVector2& vector) const
{
	return ThVector2(x * vector.x, y * vector.y);
}

inline ThVector2 ThVector2::operator / (const ThVector2& vector) const
{
	return ThVector2(x / vector.x, y / vector.y);
}

inline ThVector2 ThVector2::operator += (const ThVector2& vector)
{
	x += vector.x;
	y += vector.y;
	return *this;
}

inline ThVector2& ThVector2::operator -= (const ThVector2& vector)
{
	x -= vector.x;
	y -= vector.y;
	return *this;
}

inline ThVector2& ThVector2::operator *= (const ThVector2& vector)
{
	x *= vector.x;
	y *= vector.y;
	return *this;
}

inline ThVector2& ThVector2::operator /= (const ThVector2& vector)
{
	x /= vector.x;
	y /= vector.y;
	return *this;
}

inline bool ThVector2::operator == (const ThVector2& vector) const
{
	return x == vector.x && y == vector.y;
}

inline bool ThVector2::operator != (const ThVector2& vector) const
{
	return x != vector.x || y != vector.y;
}

// ThVector2 attributes ---------------------------------------------------------------------------

inline void ThVector2::set(float sx, float sy)
{
	x = sx;
	y = sy;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThVector3 constructors -------------------------------------------------------------------------

inline ThVector3::ThVector3(void):
	x(0.f),
	y(0.f),
	z(0.f)
{
}

inline ThVector3::ThVector3(float sx, float sy, float sz):
	x(sx),
	y(sy),
	z(sz)
{
}

// ThVector3 methods ------------------------------------------------------------------------------

inline float ThVector3::length(void) const
{
	return sqrtf(x * x + y * y + z * z);
}

inline float ThVector3::lengthSquared(void) const
{
	return x * x + y * y + z * z;
}

inline float ThVector3::dotProduct(const ThVector3& vector) const
{
	return x * vector.x + y * vector.y + z * vector.z;
}

inline ThVector3 ThVector3::crossProduct(const ThVector3& vector) const
{
	return ThVector3(vector.z * y - vector.y * z, vector.x * z - vector.z * x, vector.y * x - vector.x * y);
}

inline void ThVector3::scaleTo(float len)
{
	const float scale = len / length();

	x *= scale;
	y *= scale;
	z *= scale;
}

inline void ThVector3::normalize(void)
{
	const float scale = 1.f / length();

	x *= scale;
	y *= scale;
	z *= scale;
}

inline void ThVector3::reset(void)
{
	x = 0.f;
	y = 0.f;
	z = 0.f;
}

// ThVector3 operators ----------------------------------------------------------------------------

inline ThVector3::operator float* (void)
{
	return &(x);
}

inline ThVector3::operator const float* (void) const
{
	return &(x);
}

inline ThVector3 ThVector3::operator - (void) const
{
	return ThVector3(-x, -y, -z);
}

inline ThVector3 ThVector3::operator + (float value) const
{
	return ThVector3(x + value, y + value, z + value);
}

inline ThVector3 ThVector3::operator - (float value) const
{
	return ThVector3(x - value, y - value, z - value);
}

inline ThVector3 ThVector3::operator * (float value) const
{
	return ThVector3(x * value, y * value, z * value);
}

inline ThVector3 ThVector3::operator / (float value) const
{
	return ThVector3(x / value, y / value, z / value);
}

inline ThVector3 ThVector3::operator += (float value)
{
	x += value;
	y += value;
	z += value;
	return *this;
}

inline ThVector3& ThVector3::operator -= (float value)
{
	x -= value;
	y -= value;
	z -= value;
	return *this;
}

inline ThVector3& ThVector3::operator *= (float value)
{
	x *= value;
	y *= value;
	z *= value;
	return *this;
}

inline ThVector3& ThVector3::operator /= (float value)
{
	x /= value;
	y /= value;
	z /= value;
	return *this;
}

inline ThVector3 ThVector3::operator + (const ThVector3& vector) const
{
	return ThVector3(x + vector.x, y + vector.y, z + vector.z);
}

inline ThVector3 ThVector3::operator - (const ThVector3& vector) const
{
	return ThVector3(x - vector.x, y - vector.y, z - vector.z);
}

inline ThVector3 ThVector3::operator * (const ThVector3& vector) const
{
	return ThVector3(x * vector.x, y * vector.y, z * vector.z);
}

inline ThVector3 ThVector3::operator / (const ThVector3& vector) const
{
	return ThVector3(x / vector.x, y / vector.y, z / vector.z);
}

inline ThVector3 ThVector3::operator += (const ThVector3& vector)
{
	x += vector.x;
	y += vector.y;
	z += vector.z;
	return *this;
}

inline ThVector3& ThVector3::operator -= (const ThVector3& vector)
{
	x -= vector.x;
	y -= vector.y;
	z -= vector.z;
	return *this;
}

inline ThVector3& ThVector3::operator *= (const ThVector3& vector)
{
	x *= vector.x;
	y *= vector.y;
	z *= vector.z;
	return *this;
}

inline ThVector3& ThVector3::operator /= (const ThVector3& vector)
{
	x /= vector.x;
	y /= vector.y;
	z /= vector.z;
	return *this;
}

inline bool ThVector3::operator == (const ThVector3& vector) const
{
	return x == vector.x && y == vector.y && z == vector.z;
}

inline bool ThVector3::operator != (const ThVector3& vector) const
{
	return x != vector.x || y != vector.y || z != vector.z;
}

// ThVector3 attributes ---------------------------------------------------------------------------

inline void ThVector3::set(float sx, float sy, float sz)
{
	x = sx;
	y = sy;
	z = sz;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThVector4 constructors -------------------------------------------------------------------------

inline ThVector4::ThVector4(void):
	x(0),
	y(0),
	z(0),
	w(0)
{
}

inline ThVector4::ThVector4(float sx, float sy, float sz, float sw):
	x(sx),
	y(sy),
	z(sz),
	w(sw)
{
}

// ThVector4 methods ------------------------------------------------------------------------------

inline float ThVector4::length(void) const
{
	return sqrtf(x * x + y * y + z * z + w * w);
}

inline float ThVector4::lengthSquared(void) const
{
	return x * x + y * y + z * z + w * w;
}

inline void ThVector4::scaleTo(float len)
{
	const float scale = len / length();

	x *= scale;
	y *= scale;
	z *= scale;
	w *= scale;
}

inline void ThVector4::normalize(void)
{
	const float scale = 1.f / length();

	x *= scale;
	y *= scale;
	z *= scale;
	w *= scale;
}

inline void ThVector4::reset(void)
{
	x = 0.f;
	y = 0.f;
	z = 0.f;
	w = 0.f;
}

// ThVector4 operators ----------------------------------------------------------------------------

inline ThVector4::operator float* (void)
{
	return &(x);
}

inline ThVector4::operator const float* (void) const
{
	return &(x);
}

inline ThVector4 ThVector4::operator - (void) const
{
	return ThVector4(-x, -y, -z, -w);
}

inline ThVector4 ThVector4::operator + (float value) const
{
	return ThVector4(x + value, y + value, z + value, w + value);
}

inline ThVector4 ThVector4::operator - (float value) const
{
	return ThVector4(x - value, y - value, z - value, w - value);
}

inline ThVector4 ThVector4::operator * (float value) const
{
	return ThVector4(x * value, y * value, z * value, w * value);
}

inline ThVector4 ThVector4::operator / (float value) const
{
	value = 1.f / value;

	return ThVector4(x * value, y * value, z * value, w * value);
}

inline ThVector4 ThVector4::operator += (float value)
{
	x += value;
	y += value;
	z += value;
	w += value;
	return *this;
}

inline ThVector4& ThVector4::operator -= (float value)
{
	x -= value;
	y -= value;
	z -= value;
	w -= value;
	return *this;
}

inline ThVector4& ThVector4::operator *= (float value)
{
	x *= value;
	y *= value;
	z *= value;
	w *= value;
	return *this;
}

inline ThVector4& ThVector4::operator /= (float value)
{
	const float scale = 1.f / value;

	x *= scale;
	y *= scale;
	z *= scale;
	w *= scale;
	return *this;
}

inline ThVector4 ThVector4::operator + (const ThVector4& vector) const
{
	return ThVector4(x + vector.x, y + vector.y, z + vector.z, w + vector.w);
}

inline ThVector4 ThVector4::operator - (const ThVector4& vector) const
{
	return ThVector4(x - vector.x, y - vector.y, z - vector.z, w - vector.w);
}

inline ThVector4 ThVector4::operator * (const ThVector4& vector) const
{
	return ThVector4(x * vector.x, y * vector.y, z * vector.z, w * vector.w);
}

inline ThVector4 ThVector4::operator / (const ThVector4& vector) const
{
	return ThVector4(x / vector.x, y / vector.y, z / vector.z, w / vector.w);
}

inline ThVector4 ThVector4::operator += (const ThVector4& vector)
{
	x += vector.x;
	y += vector.y;
	z += vector.z;
	w += vector.w;
	return *this;
}

inline ThVector4& ThVector4::operator -= (const ThVector4& vector)
{
	x -= vector.x;
	y -= vector.y;
	z -= vector.z;
	w -= vector.w;
	return *this;
}

inline ThVector4& ThVector4::operator *= (const ThVector4& vector)
{
	x *= vector.x;
	y *= vector.y;
	z *= vector.z;
	w *= vector.w;
	return *this;
}

inline ThVector4& ThVector4::operator /= (const ThVector4& vector)
{
	x /= vector.x;
	y /= vector.y;
	z /= vector.z;
	w /= vector.w;
	return *this;
}

inline bool ThVector4::operator == (const ThVector4& vector) const
{
	return x == vector.x && y == vector.y && z == vector.z && w == vector.w;
}

inline bool ThVector4::operator != (const ThVector4& vector) const
{
	return x != vector.x || y != vector.y || z != vector.z || w != vector.w;
}

// ThVector4 attributes ---------------------------------------------------------------------------

inline void ThVector4::set(float sx, float sy, float sz, float sw)
{
	x = sx;
	y = sy;
	z = sz;
	w = sw;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThQuaternion constructors ----------------------------------------------------------------------

inline ThQuaternion::ThQuaternion(void)
{
}

inline ThQuaternion::ThQuaternion(float sw, float sx, float sy, float sz):
	w(sw),
	x(sx),
	y(sy),
	z(sz)
{
}

// ThQuaternion methods ---------------------------------------------------------------------------

inline void ThQuaternion::invert(void)
{
	x = -x;
	y = -y;
	z = -z;
}

inline void ThQuaternion::normalize(void)
{
	const float square = w * w + x * x + y * y + z * z;

	if (square * square != square)
	{
		const float scale = 1.f / sqrtf(square);

		w *= scale;
		x *= scale;
		y *= scale;
		z *= scale;
	}
}

inline void ThQuaternion::reset(void)
{
	w = 1.f;
	x = 0.f;
	y = 0.f;
	z = 0.f;
}

// ThQuaternion vector methods --------------------------------------------------------------------

inline void ThQuaternion::rotateVector(ThVector3& vector) const
{
	ThMatrix3 matrix;

	matrix.setQuatRotation(*this);
	matrix.rotateVector(vector);
}

// ThQuaternion operators -------------------------------------------------------------------------

inline ThQuaternion::operator float* (void)
{
	return &(w);
}

inline ThQuaternion::operator const float* (void) const
{
	return &(w);
}

inline ThQuaternion ThQuaternion::operator + (const ThQuaternion& quat) const
{
	return ThQuaternion(w + quat.w, x + quat.x, y + quat.y, z + quat.z);
}

inline ThQuaternion ThQuaternion::operator * (const ThQuaternion& quat) const
{
	ThQuaternion result;
	
	result.w = w * quat.w - x * quat.x - y * quat.y - z * quat.z;
	result.x = w * quat.x + x * quat.w + y * quat.z - z * quat.y;
	result.y = w * quat.y + y * quat.w + z * quat.x - x * quat.z;
	result.z = w * quat.z + z * quat.w + x * quat.y - y * quat.x;

	return result;
}

inline ThQuaternion& ThQuaternion::operator += (const ThQuaternion& quat)
{
	w += quat.w;
	x += quat.x;
	y += quat.y;
	z += quat.z;

	return *this;
}

inline ThQuaternion& ThQuaternion::operator *= (const ThQuaternion& quat)
{
	ThQuaternion temp;
	
	temp.w = w * quat.w - x * quat.x - y * quat.y - z * quat.z;
	temp.x = w * quat.x + x * quat.w + y * quat.z - z * quat.y;
	temp.y = w * quat.y + y * quat.w + z * quat.x - x * quat.z;
	temp.z = w * quat.z + z * quat.w + x * quat.y - y * quat.x;

	return operator = (temp);
}

inline bool ThQuaternion::operator == (const ThQuaternion& quat) const
{
	return w == quat.w && x == quat.x && y == quat.y && z == quat.z;
}

inline bool ThQuaternion::operator != (const ThQuaternion& quat) const
{
	return w != quat.w || x != quat.x || y != quat.y || z != quat.z;
}

// ThQuaternion attributes ------------------------------------------------------------------------

inline void ThQuaternion::set(float sw, float sx, float sy, float sz)
{
	w = sw;
	x = sx;
	y = sy;
	z = sz;
}

inline void ThQuaternion::setEulerRotation(const ThVector3& angles)
{
	// TODO: implement!
	/*
	const float cosX = cosf(angles.x * 0.5f);
	const float cosY = cosf(angles.y * 0.5f);
	const float cosZ = cosf(angles.z * 0.5f);

	const float sinX = sinf(angles.x * 0.5f);
	const float sinY = sinf(angles.y * 0.5f);
	const float sinZ = sinf(angles.z * 0.5f);

	w = cosX * cosY * cosZ + sinX * sinY * sinZ;
	x = sinX * cosY * cosZ - cosX * sinY * sinZ;
	y = cosX * sinY * cosZ + sinX * cosY * sinZ;
	z = cosX * cosY * sinZ - sinX * sinY * cosZ;
	*/
}

inline void ThQuaternion::setMatrixRotation(const ThMatrix3& matrix)
{
	const float trace = matrix.x.x + matrix.y.y + matrix.z.z;

	if (trace > 0.f)
	{
		float root = sqrtf(trace + 1.f);

		w = root * 0.5f;

		root = 0.5f / root;

		x = (matrix.y.z - matrix.z.y) * root;
		y = (matrix.z.x - matrix.x.z) * root;
		z = (matrix.x.y - matrix.y.x) * root;
	}
	else
	{		
		float q[4];
		int i, j, k;

		int next[3] = { 1, 2, 0 };

		i = 0;

		if (matrix.y.y > matrix.x.x)
			i = 1;

		if (matrix.z.z > matrix(i, i))
			i = 2;

		j = next[i];
		k = next[j];

		float root = sqrtf((matrix(i, i) - (matrix(j, j) + matrix(k, k))) + 1.f);

		q[i] = root * 0.5f;

		if (root != 0.f)
			root = 0.5f / root;

		q[3] = (matrix(j, k) - matrix(k, j)) * root;
		q[j] = (matrix(i, j) + matrix(j, i)) * root;
		q[k] = (matrix(i, k) + matrix(k, i)) * root;

		w = q[3];
		x = q[0];
		y = q[1];
		z = q[2];
	}
}

inline void ThQuaternion::setVectorRotation(const ThVector3& vector, float angle)
{
	ThMatrix3 matrix;

	matrix.setVectorRotation(vector, angle);
	setMatrixRotation(matrix);
}

inline void ThQuaternion::getAxisRotation(ThVector3& axis, float& angle) const
{
	const float square = w * w + x * x + y * y + z * z;

	if (square)
	{
		axis.x = x * square;
		axis.y = y * square;
		axis.z = z * square;

		angle = 2.f * acosf(w);
	}
	else
	{
		axis.x = 1.f;
		axis.y = 0.f;
		axis.z = 0.f;

		angle = 0.f;
	}
}

inline void ThQuaternion::setAxisRotation(const ThVector3& axis, float angle)
{
	const float halfAngle = angle * 0.5f;

	w = cosf(halfAngle);
	x = axis.x * sinf(halfAngle);
	y = axis.y * sinf(halfAngle);
	z = axis.z * sinf(halfAngle);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThMatrix3 constructors -------------------------------------------------------------------------

inline ThMatrix3::ThMatrix3(void)
{
}

inline ThMatrix3::ThMatrix3(const ThVector3& sx, const ThVector3& sy, const ThVector3& sz):
	x(sx),
	y(sy),
	z(sz)
{
}

// ThMatrix3 methods ------------------------------------------------------------------------------

inline void ThMatrix3::invert(void)
{
	// TODO: implement!
}

inline void ThMatrix3::transpose(void)
{
	float temp;

	temp = x.y; x.y = y.x; y.x = temp;
	temp = x.z; x.z = z.x; z.x = temp;
	temp = y.z; y.z = z.y; z.y = temp;
}

inline void ThMatrix3::normalize(void)
{
	// TODO: implement!
}

inline void ThMatrix3::concatenate(const ThMatrix3& matrix)
{
	ThMatrix3 result;

  result.x.x = x.x * matrix.x.x + y.x * matrix.x.y + z.x * matrix.x.z;
  result.y.x = x.x * matrix.y.x + y.x * matrix.y.y + z.x * matrix.y.z;
  result.z.x = x.x * matrix.z.x + y.x * matrix.z.y + z.x * matrix.z.z;

  result.x.y = x.y * matrix.x.x + y.y * matrix.x.y + z.y * matrix.x.z;
  result.y.y = x.y * matrix.y.x + y.y * matrix.y.y + z.y * matrix.y.z;
  result.z.y = x.y * matrix.z.x + y.y * matrix.z.y + z.y * matrix.z.z;

  result.x.z = x.z * matrix.x.x + y.z * matrix.x.y + z.z * matrix.x.z;
  result.y.z = x.z * matrix.y.x + y.z * matrix.y.y + z.z * matrix.y.z;
  result.z.z = x.z * matrix.z.x + y.z * matrix.z.y + z.z * matrix.z.z;

	operator = (result);
}

inline void ThMatrix3::reset(void)
{
	x.set(1.f, 0.f, 0.f);
	y.set(0.f, 1.f, 0.f);
	z.set(0.f, 0.f, 1.f);
}

// ThMatrix3 operators ----------------------------------------------------------------------------

inline ThMatrix3::operator float* (void)
{
	return &(x.x);
}

inline ThMatrix3::operator const float* (void) const
{
	return &(x.x);
}

inline float ThMatrix3::operator () (unsigned int column, unsigned int row) const
{
	return (&(x.x))[column * 3 + row];
}

inline float& ThMatrix3::operator () (unsigned int column, unsigned int row)
{
	return (&(x.x))[column * 3 + row];
}

inline ThMatrix3 ThMatrix3::operator + (const ThMatrix3& matrix) const
{
	return ThMatrix3(x + matrix.x, y + matrix.y, z + matrix.z);
}

inline ThMatrix3 ThMatrix3::operator - (const ThMatrix3& matrix) const
{
	return ThMatrix3(x - matrix.x, y - matrix.y, z - matrix.z);
}

inline ThMatrix3 ThMatrix3::operator * (const ThMatrix3& matrix) const
{
	ThMatrix3 result;

  result.x.x = x.x * matrix.x.x + y.x * matrix.x.y + z.x * matrix.x.z;
  result.y.x = x.x * matrix.y.x + y.x * matrix.y.y + z.x * matrix.y.z;
  result.z.x = x.x * matrix.z.x + y.x * matrix.z.y + z.x * matrix.z.z;

  result.x.y = x.y * matrix.x.x + y.y * matrix.x.y + z.y * matrix.x.z;
  result.y.y = x.y * matrix.y.x + y.y * matrix.y.y + z.y * matrix.y.z;
  result.z.y = x.y * matrix.z.x + y.y * matrix.z.y + z.y * matrix.z.z;

  result.x.z = x.z * matrix.x.x + y.z * matrix.x.y + z.z * matrix.x.z;
  result.y.z = x.z * matrix.y.x + y.z * matrix.y.y + z.z * matrix.y.z;
  result.z.z = x.z * matrix.z.x + y.z * matrix.z.y + z.z * matrix.z.z;

	return result;
}

inline ThMatrix3& ThMatrix3::operator += (const ThMatrix3& matrix)
{
	x += matrix.x;
	y += matrix.y;
	z += matrix.z;

	return *this;
}

inline ThMatrix3& ThMatrix3::operator -= (const ThMatrix3& matrix)
{
	x -= matrix.x;
	y -= matrix.y;
	z -= matrix.z;

	return *this;
}

inline ThMatrix3& ThMatrix3::operator *= (const ThMatrix3& matrix)
{
	ThMatrix3 result;

  result.x.x = x.x * matrix.x.x + y.x * matrix.x.y + z.x * matrix.x.z;
  result.y.x = x.x * matrix.y.x + y.x * matrix.y.y + z.x * matrix.y.z;
  result.z.x = x.x * matrix.z.x + y.x * matrix.z.y + z.x * matrix.z.z;

  result.x.y = x.y * matrix.x.x + y.y * matrix.x.y + z.y * matrix.x.z;
  result.y.y = x.y * matrix.y.x + y.y * matrix.y.y + z.y * matrix.y.z;
  result.z.y = x.y * matrix.z.x + y.y * matrix.z.y + z.y * matrix.z.z;

  result.x.z = x.z * matrix.x.x + y.z * matrix.x.y + z.z * matrix.x.z;
  result.y.z = x.z * matrix.y.x + y.z * matrix.y.y + z.z * matrix.y.z;
  result.z.z = x.z * matrix.z.x + y.z * matrix.z.y + z.z * matrix.z.z;

	return operator = (result);
}

// ThMatrix3 vector methods -----------------------------------------------------------------------

inline void ThMatrix3::rotateVector(ThVector3& vector) const
{
	ThVector3 temp;

	temp.x = vector.x * x.x + vector.y * y.x + vector.z * z.x;
	temp.y = vector.x * x.y + vector.y * y.y + vector.z * z.y;
	temp.z = vector.x * x.z + vector.y * y.z + vector.z * z.z;

	vector = temp;
}

// ThMatrix3 attributes ---------------------------------------------------------------------------

inline void ThMatrix3::set(const ThVector3& sx, const ThVector3& sy, const ThVector3& sz)
{
	x = sx;
	y = sy;
	z = sz;
}

inline void ThMatrix3::setEulerRotation(const ThVector3& angles)
{
	// TODO: implement!
	/*
	ThVector3 sin, cos;

	sin.set(sinf(angles.x), sinf(angles.y), sinf(angles.z));
	cos.set(cosf(angles.x), cosf(angles.y), cosf(angles.z));

	x.x = cos.y * cos.z;
	x.y = sin.x * sin.y * cos.z - cos.x * sin.z;
	x.z = cos.x * sin.y * cos.z + sin.x * sin.z;

	y.x = cos.y * sin.z;
	y.y = cos.x * cos.z + sin.x * sin.y * sin.z;
	y.z = cos.x * sin.y * sin.z - sin.x * cos.z;

	z.x = -sin.y;
	z.y = sin.x * cos.y;
	z.z = cos.x * cos.y;
	*/
}

inline void ThMatrix3::setQuatRotation(const ThQuaternion& quat)
{
	x.x = 1.f - 2.f * quat.y * quat.y - 2.f * quat.z * quat.z;
	x.y = 2.f * quat.x * quat.y + 2.f * quat.w * quat.z;
	x.z = 2.f * quat.x * quat.z - 2.f * quat.w * quat.y;

	y.x = 2.f * quat.x * quat.y - 2.f * quat.w * quat.z;
	y.y = 1.f - 2.f * quat.x * quat.x - 2.f * quat.z * quat.z;
	y.z = 2.f * quat.y * quat.z + 2.f * quat.w * quat.x;

	z.x = 2.f * quat.x * quat.z + 2.f * quat.w * quat.y;
	z.y = 2.f * quat.y * quat.z - 2.f * quat.w * quat.x;
	z.z = 1.f - 2.f * quat.x * quat.x - 2.f * quat.y * quat.y;
}

inline void ThMatrix3::setVectorRotation(const ThVector3& vector, float angle)
{
	const float sinAngle = sinf(angle);
	const float cosAngle = cosf(angle);

  x.x = vector.x * vector.x * (1.f - cosAngle) + cosAngle;
  x.y = vector.x * vector.y * (1.f - cosAngle) - vector.z * sinAngle;
  x.z = vector.x * vector.z * (1.f - cosAngle) + vector.y * sinAngle;
  y.x = vector.x * vector.y * (1.f - cosAngle) + vector.z * sinAngle;
  y.y = vector.y * vector.y * (1.f - cosAngle) + cosAngle;
  y.z = vector.y * vector.z * (1.f - cosAngle) - vector.x * sinAngle;
  z.x = vector.x * vector.z * (1.f - cosAngle) - vector.y * sinAngle;
  z.y = vector.y * vector.z * (1.f - cosAngle) + vector.x * sinAngle;
  z.z = vector.z * vector.z * (1.f - cosAngle) + cosAngle;
}

inline void ThMatrix3::setAxisRotation(const ThVector3& axis, float angle)
{
	const float cosAngle = cosf(angle);
	const float sinAngle = sinf(angle);

	const float oneMinusCos = 1.f - cosAngle;

	const float x2 = axis.x * axis.x;
	const float y2 = axis.y * axis.y;
	const float z2 = axis.z * axis.z;
	const float xy = axis.x * axis.y * oneMinusCos;
	const float xz = axis.x * axis.z * oneMinusCos;
	const float yz = axis.y * axis.z * oneMinusCos;
	const float xs = axis.x * sinAngle;
	const float ys = axis.y * sinAngle;
	const float zs = axis.z * sinAngle;

	x.x = x2 * oneMinusCos + cosAngle;
	y.x = xy - zs;
	z.x = xz + ys;
	x.y = xy + zs;
	y.y = y2 * oneMinusCos + cosAngle;
	z.y = yz - xs;
	x.z = xz - ys;
	y.z = yz + xs;
	z.z = z2 * oneMinusCos + cosAngle;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThMatrix4 constructors -------------------------------------------------------------------------

inline ThMatrix4::ThMatrix4(void)
{
}

inline ThMatrix4::ThMatrix4(const ThVector4& sx, const ThVector4& sy, const ThVector4& sz, const ThVector4& sw):
	x(sx),
	y(sy),
	z(sz),
	w(sw)
{
}

// ThMatrix4 methods ------------------------------------------------------------------------------

inline void ThMatrix4::invert(void)
{
	// TODO: implement!
}

inline void ThMatrix4::transpose(void)
{
	float temp;

	temp = x.y; x.y = y.x; y.x = temp;
	temp = x.z; x.z = z.x; z.x = temp;
	temp = x.w; x.w = w.x; w.x = temp;
	temp = y.z; y.z = z.y; z.y = temp;
	temp = y.w; y.w = w.y; w.y = temp;
	temp = z.w; z.w = w.z; w.z = temp;
}

inline void ThMatrix4::normalize(void)
{
	// TODO: implement!
}

inline void ThMatrix4::concatenate(const ThMatrix4& matrix)
{
	ThMatrix4 result;

  result.x.x = x.x * matrix.x.x + y.x * matrix.x.y + z.x * matrix.x.z + w.x * matrix.x.w;
  result.y.x = x.x * matrix.y.x + y.x * matrix.y.y + z.x * matrix.y.z + w.x * matrix.y.w;
  result.z.x = x.x * matrix.z.x + y.x * matrix.z.y + z.x * matrix.z.z + w.x * matrix.z.w;
  result.w.x = x.x * matrix.w.x + y.x * matrix.w.y + z.x * matrix.w.z + w.x * matrix.w.w;

  result.x.y = x.y * matrix.x.x + y.y * matrix.x.y + z.y * matrix.x.z + w.y * matrix.x.w;
  result.y.y = x.y * matrix.y.x + y.y * matrix.y.y + z.y * matrix.y.z + w.y * matrix.y.w;
  result.z.y = x.y * matrix.z.x + y.y * matrix.z.y + z.y * matrix.z.z + w.y * matrix.z.w;
  result.w.y = x.y * matrix.w.x + y.y * matrix.w.y + z.y * matrix.w.z + w.y * matrix.w.w;

  result.x.z = x.z * matrix.x.x + y.z * matrix.x.y + z.z * matrix.x.z + w.z * matrix.x.w;
  result.y.z = x.z * matrix.y.x + y.z * matrix.y.y + z.z * matrix.y.z + w.z * matrix.y.w;
  result.z.z = x.z * matrix.z.x + y.z * matrix.z.y + z.z * matrix.z.z + w.z * matrix.z.w;
  result.w.z = x.z * matrix.w.x + y.z * matrix.w.y + z.z * matrix.w.z + w.z * matrix.w.w;

  result.x.w = x.w * matrix.x.x + y.w * matrix.x.y + z.w * matrix.x.z + w.w * matrix.x.w;
  result.y.w = x.w * matrix.y.x + y.w * matrix.y.y + z.w * matrix.y.z + w.w * matrix.y.w;
  result.z.w = x.w * matrix.z.x + y.w * matrix.z.y + z.w * matrix.z.z + w.w * matrix.z.w;
  result.w.w = x.w * matrix.w.x + y.w * matrix.w.y + z.w * matrix.w.z + w.w * matrix.w.w;

	operator = (result);
}

inline void ThMatrix4::reset(void)
{
	x.set(1.f, 0.f, 0.f, 0.f);
	y.set(0.f, 1.f, 0.f, 0.f);
	z.set(0.f, 0.f, 1.f, 0.f);
	w.set(0.f, 0.f, 0.f, 1.f);
}

// ThMatrix4 vector methods -----------------------------------------------------------------------

inline void ThMatrix4::transformVector(ThVector3& vector) const
{
	ThVector3 temp;

	temp.x = vector.x * x.x + vector.y * y.x + vector.z * z.x + w.x;
	temp.y = vector.x * x.y + vector.y * y.y + vector.z * z.y + w.y;
	temp.z = vector.x * x.z + vector.y * y.z + vector.z * z.z + w.z;

	vector = temp;
}

inline void ThMatrix4::transformVector(ThVector4& vector) const
{
	ThVector4 temp;

	temp.x = vector.x * x.x + vector.y * y.x + vector.z * z.x + vector.w * w.x;
	temp.y = vector.x * x.y + vector.y * y.y + vector.z * z.y + vector.w * w.y;
	temp.z = vector.x * x.z + vector.y * y.z + vector.z * z.z + vector.w * w.z;
	temp.w = vector.x * x.w + vector.y * y.w + vector.z * z.w + vector.w * w.w;

	vector = temp;
}

inline void ThMatrix4::translateVector(ThVector3& vector) const
{
	vector.x += w.x;
	vector.y += w.y;
	vector.z += w.z;
}

inline void ThMatrix4::rotateVector(ThVector3& vector) const
{
	ThVector3 temp;

	temp.x = vector.x * x.x + vector.y * y.x + vector.z * z.x;
	temp.y = vector.x * x.y + vector.y * y.y + vector.z * z.y;
	temp.z = vector.x * x.z + vector.y * y.z + vector.z * z.z;

	vector = temp;
}

// ThMatrix4 operators ----------------------------------------------------------------------------

inline ThMatrix4::operator float* (void)
{
	return &(x.x);
}

inline ThMatrix4::operator const float* (void) const
{
	return &(x.x);
}

inline float ThMatrix4::operator () (unsigned int column, unsigned int row) const
{
	return (&(x.x))[column * 4 + row];
}

inline float& ThMatrix4::operator () (unsigned int column, unsigned int row)
{
	return (&(x.x))[column * 4 + row];
}

inline ThMatrix4 ThMatrix4::operator + (const ThMatrix4& matrix) const
{
	return ThMatrix4(x + matrix.x, y + matrix.y, z + matrix.z, w + matrix.w);
}

inline ThMatrix4 ThMatrix4::operator - (const ThMatrix4& matrix) const
{
	return ThMatrix4(x - matrix.x, y - matrix.y, z - matrix.z, w - matrix.w);
}

inline ThMatrix4 ThMatrix4::operator * (const ThMatrix4& matrix) const
{
	ThMatrix4 result;

  result.x.x = x.x * matrix.x.x + y.x * matrix.x.y + z.x * matrix.x.z;
  result.y.x = x.x * matrix.y.x + y.x * matrix.y.y + z.x * matrix.y.z;
  result.z.x = x.x * matrix.z.x + y.x * matrix.z.y + z.x * matrix.z.z;
  result.w.x = x.x * matrix.w.x + y.x * matrix.w.y + z.x * matrix.w.z + w.x;

  result.x.y = x.y * matrix.x.x + y.y * matrix.x.y + z.y * matrix.x.z;
  result.y.y = x.y * matrix.y.x + y.y * matrix.y.y + z.y * matrix.y.z;
  result.z.y = x.y * matrix.z.x + y.y * matrix.z.y + z.y * matrix.z.z;
  result.w.y = x.y * matrix.w.x + y.y * matrix.w.y + z.y * matrix.w.z + w.y;

  result.x.z = x.z * matrix.x.x + y.z * matrix.x.y + z.z * matrix.x.z;
  result.y.z = x.z * matrix.y.x + y.z * matrix.y.y + z.z * matrix.y.z;
  result.z.z = x.z * matrix.z.x + y.z * matrix.z.y + z.z * matrix.z.z;
  result.w.z = x.z * matrix.w.x + y.z * matrix.w.y + z.z * matrix.w.z + w.z;

  result.x.w = 0.f;
  result.y.w = 0.f;
  result.z.w = 0.f;
  result.w.w = 1.f;

	return result;
}

inline ThMatrix4& ThMatrix4::operator += (const ThMatrix4& matrix)
{
	x += matrix.x;
	y += matrix.y;
	z += matrix.z;
	w += matrix.w;

	return *this;
}

inline ThMatrix4& ThMatrix4::operator -= (const ThMatrix4& matrix)
{
	x -= matrix.x;
	y -= matrix.y;
	z -= matrix.z;
	w -= matrix.w;

	return *this;
}

inline ThMatrix4& ThMatrix4::operator *= (const ThMatrix4& matrix)
{
	ThMatrix4 result;

  result.x.x = x.x * matrix.x.x + y.x * matrix.x.y + z.x * matrix.x.z;
  result.y.x = x.x * matrix.y.x + y.x * matrix.y.y + z.x * matrix.y.z;
  result.z.x = x.x * matrix.z.x + y.x * matrix.z.y + z.x * matrix.z.z;
  result.w.x = x.x * matrix.w.x + y.x * matrix.w.y + z.x * matrix.w.z + w.x;

  result.x.y = x.y * matrix.x.x + y.y * matrix.x.y + z.y * matrix.x.z;
  result.y.y = x.y * matrix.y.x + y.y * matrix.y.y + z.y * matrix.y.z;
  result.z.y = x.y * matrix.z.x + y.y * matrix.z.y + z.y * matrix.z.z;
  result.w.y = x.y * matrix.w.x + y.y * matrix.w.y + z.y * matrix.w.z + w.y;

  result.x.z = x.z * matrix.x.x + y.z * matrix.x.y + z.z * matrix.x.z;
  result.y.z = x.z * matrix.y.x + y.z * matrix.y.y + z.z * matrix.y.z;
  result.z.z = x.z * matrix.z.x + y.z * matrix.z.y + z.z * matrix.z.z;
  result.w.z = x.z * matrix.w.x + y.z * matrix.w.y + z.z * matrix.w.z + w.z;

  result.x.w = 0.f;
  result.y.w = 0.f;
  result.z.w = 0.f;
  result.w.w = 1.f;

	return operator = (result);
}

// ThMatrix4 attributes ---------------------------------------------------------------------------

inline void ThMatrix4::set(const ThVector4& sx, const ThVector4& sy, const ThVector4& sz, const ThVector4& sw)
{
	x = sx;
	y = sy;
	z = sz;
	w = sw;
}

inline void ThMatrix4::setEulerRotation(const ThVector3& angles)
{
	// TODO: implement!
	/*
	ThVector3 sin, cos;

	sin.set(sinf(angles.x), sinf(angles.y), sinf(angles.z));
	cos.set(cosf(angles.x), cosf(angles.y), cosf(angles.z));

	x.x = cos.y * cos.z;
	x.y = sin.x * sin.y * cos.z - cos.x * sin.z;
	x.z = cos.x * sin.y * cos.z + sin.x * sin.z;

	y.x = cos.y * sin.z;
	y.y = cos.x * cos.z + sin.x * sin.y * sin.z;
	y.z = cos.x * sin.y * sin.z - sin.x * cos.z;

	z.x = -sin.y;
	z.y = sin.x * cos.y;
	z.z = cos.x * cos.y;
	*/
}

inline void ThMatrix4::setQuatRotation(const ThQuaternion& quat)
{
	x.x = 1.f - 2.f * quat.y * quat.y - 2.f * quat.z * quat.z;
	x.y = 2.f * quat.x * quat.y + 2.f * quat.w * quat.z;
	x.z = 2.f * quat.x * quat.z - 2.f * quat.w * quat.y;

	y.x = 2.f * quat.x * quat.y - 2.f * quat.w * quat.z;
	y.y = 1.f - 2.f * quat.x * quat.x - 2.f * quat.z * quat.z;
	y.z = 2.f * quat.y * quat.z + 2.f * quat.w * quat.x;

	z.x = 2.f * quat.x * quat.z + 2.f * quat.w * quat.y;
	z.y = 2.f * quat.y * quat.z - 2.f * quat.w * quat.x;
	z.z = 1.f - 2.f * quat.x * quat.x - 2.f * quat.y * quat.y;
}

inline void ThMatrix4::setVectorRotation(const ThVector3& vector, float angle)
{
	const float sinAngle = sinf(angle);
	const float cosAngle = cosf(angle);

  x.x = vector.x * vector.x * (1.f - cosAngle) + cosAngle;
  x.y = vector.x * vector.y * (1.f - cosAngle) - vector.z * sinAngle;
  x.z = vector.x * vector.z * (1.f - cosAngle) + vector.y * sinAngle;
  y.x = vector.x * vector.y * (1.f - cosAngle) + vector.z * sinAngle;
  y.y = vector.y * vector.y * (1.f - cosAngle) + cosAngle;
  y.z = vector.y * vector.z * (1.f - cosAngle) - vector.x * sinAngle;
  z.x = vector.x * vector.z * (1.f - cosAngle) - vector.y * sinAngle;
  z.y = vector.y * vector.z * (1.f - cosAngle) + vector.x * sinAngle;
  z.z = vector.z * vector.z * (1.f - cosAngle) + cosAngle;
}

inline void ThMatrix4::setAxisRotation(const ThVector3& axis, float angle)
{
	const float cosAngle = cosf(angle);
	const float sinAngle = sinf(angle);

	const float oneMinusCos = 1.f - cosAngle;

	const float x2 = axis.x * axis.x;
	const float y2 = axis.y * axis.y;
	const float z2 = axis.z * axis.z;
	const float xy = axis.x * axis.y * oneMinusCos;
	const float xz = axis.x * axis.z * oneMinusCos;
	const float yz = axis.y * axis.z * oneMinusCos;
	const float xs = axis.x * sinAngle;
	const float ys = axis.y * sinAngle;
	const float zs = axis.z * sinAngle;

	x.x = x2 * oneMinusCos + cosAngle;
	y.x = xy - zs;
	z.x = xz + ys;
	x.y = xy + zs;
	y.y = y2 * oneMinusCos + cosAngle;
	z.y = yz - xs;
	x.z = xz - ys;
	y.z = yz + xs;
	z.z = z2 * oneMinusCos + cosAngle;
}

inline void ThMatrix4::getMatrixRotation(ThMatrix3& matrix) const
{
	matrix.x.x = x.x;
	matrix.x.y = x.y;
	matrix.x.z = x.z;

	matrix.y.x = y.x;
	matrix.y.y = y.y;
	matrix.y.z = y.z;

	matrix.z.x = z.x;
	matrix.z.y = z.y;
	matrix.z.z = z.z;
}

inline void ThMatrix4::setMatrixRotation(const ThMatrix3& matrix)
{
	x.x = matrix.x.x;
	x.y = matrix.x.y;
	x.z = matrix.x.z;

	y.x = matrix.y.x;
	y.y = matrix.y.y;
	y.z = matrix.y.z;

	z.x = matrix.z.x;
	z.y = matrix.z.y;
	z.z = matrix.z.z;
}

inline void ThMatrix4::getScaling(ThVector3& scaling) const
{
	ThVector3 axis;

	axis.x = x.x;
	axis.y = y.x;
	axis.z = z.x;

	scaling.x = axis.length();

	axis.x = x.y;
	axis.y = y.y;
	axis.z = z.y;

	scaling.y = axis.length();

	axis.x = x.z;
	axis.y = y.z;
	axis.z = z.z;

	scaling.z = axis.length();
}

inline void ThMatrix4::setScaling(const ThVector3& scaling)
{
	x.x = scaling.x;
	y.y = scaling.y;
	z.z = scaling.z;
}

inline void ThMatrix4::getTranslation(ThVector3& vector) const
{
	vector.x = w.x;
	vector.y = w.y;
	vector.z = w.z;
}

inline void ThMatrix4::setTranslation(const ThVector3& vector)
{
	w.x = vector.x;
	w.y = vector.y;
	w.z = vector.z;
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThTransform constructors -----------------------------------------------------------------------

inline ThTransform::ThTransform(void)
{
}

inline ThTransform::ThTransform(const ThQuaternion& rotation, const ThVector3& position, const ThVector3& scaling):
	m_rotation(rotation),
	m_position(position),
	m_scaling(scaling)
{
}

// ThTransform methods ----------------------------------------------------------------------------

inline void ThTransform::invert(void)
{
	m_rotation.invert();
	
	m_scaling.x = 1.f / m_scaling.x;
	m_scaling.y = 1.f / m_scaling.y;
	m_scaling.z = 1.f / m_scaling.z;
	
	// TODO: implement!
}

inline void ThTransform::concatenate(const ThTransform& parent)
{
	m_rotation = parent.m_rotation * m_rotation;

	parent.m_rotation.rotateVector(m_position);
	m_position *= parent.m_scaling;
	m_position += parent.m_position;

	m_scaling *= parent.m_scaling;
}

inline void ThTransform::reset(void)
{
	m_rotation.reset();
	m_position.reset();
	m_scaling.set(1.f, 1.f, 1.f);
}

// ThTransform attributes -------------------------------------------------------------------------

inline void ThTransform::set(const ThQuaternion& rotation, const ThVector3& position, const ThVector3& scaling)
{
	m_rotation = rotation;
	m_position = position;
	m_scaling  = scaling;
}

inline void ThTransform::getMatrixTransform(ThMatrix4& matrix) const
{
	matrix.reset();

	matrix.setQuatRotation(m_rotation);
	matrix.setTranslation(m_position);

	ThMatrix4 scaling;

	scaling.reset();
	scaling.setScaling(m_scaling);
	matrix *= scaling;
}

inline void ThTransform::setMatrixTransform(const ThMatrix4& matrix)
{
	ThMatrix3 rotation;

	matrix.getMatrixRotation(rotation);
	m_rotation.setMatrixRotation(rotation);

	matrix.getTranslation(m_position);
	matrix.getScaling(m_scaling);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThArea constructors ----------------------------------------------------------------------------

inline ThArea::ThArea(void)
{
}

inline ThArea::ThArea(float x, float y, float width, float height):
	m_pos(x, y),
	m_size(width, height)
{
}

// ThArea methods ---------------------------------------------------------------------------------

inline bool ThArea::contains(const ThArea& area) const
{
	if (!contains(area.m_pos))
		return false;

	ThVector2 offset = area.m_pos - m_pos;

	if (offset.x + area.m_size.x > m_size.x || offset.y + area.m_size.y > m_size.y)
		return false;

	return true;
}

inline bool ThArea::contains(const ThVector2& vector) const
{
	if (vector.x < m_pos.x || vector.y < m_pos.y)
		return false;

	if (vector.x > m_pos.x + m_size.x || vector.y > m_pos.y + m_size.y)
		return false;

	return true;
}

inline bool ThArea::intersects(const ThArea& area) const
{
	if (m_pos.x + m_size.x < area.m_pos.x || m_pos.y + m_size.y < area.m_pos.y)
		return false;

	if (area.m_pos.x + area.m_size.x < m_pos.x || area.m_pos.y + area.m_size.y < m_pos.y)
		return false;

	return true;
}

inline void ThArea::reset(void)
{
	m_pos.reset();
	m_size.reset();
}

// ThArea operators -------------------------------------------------------------------------------

inline bool ThArea::operator == (const ThArea& area) const
{
	return m_pos == area.m_pos && m_size == area.m_size;
}

inline bool ThArea::operator != (const ThArea& area) const
{
	return m_pos != area.m_pos || m_size != area.m_size;
}

// ThArea attributes ------------------------------------------------------------------------------

inline void ThArea::set(float x, float y, float width, float height)
{
	m_pos.set(x, y);
	m_size.set(width, height);
}

///////////////////////////////////////////////////////////////////////////////////////////////////

// ThSphere constructors --------------------------------------------------------------------------

inline ThSphere::ThSphere(void)
{
}

inline ThSphere::ThSphere(const ThVector3& center, float radius):
	m_center(center),
	m_radius(radius)
{
}

// ThSphere methods -------------------------------------------------------------------------------

inline void ThSphere::concatenate(const ThSphere& sphere)
{
	const ThVector3 centerDiff = sphere.m_center - m_center;
	const float centerDiffSqr = centerDiff.lengthSquared();
	const float radiusDiff = sphere.m_radius - m_radius;

	if (radiusDiff * radiusDiff >= centerDiffSqr)
	{
		if (radiusDiff >= 0.f)
			operator = (sphere);
	}
	else
	{
		const float length = sqrtf(centerDiffSqr);
		const float tolerance = 0.000001f;

		ThSphere result;

		if (length > tolerance)
			result.m_center = m_center + centerDiff * (length + radiusDiff) / (2.f * length);
		else
			result.m_center = m_center;

		result.m_radius = 0.5f * (length + m_radius + sphere.m_radius);

		operator = (result);
	}
}

inline bool ThSphere::contains(const ThVector3& position) const
{
	ThVector3 vector = position - m_center;

	return vector.lengthSquared() < m_radius * m_radius;
}

inline bool ThSphere::contains(const ThSphere& sphere) const
{
	if (m_radius < sphere.m_radius)
		return false;

	ThVector3 vector = sphere.m_center - m_center;

	const float radiusDiff = m_radius - sphere.m_radius;

	return vector.lengthSquared() < radiusDiff * radiusDiff;
}

inline bool ThSphere::intersects(const ThSphere& sphere) const
{
	ThVector3 vector = sphere.m_center - m_center;

	const float radiusSum = m_radius + sphere.m_radius;

	return vector.lengthSquared() < radiusSum * radiusSum;
}

inline void ThSphere::reset(void)
{
	m_center.reset();
	m_radius = 0.f;
}

// ThSphere operators -----------------------------------------------------------------------------

inline bool ThSphere::operator == (const ThSphere& sphere) const
{
	return m_center == sphere.m_center && m_radius == sphere.m_radius;
}

inline bool ThSphere::operator != (const ThSphere& sphere) const
{
	return m_center != sphere.m_center || m_radius != sphere.m_radius;
}

// ThSphere attributes ----------------------------------------------------------------------------

inline void ThSphere::set(const ThVector3& center, float radius)
{
	m_center = center;
	m_radius = radius;
}

///////////////////////////////////////////////////////////////////////////////////////////////////
