///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Theresa core library
// Copyright (C) 2001 Camilla Drefvenborg <elmindreda@home.se>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
///////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef SHARED_THMATH_H
#define SHARED_THMATH_H
///////////////////////////////////////////////////////////////////////////////////////////////////

#define _USE_MATH_DEFINES

#if HAVE_MATH_H
	#include <math.h>
#endif

#ifndef M_PI
	#define M_PI 3.14159265358979323846
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////

float randf(void);
float randf(float min, float max);

///////////////////////////////////////////////////////////////////////////////////////////////////

template <typename T>
class ThRange
{
public:
// constructors
	inline														ThRange(void);
	inline														ThRange(T min, T max);
// methods
	inline T													generate(void) const;
	inline void												reset(void);
// attributes
	inline void												set(T min, T max);
// data
	T																	m_min;
	T																	m_max;
};

//-------------------------------------------------------------------------------------------------

template <typename T, unsigned int N>
class ThSpace
{
public:
// constructors
	inline														ThSpace(void);
// methods
	inline void												generate(T* target) const;
	inline void												reset(void);
// attributes
	inline void												set(unsigned int index, T min, T max);
// data
	T																	m_min[N];
	T																	m_max[N];
};

//-------------------------------------------------------------------------------------------------

class ThVector2
{
public:
// constructors
	inline														ThVector2(void);
	inline														ThVector2(float sx, float sy);
// methods
	inline float											length(void) const;
	inline float											lengthSquared(void) const;
	inline float											dotProduct(const ThVector2& vector) const;
	inline void												scaleBy(float factor);
	inline void												scaleTo(float length);
	inline void												normalize(void);
	inline void												reset(void);
// operators
	inline														operator float* (void);
	inline														operator const float* (void) const;
	inline ThVector2									operator - (void) const;
	inline ThVector2									operator + (float value) const;
	inline ThVector2									operator - (float value) const;
	inline ThVector2									operator * (float value) const;
	inline ThVector2									operator / (float value) const;
	inline ThVector2									operator += (float value);
	inline ThVector2&									operator -= (float value);
	inline ThVector2&									operator *= (float value);
	inline ThVector2&									operator /= (float value);
	inline ThVector2									operator + (const ThVector2& vector) const;
	inline ThVector2									operator - (const ThVector2& vector) const;
	inline ThVector2									operator * (const ThVector2& vector) const;
	inline ThVector2									operator / (const ThVector2& vector) const;
	inline ThVector2									operator += (const ThVector2& vector);
	inline ThVector2&									operator -= (const ThVector2& vector);
	inline ThVector2&									operator *= (const ThVector2& vector);
	inline ThVector2&									operator /= (const ThVector2& vector);
	inline bool												operator == (const ThVector2& vector) const;
	inline bool												operator != (const ThVector2& vector) const;
// attributes
	inline void												set(float sx, float sy);
// data
	float															x;
	float															y;
};

//-------------------------------------------------------------------------------------------------

class ThVector3
{
public:
// constructors
	inline														ThVector3(void);
	inline														ThVector3(float sx, float sy, float sz);
// methods
	inline float											length(void) const;
	inline float											lengthSquared(void) const;
	inline float											dotProduct(const ThVector3& vector) const;
	inline ThVector3									crossProduct(const ThVector3& vector) const;
	inline void												scaleBy(float factor);
	inline void												scaleTo(float length);
	inline void												normalize(void);
	inline void												reset(void);
// operators
	inline														operator float* (void);
	inline														operator const float* (void) const;
	inline ThVector3									operator - (void) const;
	inline ThVector3									operator + (float value) const;
	inline ThVector3									operator - (float value) const;
	inline ThVector3									operator * (float value) const;
	inline ThVector3									operator / (float value) const;
	inline ThVector3									operator += (float value);
	inline ThVector3&									operator -= (float value);
	inline ThVector3&									operator *= (float value);
	inline ThVector3&									operator /= (float value);
	inline ThVector3									operator + (const ThVector3& vector) const;
	inline ThVector3									operator - (const ThVector3& vector) const;
	inline ThVector3									operator * (const ThVector3& vector) const;
	inline ThVector3									operator / (const ThVector3& vector) const;
	inline ThVector3									operator += (const ThVector3& vector);
	inline ThVector3&									operator -= (const ThVector3& vector);
	inline ThVector3&									operator *= (const ThVector3& vector);
	inline ThVector3&									operator /= (const ThVector3& vector);
	inline bool												operator == (const ThVector3& vector) const;
	inline bool												operator != (const ThVector3& vector) const;
// attributes
	inline void												set(float sx, float sy, float sz);
// data
	float															x;
	float															y;
	float															z;
};

//-------------------------------------------------------------------------------------------------

class ThVector4
{
public:
// constructors
	inline														ThVector4(void);
	inline														ThVector4(float sx, float sy, float sz, float sw);
// methods
	inline float											length(void) const;
	inline float											lengthSquared(void) const;
	inline float											dotProduct(const ThVector4& vector) const;
	inline ThVector4									crossProduct(const ThVector4& vector) const;
	inline void												scaleBy(float factor);
	inline void												scaleTo(float length);
	inline void												normalize(void);
	inline void												reset(void);
// operators
	inline														operator float* (void);
	inline														operator const float* (void) const;
	inline ThVector4									operator - (void) const;
	inline ThVector4									operator + (float value) const;
	inline ThVector4									operator - (float value) const;
	inline ThVector4									operator * (float value) const;
	inline ThVector4									operator / (float value) const;
	inline ThVector4									operator += (float value);
	inline ThVector4&									operator -= (float value);
	inline ThVector4&									operator *= (float value);
	inline ThVector4&									operator /= (float value);
	inline ThVector4									operator + (const ThVector4& vector) const;
	inline ThVector4									operator - (const ThVector4& vector) const;
	inline ThVector4									operator * (const ThVector4& vector) const;
	inline ThVector4									operator / (const ThVector4& vector) const;
	inline ThVector4									operator += (const ThVector4& vector);
	inline ThVector4&									operator -= (const ThVector4& vector);
	inline ThVector4&									operator *= (const ThVector4& vector);
	inline ThVector4&									operator /= (const ThVector4& vector);
	inline bool												operator == (const ThVector4& vector) const;
	inline bool												operator != (const ThVector4& vector) const;
// attributes
	inline void												set(float sx, float sy, float sz, float sw);
// data
	float															x;
	float															y;
	float															z;
	float															w;
};

//-------------------------------------------------------------------------------------------------

class ThQuaternion
{
public:
// constructors
	inline														ThQuaternion(void);
	inline														ThQuaternion(float sw, float sx, float sy, float sz);
// methods
	inline void												invert(void);
	inline void												normalize(void);
	inline void												reset(void);
// vector methods
	inline void												rotateVector(ThVector3& vector) const;
// operators
	inline														operator float* (void);
	inline														operator const float* (void) const;
	inline ThQuaternion								operator + (const ThQuaternion& quat) const;
	inline ThQuaternion								operator * (const ThQuaternion& quat) const;
	inline ThQuaternion&							operator += (const ThQuaternion& quat);
	inline ThQuaternion&							operator *= (const ThQuaternion& quat);
	inline bool												operator == (const ThQuaternion& quat) const;
	inline bool												operator != (const ThQuaternion& quat) const;
// attributes
	inline void												set(float sw, float sx, float sy, float sz);
	inline void												setEulerRotation(const ThVector3& angles);
	inline void												setMatrixRotation(const ThMatrix3& matrix);
	inline void												setVectorRotation(const ThVector3& vector, float angle);
	inline void												getAxisRotation(ThVector3& axis, float& angle) const;
	inline void												setAxisRotation(const ThVector3& axis, float angle);
// data
	float															w;
	float															x;
	float															y;
	float															z;
};

//-------------------------------------------------------------------------------------------------

class ThMatrix3
{
public:
// constructors
	inline														ThMatrix3(void);
	inline														ThMatrix3(const ThVector3& sx, const ThVector3& sy, const ThVector3& sz);
// methods
	inline void												invert(void);
	inline void												transpose(void);
	inline void												normalize(void);
	inline void												concatenate(const ThMatrix3& matrix);
	inline void												reset(void);
// vector methods
	inline void												rotateVector(ThVector3& vector) const;
// operators
	inline														operator float* (void);
	inline														operator const float* (void) const;
	inline float											operator () (unsigned int column, unsigned int row) const;
	inline float&											operator () (unsigned int column, unsigned int row);
	inline ThMatrix3									operator + (const ThMatrix3& matrix) const;
	inline ThMatrix3									operator - (const ThMatrix3& matrix) const;
	inline ThMatrix3									operator * (const ThMatrix3& matrix) const;
	inline ThMatrix3&									operator += (const ThMatrix3& matrix);
	inline ThMatrix3&									operator -= (const ThMatrix3& matrix);
	inline ThMatrix3&									operator *= (const ThMatrix3& matrix);
	inline bool												operator == (const ThMatrix3& matrix) const;
	inline bool												operator != (const ThMatrix3& matrix) const;
// attributes
	inline void												set(const ThVector3& sx, const ThVector3& sy, const ThVector3& sz);
	inline void												setEulerRotation(const ThVector3& angles);
	inline void												setQuatRotation(const ThQuaternion& quat);
	inline void												setVectorRotation(const ThVector3& vector, float angle);
	inline void												setAxisRotation(const ThVector3& axis, float angle);
// data
	ThVector3													x;
	ThVector3													y;
	ThVector3													z;
};

//-------------------------------------------------------------------------------------------------

class ThMatrix4
{
public:
// constructors
	inline														ThMatrix4(void);
	inline														ThMatrix4(const ThVector4& sx, const ThVector4& sy, const ThVector4& sz, const ThVector4& sw);
// methods
	inline void												invert(void);
	inline void												transpose(void);
	inline void												normalize(void);
	inline void												concatenate(const ThMatrix4& matrix);
	inline void												reset(void);
// vector methods
	inline void												transformVector(ThVector3& vector) const;
	inline void												transformVector(ThVector4& vector) const;
	inline void												rotateVector(ThVector3& vector) const;
	inline void												translateVector(ThVector3& vector) const;
// operators
	inline														operator float* (void);
	inline														operator const float* (void) const;
	inline float											operator () (unsigned int column, unsigned int row) const;
	inline float&											operator () (unsigned int column, unsigned int row);
	inline ThMatrix4									operator + (const ThMatrix4& matrix) const;
	inline ThMatrix4									operator - (const ThMatrix4& matrix) const;
	inline ThMatrix4									operator * (const ThMatrix4& matrix) const;
	inline ThMatrix4&									operator += (const ThMatrix4& matrix);
	inline ThMatrix4&									operator -= (const ThMatrix4& matrix);
	inline ThMatrix4&									operator *= (const ThMatrix4& matrix);
	inline bool												operator == (const ThMatrix4& matrix) const;
	inline bool												operator != (const ThMatrix4& matrix) const;
// attributes
	inline void												set(const ThVector4& sx, const ThVector4& sy, const ThVector4& sz, const ThVector4& sw);
	inline void												setEulerRotation(const ThVector3& angles);
	inline void												setQuatRotation(const ThQuaternion& quat);
	inline void												setVectorRotation(const ThVector3& vector, float angle);
	inline void												setAxisRotation(const ThVector3& axis, float angle);
	inline void												getMatrixRotation(ThMatrix3& matrix) const;
	inline void												setMatrixRotation(const ThMatrix3& matrix);
	inline void												getScaling(ThVector3& scaling) const;
	inline void												setScaling(const ThVector3& scaling);
	inline void												getTranslation(ThVector3& vector) const;
	inline void												setTranslation(const ThVector3& vector);
// data
	ThVector4													x;
	ThVector4													y;
	ThVector4													z;
	ThVector4													w;
};

//-------------------------------------------------------------------------------------------------

class ThTransform
{
public:
// constructors
	inline														ThTransform(void);
	inline														ThTransform(const ThQuaternion& rotation, const ThVector3& position, const ThVector3& scaling);
// methods
	inline void												invert(void);
	inline void												concatenate(const ThTransform& parent);
	inline void												reset(void);
// operators
	inline bool												operator == (const ThTransform& transform) const;
	inline bool												operator != (const ThTransform& transform) const;
// attributes
	inline void												set(const ThQuaternion& rotation, const ThVector3& position, const ThVector3& scaling);
	inline void												getMatrixTransform(ThMatrix4& matrix) const;
	inline void												setMatrixTransform(const ThMatrix4& matrix);
// data
	ThQuaternion											m_rotation;
	ThVector3													m_position;
	ThVector3													m_scaling;
};

//-------------------------------------------------------------------------------------------------

class ThArea
{
public:
// constructors
	inline														ThArea(void);
	inline														ThArea(float x, float y, float width, float height);
// methods
	inline bool												contains(const ThArea& area) const;
	inline bool												contains(const ThVector2& point) const;
	inline bool												intersects(const ThArea& area) const;
	inline void												reset(void);
// operators
	inline bool												operator == (const ThArea& area) const;
	inline bool												operator != (const ThArea& area) const;
// attributes
	inline void												set(float x, float y, float width, float height);
// data
	ThVector2													m_pos;
	ThVector2													m_size;
};

//-------------------------------------------------------------------------------------------------

class ThSphere
{
public:
// constructors
	inline														ThSphere(void);
	inline														ThSphere(const ThVector3& center, float radius);
// methods
	inline void												concatenate(const ThSphere& sphere);
	inline bool												contains(const ThVector3& position) const;
	inline bool												contains(const ThSphere& sphere) const;
	inline bool												intersects(const ThSphere& sphere) const;
	inline void												reset(void);
// operators
	inline bool												operator == (const ThSphere& sphere) const;
	inline bool												operator != (const ThSphere& sphere) const;
// attributes
	inline void												set(const ThVector3& center, float radius);
// data
	ThVector3													m_center;
	float															m_radius;
};

///////////////////////////////////////////////////////////////////////////////////////////////////

#include "ThMath.hpp"

///////////////////////////////////////////////////////////////////////////////////////////////////
#endif /* SHARED_THMATH_H */
///////////////////////////////////////////////////////////////////////////////////////////////////
