// Viewport.cpp: implementation of the CViewport class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <stdio.h>

#include <windows.h>
#include <gl\gl.h>
#include <gl\glu.h>
#include <gl\glaux.h>

#include "Viewport.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

#define DEFWINDOW_CLASSNAME			"xDL Window"

#define DEFAULT_COLORBITS			16
#define DEFAULT_ZBUFFER				16
#define DEFAULT_STENCILBUFFER		0
#define DEFAULT_FARPLANEDISTANCE	5000.0f
#define DEFAULT_NEARPLANEDISTANCE	7.0f

#define DEFAULT_VIEWPORTNAME		"xDL Viewport"

#define FPS_REFRESH_TIME			0.5f

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CViewport::CViewport()
{
	this->CViewport::CViewport(DEFAULT_VIEWPORTNAME);
}

CViewport::~CViewport()
{
	DestroyDefaultWindow();	
}

bool CViewport::set(int w, int h, int cb, int zb, int sb, bool fs, bool show)
{
	if (!CreateDefaultWindow(w, h, cb, fs)) return false;
	hdc = GetDC(hwnd);
	if (hdc == NULL) return false;
	SetPixelFormat(cb, zb, sb);
	GLuint pf = ChoosePixelFormat(hdc, &pfd);
	if (pf == 0) { DestroyDefaultWindow(); return false; }
	if (!::SetPixelFormat(hdc, pf, &pfd)) { DestroyDefaultWindow(); return false; }
	hrc = wglCreateContext(hdc);
	if (hrc == 0) { DestroyDefaultWindow(); return false; }
	if (!wglMakeCurrent(hdc, hrc)) { DestroyDefaultWindow(); return false; }
	backrect.left = (GetSystemMetrics(SM_CXSCREEN) - width) / 2;
	backrect.top = (GetSystemMetrics(SM_CYSCREEN) - height) / 2;
	backrect.right = backrect.left + width;
	backrect.bottom = backrect.top + height;
	if (!fs) SetWindowPos(hwnd, NULL, backrect.left, backrect.top, width, height, SWP_NOZORDER | SWP_NOSIZE);
	if (show) { if (!CViewport::show()) return false; }
	else visible = false;
	resize(w, h);
	vp_closed = false;
	backstyle = WS_OVERLAPPEDWINDOW;
	if (resizeable) backstyle |= WS_MAXIMIZEBOX;
	else backstyle &= ~WS_MAXIMIZEBOX;
	backstyleex = WS_EX_APPWINDOW | WS_EX_WINDOWEDGE;
	oldmousex = -1;
	frames = 0;
	fps_acum = 0;
	print_counter = FPS_REFRESH_TIME;

#if COMPILE_CONSOLE
	console.start();
#endif

	nrenders = 0;

	return true;
}

LRESULT CALLBACK CViewport::WndProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam) {
	CViewport *vp;
	RECT r;
	switch(uMsg) {
		case WM_KEYDOWN:
			if (wParam == VK_ESCAPE) PostMessage(hwnd, WM_CLOSE, 0, 0);
			return 0;
		case WM_CLOSE:
			vp = (CViewport *)GetWindowLong(hwnd, 0);
			if (vp != 0) vp->vp_closed = true;
			break;
		case WM_SIZE:
			vp = (CViewport *)GetWindowLong(hwnd, 0);
			if (vp != 0) {
				if (vp->resizeable) {
					vp->resize(lParam & 0xffff, lParam >> 16);
				}
				else {
					if (wParam & SIZE_MAXIMIZED) ShowWindow(vp->hwnd, SW_RESTORE);
					GetWindowRect(vp->hwnd, &r);
					SetWindowPos(vp->hwnd, NULL, r.left, r.top, vp->width, vp->height, SWP_NOZORDER);
				}
				return 0;
			}
			break;
		case WM_SIZING:
			vp = (CViewport *)GetWindowLong(hwnd, 0);
			if (vp != 0 && !vp->resizeable) {
				GetWindowRect(vp->hwnd, &r);
				((RECT *)lParam)->top = r.top;
				((RECT *)lParam)->bottom = r.bottom;
				((RECT *)lParam)->left = r.left;
				((RECT *)lParam)->right = r.right;
				return TRUE;
			}
			break;
		case WM_SYSCOMMAND:
			if (wParam == SC_MAXIMIZE || wParam == SC_SIZE) return 0;
			break;
	}
	return DefWindowProc(hwnd, uMsg, wParam, lParam); 
}

bool CViewport::CreateDefaultWindow(int w, int h, int b, bool fs)
{
	if (hwnd != NULL) DestroyDefaultWindow();
	// Guardamos datos
	fullscreen = fs;
	firstwidth = w;
	firstheight = h;
	width = w;
	height = h;
	cbits = b;
	// Registramos la clase de ventana por defecto
	HINSTANCE hInstance = GetModuleHandle(NULL);

	WNDCLASS wc;
	bool own_registration = false;
	// Si la clase de ventana estaba ya registrada, no la registramos
	if (!GetClassInfo(hInstance, DEFWINDOW_CLASSNAME, &wc)) {
		wc.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
		wc.lpfnWndProc = (WNDPROC)WndProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = sizeof(long);
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(NULL, IDI_EXCLAMATION);
		wc.hCursor = LoadCursor(NULL, IDC_ARROW);
		wc.hbrBackground = NULL;
		wc.lpszMenuName = NULL;
		wc.lpszClassName = DEFWINDOW_CLASSNAME;
		if (!RegisterClass(&wc)) return false;
		own_registration = true;				// Nos indica que, si hay error, podemos desregistrar la clase
	}

	// Si es pantalla completa
	int style, exstyle;
	if (fs) {
		exstyle = WS_EX_APPWINDOW;
		style = WS_POPUP;
	}
	else {
		exstyle = WS_EX_APPWINDOW | WS_EX_WINDOWEDGE;
		if (resizeable) style = WS_OVERLAPPEDWINDOW;
		else style = WS_OVERLAPPEDWINDOW & ~WS_MAXIMIZEBOX;
	}
	RECT wr;
	wr.left = 0;
	wr.top = 0;
	wr.right = w;
	wr.bottom = h;
	AdjustWindowRectEx(&wr, style, FALSE, exstyle);
	// Creamos la ventana
	hwnd = CreateWindowEx(exstyle, DEFWINDOW_CLASSNAME, name, WS_CLIPSIBLINGS | WS_CLIPCHILDREN | style, 
						  0, 0, wr.right - wr.left, wr.bottom - wr.top, NULL, NULL, hInstance, NULL);
	if (hwnd == NULL) { 
		if (own_registration) UnregisterClass(DEFWINDOW_CLASSNAME, GetModuleHandle(NULL)); 
		return false; 
	}

	SetWindowLong(hwnd, 0, (long)this);
	return true;
}

void CViewport::DestroyDefaultWindow()
{
	if (fullscreen) { ChangeDisplaySettings(NULL, 0); ShowCursor(TRUE); }
	if (hrc != NULL) { wglMakeCurrent(NULL,NULL); wglDeleteContext(hrc); hrc = NULL; }
	if (hwnd != NULL && hdc != NULL) { ReleaseDC(hwnd, hdc); hdc = NULL; }
	if (hwnd != NULL) { DestroyWindow(hwnd); hwnd = NULL; }
	UnregisterClass(DEFWINDOW_CLASSNAME, GetModuleHandle(NULL));
}

void CViewport::SetPixelFormat(int cb, int zb, int sb)
{
	cbits = cb;
	zbits = zb;
	sbits = sb;
	pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER | PFD_GENERIC_ACCELERATED;
	pfd.iPixelType = PFD_TYPE_RGBA;
	if (cb == 32) cb = 24;
	pfd.cColorBits = cb;
	pfd.cRedBits = 0;
	pfd.cRedShift = 0;
	pfd.cGreenBits = 0;
	pfd.cGreenShift = 0;
	pfd.cBlueBits = 0;
	pfd.cBlueShift = 0;
	pfd.cAlphaBits = 0;
	pfd.cAlphaShift = 0;
	pfd.cAccumBits = 0;
	pfd.cAccumRedBits = 0;
	pfd.cAccumGreenBits = 0;
	pfd.cAccumBlueBits = 0;
	pfd.cAccumAlphaBits = 0;
	pfd.cDepthBits = zb;
	pfd.cStencilBits = sb;
	pfd.cAuxBuffers = 0;
	pfd.iLayerType = PFD_MAIN_PLANE;
	pfd.bReserved = 0;
	pfd.dwLayerMask = 0;
	pfd.dwVisibleMask = 0;
	pfd.dwDamageMask = 0;
}

bool CViewport::set(int w, int h, int cb, bool fs, bool show)
{
	return set(w, h, cb, DEFAULT_ZBUFFER, DEFAULT_STENCILBUFFER, fs, show);
}

bool CViewport::set(int w, int h, bool fs, bool show)
{
	return set(w, h, DEFAULT_COLORBITS, DEFAULT_ZBUFFER, DEFAULT_STENCILBUFFER, fs, show);
}

void CViewport::resize(int w, int h)
{
	width = w; height = h;
	if (h == 0) h = 1;
	glViewport(0, 0, w, h);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(45.0f, (float)width / (float)height, DEFAULT_NEARPLANEDISTANCE, DEFAULT_FARPLANEDISTANCE);

#if COMPILE_CONSOLE
	console.notifyResize();
#endif
}

CViewport::CViewport(char *n)
{
	show_fps = false;
	current_fps = 0;
	for (int i = 0; i < (sizeof(fps) / sizeof(float)); i++) fps[i] = 0;
	vp_closed = true;
	hwnd = NULL;
	hdc = NULL;
	hrc = NULL;
	int l = strlen(n);
	if (l + 1 > sizeof(name)) l = sizeof(name) - 1;
	memcpy(name, n, l + 1);
	resizeable = true;
#if COMPILE_CONSOLE
	console.setParentViewport(this);
#endif
}

bool CViewport::isClosed()
{
	return vp_closed;
}

void CViewport::swapBuffers()
{
	SwapBuffers(hdc);
}

void CViewport::setResizable(bool s)
{
	resizeable = s;
	if (resizeable) backstyle |= WS_MAXIMIZEBOX;
	else backstyle &= ~WS_MAXIMIZEBOX;
	if (!fullscreen) SetWindowLong(hwnd, GWL_STYLE, backstyle);
	if (visible) { ShowWindow(hwnd, SW_HIDE); ShowWindow(hwnd, SW_SHOWNORMAL); }
}

void CViewport::clear(int flags)
{
	glClear(flags);
	glLoadIdentity();
}

bool CViewport::set(HWND wnd, int cb, int zb, int sb)
{
	hwnd = wnd;
	hdc = GetDC(hwnd);
	if (hdc == NULL) return false;
	SetPixelFormat(cb, zb, sb);
	GLuint pf = ChoosePixelFormat(hdc, &pfd);
	if (pf == 0) return false;
	if (!::SetPixelFormat(hdc, pf, &pfd)) return false;
	hrc = wglCreateContext(hdc);
	if (hrc == 0) return false;
	if (!wglMakeCurrent(hdc, hrc)) return false;
	RECT rect;
	GetClientRect(hwnd, &rect);
	resize(rect.right - rect.left + 1, rect.bottom - rect.top + 1);
	vp_closed = false;
	backstyle = GetWindowLong(hwnd, GWL_STYLE);
	backstyleex = GetWindowLong(hwnd, GWL_EXSTYLE);
	oldmousex = -1;
	frames = 0;
	fps_acum = 0;
	print_counter = FPS_REFRESH_TIME;

#if COMPILE_CONSOLE
	console.start();
#endif

	nrenders = 0;

	return true;
}

bool CViewport::show()
{
	ShowWindow(hwnd, SW_SHOWNORMAL);
	SetForegroundWindow(hwnd);
	SetFocus(hwnd);
	visible = true;
	if (fullscreen) {
		DEVMODE ss;
		memset(&ss, 0, sizeof(ss));
		ss.dmSize = sizeof(ss);
		ss.dmPelsWidth = firstwidth;
		ss.dmPelsHeight = firstheight;
		ss.dmBitsPerPel = cbits;
		ss.dmFields = DM_BITSPERPEL | DM_PELSWIDTH | DM_PELSHEIGHT;
		if (ChangeDisplaySettings(&ss, CDS_FULLSCREEN) != DISP_CHANGE_SUCCESSFUL) return false; 
		ShowCursor(FALSE);
	}
	return true;
}

void CViewport::hide()
{
	ShowWindow(hwnd, SW_HIDE);
	visible = false;
	if (fullscreen) {
		ChangeDisplaySettings(NULL, 0);
		ShowCursor(TRUE);
	}
}

bool CViewport::setFullScreen(bool fs)
{
	if (fs && !fullscreen) {
		backstyle = GetWindowLong(hwnd, GWL_STYLE);
		backstyleex = GetWindowLong(hwnd, GWL_EXSTYLE);
		SetWindowLong(hwnd, GWL_EXSTYLE, WS_EX_APPWINDOW);
		SetWindowLong(hwnd, GWL_STYLE, WS_POPUP);
		WINDOWPLACEMENT wp;
		wp.length = sizeof(WINDOWPLACEMENT);
		GetWindowPlacement(hwnd, &wp);
		backrect.left = wp.rcNormalPosition.left;
		backrect.right = wp.rcNormalPosition.right;
		backrect.top = wp.rcNormalPosition.top;
		backrect.bottom = wp.rcNormalPosition.bottom;
		SetWindowPos(hwnd, NULL, 0, 0, width, height, SWP_NOZORDER);
	}
	if (!fs && fullscreen) {
		if (visible) ChangeDisplaySettings(NULL, 0); 
		SetWindowLong(hwnd, GWL_STYLE, backstyle);
		SetWindowLong(hwnd, GWL_EXSTYLE, backstyleex);
		ShowCursor(TRUE);
		SetWindowPos(hwnd, NULL, backrect.left, backrect.top, backrect.right - backrect.left, backrect.bottom - backrect.top, SWP_NOZORDER);
		ShowWindow(hwnd, SW_MINIMIZE);
	}

	fullscreen = fs;
	if (visible) return show();
	return true;
}

int CViewport::getWidth()
{
	return width;
}

int CViewport::getHeight()
{
	return height;
}

bool CViewport::getFullScreen()
{
	return fullscreen;
}

void CViewport::showFps(bool s) 
{
	show_fps = s;
//	if (!s) SetWindowText(hwnd, name);
}

CVector & CViewport::pixelToCartesianCoords(CVector &v)
{
	v.x = v.x / width * 2 - 1;
	v.y = -v.y / height * 2 + 1;
	return v;
}

CVector & CViewport::pixelToCartesianDistance(CVector &v)
{
	v.x = v.x / width * 2;
	v.y = v.y / height * 2;
	return v;
}

void CViewport::printf(CVector &pos, CColor &color, char *str, ...)
{
	va_list vl;
	va_start(vl, str);
	_vsnprintf(temp_printf_str, sizeof(temp_printf_str) - 1, str, vl);
	font.print2d(pos, color, temp_printf_str);
	va_end(vl);
}
