// Quaternion.cpp: implementation of the CQuaternion class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Quaternion.h"
#include <math.h>

CQuaternion temp_q;

#define THRESHOLD		1e-03f

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CQuaternion::CQuaternion()
{

}

CQuaternion::~CQuaternion()
{

}

CQuaternion::CQuaternion(CVector axis, float angle)
{
	set(axis, angle);
}

float CQuaternion::getAngle()
{
	return (float)acos(w) * 2 * 180 / 3.14159f;
}

CVector CQuaternion::getAxis()
{
	float sin_a = (float)sqrt(1 - w * w);
	if (fabs(sin_a) < THRESHOLD) sin_a = 1;
	return CVector(x / sin_a, y / sin_a, z / sin_a);
}

CMatrix4 CQuaternion::getRotMatrix()
{
	CMatrix4 matrix;
	CQuaternion q;
	q = normalize();
	matrix.value[0][0] = 1 - 2 * (q.y * q.y + q.z * q.z);
	matrix.value[0][1] = 2 * (q.x * q.y - q.z * q.w);
	matrix.value[0][2] = 2 * (q.x * q.z + q.y * q.w);
	matrix.value[0][3] = 0;

	matrix.value[1][0] = 2 * (q.x * q.y + q.z * q.w);
	matrix.value[1][1] = 1 - 2 * (q.x * q.x + q.z * q.z);
	matrix.value[1][2] = 2 * (q.y * q.z - q.x * q.w);
	matrix.value[1][3] = 0;

	matrix.value[2][0] = 2 * (q.x * q.z - q.y * q.w);
	matrix.value[2][1] = 2 * (q.y * q.z + q.x * q.w);
	matrix.value[2][2] = 1 - 2 * (q.x * q.x + q.y * q.y);
	matrix.value[2][3] = 0;

	matrix.value[3][0] = 0;
	matrix.value[3][1] = 0;
	matrix.value[3][2] = 0;
	matrix.value[3][3] = 1;

	return matrix;
}

void CQuaternion::set(CVector axis, float angle)
{
	// Quaternion unitario : q = cos(angle) + axis * sin(angle)
	float sin_a = (float)sin(angle / 2);

	x = axis.x * sin_a;
	y = axis.y * sin_a;
	z = axis.z * sin_a;
	w = (float)cos(angle / 2);
}

float CQuaternion::norm()
{
	return (float)sqrt(x * x + y * y + z * z + w * w);
}

CQuaternion CQuaternion::inverse()
{
	float n = norm();
	float nn = n * n;
	if (nn > 0)
		return conj() / (n * n);
	else 
		return CQuaternion(0, 0, 0, 0);
}

CQuaternion CQuaternion::operator *(CQuaternion &q)
{
	return CQuaternion(w * q.x + x * q.w + y * q.z - z * q.y,
				w * q.y - x * q.z + y * q.w + z * q.x,
				w * q.z + x * q.y - y * q.x + z * q.w,
				w * q.w - x * q.x - y * q.y - z * q.z);
}

CQuaternion CQuaternion::operator / (float f)
{
	return (*this) * (1.0f / f);
}

CQuaternion CQuaternion::conj()
{
	return CQuaternion(-x, -y, -z, w);
}

CQuaternion CQuaternion::operator *(float f)
{
	return CQuaternion(x * f, y * f, z * f, w * f);
}

CQuaternion::CQuaternion(CQuaternion &q)
{
	*this =	q;
}

CQuaternion CQuaternion::log()
{
    // If q = cos(A)+sin(A)*(x*i+y*j+z*k) where (x,y,z) is unit length, then
    // log(q) = A*(x*i+y*j+z*k).  If sin(A) is near zero, use log(q) =
    // sin(A)*(x*i+y*j+z*k) since sin(A)/A has limit 1.

	CQuaternion q;
	q.w = 0;

    if (fabs(w) < 1.0f)
    {
		float a = (float)acos(w);
		float sin_a = (float)sin(a); // (float)sqrt(1 - w * w);
        if (fabs(sin_a) >= THRESHOLD) {
            float k = a / sin_a;
            q.x = k * x;
            q.y = k * y;
            q.z = k * z;
			return q;
        }
    }

	q.x = x; q.y = y; q.z = z;

    return q;
}

CQuaternion CQuaternion::exp()
{
    // If q = A*(x*i+y*j+z*k) where (x,y,z) is unit length, then
    // exp(q) = cos(A)+sin(A)*(x*i+y*j+z*k).  If sin(A) is near zero,
    // use exp(q) = cos(A)+A*(x*i+y*j+z*k) since A/sin(A) has limit 1.

	CQuaternion q;
    float a = (float)sqrt(x * x + y * y + z * z);
    float sin_a = (float)sin(a);

    q.w = (float)cos(a);

    if (fabs(a) >= THRESHOLD)
    {
        float k = sin_a / a;
        q.x = k * x;
        q.y = k * y;
        q.z = k * z;
    }
	else {
		q.x = x; q.y = y; q.z = z;
	}

    return q;
}

CQuaternion CQuaternion::slerp(float t, CQuaternion &q1, CQuaternion &q2)
{
    float a = (float)acos(q1.dot(q2));

    if (fabs(a) < THRESHOLD) return CQuaternion(q1);

    float sin_a = (float)sin(a);
    float inv_sin = 1.0f / sin_a;
    float k0 = (float)sin((1.0f - t) * a) * inv_sin;
    float k1 = (float)sin(t * a) * inv_sin;
    return q1 * k0 + q2 * k1;
}

float CQuaternion::dot(CQuaternion &q)
{
	return *(CVector4 *)this * q;
}

CQuaternion CQuaternion::operator +(CQuaternion &q)
{
	return CQuaternion(x + q.x, y + q.y, z + q.z, w + q.w);
}

CQuaternion::CQuaternion(float x, float y, float z, float w)
{
	CQuaternion::x = x;
	CQuaternion::y = y;
	CQuaternion::z = z;
	CQuaternion::w = w;
}

CQuaternion CQuaternion::squad(float t, CQuaternion &p, CQuaternion &a, CQuaternion &b, CQuaternion &q)
{
    return slerp(2.0f * t * (1.0f - t), slerp(t, p, q), slerp(t, a, b));
}

CQuaternion CQuaternion::operator -()
{
	return CQuaternion(-x, -y, -z, -w);
}

CQuaternion CQuaternion::operator -(CQuaternion &q)
{
	return CQuaternion(x - q.x, y - q.y, z - q.z, w - q.w);
}

CQuaternion::CQuaternion(CMatrix4 &rotm)
{
	// Quaternion from rotation matrix
	float trace = rotm.trace();
	if (trace > THRESHOLD) {
		float s = 0.5f / (float)sqrt(trace);
		w = 0.25f / s;
		x = (rotm.value[2][1] - rotm.value[1][2]) * s;
		y = (rotm.value[0][2] - rotm.value[2][0]) * s;
		z = (rotm.value[1][0] - rotm.value[0][1]) * s;
	}
	else {
		float s;
		if (rotm.value[0][0] > rotm.value[1][1] && rotm.value[0][0] > rotm.value[2][2]) {
			s = 1.0f / (2 * (float)sqrt(1 + rotm.value[0][0] - rotm.value[1][1] - rotm.value[2][2]));
			x = 0.25f / s;
			y = (rotm.value[0][1] + rotm.value[1][0]) * s;
			z = (rotm.value[0][2] + rotm.value[2][0]) * s;
			w = (rotm.value[1][2] - rotm.value[2][1]) * s;
		}
		else if (rotm.value[1][1] > rotm.value[2][2]) {
			s = 1.0f / (2 * (float)sqrt(1 + rotm.value[1][1] - rotm.value[0][0] - rotm.value[2][2]));
			y = 0.25f / s;
			x = (rotm.value[0][1] + rotm.value[1][0]) * s;
			w = (rotm.value[0][2] - rotm.value[2][0]) * s;
			z = (rotm.value[1][2] + rotm.value[2][1]) * s;
		}
		else {
			s = 1.0f / (2 * (float)sqrt(1 + rotm.value[2][2] - rotm.value[0][0] - rotm.value[1][1]));
			z = 0.25f / s;
			w = (rotm.value[0][1] - rotm.value[1][0]) * s;
			x = (rotm.value[0][2] + rotm.value[2][0]) * s;
			y = (rotm.value[1][2] + rotm.value[2][1]) * s;
		}
	}
}

CQuaternion CQuaternion::normalize()
{
	float n = norm();
	return CQuaternion(x / n, y / n, z / n, w / n);
}
