/*
 * backgroundhandler.cpp: Static backgrounds, split into multiple square
 *                        textures so we don't have to scale them (if
 *                        they match the screen resolution, that is).
 *
 *                        Note that no rescaling is supported here, so
 *                        it essentially breaks Voodoo1/2/3 support. Tough. :-)
 *                        Ideally, we should resize the image first or
 *                        something in case of a minification... do later. :-)
 *                        
 *                        Also supports GL_NV_texture_rectangle if you
 *                        have it.
 */

#include <stdio.h>
#include <string.h>

#ifdef WIN32
#include <windows.h>
#endif
#include <GL/gl.h>

#include "main/backgroundhandler.h"
#include "opengl/texture.h"
#include "opengl/extensions.h"
#include "exception.h"
#include "demolib_prefs.h"

#if DEMOLIB_MAINLOOP 

BackgroundHandler::BackgroundHandler(MainLoop *ml, const char *title, const char *elem, Hashtable *attr) :
	Event(ml, title, elem, attr, "alpha")
{
	if (attr->exists("layer")) {
		this->layer = attr->get_float("layer");
	} else {
		this->layer = -50.0f;
	}
	
	Image *baseimg = load_image(attr->get_str("file"));
	this->xres = baseimg->get_width();
	this->yres = baseimg->get_height();
	
	GLenum fmt = texture::get_opengl_format(baseimg->get_bpp());
	GLenum ifmt = texture::get_opengl_internal_format(baseimg->get_bpp());

	/* find how many parts we need to split in :-) */
	int i;
	
	this->xparts = 0;
	for (i = 0; i < 30; i++) {
		if (xres & (1<<i)) this->xparts++;
	}
	
	this->yparts = 0;
	for (i = 0; i < 30; i++) {
		if (yres & (1<<i)) this->yparts++;
	}

	this->textures = new GLuint[xparts * yparts];
	glGenTextures(xparts * yparts, this->textures);

	const int psize = baseimg->get_bpp() / 8;
	
	/* now split the image in individual power-of-two-sized parts */
	int yoffs = 0, texnum = 0;
	for (int ybit = 0; ybit < 30; ybit++) {
		const int h = 1 << ybit;
		if (!(yres & h)) continue;
			
		int xoffs = 0;
		for (int xbit = 0; xbit < 30; xbit++) {
		const int w = 1 << xbit;
			if (!(xres & w)) continue;

			/* 
			 * generate a memory area to store the things in
			 * (simplest, we don't have to mess around with
			 * strides etc.
			 */
			unsigned char *tmpbuf = new unsigned char[w * h * psize];
			for (int y = 0; y < h; y++) {
				memcpy(tmpbuf + y * w * psize,
					baseimg->get_pixel_data() + ((y+yoffs) * xres + xoffs) * psize,
					w * psize);
			}
			
			/* no mipmapping -- save texture memory :-) */
			glBindTexture(GL_TEXTURE_2D, this->textures[texnum++]);
			glTexImage2D(GL_TEXTURE_2D, 0, ifmt, w, h, 0, fmt, GL_UNSIGNED_BYTE, tmpbuf);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
		        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
			glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

			delete[] tmpbuf;
				
			xoffs += w;
		}
		yoffs += h;
	}

	delete baseimg;
}

BackgroundHandler::~BackgroundHandler()
{
	glDeleteTextures(xparts * yparts, this->textures);
}

void BackgroundHandler::start_effect()
{
}

void BackgroundHandler::draw_scene(float progress)
{
	float alpha = this->get_val("alpha", progress);
	if (alpha < 0.0f) return;

        glMatrixMode(GL_MODELVIEW);
        glPushMatrix();
	glLoadIdentity();

        glMatrixMode(GL_PROJECTION);
        glPushMatrix();
        glLoadIdentity();

	glOrtho(0.0f, (float)(this->xres), (float)(this->yres), 0.0f, 0.0f, 1.0f);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	
	glColor4f(1.0f, 1.0f, 1.0f, alpha);
	
        glDisable(GL_LIGHTING);
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_CULL_FACE);

	glEnable(GL_TEXTURE_2D);

	/* regenerate picture from the parts */
	int yoffs = 0, texnum = 0;
	for (int ybit = 0; ybit < 30; ybit++) {
		const int h = 1 << ybit;
		if (!(yres & h)) continue;
			
		int xoffs = 0;
		for (int xbit = 0; xbit < 30; xbit++) {
			const int w = 1 << xbit;
			if (!(xres & w)) continue;

			glBindTexture(GL_TEXTURE_2D, this->textures[texnum++]);
		
       			glBegin(GL_QUADS);
			glTexCoord2f(0.0f, 0.0f);
			glVertex3f((float)(xoffs    ), (float)(yoffs    ), 0.0f);
			glTexCoord2f(0.0f, 1.0f);
			glVertex3f((float)(xoffs    ), (float)(yoffs + h), 0.0f);
			glTexCoord2f(1.0f, 1.0f);
			glVertex3f((float)(xoffs + w), (float)(yoffs + h), 0.0f);
			glTexCoord2f(1.0f, 0.0f);
			glVertex3f((float)(xoffs + w), (float)(yoffs    ), 0.0f);
			glEnd();
			
			xoffs += w;
		}
	
		yoffs += h;
	}
		
	glPopMatrix();

	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
}

void BackgroundHandler::end_effect() {}

#endif
