// This is a sound generator imitating human voice. It uses a glottis
// oscillator and five-stage formant filter imitating the
// mouth and larynx. A zero filter is added to imitate the release
// of sound into open air. The mathematical models in this
// program come mostly from Johan Sundberg's book "The Science
// of the Singing Voice", ISBN 0-87580-542-6.

// The program reads control functions and uses them to control
// sound generation. The time resolution of the control functions
// is one sound sample. The control functions are floating-point
// MIDI note value (*.N), amplitude in dB (*.A), frequency modulation
// in cents (*.FM), amplitude modulation in dB (*.AM), formant
// frequency modulation as a coefficient (*.ZM), sound expressivity level
// (*.E, see glottis oscillator for details), low frequency oscillator
// frequency in Hz (*.LF), and the five formants, for which
// is given the pole radius (*.Z?R) and the center frequency in Hz (*.Z?F),
// where the question mark is replaced with the formant number from 1 to 5.
// The zero filter has fixed radius ZEROATTC, and its center frequency
// is zero.

// For each sample the low and high frequency oscillators are run
// in parallel, used for modulation and glottis oscillation.
// The glottis output, modified by the formant filters, is obtained
// from osc () and modulated as needed by the low frequency oscillator
// output. The oscillator arguments advance according to the
// current frequency, which may be modulated, too. The routine cbtask ()
// is called to update those random values which change only once per
// glottis cycle. One of these is the glottis impulse delay.

// The output of the generator is a WAV file, which is not guarded against
// overflow. Use level adjustment variable lvladj as needed.

// The glottis oscillator uses two ad-hoc functions for the ascending
// and descending side (aslope () and dslope ()) of the glottis pulse.
// The expressivity, or sharpness, of the voice is controlled by the slope
// of the descending side. There is also a RASP control constant which
// causes random displacement of the glottal pulses. If you need
// white noise mixed into the glottal pulse, modify this program
// or use the modified version HVOC_GEN.C, which uses a hiss level
// parameter instead of the frequency modulation.

// Some of the variables have been explained in the main () function.

// All modulations are done downwards from the nominal value. If this does
// not fit your purpose, add a bias to the nominal values or modify the
// subtractions (b_am, b_fm, b_zm) in main ().

// Obtaining the desired voice quality is hard work with many nasty
// surprises. Patient experimenting will improve the results. Note also
// that a voice should be listened in its final environment. What sounds
// best with a quiet background and dry acoustics is not necessarily
// optimal for actual performance situation.

// Writing a front-end processor creating *.SRC input files for the voice
// generator could be considered.

#include <stdio.h>
#include <stdlib.h>
#include <io.h>
#include <math.h>
#include <string.h>
#include <values.h>



#define DPI          (2.0 * 3.141592653587973238)

#define FORMANTS     6            // 0 Hz zero included
#define FILEFORMANTS 5
#define BUFLEN       0x00100000

// Glottis oscillator control constants

#define ZEROATTC     0.95         // Max 1.0
#define ARATE        0.3
#define DRATE        0.2
#define ERATE        0.15
#define RASP         0.01

#define BOOST        8.0



double ran1,
       ran2;
double prevoffs = 0.0;
double srate    = 44100.0;

FILE   *df;

// Formant filter. r is pole radius, f is center frequency, and
// z1 and z2 are the delay registers.

struct filt {
  double r;
  double f;
  double z1;
  double z2; } z [FORMANTS];

// Buffers for the control function files

float b_n  [BUFLEN];
float b_a  [BUFLEN];
float b_fm [BUFLEN];
float b_am [BUFLEN];
float b_zm [BUFLEN];
float b_e  [BUFLEN];
float b_lf [BUFLEN];
float b_zr [FORMANTS] [BUFLEN];
float b_zf [FORMANTS] [BUFLEN];



double pole (double v, short k) {

  double y, a, b, c;

  a = 2.0 * z[k].r * cos (DPI * z[k].f / srate);
  b = z[k].r * z[k].r;
  c = 1.0 - a + b;
  y = v + a * z[k].z1 - b * z[k].z2;
  z[k].z2 = z[k].z1;
  z[k].z1 = y;
  return (c * y); } 



double zero (double v, short k) {

  double y, a, b, c;

  a = 2.0 * z[k].r * cos (DPI * z[k].f / srate);
  b = z[k].r * z[k].r;
  c = 1.0 - a + b;
  z[k].z2 = z[k].z1;
  z[k].z1 = v;
  y = v - a * z[k].z1 + b * z[k].z2;
  return (BOOST * y); } 



void cbtask (void) {
  ran1 = (double) random () / (double) MAXINT;
  ran2 = (double) random () / (double) MAXINT; }



double aslope (double x) {
  double v;
  x  = 2.0 * x - 1.0;
  v  = - 1.0 / 5.0 * pow (x, 5.0) + x;
  v *= (5.0 / 4.0);
  v  = (v + 1.0) / 2.0;
  return (v); }



double dslope (double x) {
  double v;
  x  = - 2.0 * x + 1.0;
  v  = - 1.0 / 5.0 * pow (x, 5.0) + x;
  v *= (5.0 / 4.0);
  v  = (v + 1.0) / 2.0;
  return (v); }



double osc (double a, double e) {

  double v, drate, aa;

  // Glottis

  a /= DPI;
  a -= ran1 * RASP;
  drate = DRATE - ERATE * sqrt (e);
  v = 0.0;
  if (a < ARATE) v = aslope (a / ARATE);
  if ((a >= ARATE) && (a < ARATE + drate))
    v = dslope ((a - ARATE) / drate);

  // Formants

  v = zero (v, 0);
  v = pole (v, 1);
  v = pole (v, 2);
  v = pole (v, 3);
  v = pole (v, 4);
  v = pole (v, 5);

  return (v); }



void hdr (FILE *f, char *proj, char *suffix) {

  char           nb [256];
  FILE           *rf;
  unsigned long  d, bc;
  unsigned short w;

  sprintf (nb, "%s.%s", proj, suffix);
  rf = fopen (nb, "rb");
  bc =   (  filelength (fileno (rf))
          * (long) sizeof (short) )
       / (long) sizeof (float);
  fclose (rf);

  // Monophonic WAV header

                 fwrite ("RIFF", 1, 4, f);
  d =  bc + 44L; fwrite (&d,     4, 1, f);       /* Bytes + header  */
                 fwrite ("WAVE", 1, 4, f);
                 fwrite ("fmt ", 1, 4, f);
  d =       16L; fwrite (&d,     4, 1, f);       /* ???             */
  w =         1; fwrite (&w,     2, 1, f);       /* WAV format      */
  w =         1; fwrite (&w,     2, 1, f);       /* Mono            */
  d =     srate; fwrite (&d,     4, 1, f);       /* Sample rate     */
  d = 2 * srate; fwrite (&d,     4, 1, f);       /* Avg BPS         */
  w =         2; fwrite (&w,     2, 1, f);       /* Block align     */
  w =        16; fwrite (&w,     2, 1, f);       /* Bits per sample */
                 fwrite ("data", 1, 4, f);
  d =        bc; fwrite (&d,     4, 1, f); }     /* Bytes to go     */



char *makefn (char *buf, char *proj, char *e) {
  sprintf (buf, "%s.%s", proj, e);
  return (buf); }



long loadbuf (float *b, char *proj, char *suffix, long len) {
  char nb [256];
  FILE *f;
  long i, rv;
  f = fopen (makefn (nb, proj, suffix), "rb");
  if (f == NULL) return (0L);
  for (i = 0 ; i < BUFLEN ; i++) {
    rv = fread (b + i, sizeof (float), 1, f);
    if (rv != 1) break; }
  fclose (f);
  if (len > i) len = i;
  return (len); }



int main (int argc, char *argv []) {

  double a,            // High frequency oscillator argument
         lfo,          // Low frequency oscillator output
         lfoa,         // Low frequency oscillator argument
         lvladj;       // Output level adjustment [dB, zero level unknown]
  FILE   *sf;          // Control function file
  long   i,
         j,
         sn,           // The number of the current sample
         len;          // Buffer length in samples
  short  s;            // Output sample value
  char   proj [256];   // Project tag (Common file name stem part)
  char   nb   [256];   // File name buffer

  printf ("voc_gen - Voice tone generator 2.0\n\n");

  if (argc < 2) {
    printf ("Use: voc_gen <projecttag> [leveladj] [samplerate]\n");
    return (1); }

  if (argc >= 3) sscanf (argv [2], "%lf", &lvladj);
  if (argc >= 4) sscanf (argv [3], "%lf", &srate);

  lvladj = pow (10.0, lvladj / 20.0);

  strcpy (proj, argv [1]);
  i = strlen (proj);
  while ((i >= 0) && (proj [i] != '.')) i--;
  if (i >= 0) proj [i] = '\0';

  sprintf (nb, "%s.WAV", proj);
  sf = fopen (nb,  "wb");
  hdr (sf, proj, "N");

  // Initialize

  a = 0.0; lfoa = 0.0;
  cbtask ();
  z[0].f = 0.0;
  z[0].r = ZEROATTC;
  for (i = 0 ; i < FORMANTS ; i++) {z[i].z1 = 0.0; z[i].z2 = 0.0;}

  // Run

  sn = 0L;
  while (1) {
    printf ("%8.3lf s\n", (double) sn / srate);
    len = loadbuf (b_n,  proj, "N",  0x7fffffff);
    len = loadbuf (b_a,  proj, "A",  len);
    len = loadbuf (b_fm, proj, "FM", len);
    len = loadbuf (b_am, proj, "AM", len);
    len = loadbuf (b_zm, proj, "ZM", len);
    len = loadbuf (b_e,  proj, "E",  len);
    len = loadbuf (b_lf, proj, "LF", len);
    for (i = 1 ; i <= FILEFORMANTS ; i++) {
      char b [4];
      sprintf (b, "Z%1ldR", i);
      len = loadbuf (b_zr [i], proj, b, len);
      sprintf (b, "Z%1ldF", i);
      len = loadbuf (b_zf [i], proj, b, len); }
    for (i = 0 ; i < len ; i++) {
      lfo = (1.0 - cos (lfoa)) / 2.0;
      lfoa += DPI * b_lf [i] / srate;
      if (lfoa > DPI) lfoa -= DPI;
      for (j = 1 ; j <= FILEFORMANTS ; j++) {
        z[j].r = b_zr [j] [i];
        z[j].f = b_zf [j] [i] * (1.0 - lfo * (1.0 - b_zm [i])); }
      s = (short) (  lvladj
                   * 32767.0
                   * pow (10.0, b_a [i] / 20.0)
                   * pow (10.0, (- lfo * b_am [i]) / 20.0)
                   * osc (a, b_e [i]) );
      a +=   DPI
           * 440.0
           * pow (2.0, (b_n [i] - 69.0 - lfo * b_fm [i] / 100.0) / 12.0)
           / srate;
      if (a > DPI) {a -= DPI; cbtask ();}
      fwrite (&s, sizeof (short), 1, sf); }
    sn += len;
    if (len < BUFLEN) break; }
  printf ("%ld samples (%8.3lf s)\n", sn, (double) sn / srate);

  fclose (sf);

  return (0); }
