// Early reflection impulse response generator, hardcoded for single hall at
// sample rate 44100 Hz. Stereo output.

// The sound source position and devaluation are given as program
// parameters. Direct sound is at the table index 0. The indices
// 1 to 8 are for the reflections from rectangular walls as heard
// by the closer one of the two ears. The reflections as detected by the
// opposite ear are indexed from 9 upwards. The reflections of an
// impulse are calculated and written to an ASCII file to be used
// by the convolver program.

// No guarantee is given about the correctness of the model or
// of the program code.

#include <stdio.h>
#include <stdlib.h>
#include <io.h>
#include <math.h>
#include <string.h>
#include <values.h>

#define DPI           (2.0 * PI)
#define HPI           (PI / 2.0)
#define SRATE         44100.0
#define SOUNDSPEED   340.0
#define SC            (SRATE / SOUNDSPEED)
#define ERCNT         18
#define MAXAOFFS      40   /* Head width [samples] */

// Hall dimensions [m]

#define WIDTH         20.0
#define DEPTH         40.0
#define LATLOC         0.0
#define PALOC          5.0

// Distance to walls [m]

#define LEFTCL        (WIDTH / 2.0 - LATLOC)
#define RIGHTCL       (WIDTH / 2.0 + LATLOC)
#define FRONTCL       (DEPTH / 2.0 - PALOC)
#define BACKCL        (DEPTH / 2.0 + PALOC)



// Distance (r) and direction (beta) of a lateral reflection

void imglat (double d, double alpha, double wall, double *r, double *beta) {
  if (fabs (cos (alpha)) < 0.000001) {
    *beta = HPI; if (alpha < 0.0) *beta = -HPI;
    *r = 2.0 * wall - d;
    return; }
  *beta = atan ((2.0 * wall - d * sin (alpha)) / (d * cos (alpha)));
  *r    = sqrt (d * d - 2.0 * wall * d * sin (alpha) + wall * wall); }



// Distance (r) and direction (beta) of a posteroanterior reflection

void imgpa (double d, double alpha, double wall, double *r, double *beta) {
  if (fabs (cos (alpha)) < 0.000001) {
    *beta = HPI; if (alpha < 0.0) *beta = -HPI;
    *r = 2.0 * wall - d;
    return; }
  *beta = atan (d * sin (alpha) / (2.0 * wall - d * cos (alpha)));
  *r    = sqrt (d * d - 2.0 * wall * d * sin (alpha) + wall * wall); }



int main (int argc, char *argv []) {

  double edir [ERCNT],   // Reflection panoration
         edel [ERCNT],   // Reflection delay [samples]
         eatt [ERCNT],   // Reflection attenuation by distance [dB]
         ei [ERCNT];     // Reflection amplitude
  double dist,           // Sound source distance [m]
         pan,            // Sound source direction
         eimax,          // Largest found amplitude
         imgdist,        // Distance to a source reflection [m]
         imgpan,         // Direction of a source reflection
         srcatt;         // Sound source devaluation [dB]
  FILE   *tgt;           // Output file
  long   i;

  if (argc < 5) {
    printf ("Use: asme <outputfile> <distance_meters> <direction_rad> <srcatten_dB>\n");
    printf ("\n");
    return (1); }

  tgt = fopen (argv [1], "w");
  if (tgt == NULL) {
    printf ("Cannot open output file '%s'\n", argv [1]);
    return (2); }

  sscanf (argv [2], "%lf", &dist);
  sscanf (argv [3], "%lf", &pan);
  sscanf (argv [4], "%lf", &srcatt);

  edel [0] = SC * dist;                                           // Direct sound
  edir [0] = pan;

  imglat (dist, pan, LEFTCL, &imgdist, &imgpan);
  edel [1] = imgdist * SC;                                        // Left wall
  edir [1] = imgpan;

  imglat (dist, pan, - RIGHTCL, &imgdist, &imgpan);
  edel [2] = imgdist * SC;                                        // Right wall
  edir [2] = imgpan;

  imgpa (dist, pan, FRONTCL, &imgdist, &imgpan);
  edel [3] = imgdist * SC;                                        // Front wall
  edir [3] = imgpan;

  edel [4] = sqrt (LEFTCL * LEFTCL + FRONTCL * FRONTCL) * SC;     // Left front corner
  edir [4] = atan (LEFTCL / FRONTCL);

  edel [5] = sqrt (RIGHTCL * RIGHTCL + FRONTCL * FRONTCL) * SC;   // Right front corner
  edir [5] = atan (- RIGHTCL / FRONTCL);

  imgpa (dist, pan, BACKCL, &imgdist, &imgpan);
  edel [6] = imgdist * SC;                                        // Back wall, reflected front
  edir [6] = imgpan;

  edel [7] = sqrt (LEFTCL * LEFTCL + BACKCL * BACKCL) * SC;       // Left back corner
  edir [7] = atan (LEFTCL / BACKCL);

  edel [8] = sqrt (RIGHTCL * RIGHTCL + BACKCL * BACKCL) * SC;     // Right back corner
  edir [8] = atan (- RIGHTCL / BACKCL);

  for (i = 0 ; i < 9 ; i++) eatt [i] = 0.0;

  for (i = 9 ; i < 18 ; i++) {
    edel [i] =   edel [i - 9] + fabs (MAXAOFFS * sin (edir [i - 9]));   // Opposite ear
    edir [i] = - edir [i - 9];
    eatt [i] = - 20.0 * (double) (edel [i] - edel [i - 9]) / (double) MAXAOFFS; }

  eimax = 0.0;
  for (i = 0L ; i < ERCNT ; i++) {
    ei [i] = pow (10.0, eatt [i] / 20.0) / (edel [i] * edel [i]);
    if (i == 0L) ei [i] *= pow (10.0, srcatt / 20.0);
    if (i == 9L) ei [i] *= pow (10.0, srcatt / 20.0);
    if (eimax < ei [i]) eimax = ei [i]; }

  for (i = 0L ; i < ERCNT ; i++) ei [i] = 32767.0 * ei [i] / eimax;

  for (i = 0L ; i < ERCNT ; i++)
    fprintf (tgt, "%c   %6.0lf   %6.0lf\n", "RL" [edir [i] >= 0.0], edel [i], ei [i]);

  fprintf (tgt, "END\n");

  fclose (tgt);

  return (0); }
