#include <GL/gl.h>
#include <math.h>
#include <stdio.h>
#include <string>
#include <string.h>
#include <vector>

#include "coords.h"
#include "font.h"
#include "textures.h"
#include "textview.h"

class TextItem
{
public:
	int id;

	std::string text;
	int x, y;
	int zoom;

	GLuint displayList;

	TextItem(): displayList( 0 ) { }

	// Frees the item
	void free() { if ( displayList > 0 ) glDeleteLists( displayList, 1 ); displayList = 0; }
};

class TextPoint
{
public:
	// Angle
	float ang;
	// Between [0,1]
	float x, y;
};

typedef std::vector< TextItem > TextItemList;

// The text points
static TextPoint textPoints[ FONTH ][ FONTW ];

// The text item list
static TextItemList textItemList;
static int runningId = 0;

// Renders the given text
static void render( const TextItem &item );

////////

// Initializes the text view
void textviewInit()
{
	textviewClear();

	// Init the points
	for ( int y = 0; y < FONTH; ++y )
		for ( int x = 0; x < FONTW; ++x )
		{
			textPoints[ y ][ x ].ang = x + y;
			textPoints[ y ][ x ].x = 0.5f;
			textPoints[ y ][ x ].y = 0.5f;
		}
}

// Clears all the text
void textviewClear()
{
	TextItemList::iterator pos = textItemList.begin();
	TextItemList::iterator end = textItemList.end();

	for ( ; pos != end; ++pos )
	{
		pos->free();
	}

	textItemList.clear();
}

// Removes a certain item from the screen
void textviewRemove( int id )
{
	TextItemList::iterator pos = textItemList.begin();
	TextItemList::iterator end = textItemList.end();

	for ( ; pos != end; ++pos )
	{
		if ( pos->id == id )
		{
			pos->free();
			textItemList.erase( pos );
			return;
		}
	}
}

// Returns the text width/height
int textviewWidth( const char *text, int zoom )
{
	return strlen( text ) * FONTW * zoom;
}

int textviewHeight( const char *text, int zoom )
{
	return FONTH * zoom;
}

// Adds text to the view (returns an unique id)
int textviewWrite( const char *text, int x, int y, int zoom )
{
	TextItem item;

	// Create the struct
	item.id = ++runningId;
	item.text = text;
	item.x = x;
	item.y = y;
	item.zoom = zoom;

	// Create the display list
/*	item.displayList = glGenLists( 1 );
	glNewList( item.displayList, GL_COMPILE );
	render( item );
	glEndList();*/

	// Add it
	textItemList.push_back( item );
	return item.id;
}

// Updates the textview for one frame
void textviewUpdate()
{
	// Update the points
	for ( int y = 0; y < FONTH; ++y )
	{
		for ( int x = 0; x < FONTW; ++x )
		{
			float add = ( float ) ( ( x + y ) & 3 ) / 6.f;
			textPoints[ y ][ x ].ang += 0.5f + add;

			float a1 = textPoints[ y ][ x ].ang * 3.1415926f / 180.f;
			float a2 = a1 * 2;
			float a3 = a2 * 3;

			float fx = cosf( a1 ) * 0.3f + cosf( a2 ) * 0.5f + cosf( a3 ) * 0.1f;
			float fy = sinf( a3 ) * 0.3f + sinf( a1 ) * 0.5f + sinf( a2 ) * 0.1f;

			textPoints[ y ][ x ].x = ( float ) x + ( 1.f + fx ) / 2.f;
			textPoints[ y ][ x ].y = ( float ) y + ( 1.f + fy ) / 2.f;
		}
	}
}

// Renders the textview
void textviewRender()
{
	// Init
	glMatrixMode( GL_MODELVIEW );
	glLoadIdentity();

	glMatrixMode( GL_PROJECTION );
	glPushMatrix();
	glLoadIdentity();
	glOrtho( 0, 639, 479, 0, 0, 10 );

	glDisable( GL_LIGHTING );
//	glDisable( GL_DEPTH_TEST );
	glEnable( GL_BLEND );
	glBlendFunc( GL_SRC_ALPHA, GL_ONE /*GL_ONE_MINUS_SRC_ALPHA*/ );

	setTexture( textviewTexture );

	// Draw
	int size = textItemList.size();
	for ( int i = 0; i < size; ++i )
	{
		render( textItemList[ i ] );
//		glCallList( textItemList[ i ].displayList );
	}

	// Clean up
	glPopMatrix();
	glMatrixMode( GL_MODELVIEW );

//	glEnable( GL_DEPTH_TEST );
	glDisable( GL_TEXTURE_2D );
	glDisable( GL_BLEND );
}

// Renders one text
void textviewRender( const char *text, int x, int y, int zoom )
{
	TextItem item;

	// Create the struct
	item.id = 0;
	item.text = text;
	item.x = x;
	item.y = y;
	item.zoom = zoom;

	// Init
	glMatrixMode( GL_MODELVIEW );
	glLoadIdentity();

	glMatrixMode( GL_PROJECTION );
	glPushMatrix();
	glLoadIdentity();
	glOrtho( 0, 639, 479, 0, 0, 10 );

	glDisable( GL_LIGHTING );
	glEnable( GL_BLEND );
	glBlendFunc( GL_SRC_ALPHA, GL_ONE /*GL_ONE_MINUS_SRC_ALPHA*/ );

	// Draw
	setTexture( textviewTexture );

	render( item );

	// Clean up
	glPopMatrix();
	glMatrixMode( GL_MODELVIEW );

	glDisable( GL_TEXTURE_2D );
	glDisable( GL_BLEND );
}

////////

// Draws one block (GL_QUADS must be begun)
static void drawBlock( float x1, float y1, float x2, float y2, float width )
{
	float dx = ( x2 - x1 );
	float dy = ( y2 - y1 );
	
	float w2 = width / 2.f;

	// Normalize
	float len = sqrtf( dx * dx + dy * dy );
	dx /= len;
	dy /= len;

	float nx = -dy * ( width / 2.f );
	float ny =  dx * ( width / 2.f );

	// Lenghten the line
	x1 -= dx * w2;
	y1 -= dy * w2;

	x2 += dx * w2;
	y2 += dy * w2;

	// Draw
	glTexCoord2f( 0, 0 );
	glVertex3f( x1 - nx, y1 - ny, -1 );

	glTexCoord2f( 1, 0 );
	glVertex3f( x2 - nx, y2 - ny, -1 );

	glTexCoord2f( 1, 1 );
	glVertex3f( x2 + nx, y2 + ny, -1 );

	glTexCoord2f( 0, 1 );
	glVertex3f( x1 + nx, y1 + ny, -1 );
}

// Renders the given text
static void render( const TextItem &item )
{
	const char *text = item.text.c_str();
	int x = item.x;
	int y = item.y;
	int zoom = item.zoom;
	int blockSize = zoom;

	float width = ( float ) zoom * 2.f;

	if ( *text == '\0' )
		return;

	glBegin( GL_QUADS );
	glNormal3f( 0, 0, 1 );

	while ( *text != '\0' )
	{
		const char *font = fontGet( *text++ );

		for ( int fy = 0; fy < FONTH; ++fy )
		{
			for ( int fx = 0; fx < FONTW; ++fx )
			{
#define GETPT( comp, addX, addY ) \
	( float ) comp + textPoints[ fy + addY ][ fx + addX ].comp * ( float ) blockSize

				float x1 = GETPT( x, 0, 0 );
				float y1 = GETPT( y, 0, 0 );

				int idx = fx + fy * FONTW;
				bool isSet = font[ idx ] != FONT_CLEAR;
				bool leftSet = ( fx > 0 ) ? ( font[ idx - 1 ] != FONT_CLEAR ) : false;
				bool rightSet = ( fx < ( FONTW - 1 ) ) ? ( font[ idx + 1 ] != FONT_CLEAR ) : false;
				bool upSet = ( fy > 0 ) ? ( font[ idx - FONTW ] != FONT_CLEAR ) : false;
				bool downSet = ( fy < ( FONTH - 1 ) ) ? ( font[ idx + FONTW ] != FONT_CLEAR ) : false;

				if ( !isSet )
					continue;

				if ( !leftSet && !rightSet && !upSet && !downSet )
					drawBlock( x1, y1, x1, y1 + ( float ) width / 2.f, width );

				if ( leftSet )
					drawBlock( x1, y1, GETPT( x, -1, 0 ), GETPT( y, -1, 0 ), width );
				if ( rightSet )
					drawBlock( x1, y1, GETPT( x, 1, 0 ), GETPT( y, 1, 0 ), width );
				if ( upSet )
					drawBlock( x1, y1, GETPT( x, 0, -1 ), GETPT( y, 0, -1 ), width );
				if ( downSet )
					drawBlock( x1, y1, GETPT( x, 0, 1 ), GETPT( y, 0, 1 ), width );

#undef GETPT
			}
		}

		// Next
		x += FONTW * blockSize;
	}

	glEnd();
}
