#include <fcntl.h>
#include <linux/soundcard.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <stdio.h>
#include <unistd.h>
#include <mad.h>

#include "sound.h"

//// STATIC DEFINITION ////

typedef unsigned char UInt8;

// Internal file buffer
struct MP3Buffer
{
	unsigned char const *start;
	unsigned long length;
};

// The sound handle
static int globDSP = -1;

// The sound buffer
static UInt8 *globDSPBuffer = 0;
static int globDSPBufferSize = 0;


// The open MP3 file
static int globMP3File = -1;

// The MP3 file mapped memory
static void *globMP3FileMem = 0;
static int globMP3FileMemSize = 0;

// The MP3 buffer
static struct MP3Buffer globMP3Buffer;
// The MP3 decoder
static struct mad_decoder globMP3Decoder;
static bool globMP3DecoderInit = false;



// Fetches new data for the buffer
static mad_flow input( void *data, struct mad_stream *stream );

// Plays the data on the output
static mad_flow output( void *data, struct mad_header const *header, struct mad_pcm *pcm );

//// GLOBAL IMPLEMENTATION ////

// Inits the sound
bool soundInit()
{
	soundDeinit();

	// Open the device
	globDSP = open( "/dev/dsp", O_WRONLY );
	if ( globDSP == -1 )
		return false;

	// Init
	int format = AFMT_S16_LE;
	int stereo = 1;
	int channels = 2;
	int speed = 44100;

	ioctl( globDSP, SNDCTL_DSP_SYNC, 0 );
	ioctl( globDSP, SNDCTL_DSP_SETFMT, &format );

	ioctl( globDSP, SNDCTL_DSP_STEREO, &stereo );
	ioctl( globDSP, SNDCTL_DSP_CHANNELS, &channels );
	ioctl( globDSP, SNDCTL_DSP_SPEED, &speed );

	// Allocate the buffer
	globDSPBufferSize = 4096;
	globDSPBuffer = new UInt8[ globDSPBufferSize + 64 ];

	return true;
}

// Deinits the sound
void soundDeinit()
{
	soundStopMP3();

	// Release the device
	if ( globDSP != -1 )
	{
		ioctl( globDSP, SNDCTL_DSP_RESET, 0 );
		close( globDSP );
		globDSP = -1;
	}

	// Release the buffers
	if ( globDSPBuffer != 0 )
	{
		delete [] globDSPBuffer;
		globDSPBuffer = 0;
	}
}

// Starts playing an MP3 file
bool soundStartMP3( const char *file )
{
	soundStopMP3();

	if ( globDSP == -1 )
		return true;

	// Open the file
	globMP3File = open( file, O_RDONLY );
	if ( globMP3File == -1 )
		return false;

	// Get the size
	struct stat stat;

	if ( fstat( ( int ) globMP3File, &stat ) == -1 ||
		 stat.st_size == 0 )
	{
		soundStopMP3();
		return false;
	}
	globMP3FileMemSize = stat.st_size;

	// Map the file
	globMP3FileMem = mmap( 0, globMP3FileMemSize, PROT_READ, MAP_SHARED, ( int ) globMP3File, 0 );
	if ( globMP3FileMem == MAP_FAILED )
	{
		globMP3FileMem = 0;
		soundStopMP3();
		return false;
	}

	// Init the buffer
	globMP3Buffer.start = ( unsigned char * ) globMP3FileMem;
	globMP3Buffer.length = globMP3FileMemSize;

	// Init the decoder
	mad_decoder_init( &globMP3Decoder, &globMP3Buffer,
					  input, 0 /* header */, 0 /* filter */, output,
					  0 /* error */, 0 /* message */);
	globMP3DecoderInit = true;

	// Play
	mad_decoder_run( &globMP3Decoder, MAD_DECODER_MODE_ASYNC );

    return true;
}

// Stops the MP3 playback
void soundStopMP3()
{
	// Deinit the decoder
	if ( globMP3DecoderInit )
	{
		mad_decoder_finish( &globMP3Decoder );
		globMP3DecoderInit = false;
	}

	// Unmap the memory
	if ( globMP3FileMem != 0 )
	{
		munmap( globMP3FileMem, globMP3FileMemSize );
		globMP3FileMem = 0;
		globMP3FileMemSize = 0;
	}

	// Close the file
	if ( globMP3File != -1 )
	{
		close( globMP3File );
		globMP3File = -1;
	}
}

//// STATIC IMPLEMENTATION ////


// Scales the samples
static inline signed int scale( mad_fixed_t sample )
{
	// round
	sample += ( 1L << (MAD_F_FRACBITS - 16 ) );

	// clip
	if ( sample >= MAD_F_ONE )
		sample = MAD_F_ONE - 1;
	else if ( sample < -MAD_F_ONE )
		sample = -MAD_F_ONE;

	// quantize
	return sample >> ( MAD_F_FRACBITS + 1 - 16 );
}

// Writes data to the sound device
static void outputSound( UInt8 *buf, int len )
{
	if ( globDSP == -1 || buf == 0 )
		return;

	while ( len > 0 )
	{
		int amt = write( globDSP, buf, len );
		if ( amt < 0 )
			break;
		
		// Advance
		buf += amt;
		len -= amt;
	}
}

/////

// Fetches new data for the buffer
static mad_flow input( void *data, struct mad_stream *stream )
{
	MP3Buffer *buffer = ( MP3Buffer * ) data;

	if ( buffer->length <= 0 )
	{
		// HOX: Should return MAD_FLOW_STOP, but stopping seems to make ODE crash?!??
		return MAD_FLOW_IGNORE;
	}

	mad_stream_buffer( stream, buffer->start, buffer->length );

	buffer->length = 0;

	return MAD_FLOW_CONTINUE;
}

// Plays the data on the output
static mad_flow output( void *data, struct mad_header const *header, struct mad_pcm *pcm )
{
	unsigned int nchannels, nsamples;
	mad_fixed_t const *left_ch, *right_ch;

	if ( globDSP == -1 )
		return MAD_FLOW_STOP;

	/* pcm->samplerate contains the sampling frequency */
	nchannels = pcm->channels;
	nsamples  = pcm->length;
	left_ch   = pcm->samples[0];
	right_ch  = pcm->samples[1];

	// Init the buffers
	UInt8 *bufPos = globDSPBuffer;
	int bufSize = 0;
	int bufLeft = globDSPBufferSize - 30;

	// Downsample the bits
	while ( nsamples-- )
	{
		signed int sample;

		/* output sample(s) in 16-bit signed little-endian PCM */
		sample = scale( *left_ch++ );

		// Left
		*bufPos++ = ( UInt8 ) ( ( sample >> 0 ) & 0xff );
		*bufPos++ = ( UInt8 ) ( ( sample >> 8 ) & 0xff );

		if ( nchannels == 2 )
			sample = scale( *right_ch++ );

		// Right
		*bufPos++ = ( UInt8 ) ( ( sample >> 0 ) & 0xff );
		*bufPos++ = ( UInt8 ) ( ( sample >> 8 ) & 0xff );

		// Flush buffers?
		bufSize += 4;
		bufLeft -= 4;

		if ( bufLeft <= 0 )
		{ // Output to sound device
			outputSound( globDSPBuffer, bufSize );

			// Init new buffer
			bufPos = globDSPBuffer;
			bufSize = 0;
			bufLeft = globDSPBufferSize - 30;
		}
	}

	// Output the rest
	outputSound( globDSPBuffer, bufSize );

	return MAD_FLOW_CONTINUE;
}
