#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "GL/gl.h"

#include "fadeeffect.h"
#include "pipeeffect.h"
#include "textures.h"
#include "textview.h"

#define TIMEOUT_CODE1 ( 150 )
#define TIMEOUT_CODE2 ( TIMEOUT_CODE1 + 90 )
#define TIMEOUT_MUSIC1 ( TIMEOUT_CODE2 + 130 )
#define TIMEOUT_MUSIC2 ( TIMEOUT_MUSIC1 + 90 )
#define TIMEOUT_CLEAR ( TIMEOUT_MUSIC2 + 150 )
#define TIMEOUT_NEXT ( TIMEOUT_CLEAR + 50 )

////////

class Point
{
public:
	float x, y, z;

	Point() { }
	Point( float x, float y, float z ): x( x ), y( y ), z( z ) { }

	Point operator * ( float val ) const { return Point( x * val, y * val, z * val ); }
	Point operator + ( Point pt ) const { return Point( x + pt.x, y + pt.y, z + pt.z ); }
};

// Calculates one point in a hermite spline
static inline Point hermite( Point p1, Point n1, Point p2, Point n2, float t )
{
	float t2 = t * t;
	float t3 = t2 * t;

	return p1 * (  2 * t3 - 3 * t2 +     1 ) +
	       n1 * (      t3 - 2 * t2 + t     ) +
	       n2 * (      t3 -     t2         ) +
               p2 * ( -2 * t3 + 3 * t2         );
}

// Returns a random float [0,1]
static inline float getRandom()
{
	return ( float ) ( rand() % 20001 ) / 20000.f;
}

// Creates one ring of data
static inline void createRing( Point *dest, int ptNum, float height, const Point *normals )
{
	for ( int i = 0; i < ptNum; ++i )
	{
		float mul = 0.2f + getRandom() / 2.f;
		dest[ i ] = normals[ i ] * mul;
		dest[ i ].y = height;
	}
}

// Adds a point to the gl
static inline void glNormal( Point pt ) { glNormal3f( pt.x, pt.y, pt.z ); }
static inline void glVertex( Point pt ) { glVertex3f( pt.x, pt.y, pt.z ); }

///////

PipeEffect::PipeEffect():
	myPipeDisplayList( 0 ), myPipeDist( 6.f ), myPipeHeight( 0 ),
	myTimeout( 0 ), myLeaving( false ), myAngle( 0 ), myPipePos( 0 )
{
	textviewClear();

	createPipe();

	initLight( 0,   5, 5, 5,   56, 190, 255,   255, 60, 60 );
}

PipeEffect::~PipeEffect()
{
	if ( myPipeDisplayList != 0 )
		glDeleteLists( myPipeDisplayList, 1 );
}

// Updates the effect for one frame (returns the next effect; 0 (quit), this (continue) or a new effect)
Effect *PipeEffect::update()
{
	myAngle += 0.5f;
	myPipePos += 0.005f;
	if ( myPipePos > ( myPipeHeight / 2.f ) )
		myPipePos = -myPipeHeight / 2.f;

	// Move the pipes in/out
	if ( !myLeaving )
	{
		myPipeDist -= 0.07f;
		if ( myPipeDist < 2.f )
			myPipeDist = 2.f;
	}
	else
	{
		myPipeDist += 0.07f;
		if ( myPipeDist > 6.f )
		{
			return new FadeEffect( 0.f, 0.f, 0.f,
			                       FadeEffect::Effect_Trail,
			                       0.05f, 0.15f, 0.15f,
			                       140 );
		}
	}

	int height4 = textviewHeight( "text", 4 );
	
	// Timeouts
	switch ( myTimeout++ )
	{
	case TIMEOUT_CODE1:
		textviewWrite( "code", 400 - textviewWidth( "code", 4 ) / 2, 200 - height4, 4 );
		textviewWrite( "music", 260 - textviewWidth( "music", 4 ) / 2, 300 - height4, 4 );
		break;

	case TIMEOUT_CODE2:
		textviewWrite( "xmunkki", 400 - textviewWidth( "xmunkki", 3 ) / 2, 200, 3 );
		textviewWrite( "crud", 260 - textviewWidth( "crud", 3 ) / 2, 300, 3 );
		break;

	case TIMEOUT_MUSIC1:
		textviewWrite( "eye", 600 - textviewWidth( "eye", 4 ), 40, 4 );
		textviewWrite( "title", 40, 440 - textviewHeight( "abyss", 3 ) - height4, 4 );
		break;

	case TIMEOUT_MUSIC2:
		textviewWrite( "st rana", 600 - textviewWidth( "st rana", 3 ), 40 + height4, 3 );
		textviewWrite( "abyss", 40, 440 - textviewHeight( "abyss", 3 ), 3 );
		break;

	case TIMEOUT_CLEAR:
		textviewClear();
		break;

	case TIMEOUT_NEXT:
		myLeaving = true;
		break;
	}

	return this;
}

// Renders the effects current frame
void PipeEffect::render()
{
	// Camera
	glTranslatef( 0, 0, -5.f );

	glEnable( GL_LIGHTING );

	setTexture( texturePipe, texturePipeCLUT );

	// Draw the pipes (lower right)
	glPushMatrix();
	glTranslatef( myPipeDist, 0, 0 );
	glRotatef( -45, 0, 0, 1 );
	glTranslatef( 0, myPipePos, 0 );

	glRotatef( myAngle, 0, 1, 0 );
	setTexture( texturePipe, texturePipeCLUT );
	glCallList( myPipeDisplayList );

	glTranslatef( 0, -myPipeHeight, 0 );
	setTexture( texturePipe, texturePipeCLUT );
	glCallList( myPipeDisplayList );
	glPopMatrix();


	// Draw the pipes (upper left)
	glPushMatrix();
	glTranslatef( -myPipeDist, 0, 0 );
	glRotatef( -45, 0, 0, 1 );
	glTranslatef( 0, -myPipePos, 0 );

	setTexture( texturePipe, texturePipeCLUT );

	glRotatef( myAngle, 0, 1, 0 );
	setTexture( texturePipe, texturePipeCLUT );
	glCallList( myPipeDisplayList );

	glTranslatef( 0, -myPipeHeight, 0 );
	setTexture( texturePipe, texturePipeCLUT );
	glCallList( myPipeDisplayList );
	glPopMatrix();
}

// Creates the pipe
void PipeEffect::createPipe()
{
	int i;

	myPipeDisplayList = glGenLists( 1 );
	glNewList( myPipeDisplayList, GL_COMPILE );

	const int ptNum = 15;
	const int ringNum = 6;
	const int ringSubdiv = 10;
	const float ringHeight = 0.8f;

	myPipeHeight = ringNum * ringHeight;

	// Create the normals
	Point *normals = new Point[ ptNum ];

	for ( i = 0; i < ptNum; ++i )
	{
		float ang = ( float ) i * ( 360.f / ( float ) ptNum );
		ang *= 3.1415926f / 180.f;
		normals[ i ] = Point( cosf( ang ), 0, sinf( ang ) );
	}

	// Allocate memory for the rings
	Point *ringFirst = new Point[ ptNum ];
	Point *ring1 = new Point[ ptNum ];
	Point *ring2 = new Point[ ptNum ];

	// Create the first ring
	float y = 0;

	createRing( ringFirst, ptNum, y, normals );
	y += ringHeight;

	memcpy( ring1, ringFirst, sizeof( Point ) * ptNum );

	// Create all the rings
	for ( int ring = 0; ring < ringNum; ++ring, y += ringHeight )
	{
		// Create the "to" ring (if not last)
		if ( ring < ( ringNum - 1 ) )
		{
			createRing( ring2, ptNum, y, normals );
		}
		else
		{ // Copy the first ring to last)
			for ( i = 0; i < ptNum; ++i )
			{
				ring2[ i ] = ringFirst[ i ];
				ring2[ i ].y = y;
			}
		}

		// Create the subdivisions
		const float tAdd = 1.f / ( float ) ringSubdiv;
		float t = 0;

		for ( int sub = 0; sub < ringSubdiv; ++sub, t += tAdd )
		{
			float tNext = t + tAdd;

			glBegin( GL_TRIANGLE_STRIP );
			for ( i = 0; i < ptNum; ++i )
			{
				int iNext = ( i + 1 ) % ptNum;

				// First point?
				if ( i == 0 )
				{
					glNormal( normals[ i ] );
					glTexCoord2f( ( float ) i / ( float ) ptNum, t );
					glVertex( hermite( ring1[ i ], normals[ i ], ring2[ i ], normals[ i ], t ) );

					glNormal( normals[ i ] );
					glTexCoord2f( ( float ) i / ( float ) ptNum, tNext );
					glVertex( hermite( ring1[ i ], normals[ i ], ring2[ i ], normals[ i ], tNext ) );
				}

				// Next points
				glNormal( normals[ iNext ] );
				glTexCoord2f( ( float ) ( i + 1 ) / ( float ) ptNum, t );
				glVertex( hermite( ring1[ iNext ], normals[ iNext ], ring2[ iNext ], normals[ iNext ], t ) );

				glNormal( normals[ iNext ] );
				glTexCoord2f( ( float ) ( i + 1 ) / ( float ) ptNum, tNext );
				glVertex( hermite( ring1[ iNext ], normals[ iNext ], ring2[ iNext ], normals[ iNext ], tNext ) );
			}
			glEnd();
		}

		// Next ring
		memcpy( ring1, ring2, sizeof( Point ) * ptNum );
	}

	// Free the memory
	delete [] normals;
	delete [] ringFirst;
	delete [] ring1;
	delete [] ring2;

	glEndList();
}
