#include <math.h>
#include "GL/gl.h"

#include "fadeeffect.h"
#include "geareffect.h"
#include "pipeeffect.h"
#include "textview.h"

#define TEXT_TIMEOUT	60 * 5
#define TIMEOUT		60 * 20

GearEffect::GearEffect():
	myGearDisplayList( 0 ), myGearPosY( -3.f ), myGearDist( 0 ),
	myGearAngle( 0 ), myTextShowTimeout( TEXT_TIMEOUT ),
	myTimeoutCount( TIMEOUT )
{
	createGear();

	glClearColor( 0.1f, 0.05f, 0.15f, 0.0f );
	initLight( 0, 0,  10, -10, 100, 100, 255 );
	initLight( 1, 0, -10,  10, 100, 255, 100 );

	textviewClear();
}

GearEffect::~GearEffect()
{
	if ( myGearDisplayList != 0 )
		glDeleteLists( myGearDisplayList, 1 );
}

// Updates the effect for one frame (returns the next effect; 0 (quit), this (continue) or a new effect)
Effect *GearEffect::update()
{
	myGearPosY += 0.1f;
	if ( myGearPosY > 0.f )
		myGearPosY = 0.f;

	myGearAngle += 0.4f;

	// Move the distance between the gears?
	if ( myGearDist > 0.f )
	{
		myGearDist += 0.1f;
		if ( myGearDist > 20.f )
			return new FadeEffect( 0.1f, 0.05f, 0.15f,
                                               FadeEffect::Effect_Pipe,
			                       0.f, 0.f, 0.f,
			                       140 );
	}

	// Show text?
	--myTextShowTimeout;
	if ( myTextShowTimeout == 0 )
	{
		textviewWrite( "rattaat on", 20, 20, 4 );
		textviewWrite( "muodokkaita", 20, 20 + textviewHeight( "rattaat", 4 ), 4 );
	}

	if ( myTextShowTimeout == -200 )
	{
		const char *text = "gears are formy";
		textviewWrite( text, 620 - textviewWidth( text, 3 ), 460 - textviewHeight( text, 3 ), 3 );
	}

	// Next effect?
	if ( myTimeoutCount-- == 0 )
	{
		textviewClear();
		myGearDist += 0.1f;
	}

	return this;
}

// Renders the effects current frame
void GearEffect::render()
{
	// Camera
	glTranslatef( 0, 0, -7 );
	glRotatef( 30, 1, 0, 0 );
	glRotatef( 30, 0, 1, 0 );

	glEnable( GL_LIGHTING );

	// Gears
	float dist = myGearDist + ( 3.5f + 4.f ) / 2.f;

	glPushMatrix();
	glTranslatef( -( dist / 2.f ), myGearPosY, 0 );
	glRotatef( myGearAngle, 0, 1, 0 );
	glCallList( myGearDisplayList );
	glPopMatrix();

	glPushMatrix();
	glTranslatef(  ( dist / 2.f ), -myGearPosY, 0 );
	glRotatef( -myGearAngle - 10.f, 0, 1, 0 );
	glCallList( myGearDisplayList );
	glPopMatrix();
}

// Creates the gear
void GearEffect::createGear()
{
	int i;

	myGearDisplayList = glGenLists( 1 );
	glNewList( myGearDisplayList, GL_COMPILE );

	float *xList = new float[ 360 ];
	float *zList = new float[ 360 ];

	float radiusInner = 1.f / 2.f;
	float radiusOuter = 3.5f / 2.f;
	float radiusOuter2 = 4.0f / 2.f;
	float height = 0.4f / 2.f;

	// Generate a circle (normalized)
	for ( i = 0; i < 360; ++i )
	{
		xList[ i ] = cosf( ( float ) i * 3.1415926f / 180.f );
		zList[ i ] = sinf( ( float ) i * 3.1415926f / 180.f );
	}

	// The inner circle
	glBegin( GL_TRIANGLE_STRIP );
	for ( i = 0; i < 360; i += 5 )
	{
		int next = ( i + 5 ) % 360;

		if ( i == 0 )
		{ // Two first vertices
			glNormal3f( -xList[ i ], 0, -zList[ i ] );
			glVertex3f( xList[ i ] * radiusInner, -height, zList[ i ] * radiusInner );
			glNormal3f( -xList[ i ], 0, -zList[ i ] );
			glVertex3f( xList[ i ] * radiusInner,  height, zList[ i ] * radiusInner );
		}

		glNormal3f( -xList[ i ], 0, -zList[ i ] );
		glVertex3f( xList[ next ] * radiusInner, -height, zList[ next ] * radiusInner );
		glNormal3f( -xList[ i ], 0, -zList[ i ] );
		glVertex3f( xList[ next ] * radiusInner,  height, zList[ next ] * radiusInner );
	}
	glEnd();

	// The outer circle
	bool lastIn = true;
	glBegin( GL_TRIANGLE_STRIP );
	for ( i = 0; i < 360; i += 5 )
	{
		int next = ( i + 5 ) % 360;
		bool thisIn = ( ( i % 20 ) == 0 ) ? !lastIn : lastIn;

		if ( i == 0 )
		{ // Two first vertices
			glNormal3f( xList[ i ], 0, zList[ i ] );
			glVertex3f( xList[ i ] * radiusOuter, -height, zList[ i ] * radiusOuter );
			glNormal3f( xList[ i ], 0, zList[ i ] );
			glVertex3f( xList[ i ] * radiusOuter,  height, zList[ i ] * radiusOuter );
		}

		float normalX = xList[ i ];
		float normalZ = zList[ i ];

		// Transitioning here (out<->in)?
		if ( thisIn != lastIn )
		{
			float lastRadius = lastIn ? radiusOuter : radiusOuter2;

			// Affect the normal
			float n2X = -zList[ i ];
			float n2Z =  xList[ i ];
			normalX = ( normalX + n2X ) / 2.f;
			normalZ = ( normalZ + n2Z ) / 2.f;

			// Vertices
			glNormal3f( normalX, 0, normalZ );
			glVertex3f( xList[ next ] * lastRadius, -height, zList[ next ] * lastRadius );
			glNormal3f( normalX, 0, normalZ );
			glVertex3f( xList[ next ] * lastRadius,  height, zList[ next ] * lastRadius );
		}
		
		// Next vertices
		float radius = thisIn ? radiusOuter : radiusOuter2;

		glNormal3f( normalX, 0, normalZ );
		glVertex3f( xList[ next ] * radius, -height, zList[ next ] * radius );
		glNormal3f( normalX, 0, normalZ );
		glVertex3f( xList[ next ] * radius,  height, zList[ next ] * radius );

		// Next
		lastIn = thisIn;
	}
	glEnd();

	// Top (inner)
	glBegin( GL_TRIANGLE_STRIP );
	for ( i = 0; i < 360; i += 5 )
	{
		int next = ( i + 5 ) % 360;

		if ( i == 0 )
		{ // Two first vertices
			glNormal3f( 0, 1, 0 );
			glVertex3f( xList[ i ] * radiusInner, height, zList[ i ] * radiusInner );
			glNormal3f( 0, 1, 0 );
			glVertex3f( xList[ i ] * radiusOuter, height, zList[ i ] * radiusOuter );
		}

		glNormal3f( 0, 1, 0 );
		glVertex3f( xList[ next ] * radiusInner, height, zList[ next ] * radiusInner );
		glNormal3f( 0, 1, 0 );
		glVertex3f( xList[ next ] * radiusOuter, height, zList[ next ] * radiusOuter );
	}
	glEnd();

	// Top (spikes)
	for ( i = 0; i < 360; i += 40 )
	{
		glBegin( GL_TRIANGLE_STRIP );
		for ( int j = i; j <= ( i + 20 ); j += 5 )
		{
			int next = ( j + 5 ) % 360;
	
			if ( j == 0 )
			{ // Two first vertices
				glNormal3f( 0, 1, 0 );
				glVertex3f( xList[ i ] * radiusOuter, height, zList[ i ] * radiusOuter );
				glNormal3f( 0, 1, 0 );
				glVertex3f( xList[ i ] * radiusOuter2, height, zList[ i ] * radiusOuter2 );
			}
	
			glNormal3f( 0, 1, 0 );
			glVertex3f( xList[ next ] * radiusOuter, height, zList[ next ] * radiusOuter );
			glNormal3f( 0, 1, 0 );
			glVertex3f( xList[ next ] * radiusOuter2, height, zList[ next ] * radiusOuter2 );
		}
		glEnd();
	}

	delete [] xList;
	delete [] zList;

	glEndList();
}
