#include <ode/ode.h>
#include <vector>
#include "GL/gl.h"

#include <stdio.h>

#include "boxfield.h"
#include "textures.h"

struct BoxObject
{
	// The texture id
	int texId;

	// The body
	dBodyID body;

	// Geometries representing this body
	dGeomID geom;
};

typedef std::vector< BoxObject > BoxObjectList;

static dWorldID world;
static dSpaceID space;
static dJointGroupID contactgroup;

static BoxObjectList objectList;

// The collision callback
static void nearCallback( void *data, dGeomID o1, dGeomID o2 );

// Pushes the position/rotation matrix to the stack
static void pushTransform( const float pos[3], const float rot[12] );

// Draws a box
static void drawBox( const float sides[ 3 ] );

////////

// Initializes the box field
void boxfieldInit()
{
	world = dWorldCreate();
	space = dSimpleSpaceCreate();
	contactgroup = dJointGroupCreate( 0 );

	dWorldSetGravity( world, 0, -0.5, 0 );
	dWorldSetCFM( world, 1e-5 );

	// Create the "world"
	dCreatePlane( space, 0, 1, 0, 0 );

	float sidePos = 3;
	dCreatePlane( space,  1, 0,  0, -sidePos );
	dCreatePlane( space, -1, 0,  0, -sidePos );
	dCreatePlane( space,  0, 0,  1, -sidePos );
	dCreatePlane( space,  0, 0, -1, -sidePos );

	// Init the random
	for ( int i = 0; i < 10; ++i )
		dRandReal();

	objectList.clear();
}

// Adds a box to the field
void boxfieldAdd( int texId )
{
	boxfieldAdd( texId, dRandReal() * 1.4 - 0.7, dRandReal() * 2 + 5, dRandReal() * 1.4 - 0.7 );
}

void boxfieldAdd( int texId, float x, float y, float z )
{
	BoxObject obj;

	dReal sides[ 3 ];
	dMatrix3 rotMat;
	dMass mass;

	obj.texId = texId;

	// Pick the side sizes
	for ( int i = 0; i < 3; ++i )
		sides[ i ] = dRandReal() * 0.2 + 0.5;
//	sides[ 0 ] = sides[ 1 ] = sides[ 2 ] = 0.6;

	// Create the body
	obj.body = dBodyCreate( world );
	dBodySetPosition( obj.body, x, y, z );

	dRFromAxisAndAngle( rotMat, dRandReal() * 2.0 - 1.0,dRandReal() * 2.0 - 1.0, dRandReal() * 2.0 - 1.0, dRandReal() * 10.0 - 5.0 );
	dBodySetRotation( obj.body, rotMat );

	dMassSetBox( &mass, 5, sides[ 0 ], sides[ 1 ], sides[ 2 ] );
	dBodySetMass( obj.body, &mass );

	// Create the geometry
	obj.geom = dCreateBox( space, sides[ 0 ], sides[ 1 ], sides[ 2 ] );

	// Attach
	dGeomSetBody( obj.geom, obj.body );

	// Add
	objectList.push_back( obj );
}

// Updates the box field for one frame
void boxfieldUpdate()
{
	int loops = 2;
	float step = 0.08f / ( float ) loops;

	while ( loops-- > 0 )
	{
		dSpaceCollide( space, 0, nearCallback );
		dWorldStep( world, step );
		dJointGroupEmpty( contactgroup );
	}
}

// Renders the box field
void boxfieldRender()
{
	dVector3 sides;

	// Setup

	// Draw each box
	int size = objectList.size();
	for ( int i = 0; i < size; ++i )
	{
		BoxObject obj = objectList[ i ];

		// Get the side lengths
		dGeomBoxGetLengths( obj.geom, sides );
		// Get the position/rotation
		const dReal *pos = dGeomGetPosition( obj.geom );
		const dReal *rot = dGeomGetRotation( obj.geom );

		setTexture( obj.texId, textureGreetCLUT );

		// Draw it
		pushTransform( pos, rot );
		drawBox( sides );
		glPopMatrix();
	}
}

////////

// The collision callback
static void nearCallback( void *data, dGeomID o1, dGeomID o2 )
{
	dContact contact[ 3 ];
	int i;

	dBodyID b1 = dGeomGetBody( o1 );
	dBodyID b2 = dGeomGetBody( o2 );

	if ( b1 == 0 && b2 == 0 )
		return;

	for ( i=0; i < 3; ++i )
	{
		contact[ i ].surface.mode = dContactBounce;
		contact[ i ].surface.mu = dInfinity;
		contact[ i ].surface.mu2 = 0;
		contact[ i ].surface.bounce = 0.6;
		contact[ i ].surface.bounce_vel = 0.2;
	}

	int numc = dCollide( o1, o2, 3, &contact[ 0 ].geom, sizeof( dContact ) );

	if ( numc > 0 )
	{
		for ( i = 0; i < numc; ++i )
		{
			dJointID c = dJointCreateContact( world, contactgroup, contact + i );
			dJointAttach( c, b1, b2 );
		}
	}
}

// Pushes the position/rotation matrix to the stack
static void pushTransform( const float pos[3], const float rot[12] )
{
	GLfloat matrix[ 16 ];

	matrix[ 0 ] = rot[ 0 ];
	matrix[ 1 ] = rot[ 4 ];
	matrix[ 2 ] = rot[ 8 ];
	matrix[ 3 ] = 0;

	matrix[ 4 ] = rot[ 1 ];
	matrix[ 5 ] = rot[ 5 ];
	matrix[ 6 ] = rot[ 9 ];
	matrix[ 7 ] = 0;

	matrix[ 8 ] = rot[ 2 ];
	matrix[ 9 ] = rot[ 6 ];
	matrix[ 10 ] = rot[ 10 ];
	matrix[ 11 ] = 0;

	matrix[ 12 ] = pos[ 0 ];
	matrix[ 13 ] = pos[ 1 ];
	matrix[ 14 ] = pos[ 2 ];
	matrix[ 15 ] = 1;

	glPushMatrix();
	glMultMatrixf( matrix );
}

// Draws a box
static void drawBox( const float sides[ 3 ] )
{
	float lx = sides[0]*0.5f;
	float ly = sides[1]*0.5f;
	float lz = sides[2]*0.5f;
	
	// sides
	glBegin (GL_TRIANGLE_STRIP);
	glNormal3f (-1,0,0);
	glTexCoord2f( 0, 0 );
	glVertex3f (-lx,-ly,-lz);
	glTexCoord2f( 0, 1 );
	glVertex3f (-lx,-ly,lz);
	glTexCoord2f( 1, 0 );
	glVertex3f (-lx,ly,-lz);
	glTexCoord2f( 1, 1 );
	glVertex3f (-lx,ly,lz);
	glNormal3f (0,1,0);
	glTexCoord2f( 0, 1 );
	glVertex3f (lx,ly,-lz);
	glTexCoord2f( 0, 0 );
	glVertex3f (lx,ly,lz);
	glNormal3f (1,0,0);
	glTexCoord2f( 1, 0 );
	glVertex3f (lx,-ly,-lz);
	glTexCoord2f( 1, 1 );
	glVertex3f (lx,-ly,lz);
	glNormal3f (0,-1,0);
	glTexCoord2f( 0, 1 );
	glVertex3f (-lx,-ly,-lz);
	glTexCoord2f( 0, 0 );
	glVertex3f (-lx,-ly,lz);
	glEnd();
	
	// top face
	glBegin( GL_TRIANGLE_FAN );
	glNormal3f( 0, 0, 1 );
	glTexCoord2f( 0, 0 );
	glVertex3f( -lx, -ly, lz );
	glTexCoord2f( 1, 0 );
	glVertex3f( lx, -ly, lz );
	glTexCoord2f( 1, 1 );
	glVertex3f( lx, ly, lz );
	glTexCoord2f( 0, 1 );
	glVertex3f( -lx, ly, lz );
	glEnd();
	
	// bottom face
	glBegin( GL_TRIANGLE_FAN );
	glNormal3f( 0, 0, -1 );
	glTexCoord2f( 1, 1 );
	glVertex3f( -lx, -ly, -lz );
	glTexCoord2f( 0, 1 );
	glVertex3f( -lx, ly, -lz );
	glTexCoord2f( 0, 0 );
	glVertex3f( lx, ly, -lz );
	glTexCoord2f( 1, 0 );
	glVertex3f( lx, -ly, -lz );
	glEnd();
}
