/* buffer.c
 *
 * Copyright 2001-2002 Vesa Halttunen (Vesuri/dA JoRMaS)
 *
 * This file is part of JRm-core.
 *
 * JRm-core is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * JRm-core is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JRm-core; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <png.h>
#include <jpeglib.h>
#include "buffer.h"

/* Note: there are endianess problems. */

/* Allocates a new empty buffer */
buffer *buffer_new(int width, int height) {
  buffer *b;

  /* Allocate a buffer and fill in supplied information */
  b=(buffer *)malloc(sizeof(buffer));
  b->width=width;
  b->height=height;
  b->data=(unsigned long *)calloc(width*height, sizeof(unsigned long));

  return b;
}

/* Frees a buffer */
void buffer_free(buffer *b) {
  if(b) {
    free(b->data);
    free(b);
  }
}

/* Allocates a buffer and loads the contents from a PNG or a JPEG file */
buffer *buffer_load(char *filename) {
  unsigned char header[8], *src, *dest;
  png_structp png_ptr;
  png_infop info_ptr, end_info;
  struct jpeg_decompress_struct cinfo;
  struct jpeg_error_mgr jerr;
  FILE *fp;
  int width, height, depth, colortype, i, j;
  png_bytep *row_pointers;
  buffer *b;

  /* Allocate a buffer structure - fill in information later */
  b=(buffer *)malloc(sizeof(buffer));

  /* Open the file */
  if(!(fp=fopen(filename, "rb")))
    return NULL;

  /* Read 8 bytes (header) */
  fread(header, 1, 8, fp);
  fseek(fp, 0, SEEK_SET);

  /* Check if it's a PNG file */
  if(!png_sig_cmp(header, 0, 8)) {
    /* Yes, do some more PNG checks */
    if(!(png_ptr=png_create_read_struct(PNG_LIBPNG_VER_STRING,
					(png_voidp)NULL,
					NULL, NULL)))
      return NULL;
    
    if(!(info_ptr=png_create_info_struct(png_ptr))) {
      png_destroy_read_struct(&png_ptr, (png_infopp)NULL, (png_infopp)NULL);
      return NULL;
    }
    
    if(!(end_info=png_create_info_struct(png_ptr))) {
      png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
      return NULL;
    }

    /* Initialize PNG file I/O */
    png_init_io(png_ptr, fp);

    /* Read PNG information */
    png_read_info(png_ptr, info_ptr);

    png_get_IHDR(png_ptr, info_ptr, &width, &height,
		 &depth, &colortype, NULL, NULL, NULL);

    /* Set transformations to ARGB */
    if((colortype==PNG_COLOR_TYPE_PALETTE && depth<=8) ||
       (colortype==PNG_COLOR_TYPE_GRAY && depth<8) ||
       (png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS)))
      png_set_expand(png_ptr);

    if(colortype==PNG_COLOR_TYPE_RGB ||
	colortype==PNG_COLOR_TYPE_RGB_ALPHA)
      png_set_bgr(png_ptr);

    if(colortype==PNG_COLOR_TYPE_RGB)
      png_set_filler(png_ptr, 0xffffffff, PNG_FILLER_AFTER);

    /*
    if(colortype==PNG_COLOR_TYPE_RGB_ALPHA)
      png_set_swap_alpha(png_ptr);
    */

    if(depth==16)
      png_set_strip_16(png_ptr);

    /* Update transformations */
    png_read_update_info(png_ptr, info_ptr);

    /* Fill in buffer information */
    b->width=width;
    b->height=height;
    b->data=(unsigned long *)calloc(width*height, sizeof(unsigned long));
    
    /* Allocate row pointers and fill them in */
    row_pointers=malloc(b->height*sizeof(png_bytep));
    for(i=0; i<b->height; i++)
      row_pointers[i]=&b->data[i*b->width];

    /* Read in the image */
    png_read_image(png_ptr, row_pointers);

    /* Unallocate stuff */
    png_read_end(png_ptr, end_info);
    png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);
    free(row_pointers);

    return b;
  } else {
    /* It wasn't a PNG file so assume it's a JPEG file */
    cinfo.err=jpeg_std_error(&jerr);
    jpeg_create_decompress(&cinfo);

    /* Initialize JPEG file I/O */
    jpeg_stdio_src(&cinfo, fp);

    /* Read JPEG information */
    jpeg_read_header(&cinfo, TRUE);

    jpeg_start_decompress(&cinfo);

    /* Fill in buffer information */
    b->width=cinfo.output_width;
    b->height=cinfo.output_height;
    b->data=(unsigned long *)calloc(cinfo.output_width*cinfo.output_height,
				    sizeof(unsigned long));

    /* Allocate row pointers and fill them in */
    row_pointers=malloc(b->height*sizeof(png_bytep));
    for(i=0; i<b->height; i++)
      row_pointers[i]=&b->data[i*b->width];

    /* Read in the image */
    while(cinfo.output_scanline<cinfo.output_height)
      jpeg_read_scanlines(&cinfo, &row_pointers[cinfo.output_scanline],
			  b->height);

    /* Transform BGR -> ARGB... or something */
    for(i=0; i<b->height; i++) {
      src=row_pointers[i]+(b->width-1)*3;
      dest=row_pointers[i]+(b->width-1)*4;
      for(j=0; j<b->width; j++) {
	dest[0]=src[2];
	dest[1]=src[1];
	dest[2]=src[0];
	dest[3]=0xff;
	dest-=4;
	src-=3;
      }
    }

    /* Unallocate stuff */
    jpeg_finish_decompress(&cinfo);
    jpeg_destroy_decompress(&cinfo);
    free(row_pointers);
 
    return b;
  }
}

/* Clears a buffer */
void buffer_clear(buffer *b) {
  if(b)
    memset(b->data, 0, b->width*b->height*sizeof(unsigned long));
}
