//
// 3D Studio MAX ASE Import routines by Morten Hustveit
//
// Poorly designed, based on consistency in input data.
//

#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <fstream>
#include "import.h"

#ifdef WIN32
using std::ifstream;
#endif

static char   Key[32];	// XXX: Multithread unfriendly
static char   Value[128];
static size_t CurrentMaterial = ~0;
static size_t BaseVertex = 0;
static size_t VertexCount = 0;
static size_t BaseColormapCoord = 0;
static size_t ColormapCoordCount = 0;

static bool split(const char* Line);
static void read_unknown(const char* Name, ifstream& File);
static void read_scene(import* Import, ifstream& File);
static void read_material_list(import* Import, ifstream& File);
static void read_material(import* Import, ifstream& File);
static void read_map_diffuse(i_material* Material, ifstream& File);
static void read_geomobject(import* Import, ifstream& File);
static void read_geom_node_tm(import* Import, ifstream& File);
static void read_mesh(import* Import, ifstream& File);
static void read_mesh_vertex_list(import* Import, ifstream& File);
static void read_mesh_face_list(import* Import, ifstream& File);
static void read_mesh_tvertlist(import* Import, ifstream& File);
static void read_mesh_tfacelist(import* Import, ifstream& File);
static void read_mesh_normals(import* Import, ifstream& File);

import* ase_load(const char* FileName)
{
	cerr << "INFO: ase_load: Loading '" << FileName << "'..." << endl;

	CurrentMaterial = ~0;
	VertexCount = 0;
	BaseVertex = 0;
	BaseColormapCoord = 0;
	ColormapCoordCount = 0;

	ifstream File(FileName);

	import* Import = new import;

	Import->Fields = IMPORT_VERTICES | IMPORT_COLORMAP_COORDS | IMPORT_FACES | IMPORT_MATERIALS | IMPORT_TEXTURE_FACES;


	string Line;

	while(std::getline(File, Line))
	{
		if(!split(Line.c_str()))
		{
			cerr << "WARNING: ase_load: Corrupt file." << endl;
			continue;
		}

		if(Value[strlen(Value) - 1] == '{')
		{
    		if(!strcmp(Key, "SCENE")) read_scene(Import, File);
			else if(!strcmp(Key, "MATERIAL_LIST")) read_material_list(Import, File);
			else if(!strcmp(Key, "GEOMOBJECT")) read_geomobject(Import, File);
			else read_unknown(Key, File);
		}
		else
		{
			if(!strcmp(Key, "3DSMAX_ASCIIEXPORT"))
			{
				if(strcmp(Value, "200"))
					cerr << "WARNING: ase_load: Probably incompatible file format version. Wanted: '200', actual: '" << Value << "'" << endl;
			}
			else if(!strcmp(Key, "COMMENT"))
			{
				if(strcmp(Value, "\"\""))
					cerr << "INFO: ase_load: Comment: " << Value << endl;
			}
			else cerr << "WARNING: ase_load: Unknown key '" << Key << "'." << endl;
		}
	}

	File.close();

	return Import;
}

static bool split(const char* Line)
{
	size_t i;

	Key[0] = Value[0] = 0;

	for(i = 0; Line[i]; i++)
	{
		if(Line[i] == '}')
			return false;

		if(!isspace(Line[i]))
			break;
	}

	char* Str = (char*) Line;

	while(*Str++ != '*');


	memset(Value, 0, 128);
	if(2 != sscanf(Str, "%[A-Z0-9_] %127c", Key, Value))
	{
		cerr << "WARNING: ase_load: Syntax error." << endl;
	}

	if(Value[0] == '"' && Value[strlen(Value) - 1] == '"')
	{
		for(i = 0; Value[i]; i++)
			Value[i] = Value[i + 1];

		if(i >= 2)
			Value[i - 2] = 0;
	}

	return true;
}

static void read_unknown(const char* Name, ifstream& File)
{
	cerr << "WARNING: ase_load: Unknown chunk '" << Name << "'" << endl;

	string Line;

	while(true)
	{
		if(!std::getline(File, Line))
		{
			cerr << "WARNING: ase_load: Premature end of file." << endl;
			break;
		}

		if(!split(Line.c_str()))
			break;

		if(Value[strlen(Value) - 1] == '{')
		{
			read_unknown(Key, File);
		}
		else
		{
			/* cerr << "WARNING: ase_load: Unknown key '" << Key << "'." << endl; */
		}
	}
}

static void read_scene(import* Import, ifstream& File)
{
	string Line;

	while(true)
	{
		if(!std::getline(File, Line))
		{
			cerr << "WARNING: ase_load: Premature end of file." << endl;
			break;
		}

		if(!split(Line.c_str()))
			break;

		if(Value[strlen(Value) - 1] == '{')
		{
			read_unknown(Key, File);
		}
		else
		{
			if(!strcmp(Key, "SCENE_FILENAME")) { }
			else if(!strcmp(Key, "SCENE_FIRSTFRAME")) {	}
			else if(!strcmp(Key, "SCENE_LASTFRAME")) { }
			else if(!strcmp(Key, "SCENE_FRAMESPEED")) { }
			else if(!strcmp(Key, "SCENE_TICKSPERFRAME")) { }
			else if(!strcmp(Key, "SCENE_BACKGROUND_STATIC")) { }
			else if(!strcmp(Key, "SCENE_AMBIENT_STATIC")) { }
			else cerr << "WARNING: ase_load: Unknown key '" << Key << "'." << endl;
		}
	}
}

static void read_material_list(import* Import, ifstream& File)
{
	string Line;

	while(true)
	{
		if(!std::getline(File, Line))
		{
			cerr << "WARNING: ase_load: Premature end of file." << endl;
			break;
		}

		if(!split(Line.c_str()))
			break;

		if(Value[strlen(Value) - 1] == '{')
		{
			if(!strcmp(Key, "MATERIAL")) read_material(Import, File);
			else read_unknown(Key, File);
		}
		else
		{
			if(!strcmp(Key, "MATERIAL_COUNT")) { }
			else cerr << "WARNING: ase_load: Unknown key '" << Key << "'." << endl;
		}
	}
}

static void read_material(import* Import, ifstream& File)
{
	i_material Material;

	size_t Index;

	if(1 != sscanf(Value, "%u", &Index))
	{
		cerr << "WARNING: ase_load: Syntax error." << endl;

		return;
	}

	Material.Index = Index;

	string Line;

	while(true)
	{
		if(!std::getline(File, Line))
		{
			cerr << "WARNING: ase_load: Premature end of file." << endl;
			break;
		}

		if(!split(Line.c_str()))
			break;

		if(Value[strlen(Value) - 1] == '{')
		{
			if(!strcmp(Key, "MAP_DIFFUSE")) read_map_diffuse(&Material, File);
			else read_unknown(Key, File);
		}
		else
		{
			if(!strcmp(Key, "MATERIAL_NAME")) { }
			else if(!strcmp(Key, "MATERIAL_CLASS")) { }
			else if(!strcmp(Key, "MATERIAL_AMBIENT")) { }
			else if(!strcmp(Key, "MATERIAL_DIFFUSE")) { }
			else if(!strcmp(Key, "MATERIAL_SPECULAR")) { }
			else if(!strcmp(Key, "MATERIAL_SHINE")) { }
			else if(!strcmp(Key, "MATERIAL_SHINESTRENGTH")) { }
			else if(!strcmp(Key, "MATERIAL_TRANSPARENCY")) { }
			else if(!strcmp(Key, "MATERIAL_WIRESIZE")) { }
			else if(!strcmp(Key, "MATERIAL_SHADING")) { }
			else if(!strcmp(Key, "MATERIAL_XP_FALLOFF")) { }
			else if(!strcmp(Key, "MATERIAL_SELFILLUM")) { }
			else if(!strcmp(Key, "MATERIAL_FALLOFF")) { }
			else if(!strcmp(Key, "MATERIAL_XP_TYPE")) { }
			else cerr << "WARNING: ase_load: Unknown key '" << Key << "'." << endl;
		}
	}

	Import->Materials.push_back(Material);
}

static void read_map_diffuse(i_material* Material, ifstream& File)
{
	string Line;

	while(true)
	{
		if(!std::getline(File, Line))
		{
			cerr << "WARNING: ase_load: Premature end of file." << endl;
			break;
		}

		if(!split(Line.c_str()))
			break;

		if(Value[strlen(Value) - 1] == '{')
		{
			read_unknown(Key, File);
		}
		else
		{
			if(!strcmp(Key, "MAP_NAME")) { }
			else if(!strcmp(Key, "MAP_CLASS")) { }
			else if(!strcmp(Key, "MAP_SUBNO")) { }
			else if(!strcmp(Key, "MAP_AMOUNT")) { }
			else if(!strcmp(Key, "BITMAP"))
			{
				Material->Colormap = Value;
			}
			else if(!strcmp(Key, "MAP_TYPE")) { }
			else if(!strcmp(Key, "UVW_U_OFFSET")) { }
			else if(!strcmp(Key, "UVW_V_OFFSET")) { }
			else if(!strcmp(Key, "UVW_U_TILING")) { }
			else if(!strcmp(Key, "UVW_V_TILING")) { }
			else if(!strcmp(Key, "UVW_ANGLE")) { }
			else if(!strcmp(Key, "UVW_BLUR")) { }
			else if(!strcmp(Key, "UVW_BLUR_OFFSET")) { }
			else if(!strcmp(Key, "UVW_NOUSE_AMT")) { } // Yep, 'nouse' is right.
			else if(!strcmp(Key, "UVW_NOISE_SIZE")) { }
			else if(!strcmp(Key, "UVW_NOISE_LEVEL")) { }
			else if(!strcmp(Key, "UVW_NOISE_PHASE")) { }
			else if(!strcmp(Key, "BITMAP_FILTER")) { }
			else cerr << "WARNING: ase_load: Unknown key '" << Key << "'." << endl;
		}
	}
}

static void read_geomobject(import* Import, ifstream& File)
{
	string Line;

	CurrentMaterial = ~0;

	while(true)
	{
		if(!std::getline(File, Line))
		{
			cerr << "WARNING: ase_load: Premature end of file." << endl;
			break;
		}

		if(!split(Line.c_str()))
			break;

		if(Value[strlen(Value) - 1] == '{')
		{
			if(!strcmp(Key, "NODE_TM")) read_geom_node_tm(Import, File);
			else if(!strcmp(Key, "MESH")) read_mesh(Import, File);
			else read_unknown(Key, File);
		}
		else
		{
			if(!strcmp(Key, "NODE_NAME")) { }
			else if(!strcmp(Key, "PROP_MOTIONBLUR")) { }
			else if(!strcmp(Key, "PROP_CASTSHADOW")) { }
			else if(!strcmp(Key, "PROP_RECVSHADOW")) { }
			else if(!strcmp(Key, "WIREFRAME_COLOR")) { }
			else if(!strcmp(Key, "MATERIAL_REF"))
			{
				sscanf(Value, "%u", &CurrentMaterial);
				
				std::vector<i_face>::iterator Face;

				for(Face = Import->Faces.begin(); Face != Import->Faces.end(); Face++)
				{
					if(Face->MaterialIndex == (size_t) ~0)
						Face->MaterialIndex = CurrentMaterial;
				}
			}
			else cerr << "WARNING: ase_load: Unknown key '" << Key << "'." << endl;
		}
	}
}

static void read_geom_node_tm(import* Import, ifstream& File)
{
	string Line;

	while(true)
	{
		if(!std::getline(File, Line))
		{
			cerr << "WARNING: ase_load: Premature end of file." << endl;
			break;
		}

		if(!split(Line.c_str()))
			break;

		if(Value[strlen(Value) - 1] == '{')
		{
			read_unknown(Key, File);
		}
		else
		{
			if(!strcmp(Key, "NODE_NAME")) {	}
			else if(!strcmp(Key, "INHERIT_POS")) { }
			else if(!strcmp(Key, "INHERIT_ROT")) { }
			else if(!strcmp(Key, "INHERIT_SCL")) { }
			else if(!strcmp(Key, "TM_ROW0")) {
				// sscanf(Value, "%f %f %f", &Import->Models.back()->Matrix(0, 0), &Import->Models.back()->Matrix(0, 1), &Import->Models.back()->Matrix(0, 2));
			}
			else if(!strcmp(Key, "TM_ROW1"))
			{
				// sscanf(Value, "%f %f %f", &Import->Models.back()->Matrix(1, 0), &Import->Models.back()->Matrix(1, 1), &Import->Models.back()->Matrix(1, 2));
			}
			else if(!strcmp(Key, "TM_ROW2"))
			{
				// sscanf(Value, "%f %f %f", &Import->Models.back()->Matrix(2, 0), &Import->Models.back()->Matrix(2, 1), &Import->Models.back()->Matrix(2, 2));
			}
			else if(!strcmp(Key, "TM_ROW3"))
			{
				// sscanf(Value, "%f %f %f", &Import->Models.back()->Matrix(3, 0), &Import->Models.back()->Matrix(3, 1), &Import->Models.back()->Matrix(3, 2));
			}
			else if(!strcmp(Key, "TM_POS")) { }
			else if(!strcmp(Key, "TM_ROTAXIS")) { }
			else if(!strcmp(Key, "TM_ROTANGLE")) { }
			else if(!strcmp(Key, "TM_SCALE")) { }
			else if(!strcmp(Key, "TM_SCALEAXIS")) { }
			else if(!strcmp(Key, "TM_SCALEAXISANG")) { }
			else cerr << "WARNING: ase_load: Unknown key '" << Key << "'." << endl;
		}
	}
}

static void read_mesh(import* Import, ifstream& File)
{
	string Line;

	while(true)
	{
		if(!std::getline(File, Line))
		{
			cerr << "WARNING: ase_load: Premature end of file." << endl;
			break;
		}

		if(!split(Line.c_str()))
			break;

		if(Value[strlen(Value) - 1] == '{')
		{
			if(!strcmp(Key, "MESH_VERTEX_LIST")) read_mesh_vertex_list(Import, File);
			else if(!strcmp(Key, "MESH_FACE_LIST")) read_mesh_face_list(Import, File);
			else if(!strcmp(Key, "MESH_NORMALS")) read_mesh_normals(Import, File);
			else if(!strcmp(Key, "MESH_TVERTLIST")) read_mesh_tvertlist(Import, File);
			else if(!strcmp(Key, "MESH_TFACELIST")) read_mesh_tfacelist(Import, File);
			else read_unknown(Key, File);
		}
		else
		{
			if(!strcmp(Key, "TIMEVALUE")) { }
			else if(!strcmp(Key, "MESH_NUMVERTEX")) { }
			else if(!strcmp(Key, "MESH_NUMFACES")) { }
			else if(!strcmp(Key, "MESH_NUMTVERTEX")) { }
			else if(!strcmp(Key, "MESH_NUMTVFACES")) { }
			else if(!strcmp(Key, "MESH_NUMCVERTEX")) { }
			else cerr << "WARNING: ase_load: Unknown key '" << Key << "'." << endl;
		}
	}
}

static void read_mesh_vertex_list(import* Import, ifstream& File)
{
	string Line;

	while(true)
	{
		if(!std::getline(File, Line))
		{
			cerr << "WARNING: ase_load: Premature end of file." << endl;
			break;
		}

		if(!split(Line.c_str()))
			break;

		if(Value[strlen(Value) - 1] == '{')
		{
			read_unknown(Key, File);
		}
		else
		{
			if(!strcmp(Key, "MESH_VERTEX"))
			{
				size_t Index;
				i_vector Vertex;

				sscanf(Value, "%u %f %f %f", &Index, &Vertex.x, &Vertex.z, &Vertex.y);

				Import->Vertices.push_back(Vertex);

				VertexCount++;
			}
			else cerr << "WARNING: ase_load: Unknown key '" << Key << "'." << endl;
		}
	}
}

static void read_mesh_face_list(import* Import, ifstream& File)
{
	string Line;

	while(true)
	{
		if(!std::getline(File, Line))
		{
			cerr << "WARNING: ase_load: Premature end of file." << endl;
			break;
		}

		if(!split(Line.c_str()))
			break;

		if(Value[strlen(Value) - 1] == '{')
		{
			read_unknown(Key, File);
		}
		else
		{
			if(!strcmp(Key, "MESH_FACE"))
			{
				size_t Index;
				i_face Face;

				Face.IndexCount = 3;
				Face.Indices = new size_t[Face.IndexCount];

				sscanf(Value, "%u: A: %u B: %u C: %u", &Index, &Face.Indices[0], &Face.Indices[1], &Face.Indices[2]);

				Face.MaterialIndex = CurrentMaterial;
				Face.Indices[0] += BaseVertex;
				Face.Indices[1] += BaseVertex;
				Face.Indices[2] += BaseVertex;

				Import->Faces.push_back(Face);
			}
			else cerr << "WARNING: ase_load: Unknown key '" << Key << "'." << endl;
		}
	}

	BaseVertex += VertexCount;
	VertexCount = 0;
}

static void read_mesh_normals(import* Import, ifstream& File)
{
	string Line;

	while(true)
	{
		if(!std::getline(File, Line))
		{
			cerr << "WARNING: ase_load: Premature end of file." << endl;
			break;
		}

		if(!split(Line.c_str()))
			break;

		if(Value[strlen(Value) - 1] == '{')
		{
			read_unknown(Key, File);
		}
		else
		{
			if(!strcmp(Key, "MESH_FACENORMAL"))
			{
				size_t   Index;
				i_vector Normal;

				sscanf(Value, "%u %f %f %f", &Index, &Normal.x, &Normal.y, &Normal.z);

				Import->FaceNormals.push_back(Normal);
			}
			else if(!strcmp(Key, "MESH_VERTEXNORMAL"))
			{
				size_t   Index;
				i_vector Normal;

				sscanf(Value, "%u %f %f %f", &Index, &Normal.x, &Normal.y, &Normal.z);

				Import->VertexNormals.push_back(Normal);
			}
			else cerr << "WARNING: ase_load: Unknown key '" << Key << "'." << endl;
		}
	}
}

static void read_mesh_tvertlist(import* Import, ifstream& File)
{
	string Line;

	while(true)
	{
		if(!std::getline(File, Line))
		{
			cerr << "WARNING: ase_load: Premature end of file." << endl;
			break;
		}

		if(!split(Line.c_str()))
			break;

		if(Value[strlen(Value) - 1] == '{')
		{
			read_unknown(Key, File);
		}
		else
		{
			if(!strcmp(Key, "MESH_TVERT"))
			{
				size_t Index;
				i_map_coord ColormapCoord;

				sscanf(Value, "%u %f %f %*f", &Index, &ColormapCoord.u, &ColormapCoord.v);

				ColormapCoord.v = -ColormapCoord.v;

				Import->ColormapCoords.push_back(ColormapCoord);

				ColormapCoordCount++;
			}
			else cerr << "WARNING: ase_load: Unknown key '" << Key << "'." << endl;
		}
	}
}

static void read_mesh_tfacelist(import* Import, ifstream& File)
{
	string Line;

	while(true)
	{
		if(!std::getline(File, Line))
		{
			cerr << "WARNING: ase_load: Premature end of file." << endl;
			break;
		}

		if(!split(Line.c_str()))
			break;

		if(Value[strlen(Value) - 1] == '{')
		{
			read_unknown(Key, File);
		}
		else
		{
			if(!strcmp(Key, "MESH_TFACE"))
			{
				i_face Face;
				size_t Index;

				Face.IndexCount = 3;
				Face.Indices = new size_t[Face.IndexCount];

				sscanf(Value, "%u %u %u %u", &Index, &Face.Indices[0], &Face.Indices[1], &Face.Indices[2]);

				Face.MaterialIndex = CurrentMaterial;
				Face.Indices[0] += BaseColormapCoord;
				Face.Indices[1] += BaseColormapCoord;
				Face.Indices[2] += BaseColormapCoord;

				Import->TextureFaces.push_back(Face);
			}
			else cerr << "WARNING: ase_load: Unknown key '" << Key << "'." << endl;
		}
	}

	BaseColormapCoord += ColormapCoordCount;
	ColormapCoordCount = 0;
}

