/*-------------------------------------------------------------------------------------------*\
|                                                                                             |
|    Copyright (C) 1999 by Punk Productions Electronic Entertainment. All Rights Reserved.    | 
|                                                                                             |
|    Purpose.....: Shows how to load JPEG files into a DDsurface                              |
|    Programmer..: Nikolaus Brennig, 19, (Nikolaus.Brennig@nol.at)                            |
|	 Contact.....: virtalnik@nol.at || ICQ: 20345533										  |
|	 Internet....: http://cust.nol.at/ppee													  |
|    Help........: Joine the PPEE-GamesDev MailingList			                              |
|    Day created.: February 17th, 1999                                                        |
|    Update......: March, 30th, 1999                                                          |
|				   Can't remember anymore what i've changed :)								  |
|				   October, 3rd, 1999:														  |
|				   Completely rewritten, Faster Loading/Saving, C++ Interface                 |
|				   now the original JPEGLib 6.2 is used, also Support for                     |
|				   loading PPM/PGM files. Written for DirectX7		                          |
|				   October, 29th, 1999:	      												  |
|				   Fixed a small memorybug in LoadJPEG. Thanks to Fabien Pigre.              |
|				   Also converted the PseudoJPGClass into a NameSpace. Much better here          |
|                                                                                             |
\*-------------------------------------------------------------------------------------------*/

#include <stdio.h>
#include "JpgLoader.h"

extern "C" 
{
//	#define HAVE_BOOLEAN, enable this if you have to...
	#include <cdjpeg.h>
	#include <jpeglib.h>
	#include <setjmp.h>
}

//---------------------------------------------------------------------------------------------
// Externals...
//---------------------------------------------------------------------------------------------
//extern Error( char *err );

//---------------------------------------------------------------------------------------------
// error handler...
//---------------------------------------------------------------------------------------------
struct my_error_mgr
{
	struct	jpeg_error_mgr pub;
	jmp_buf setjmp_buffer; 
};

typedef struct my_error_mgr *my_error_ptr;

METHODDEF(void) my_error_exit( j_common_ptr cinfo );

//---------------------------------------------------------------------------------------------
// to handle fatal errors...
//---------------------------------------------------------------------------------------------
METHODDEF(void) my_error_exit( j_common_ptr cinfo )
{
    my_error_ptr  myerr = (my_error_ptr) cinfo->err;
    CHAR          buffer[JMSG_LENGTH_MAX];
   
    (*cinfo->err->format_message) (cinfo, buffer);
    MessageBox( NULL, buffer, "JPEG Fatal Error", MB_ICONSTOP );
    longjmp(myerr->setjmp_buffer, 1);
}


//---------------------------------------------------------------------------------------------
// read a JPEG file into a 24bit Buffer...
//---------------------------------------------------------------------------------------------
BYTE *JPEG::LoadJPEG( LPSTR fileName, UINT *width, UINT *height, INT *Bits )
{
	struct			jpeg_decompress_struct cinfo;
    struct			my_error_mgr jerr;
    FILE			*infile=NULL;       
    JSAMPARRAY		buffer;	
    BYTE			*dataBuf;


	char debug[260];
	// First open the file...
    infile = fopen(fileName, "rb");
	if( infile == NULL ) 
	{
		sprintf(debug,"error while loading: %s",fileName);
		MessageBox(NULL, debug,"z",MB_OK );
	}


    // We set up the normal JPEG error routines, then override error_exit...
    cinfo.err = jpeg_std_error(&jerr.pub);
    jerr.pub.error_exit = my_error_exit;

    // Establish the setjmp return context for my_error_exit to use...
    if( setjmp(jerr.setjmp_buffer) )
	{
		jpeg_destroy_decompress(&cinfo);
        if( infile != NULL ) fclose(infile);
        return NULL;
	}

    // Now we can initialize the JPEG decompression object...
    jpeg_create_decompress(&cinfo);

    // Tell JPEG Lib that we get the data from a file...
    jpeg_stdio_src(&cinfo, infile);

    // Now read the Header...
    jpeg_read_header(&cinfo, TRUE);

    // Start the decompressor...
    jpeg_start_decompress(&cinfo);

	// Compute WidthBytes for 24bit...
	INT WidthBytes = cinfo.output_width*3;
	if(WidthBytes & 0x003) WidthBytes = (WidthBytes | 3) + 1;

	// Alloc Memory to hold the decompressed pic...
	dataBuf = new BYTE[WidthBytes * cinfo.output_height];

    // how big is this thing gonna be?
    *width  = cinfo.output_width;
    *height = cinfo.output_height;
	*Bits   = cinfo.out_color_components << 3;

    // Make a one-row-high sample array that will go away when done with image...
    buffer = (*cinfo.mem->alloc_sarray) ((j_common_ptr) &cinfo, JPOOL_IMAGE, WidthBytes, 1);

	// Now read row by row the data into our buffer...
    if (cinfo.out_color_components==3) // and RGB Image...
	{
		INT OurRow = (cinfo.output_height-1)*WidthBytes;
		while (cinfo.output_scanline < cinfo.output_height)
		{
			// Read Scanline...
			jpeg_read_scanlines(&cinfo, buffer, 1);

			// Write Scanline, plus RGB->BGR and flipping...
			INT		col = 0;
			BYTE	*jpegline = buffer[0];
			for( UINT count=0; count<cinfo.output_width; count++, col += 3 ) 
			{
				dataBuf[OurRow + col + 0] = *(jpegline + col + 2); 
				dataBuf[OurRow + col + 1] = *(jpegline + col + 1);
				dataBuf[OurRow + col + 2] = *(jpegline + col + 0);
			}
			OurRow -= WidthBytes;
		}
	}
	else // Its a Grayscale one...
	{
		INT OurRow = (cinfo.output_height-1)*WidthBytes;
		while (cinfo.output_scanline < cinfo.output_height)
		{
			// Read Scanline...
			jpeg_read_scanlines(&cinfo, buffer, 1);

			// Write Scanline, plus flipping...
			INT		col = 0;
			BYTE	iGray = 0;
			BYTE	*jpegline = buffer[0];
			for( UINT count=0; count<cinfo.output_width; count++, col+=3 ) 
			{
		        iGray = *(jpegline + count);
				dataBuf[OurRow + col + 0] = iGray; 
				dataBuf[OurRow + col + 1] = iGray;
				dataBuf[OurRow + col + 2] = iGray;
			}
			OurRow -= WidthBytes;
		}
	}

	// Free the stuff...
    jpeg_finish_decompress(&cinfo);
    jpeg_destroy_decompress(&cinfo);
    fclose(infile);

	// Yeah...
	return dataBuf;
}


//---------------------------------------------------------------------------------------------
// read a PPM/PGM file...
//---------------------------------------------------------------------------------------------
BYTE *JPEG::LoadPPM( LPSTR fileName, UINT *width, UINT *height, INT *Bits )
{
	cjpeg_source_ptr	src_mgr;
	struct				jpeg_error_mgr jerr;
	struct				jpeg_compress_struct cinfo;
    FILE				*infile = NULL;
    BYTE				*dataBuf;


	// Zero down...
    *width=0; *height=0;

	// Open the file...
    if( (infile = fopen(fileName, "rb")) == NULL )
	{
		CHAR	buf[250];
		sprintf( buf, "JPEG :\nCan't open %s\n", fileName );
        MessageBox(NULL,buf,"error",MB_OK);
        return NULL;
	}

	// Create compression object, and default errhandler...
	cinfo.err = jpeg_std_error(&jerr);
	jpeg_create_compress(&cinfo);

	// Set some defaults, we don't really need them...
	cinfo.in_color_space = JCS_RGB;
	jpeg_set_defaults(&cinfo);

	// Init the JPEG internal PPM/PGM Loader...
	src_mgr = jinit_read_ppm(&cinfo);
	src_mgr->input_file = infile;

	// Read the input file header to obtain file size & colorspace...
	(*src_mgr->start_input) (&cinfo, src_mgr);

	// how big is this thing gonna be?
    *width  = cinfo.image_width;
    *height = cinfo.image_height;
	*Bits   = cinfo.input_components << 3;
	INT WidthBytes = cinfo.image_width*3;
	if(WidthBytes & 0x003) WidthBytes = (WidthBytes | 3) + 1;

    // alloc and open our new buffer...
	dataBuf = new BYTE[WidthBytes * cinfo.image_height];
    if (dataBuf==NULL)
	{
		MessageBox(NULL,"JpegFile :\nOut of memory","error",MB_OK);
        fclose(infile);
        return NULL;
	}

	// Now load the Picdata into our buffer we will later return...
	INT		w = *width, h = *height;
	UINT	Row = 0;

    if( cinfo.in_color_space == JCS_GRAYSCALE ) // It's a pgm file...
	{
		do
		{
			Row++;
			UINT num_scanlines = (*src_mgr->get_pixel_rows) (&cinfo, src_mgr);
			WriteGRAYScanline(*src_mgr->buffer, *width, dataBuf, cinfo.image_height-Row);
		}
		while( Row < cinfo.image_height );
	}
	else // Its a ppm...
	{
		do
		{
			Row++;
			UINT num_scanlines = (*src_mgr->get_pixel_rows) (&cinfo, src_mgr);
			WriteRGBScanline(*src_mgr->buffer, *width, dataBuf, cinfo.image_height-Row);
		}
		while( Row < cinfo.image_height );
	}

	// Free everything...
	(*src_mgr->finish_input) (&cinfo, src_mgr);
	jpeg_destroy_compress(&cinfo);
    fclose(infile);

	return dataBuf;
}


//---------------------------------------------------------------------------------------------
// Write the JPEG file...
//---------------------------------------------------------------------------------------------
BOOL JPEG::SaveJPEG( LPSTR fileName, BYTE *dataBuf, UINT widthPix, UINT height, 
						   BOOL color, int quality )
{
	struct	jpeg_compress_struct cinfo;
	struct	my_error_mgr jerr;
	LPBYTE  tmp;
	FILE    *outfile=NULL;


	// Get WidthBytes for 24bit...
	INT WidthBytes = widthPix*3;
	if(WidthBytes & 0x003) WidthBytes = (WidthBytes | 3) + 1;

	// Get WidthBytes for 8bit (GrayScale)...
	INT WidthBytes8bit = widthPix;
	if(WidthBytes8bit & 0x003) WidthBytes8bit = (WidthBytes8bit | 3) + 1;

	// Grayscale image if reqired...
	if (!color) 
	{
		tmp = new BYTE[WidthBytes8bit*height];

		UINT row,col;
		UINT Row8Bit = 0;
		UINT Row24Bit = 0;
		UINT Col24Bit = 0;
		for (row=0;row<height;row++) 
		{
			for (col=0;col<widthPix;col++) 
			{
				LPBYTE pRed, pGrn, pBlu;
				pRed = dataBuf + Row24Bit + Col24Bit;
				pGrn = dataBuf + Row24Bit + Col24Bit + 1;
				pBlu = dataBuf + Row24Bit + Col24Bit + 2;

				// luminance
				int lum = (int)(.299 * (double)(*pRed) + 
								.587 * (double)(*pGrn) + 
								.114 * (double)(*pBlu));

				LPBYTE pGray;
				pGray = tmp + Row8Bit + col;
				*pGray = (BYTE)lum;

				Col24Bit += 3;
			}
			Row8Bit += WidthBytes8bit;
			Row24Bit += WidthBytes;
			Col24Bit = 0;
		}
	}

    /* Step 1: allocate and initialize JPEG compression object */
    cinfo.err = jpeg_std_error(&jerr.pub);
    jerr.pub.error_exit = my_error_exit;

    /* Establish the setjmp return context for my_error_exit to use. */
    if (setjmp(jerr.setjmp_buffer))
	{
		jpeg_destroy_compress(&cinfo);
        if (outfile!=NULL) fclose(outfile);
        if (!color) delete [] tmp;
        return FALSE;
	}

    /* Now we can initialize the JPEG compression object. */
    jpeg_create_compress(&cinfo);

    /* Step 2: specify data destination (eg, a file) */
    if ((outfile = fopen(fileName, "wb")) == NULL)
	{
		char buf[250];
        sprintf(buf, "JpegFile :\nCan't open %s\n", fileName);
        MessageBox(NULL,buf,"error",MB_OK);
        return FALSE;
	}

    jpeg_stdio_dest(&cinfo, outfile);

    /* Step 3: set parameters for compression */
    cinfo.image_width = widthPix;   /* image widthPix and height, in pixels */
    cinfo.image_height = height;
    if (color)
	{
		cinfo.input_components = 3;             /* # of color components per pixel */
        cinfo.in_color_space = JCS_RGB;         /* colorspace of input image */
	}
    else
	{
        cinfo.input_components = 1;             /* # of color components per pixel */
        cinfo.in_color_space = JCS_GRAYSCALE;   /* colorspace of input image */
	}

    jpeg_set_defaults(&cinfo);
    jpeg_set_quality(&cinfo, quality, TRUE);

    /* Step 4: Start compressor */
    jpeg_start_compress(&cinfo, TRUE);

    /* Step 5: while (scan lines remain to be written) */
    /*           jpeg_write_scanlines(...); */
	while (cinfo.next_scanline < cinfo.image_height) 
	{
		LPBYTE outRow;
		if (color) 
		{
			outRow = dataBuf + (cinfo.next_scanline * WidthBytes);
		} 
		else 
		{
			outRow = tmp + (cinfo.next_scanline * WidthBytes8bit);
		}

		(void) jpeg_write_scanlines(&cinfo, &outRow, 1);
	}

    /* Step 6: Finish compression */
    jpeg_finish_compress(&cinfo);

    /* After finish_compress, we can close the output file. */
    fclose(outfile);

    /* This is an important step since it will release a good deal of memory. */
    jpeg_destroy_compress(&cinfo);

    if( !color ) delete tmp;
  
	return TRUE;
}


//---------------------------------------------------------------------------------------------
// Make a RGB Buffer....
//---------------------------------------------------------------------------------------------
BYTE *JPEG::MakeRGBAligned( BYTE *inBuf, UINT widthPix, UINT widthBytes, UINT height )
{
	BYTE *tmp;
	UINT row;

    if( inBuf == NULL ) return NULL;

	INT WidthBytes = widthPix*3;
	if(WidthBytes & 0x003) WidthBytes = (WidthBytes | 3) + 1;
	tmp = new BYTE[WidthBytes * height];
    if( tmp == NULL )   return NULL;

    for( row=0; row<height; row++ )
	{
		memcpy( (tmp + row * WidthBytes), (inBuf + row * WidthBytes), WidthBytes );
	}

    return tmp;
}


//---------------------------------------------------------------------------------------------
// copies BYTE buffer into DWORD-aligned BYTE buffer
// return addr of new buffer
//---------------------------------------------------------------------------------------------
BYTE *JPEG::MakeDWORDAligned( BYTE *dataBuf, UINT widthPix, UINT height, 
							   UINT *uiOutWidthBytes )
{
	UINT  uiWidthBytes;
	DWORD dwNewsize;
	BYTE  *pNew;
	UINT  uiInWidthBytes;
	UINT  uiCount;
	BYTE  *bpInAdd;
	BYTE  *bpOutAdd;
	ULONG lInOff;
	ULONG lOutOff;

	uiWidthBytes = widthPix*3;
	if(uiWidthBytes & 0x003) uiWidthBytes = (uiWidthBytes | 3) + 1;
    dwNewsize      = (DWORD)( (DWORD)uiWidthBytes * (DWORD)height );
    uiInWidthBytes = widthPix * 3;
 	if(uiInWidthBytes & 0x003) uiInWidthBytes = (uiInWidthBytes | 3) + 1;
   
	pNew = new BYTE[dwNewsize];
        
	lInOff = 0;
	lOutOff = 0;
    for( uiCount=0; uiCount<height; uiCount++ )
	{
		bpInAdd  = dataBuf + lInOff;
        bpOutAdd = pNew + lOutOff;

        memcpy( bpOutAdd, bpInAdd, uiInWidthBytes );

		lInOff += uiInWidthBytes;
		lOutOff += uiWidthBytes;
	}

    *uiOutWidthBytes=uiWidthBytes;
  
	return pNew;
}


//---------------------------------------------------------------------------------------------
// vertically flip a buffer 
//---------------------------------------------------------------------------------------------
BOOL JPEG::FlipVertically( BYTE *inbuf, UINT widthBytes, UINT height )
{   
	BYTE  *tb1;
	BYTE  *tb2;
	UINT  row_cnt;     
	ULONG off1=0;
	ULONG off2=0;
	UINT  bufsize;

    bufsize = widthBytes;

	tb1 = new BYTE[bufsize];
	tb2 = new BYTE[bufsize];

	off2 = (height-1) * bufsize;
    for( row_cnt=0; row_cnt<((height+1)>>1); row_cnt++ )
	{
        memcpy( tb1, inbuf+off1, bufsize );
        memcpy( tb2, inbuf+off2, bufsize ); 
        memcpy( inbuf+off1, tb2, bufsize );
        memcpy( inbuf+off2, tb1, bufsize );

		off1 += bufsize;
		off2 -= bufsize;
	}       

    delete tb1;
    delete tb2;

	return TRUE;
}        


//---------------------------------------------------------------------------------------------
// swap Rs and Bs...
//---------------------------------------------------------------------------------------------
BOOL JPEG::BGRFromRGB( BYTE *buf, UINT widthPix, UINT height )
{
	UINT   col, row;
	LPBYTE pRed, pGrn, pBlu;
	BYTE   tmp;
	INT TmpRow = 0;
	INT WidthBytes = widthPix*3;
	if( WidthBytes & 0x003 ) WidthBytes = (WidthBytes | 3) + 1;
	INT OurCol = 0;
    for( row=0; row<height; row++ )
	{
		for( col=0; col<widthPix; col++ )
        {
			pRed = buf + TmpRow + OurCol;
            pGrn = pRed + 1;
            pBlu = pRed + 2;

            tmp = *pRed;
            *pRed = *pBlu;
            *pBlu = tmp;

			OurCol += 3;
		}
		TmpRow += WidthBytes;
		OurCol = 0;
	}
  
	return TRUE;
}


//---------------------------------------------------------------------------------------------
// swap Rs and Bs...
//---------------------------------------------------------------------------------------------
BOOL JPEG::RGBFromBGR( BYTE *buf, UINT widthPix, UINT height )
{
	UINT   col, row;
	LPBYTE pRed, pGrn, pBlu;
	BYTE   tmp;
	INT TmpRow = 0;
	INT WidthBytes = widthPix*3;
	if( WidthBytes & 0x003 ) WidthBytes = (WidthBytes | 3) + 1;
	INT OurCol = 0;
    for( row=0; row<height; row++ )
	{
		for( col=0; col<widthPix; col++ )
        {
			pRed = buf + TmpRow + OurCol;
            pGrn = pRed + 1;
            pBlu = pRed + 2;

            tmp = *pBlu;
            *pBlu = *pRed;
            *pRed = tmp;

			OurCol += 3;
		}
		TmpRow += WidthBytes;
		OurCol = 0;
	}
  
	return TRUE;
}


//---------------------------------------------------------------------------------------------
// stash a scanline...
//---------------------------------------------------------------------------------------------
VOID JPEG::WriteRGBScanline( BYTE *jpegline, int widthPix, BYTE *outBuf, int row )
{
	INT WidthBytes = widthPix*3;
	if(WidthBytes & 0x003) WidthBytes = (WidthBytes | 3) + 1;
	int offset = row * WidthBytes;
	int col = 0;
    
	for( int count=0; count<widthPix; count++ ) 
	{
		*(outBuf + offset + col + 0) = *(jpegline + col + 2); 
        *(outBuf + offset + col + 1) = *(jpegline + col + 1);
        *(outBuf + offset + col + 2) = *(jpegline + col + 0);
		col += 3;
	}
}


//---------------------------------------------------------------------------------------------
// stash a gray scanline
//---------------------------------------------------------------------------------------------
VOID JPEG::WriteGRAYScanline( BYTE *jpegline, int widthPix, BYTE *outBuf, int row )
{
	INT WidthBytes = widthPix*3;
	if(WidthBytes & 0x003) WidthBytes = (WidthBytes | 3) + 1;
	int		offset = row * WidthBytes;
	int		col = 0;
	BYTE	iGray = 0;

    for( int count=0; count<widthPix; count++ )
	{
        iGray = *(jpegline + count);
        *(outBuf + offset + col + 0) = iGray;
        *(outBuf + offset + col + 1) = iGray;
        *(outBuf + offset + col + 2) = iGray;
		col += 3;
	}
}


// End...
