//
// Java Console class for OpenPTC 1.0 Java Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
// See http://www.gnu.org/copyleft/lgpl.html for details
//

// package name
package ptc.java;

// import classes
import ptc.Key;
import ptc.Area;
import ptc.Mode;
import ptc.Error;
import ptc.Color;
import ptc.Format;
import ptc.Palette;
import ptc.Clipper;
import java.awt.Graphics;
import java.awt.Component;
import java.awt.Dimension;



public class Console implements ptc.base.Console
{
    public Console() throws Error
    {
        // register console
        Error.register(this);

        // clear mode list
        modes[0] = new Mode();

        // create palette
        palette = new Palette();

        // setup defaults
        default_width = 320;
        default_height = 200;
        default_pages = 2;
        default_format = new Format(32,0x00FF0000,0x0000FF00,0x000000FF);
    }

    protected void finalize() throws Error
    {
        // unregister console
        Error.unregister(this);

        // close
        close();
    }

    public void configure(String file) throws Error
    {
        // configure console?
    }

    public boolean option(String option) throws Error
    {
        // not recognized
        return false;
    }

    public Mode[] modes() throws Error
    {
        // get modes array clone
        return (Mode[]) modes.clone();
    }

    public void open(String title) throws Error
    {
        // open console
        open(title,default_format);
    }

    public void open(String title,int pages) throws Error
    {
        // open console
        open(title,default_format,pages);
    }

    public void open(String title,Format format) throws Error
    {
        // open console
        open(title,default_width,default_height,format);
    }

    public void open(String title,Format format,int pages) throws Error
    {
        // open console
        open(title,default_width,default_height,format,pages);
    }

    public void open(String title,int width,int height,Format format) throws Error
    {
        // open console
        open(title,new Mode(width,height,format));
    }

    public void open(String title,int width,int height,Format format,int pages) throws Error
    {
        // open console
        open(title,new Mode(width,height,format),pages);
    }

    public void open(String title,Mode mode) throws Error
    {
        // open console
        open(title,mode,default_pages);
    }

    public void open(String title,Mode mode,int pages) throws Error
    {
        // close
        close();

        // create console window
        window = new Window(title,mode.width(),mode.height());

        // internal console open
        internal_open(mode.width(),mode.height(),mode.format(),pages);
    }

    public void close() throws Error
    {
        // check
        if (open)
        {
            // internal close
            internal_close();

            // close display
            display.close();

            // close window
            window.close();

            // closed
            open = false;
        }
    }

    public void flush() throws Error
    {
        // flush display
        display.flush();
    }

    public void finish() throws Error
    {
        // finish display
        display.finish();
    }

    public void update() throws Error
    {
        // update display
        display.update();

        // update window
        window.update();
    }   

    public void update(Area area) throws Error
    {
        // update display area
        display.update(area);

        // update window
        window.update();
    }

    public boolean key() throws Error
    {
        // check for key press
        return keyboard.key();
    }

    public Key read() throws Error
    {
        // read key press
        return keyboard.read();
    }

    public void copy(ptc.base.Surface surface) throws Error
    {
        // error message
        throw new Error("copy console to surface is not implemented");
    }

    public void copy(ptc.base.Surface surface,Area source,Area destination) throws Error
    {
        // error message
        throw new Error("copy console to surface area is not implemented");
    }

    public Object lock() throws Error
    {
        // error message
        throw new Error("console lock is not supported");
    }

    public void unlock() throws Error
    {
        // unlock surface
        throw new Error("console unlock is not supported"); 
    }

    public void load(Object pixels,int width,int height,int pitch,Format format,Palette palette) throws Error
    {
        // check clip area
        if (clip.compare(area))
        {
            // load pixels to display
            display.load(pixels,width,height,pitch,format,palette);
        }
        else
        {
            // load explicit areas
            load(pixels,width,height,pitch,format,palette,new Area(0,0,width,height),this.area);
        }
    }

    public void load(Object pixels,int width,int height,int pitch,Format format,Palette palette,Area source,Area destination) throws Error
    {
        // clip source and destination areas
        Area clipped_source = new Area();
        Area clipped_destination = new Area();
        Clipper.clip(source,new Area(0,0,width,height),clipped_source,destination,this.clip,clipped_destination);
        
        // load pixels to display area
        display.load(pixels,width,height,pitch,format,palette,clipped_source,clipped_destination);
    }

    public void save(Object pixels,int width,int height,int pitch,Format format,Palette palette) throws Error
    {
        // check clip area
        if (clip.compare(area))
        {
            // save display pixels
            display.save(pixels,width,height,pitch,format,palette);
        }
        else
        {
            // save explicit areas
            save(pixels,width,height,pitch,format,palette,this.area,new Area(0,0,width,height));
        }
    }

    public void save(Object pixels,int width,int height,int pitch,Format format,Palette palette,Area source,Area destination) throws Error
    {
        // clip source and destination areas
        Area clipped_source = new Area();
        Area clipped_destination = new Area();
        Clipper.clip(source,this.clip,clipped_source,destination,new Area(0,0,width,height),clipped_destination);

        // save display area pixels
        display.save(pixels,width,height,pitch,format,palette,clipped_source,clipped_destination);
    }

    public void clear() throws Error
    {
        // check console format
        if (format().direct())
        {
            // direct color
            clear(new Color(0,0,0,0));
        }
        else
        {
            // indexed color
            clear(new Color(0));
        }
    }

    public void clear(Color color) throws Error
    {
        // clear console
        clear(color,area);
    }

    public void clear(Color color,Area area) throws Error
    {
        // clip clear area
        Area clipped_area = Clipper.clip(area,clip);

        // clear display area
        display.clear(color,clipped_area);
    }

    public void palette(Palette palette) throws Error
    {
        // get console palette object
        Palette p = this.palette;

        // update palette data
        p.load(palette.data());

        // update display palette
        display.palette(p);
    }

    public Palette palette() throws Error
    {
        // get palette copy
        return palette.copy();
    }

    public void clip(Area area) throws Error
    {
        // set new clip area
        clip = Clipper.clip(area,this.area);
    }

    public int width() throws Error
    {
        // get width
        return width;
    }

    public int height() throws Error
    {
        // get height
        return height;
    }

    public int pages() throws Error
    {
        // get pages
        return pages;
    }

    public int pitch() throws Error
    {
        // get pitch
        return pitch;
    }

    public Area area() throws Error
    {
        // get area copy
        return area.copy();
    }

    public Area clip() throws Error
    {
        // get clip copy
        return clip.copy();
    }
    
    public Format format() throws Error
    {
        // get format copy
        return format.copy();
    }

    public String name() throws Error
    {
        // get name
        return "Java";
    }

    public String title() throws Error
    {
        // get window title
        return window.title();
    }

    public String information() throws Error
    {   
        // setup java system information
        return property("java.version") + 
               property("java.vendor") + 
               property("java.vendor.url") +
               property("java.home") +
               property("java.class.version") +
               property("java.class.path") +
               property("os.name") +
               property("os.arch") +
               property("os.version") +
               property("user.name") +
               property("user.home") +
               property("user.dir") +
               property("file.separator") +
               property("path.separator") +
               property("line.separator");
    }

    public void open(Component component,Format format) throws Error
    {
        // open console
        open(component,format,default_pages);
    }

    public void open(Component component,Format format,int pages) throws Error
    {
        // close
        close();

        // get component size
        Dimension size = component.getSize();
        
        // create console window
        window = new Window(component,size.width,size.height);

        // internal console open
        internal_open(size.width,size.height,format,pages);
    }

    public void paint(Graphics graphics) throws Error
    {
        // paint window
        window.paint(graphics);
    }

    public Component component()
    {
        // get window component
        return window.component();
    }

    private String property(String key)
    {
        try
        {
            // get system property
            String property = System.getProperty(key);

            // check for null property string
            if (property==null) property = "(unknown)";

            // setup system property string
            return key + ": " + property + "\n";
        }
        catch (SecurityException e)
        {
            // security exception
            return key + ": (unknown)\n";
        }
    }

    private void internal_open(int width,int height,Format format,int pages) throws Error
    {
        // setup data
        this.width  = width;
        this.height = height;
        this.format = format;
        this.pages  = pages;

        // setup console area
        area = new Area(0,0,width,height);

        // setup clip area
        clip = area.copy();

        // calculate pitch from width
        pitch = width * format.bytes();

        // create keyboard
        keyboard = new Keyboard();

        // register keyboard
        window.register(keyboard);

        // create display
        display = new Display(width,height,format,pages);

        // register display
        window.register(display);

        // update all pages (fixes dropped 1st frame?)
        for (int i=0; i<pages; i++) update();
    }

    private void internal_close()
    {
        // unregister display
        window.unregister(display);

        // unregister keyboard
        window.unregister(keyboard);

        // null references
        window = null;
        display = null;
        keyboard = null;
    }

    // console data
    private int width;
    private int height;
    private int pitch;
    private int pages;
    private Format format;
    private Area area;
    private Area clip;
    private Palette palette;
    private boolean open = false;
    private Mode modes[] = new Mode[1];

    // console objects
    private Window window;
    private Display display;
    private Keyboard keyboard;

    // default data
    private int default_width;
    private int default_height;
    private int default_pages;
    private Format default_format;
}
