//
// Surface class for OpenPTC 1.0 Java Implementation
// Copyright (c) 1999 Glenn Fiedler (ptc@gaffer.org)
// This source code is licensed under the GNU LGPL
// See http://www.gnu.org/copyleft/lgpl.html for details
//

// package
package ptc;



public class Surface implements ptc.base.Surface
{
    public Surface(int width,int height,Format format) throws Error
    {
        // allocate objects
        copy = new Copy();
        clear = new Clear();
        palette = new Palette();

        // unlocked
        locked = false;

        // setup data
        this.width  = width;
        this.height = height;
        this.format = format.copy();

        // setup surface area
        area = new Area(0,0,width,height);

        // setup clip area
        clip = area.copy();

        // surface memory is always linear
        pitch = width * format.bytes();

        // calculate surface size
        int size = width * height;

        // check size
        if (size==0) throw new Error("zero surface size");

        // allocate pixels
        switch (format.bits())
        {
            case 8:  pixels = new byte[size];  break;
            case 16: pixels = new short[size]; break;
            case 32: pixels = new int[size];   break;
            default: throw new Error("unsupported bits per pixel");
        }

        // clear
        clear();
    }

    protected void finalize() throws Error
    {
        // surface must be unlocked when it is destroyed
        if (locked) throw new Error("surface is still locked");
    }

    public void copy(ptc.base.Surface surface) throws Error
    {
        // load surface pixels to other surface
        surface.load(pixels,width,height,pitch,format,palette);
    }

    public void copy(ptc.base.Surface surface,Area source,Area destination) throws Error
    {
        // load surface pixels to other surface
        surface.load(pixels,width,height,pitch,format,palette,source,destination);
    }

    public Object lock() throws Error
    {
        // fail if surface is already locked
        if (locked) throw new Error("surface is already locked");

        // set locked flag
        locked = true;

        // lock surface
        return pixels;
    }

    public void unlock() throws Error
    {
        // fail if surface is not locked
        if (!locked) throw new Error("surface is not locked");

        // reset locked flag
        locked = false;
    }

    public void load(Object pixels,int width,int height,int pitch,Format format,Palette palette) throws Error
    {
        // check clip area
        if (clip.equals(area))
        {
            // request format conversion
            copy.request(format,this.format);
    
            // set copy palettes
            copy.palette(palette,this.palette);
    
            // copy pixels to surface
            copy.copy(pixels,0,0,width,height,pitch,this.pixels,0,0,this.width,this.height,this.pitch);
        }
        else
        {
            // load explicit areas
            load(pixels,width,height,pitch,format,palette,new Area(0,0,width,height),this.area);
        }
    }

    public void load(Object pixels,int width,int height,int pitch,Format format,Palette palette,Area source,Area destination) throws Error
    {
        // clip source and destination areas
        Area clipped_source = new Area();
        Area clipped_destination = new Area();
        Clipper.clip(source,new Area(0,0,width,height),clipped_source,destination,this.clip,clipped_destination);

        // request format conversion
        copy.request(format,this.format);
    
        // set copy palettes
        copy.palette(palette,this.palette);

        // copy pixels to surface
        copy.copy(pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),pitch,
                  this.pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),this.pitch);
    }

    public void save(Object pixels,int width,int height,int pitch,Format format,Palette palette) throws Error
    {
        // check clip area
        if (clip.equals(area))
        {
            // request format conversion
            copy.request(this.format,format);
    
            // set copy palettes
            copy.palette(this.palette,palette);

            // copy surface pixels to 'pixels' buffer
            copy.copy(this.pixels,0,0,this.width,this.height,this.pitch,pixels,0,0,width,height,pitch);
        }
        else
        {
            // save explicit areas
            save(pixels,width,height,pitch,format,palette,this.area,new Area(0,0,width,height));
        }
    }

    public void save(Object pixels,int width,int height,int pitch,Format format,Palette palette,Area source,Area destination) throws Error
    {
        // clip source and destination areas
        Area clipped_source = new Area();
        Area clipped_destination = new Area();
        Clipper.clip(source,this.clip,clipped_source,destination,new Area(0,0,width,height),clipped_destination);

        // request format conversion
        copy.request(this.format,format);
    
        // set copy palettes
        copy.palette(this.palette,palette);

        // copy surface pixels to 'pixels' buffer
        copy.copy(this.pixels,clipped_source.left(),clipped_source.top(),clipped_source.width(),clipped_source.height(),this.pitch,
                  pixels,clipped_destination.left(),clipped_destination.top(),clipped_destination.width(),clipped_destination.height(),pitch);
    }

    public void clear() throws Error
    {
        // check surface format
        if (format().direct())
        {
            // direct color
            clear(new Color(0,0,0,0));
        }
        else
        {
            // indexed color
            clear(new Color(0));
        }
    }

    public void clear(Color color) throws Error
    {
        // clear surface
        clear(color,area);
    }

    public void clear(Color color,Area area) throws Error
    {
        // clip clear area
        Area clipped_area = Clipper.clip(area,clip);

        // request clear
        clear.request(format);

        // clear surface area
        clear.clear(pixels,(int)clipped_area.left(),(int)clipped_area.top(),(int)clipped_area.width(),(int)clipped_area.height(),pitch,color);
    }

    public void palette(Palette palette) throws Error
    {
        // set palette
        palette.load(palette.data());
    }

    public Palette palette() throws Error
    {
        // get palette
        return palette;
    }

    public void clip(Area area) throws Error
    {
        // set new clip area
        clip = Clipper.clip(area,this.area);
    }

    public int width() throws Error
    {
        // get width
        return width;
    }

    public int height() throws Error
    {
        // get height
        return height;
    }

    public int pitch() throws Error
    {
        // get pitch
        return pitch;
    }

    public Area area() throws Error
    {
        // get area copy
        return area.copy();
    }

    public Area clip() throws Error
    {
        // get clip copy
        return clip.copy();
    }

    public Format format() throws Error
    {
        // get format copy
        return format.copy();
    }

    public boolean option(String option) throws Error
    {
        // unrecognized
        return false;
    }

    // surface data
    private int width;
    private int height;
    private int pitch;
    private Area area;
    private Area clip;
    private Format format;
    private boolean locked;
    private Object pixels;
    private Copy copy;
    private Clear clear;
    private Palette palette;
}
