#include "platform.c"

#include "gcube.h"
#include "poly.h"

void initTracks();//extern

//--- global

#define points 8
#define clothPoints 16

Vector cubeVerts[points*points];
Vector cubeNormals[points*points];

extern int timecount;
int countpos;

Wall walls[6];

int nObjects;
Object objects[2];

int nParticles;
Particle *particles;

Cloth cloth;

ParamPoint clipArea[4];
//extern int xres;                   in platform.c
int yres;

//float mxres, myres;
//float xyratio;

float camMul;
float xAdd;
float yMul;
float yAdd;

//extern unsigned int *screenBuf;    in platform.c
//extern float *divTab;              in platform.c

unsigned int *cubeMap;
unsigned short *bumpMap;
unsigned int *phongMaps[numLights];
unsigned int *miniPhongMaps[numLights];
unsigned int *particleMap;

int recursionLevel;

Track3 camPosTrack;
Track3 camTargetTrack;
Track3 pSourceTrack;
SwitchTrack pSourceActiveTrack;
Track3 lightTrack[numLights];

float camRoll = 0.0;
float camFOV = 1.0;

Vector lightColor[] = {{0.3, 0.7, 0.0}, {0.0, 0.1, 0.9}};

Vector lightPos[numLights];

int clothVisible;
int cubeGravity;
Object *startLockObj;

int renderFlag;

unsigned int picTabs[3][256];
int picControl[5];
int textControl[5];

int endFlag;
//--- global heap
extern void *globalHeap;

void initmem(int size);
void *getalignmem(int size, int align);
void *getmem(int size);


//--- vector functions
void vecAssign(Vector *v1, Vector *v2) {
  v1->x = v2->x;
  v1->y = v2->y;
  v1->z = v2->z;
}
void dvecAssign(DVector *v1, DVector *v2) {
  v1->x = v2->x;
  v1->y = v2->y;
  v1->z = v2->z;
}

void vecAssign4(Vector *v, float x, float y, float z) {
  v->x = x;
  v->y = y;
  v->z = z;
}
void dvecAssign4(DVector *v, double x, double y, double z) {
  v->x = x;
  v->y = y;
  v->z = z;
}

void vecNeg(Vector *v) {
  v->x = -v->x;
  v->y = -v->y;
  v->z = -v->z;
}

void vecAdd(Vector *v1, Vector *v2) {
  v1->x += v2->x;
  v1->y += v2->y;
  v1->z += v2->z;
}
void dvecAdd(DVector *v1, DVector *v2) {
  v1->x += v2->x;
  v1->y += v2->y;
  v1->z += v2->z;
}

void vecAdd3(Vector *r, Vector *v1, Vector *v2) {
  r->x = v1->x + v2->x;
  r->y = v1->y + v2->y;
  r->z = v1->z + v2->z;
}
void dvecAdd3(DVector *r, DVector *v1, DVector *v2) {
  r->x = v1->x + v2->x;
  r->y = v1->y + v2->y;
  r->z = v1->z + v2->z;
}

void vecSub(Vector *v1, Vector *v2) {
  v1->x -= v2->x;
  v1->y -= v2->y;
  v1->z -= v2->z;
}
void dvecSub(DVector *v1, DVector *v2) {
  v1->x -= v2->x;
  v1->y -= v2->y;
  v1->z -= v2->z;
}

void vecSub3(Vector *r, Vector *v1, Vector *v2) {
  r->x = v1->x - v2->x;
  r->y = v1->y - v2->y;
  r->z = v1->z - v2->z;
}
void dvecSub3(DVector *r, DVector *v1, DVector *v2) {
  r->x = v1->x - v2->x;
  r->y = v1->y - v2->y;
  r->z = v1->z - v2->z;
}

void vecMulf(Vector *v, float f) {
  v->x *= f;
  v->y *= f;
  v->z *= f;
}
void dvecMulf(DVector *v, double f) {
  v->x *= f;
  v->y *= f;
  v->z *= f;
}

void vecMul3(Vector *r, Vector *v, float f) {
  r->x = v->x*f;
  r->y = v->y*f;
  r->z = v->z*f;
}
void dvecMul3(DVector *r, DVector *v, double f) {
  r->x = v->x*f;
  r->y = v->y*f;
  r->z = v->z*f;
}

void vecMulAdd(Vector *r, Vector *v, float f) {
  r->x += v->x*f;
  r->y += v->y*f;
  r->z += v->z*f;
}
void dvecMulAdd(DVector *r, DVector *v, double f) {
  r->x += v->x*f;
  r->y += v->y*f;
  r->z += v->z*f;
}

void vecMulC(Vector *r, Vector *v) {
  r->x *= v->x;
  r->y *= v->y;
  r->z *= v->z;
}

float vecMul(Vector *v1, Vector *v2) {
  return v1->x*v2->x + v1->y*v2->y + v1->z*v2->z;
}
double dvecMul(DVector *v1, DVector *v2) {
  return v1->x*v2->x + v1->y*v2->y + v1->z*v2->z;
}
float dnvecMul(DVector *v1, Vector *v2) {
  return v1->x*v2->x + v1->y*v2->y + v1->z*v2->z;
}

void vecCross(Vector *r, Vector *v1, Vector *v2) {
  r->x = v1->y*v2->z - v1->z*v2->y;
  r->y = v1->z*v2->x - v1->x*v2->z;
  r->z = v1->x*v2->y - v1->y*v2->x;
}
void dvecCross(DVector *r, DVector *v1, DVector *v2) {
  r->x = v1->y*v2->z - v1->z*v2->y;
  r->y = v1->z*v2->x - v1->x*v2->z;
  r->z = v1->x*v2->y - v1->y*v2->x;
}

float vecAbs(Vector *v) {
  return sqrt(v->x*v->x + v->y*v->y + v->z*v->z);
}
double dvecAbs(DVector *v) {
  return sqrt(v->x*v->x + v->y*v->y + v->z*v->z);
}

void vecNorm(Vector *v) {
  float f;
  f = 1.0/sqrt(v->x*v->x + v->y*v->y + v->z*v->z);
  v->x *= f;
  v->y *= f;
  v->z *= f;
}
void dvecNorm(DVector *v) {
  double f;
  f = 1.0/sqrt(v->x*v->x + v->y*v->y + v->z*v->z);
  v->x *= f;
  v->y *= f;
  v->z *= f;
}

/*
void vecPrint(Vector *v) {
  printf("x = %7.3f, y = %7.3f, z = %7.3f\n", v->x, v->y, v->z);
}
*/

//--- tracks

void initTrack1(Track1 *track, Key1 *key) {
  track->startFrame = key->frame;
  track->P1 = key->P;
  key++;
  track->endFrame = key->frame;
  track->P2 = key->P;
  track->R1 = track->P2 - track->P1;
  key++;
  track->R2 = (key->P - track->P1)*0.5;
  track->key = key;
}

float doTrack1(Track1 *track, int frame) {
  Key1 *key;
  float t, t2, t3;
  float p;

  if (frame >= track->endFrame) {
    track->startFrame = track->endFrame;
    track->P1 = track->P2;
    track->R1 = track->R2;
    key = track->key;
    track->endFrame = key->frame;
    track->P2 = key->P;
    key++;
    track->R2 = (key->P - track->P1)*0.5;
    track->key = key;
  }

  t = (frame - track->startFrame)/(float) (track->endFrame - track->startFrame);
  t2 = t*t;
  t3 = t2*t;
  p = track->P1*(2.0*t3 - 3.0*t2 + 1.0)
    + track->R1*(t3 - 2.0*t2 + t)
    + track->P2*(-2.0*t3 + 3.0*t2)
    + track->R2*(t3 - t2);
  return p;
}

void initTrack3(Track3 *track, Key3 *key) {
  float f;

  track->startFrame = key->frame;
  vecAssign(&track->P1, &key->P);
  track->adj1 = 0.0;
  key++;
  track->endFrame = key->frame;
  vecAssign(&track->P2, &key->P);
  vecSub3(&track->R1, &track->P2, &track->P1);
  key++;
  vecSub3(&track->R2, &key->P, &track->P1);
//  vecMul(&track->R2, 0.5);
  f = key->frame - track->startFrame;
  track->adj2 = (track->endFrame - track->startFrame)/f;
  track->nextAdj = (key->frame - track->endFrame)/f;
  track->key = key;
}

void doTrack3(Track3 *track, int frame, Vector *p) {
  Key3 *key;
  float f, t, t2, t3;

  while (frame >= track->endFrame) {
    track->startFrame = track->endFrame;
    vecAssign(&track->P1, &track->P2);
    vecAssign(&track->R1, &track->R2);
    track->adj1 = track->nextAdj;
    key = track->key;
    track->endFrame = key->frame;
    vecAssign(&track->P2, &key->P);
    key++;
    vecSub3(&track->R2, &key->P, &track->P1);
//    vecMul(&track->R2, 0.5);
    f = key->frame - track->startFrame;
    track->adj2 = (track->endFrame - track->startFrame)/f;
    track->nextAdj = (key->frame - track->endFrame)/f;
    track->key = key;
  }

  t = (frame - track->startFrame)/(float) (track->endFrame - track->startFrame);
  t2 = t*t;
  t3 = t2*t;
  vecMul3(p, &track->P1, 2.0*t3 - 3.0*t2 + 1.0);
  vecMulAdd(p, &track->R1, (t3 - 2.0*t2 + t)*track->adj1);
  vecMulAdd(p, &track->P2, -2.0*t3 + 3.0*t2);
  vecMulAdd(p, &track->R2, (t3 - t2)*track->adj2);
//printf("adj %7.3f %7.3f\n", track->adj1, track->adj2);
}

void initSwitchTrack(SwitchTrack *track, int *keys) {
  track->key = keys;
  track->state = 0;
}

int doSwitchTrack(SwitchTrack *track, int frame) {
  while (*track->key < frame) {
    track->key++;
    track->state = !track->state;
  }
  return track->state;
}

//--- camera
void makeCamera(Camera *camera, Vector *p, Vector *target, float roll, float FOV) {
  Vector lx, ly;
  float s, c;

  vecAssign(&camera->p, p);
  vecSub3(&camera->lz, target, p);
  vecAssign4(&lx, camera->lz.y, -camera->lz.x, 0.0);
  vecNorm(&camera->lz);
  vecNorm(&lx);
  vecCross(&ly, &lx, &camera->lz);

  FOV *= camMul;
  s = sin(roll)*FOV;
  c = cos(roll)*FOV;
  vecMul3(&camera->lx, &lx, c);
  vecMulAdd(&camera->lx, &ly, -s);
//  s *= xyratio;//1.33333;  //x/y ratio
//  c *= xyratio;//1.33333;
  vecMul3(&camera->ly, &ly, c);
  vecMulAdd(&camera->ly, &lx, s);
}


//--- texture map generator
void mapInit(MapInfo *info, int shift) {
  int res, z;
  float *filter;
  float s, x, f;

  res = 1 << shift;
  bzero(globalHeap, res*res);
  info->r = (unsigned char *) globalHeap;
//  info->g = (unsigned char *) globalHeap + res*res;
//  info->b = (unsigned char *) globalHeap + 2*res*res;
  info->mask = res -1;
  info->shift = shift;
  info->filter = filter = (float *) (info->r + res*res);
  info->result = (unsigned char *) (filter + 8192);

  s = 1 << (8 - shift);
  filter[0] = 1.0;
  for (z = 1; z < /*2304*/ 3000; z++) {
    x = sqrt(z)/s;
    x *= 3.141592654;
    f = sin(x)/x;
    f *= cos(x/6.85);
    filter[z] = f;
  }
  for (; z < 8192; z++) filter[z] = 0.0;
}


void mapSet(MapInfo *info, int x, int y, int r/*, int g, int b*/) {
  int idx;
  if (r > 255) r = 255;
//  if (r < 0) r = 0; else if (r > 255) r = 255;
//  if (g < 0) r = 0; else if (g > 255) g = 255;
//  if (b < 0) r = 0; else if (b > 255) b = 255;
  idx = (x & info->mask) + ((y & info->mask) << info->shift);
  info->r[idx] = r;
//  info->g[idx] = g;
//  info->b[idx] = b;
}

int random() {
  static int seed = 80;

  seed += 55;
  seed *= seed/9;
  return (seed ^ (seed >> 8)) & 255;
}


unsigned char *mapFilter(MapInfo *info) {
  int shift, x, y;
  float s, w, r;//, g, b;
  int xi, yi, xr, yr, xp, yp;
  int yidx, idx;
  unsigned char *result;

  result = info->result;
  shift = 8 - info->shift;
  s = 1.0/(float) (1 << shift);
  for (y = 0; y < 256; y++) {
    for (x = 0; x < 256; x++) {
      xr = x >> shift;
      yr = y >> shift;
      r = /*g = b =*/ 0.0;
      for (yi = yr -2; yi <= yr +3; yi++) {
        yp = (yi << shift) - y;
        yp *= yp;
        yidx = (yi & info->mask) << info->shift;
        for (xi = xr -2; xi <= xr +3; xi++) {
          xp = (xi << shift) - x;
          w = info->filter[xp*xp + yp];

          idx = (xi & info->mask) + yidx;
          r += w*info->r[idx];
//          g += w*info->g[idx];
//          b += w*info->b[idx];
        }
      }
      *result = (int) floor(r);
      result++;
//      *map = clip(r) + (clip(g) << 11) + (clip(b) << 22);
//      map++;
    }
  }
  return info->result;
}
/*
int random2(int res) {
  return (res*(random() - 128))/(256*4);// + res/2;
}
*/
int clip(int i) {
  if (i < 0) i = 0; else if (i > 255) i = 255;
  return i;
}

void genMaps() {
  unsigned int *cm;
  unsigned short *bm;
  int shift, res;
  MapInfo info;
  int x, y;
  unsigned char *pattern;
  int i, i1, i2;

  cm = cubeMap = (unsigned int *) getalignmem(65536*4, 65536*4);
  bm = bumpMap = (unsigned short *) getalignmem(65536*2, 65536*2);

  shift = 6;
  res = 1 << shift;
  mapInit(&info, shift);

  for (y = 0; y < res; y++) {
    for (x = 0; x < res; x++) {
//      col = random()/4;
//      mapSetRgb(&info, x, y, col + 10, col + 20, col + 15);
      mapSet(&info, x, y, random()/2 + 64);
    }
  }

/*  letzte version
  x = res/8;
  for (z = x; z < res - x; z++) {
    mapSetRgb(&info, z + random2(res), z + random2(res), 200, 100, 0);
    mapSetRgb(&info, res -1 - z + random2(res), z + random2(res), 200, 100, 0);
  }
*/
/*  for (z = 0; z < res; z++) {
    mapSetRgb(&info, random2(res), random2(res), 200, 100, 0);
  }*/

/*
  mapSetRgb(&info, res/2, res/2, 255, 0, 0);
  mapSetRgb(&info, res/2 +1, res/2, 255, 0, 0);
  mapSetRgb(&info, res/2 +2, res/2 +1, 255, 0, 0);
  mapSetRgb(&info, res/2 +2, res/2 +2, 255, 0, 0);
  mapSetRgb(&info, res/2 +1, res/2 +3, 255, 0, 0);
  mapSetRgb(&info, res/2, res/2 +3, 255, 0, 0);
*/
  pattern = mapFilter(&info);

  for (y = 0; y < 256; y++) {
    for (x = 0; x < 256; x++) {
      i = pattern[x + (y << 8)] - 64;
      //*cm = clip(i + 90) | clip(40) << 11 | clip(i + 120) << 22;
//      *cm = 100 | 80 << 11;
      *cm = clip(i/4 + 100) | clip(i/4 + 80) << 11;

      i1 = pattern[((x -1) & 0xFF) + (y << 8)];
      i2 = pattern[((x +1) & 0xFF) + (y << 8)];
      i = i1 - i2;
      *bm = (i/4 + 96 + (x >> 3) -16) & 0xFF;
      i1 = pattern[x + ((y -1) << 8 & 0xFF00)];
      i2 = pattern[x + ((y +1) << 8 & 0xFF00)];
      i = i1 - i2;
      *bm |= (i/4 + 96 + (y >> 3) -16) << 8;

      cm++;
      bm++;
    }
  }
}


unsigned int *genPhongMap(Vector *col) {
  unsigned int *map, *dest;
  int x, y;
  float yf, f, c, w;
  float r, g, b, rw, gw, bw;

  dest = map = (unsigned int *) getalignmem(65536*4, 65536*4);

  r = col->x;
  g = col->y;
  b = col->z;
  rw = 1.0 - r;
  gw = 1.0 - g;
  bw = 1.0 - b;
  for (y = 0; y < 256; y++) {
    yf = y - 127.5;
    yf *= yf;
    for (x = 0; x < 256; x++) {
      f = x - 127.5;
      f = f*f + yf;
      if (f < /*16384.0*/4096.0) {
        f = cos(sqrt(f)* /*0.0122*/0.0245);
        c = pow(f, 4.0)*250.0;
        w = pow(f, 100.0)*100.0;
        *dest = (int) (c*r + w*rw) | (int) (c*g + w*gw) << 11 | (int) (c*b + w*bw) << 22;
      }
//    *dest = -1;
      dest++;
    }
  }

  return map;
}

unsigned int *scaleMap(unsigned int *tmap, int res) {
  unsigned int *map, *dest;
  int add, mul, x, y, i, r, g, b;

  map = (unsigned int *) getmem(65536*4);
  add = (256 - res)/2;
  dest = map + add + (add << 8);
  mul = 256/res;
  for (y = 0; y < res; y++) {
    for (x = 0; x < res; x++) {
      i = tmap[(x + (y << 8))*mul];
      r = i & 0xFF;
      g = (i >> 11) & 0xFF;
      b = i >> 22;
      r /= 3;
      g /= 3;
      b /= 3;
      dest[x + (y << 8)] = r | g << 11 | b << 22;
//      i = (i >> 1) & 0xFF*(1 | 1 << 11 | 1 << 22);
//      dest[x + (y << 8)] = i;
    }
  }

  return map;
}

unsigned int *genParticleMap() {
  unsigned int *map;
  int x, y, h1, h, r, g;
  float f1, f;
  unsigned int *m;
//  unsigned char *tex;
  int col;

  map = (unsigned int *) getalignmem(65536*4,65536*4);
/*
  for (x = 0; x < 256; x++) {
    h1 = x*(255 - x)/254;
    f1 = sin(3.141*x/255.0)*0.75;
    for (y = 0; y < 256-64; y++) {
      h = y + h1;
      f = f1*sin(3.141*y/(255.0-64.0)) + 0.25;
      r = (int) floor(f*240.0);
      g = (int) floor(f*((y >> 2) + 50));
      map[x + (h << 8)] = r | g << 11;
    }
  }*/
  for (x = 0; x < 128; x++) {
    h1 = x*(127 - x)/126;
    f1 = sin(3.141*x/127.0)*0.75;
    for (y = 0; y < 128-32; y++) {
      h = y + h1;
      f = f1*sin(3.141*y/(127.0-32.0)) + 0.25;
      r = (int) floor(f*240.0);
      g = (int) floor(f*((y >> 2) + 50));
      map[x + (h << 8)] = r | g << 11;
    }
  }

  //cloth map
/*  x = 0;
  h = 2;
  do {
    r = 150 + random()/8;
    g = 0;//100 + random()/4;
    while (h > 0) {
      for (y = 0; y < 128; y++) {
        map[x + (y << 8) + 128] = r | g << 11;
      }
      h--;
      x++;
    }
    x += random()/32 +1;
    h = random()/32 +1;
  } while (x + h < 128);

  x = 0;
  do {
    r = 150 + random()/4;
    g = 50;
    b = 100 + random()/8;
    while (h > 0) {
      for (y = 0; y < 128; y++) {
        m = &map[y + (x << 8) + 128];
        if (*m) {
          *m = 150 << 11 | 50 << 22;
        } else {
          *m = r | g << 11 | b << 22;
        }
      }
      h--;
      x++;
    }
    x += random()/32 +1;
    h = random()/32 +1;
  } while (x + h < 128);
*/
  x = 0;
  do {
    h = random()/32 +1;
    col = picTabs[0][150 + random()/8];
    while (h > 0 && x < 128) {
      for (y = 0; y < 128; y++) {
        map[x + (y << 8) + 128] = col;
      }
      h--;
      x++;
    }
    x += random()/32 +1;
  } while (x < 128);

  x = 0;
  do {
    h = random()/32 +1;
    col = picTabs[1][150 + random()/8];
    while (h > 0 && x < 128) {
      for (y = 0; y < 128; y++) {
        m = &map[y + (x << 8) + 128];
        if (*m) {
          *m = picTabs[2][200];
        } else {
          *m = col;
        }
      }
      h--;
      x++;
    }
    x += random()/32 +1;
  } while (x < 128);

/*  tex = textur;
  for (x = 0; x < 128; x++) {
    for (y = 0; y < 128; y++) {
      h = 3*(*tex);
      r = texturPal[h];
      g = texturPal[h +1];
      b = texturPal[h +2];
      map[y + (x << 8) + 128] = r | g << 11 | b << 22;
      tex++;
    }
  }*/

  return map;
}


//--- wall
void initWall(Wall *wall, Vector *p, Vector *u, Vector *v, int wPoints) {
  float lu, lv;
  int p1, x, y, i;
  float mu, mv;
  Vector *vertex;
  WVertexAttr *attr;
  WFace *face;

  lu = vecAbs(u);
  lv = vecAbs(v);
  vecAssign(&wall->p, p);
//  vecMulAdd(&wall->p, u, 0.5);
//  vecMulAdd(&wall->p, v, 0.5);
  vecAssign(&wall->u, u);
  vecAssign(&wall->v, v);
  vecCross(&wall->n, u, v);
  vecNorm(&wall->n);
/*  vecNorm(&wall->u);
  vecCross(&wall->v, &wall->n, &wall->u);
  vecMul(&wall->u, lu);
  vecMul(&wall->v, lv);*/
  wall->xscale = 63.0/(lu*lu);
  wall->yscale = 63.0/(lv*lv);

//  wall->nVerts = 0;
//  wall->nFaces = 0;

  //make verts
  p1 = wPoints -1;
  wall->nVerts = wPoints*wPoints;
  vertex = wall->verts = (Vector *) getmem(wall->nVerts*sizeof(Vector));
  attr = wall->attribs = (WVertexAttr *) getmem(wall->nVerts*sizeof(WVertexAttr));
  for (y = 0; y < wPoints; y++) {
    mv = y/(float) p1;
    for (x = 0; x < wPoints; x++) {
      mu = x/(float) p1;
      vecAssign(vertex, p);
      vecMulAdd(vertex, u, mu);
      vecMulAdd(vertex, v, mv);
      attr->u = (1.0 - mu)*254.0*256.0; //param scale
      attr->v = (1.0 - mv)*254.0*256.0;
      vertex++;
      attr++;
    }
  }
  //make faces
  wall->nFaces = p1*p1;
  face = wall->faces = (WFace *) getmem(wall->nFaces*sizeof(WFace));
  for (y = 0; y < p1; y++) {
    for (x = 0; x < p1; x++) {
      i = y*wPoints + x;
      face->idx[3] = i;
      face->idx[2] = i + 1;
      face->idx[1] = i + wPoints + 1;
      face->idx[0] = i + wPoints;
      face++;
    }
  }
}
/*
void makeWalls() {
  wallCurrentVertex = wallVerts;
  wallCurrentVertexAttr = wallVertexAttribs;
  wallCurrentFace = wallFaces;
}

float max(float f1, float f2) {
  return (f1 > f2) ? f1 : f2;
}

int makeVertex(Wall *wall, float x, float y) {
  Vector *p;
  WVertexAttr *vertexAttr;
  int num;

//  num = nWallVerts;
//  nWallVerts++;
  p = wallCurrentVertex;
  wallCurrentVertex++;
  vertexAttr = wallCurrentVertexAttr;
  wallCurrentVertexAttr++;
  num = wall->nVerts;
  wall->nVerts++;
  vecAssign(p, &wall->p);
  vecMulAdd(p, &wall->u, x);
  vecMulAdd(p, &wall->v, y);
  vertexAttr->u = x*255.0;
  vertexAttr->v = y*255.0;
  return num;
}

void makeWallR(Wall *wall, int level, float x, float y, float w,
  int a, int b, int c, int d) {
  float xm, ym, dist;
  int e, f, g, h, i;
  WFace *face;
 float ow;

  level++;
  ow = w;
  w *= 0.5;
  xm = x + w;
  ym = y + w;
  dist = max(max(fabs(xm - wall->cx), fabs(ym - wall->cy)) - w, wall->cd);

  if (level <= 2 || level < 5 && ow > dist) {
    //locations of points:
    // a e b
    // h i f
    // d g c
    e = makeVertex(wall, xm, y);
    f = makeVertex(wall, xm + w, ym);
    g = makeVertex(wall, xm, ym + w);
    h = makeVertex(wall, x, ym);
    i = makeVertex(wall, xm, ym);

    //recursion
    makeWallR(wall, level,  x,  y, w, a, e, i, h);
    makeWallR(wall, level, xm,  y, w, e, b, f, i);
    makeWallR(wall, level, xm, ym, w, i, f, c, g);
    makeWallR(wall, level,  x, ym, w, h, i, g, d);
  } else {
    //create face
    face = wallCurrentFace;
    wallCurrentFace++;
    wall->nFaces++;
    face->idx[3] = a;
    face->idx[2] = b;
    face->idx[1] = c;
    face->idx[0] = d;
  }
}

void makeWall(Wall *wall, Vector *camera) {
  int a, b, c, d;
  Vector v;
  int z;
  WVertexAttr *attr;

  wall->nVerts = 0;
  wall->verts = wallCurrentVertex;
  wall->attribs = wallCurrentVertexAttr;
  wall->nFaces = 0;
  wall->faces = wallCurrentFace;

  vecSub(&v, camera, &wall->p);
  wall->cx = vecMul(&v, &wall->u)*wall->xscale;
  wall->cy = vecMul(&v, &wall->v)*wall->yscale;
  wall->cd = vecMul(&v, &wall->n)/16.0;
  if (wall->cd < 0.0) wall->cd = 0.0;

  a = makeVertex(wall, 0.0, 0.0);
  b = makeVertex(wall, 1.0, 0.0);
  c = makeVertex(wall, 1.0, 1.0);
  d = makeVertex(wall, 0.0, 1.0);
  makeWallR(wall, 0, 0.0, 0.0, 1.0, a, b, c, d);

//  printf("nVerts %d nFaces %d\n", wall->nVerts, wall->nFaces);

  attr = wall->attribs;
  z = wall->nVerts;
  do {
    attr->r = 0.0;
    attr->g = 0.0;
    attr->b = 0.0;
    attr++;
    z--;
  } while (z != 0);
}

void clearWall(Wall *wall) {
  int z;
  WVertexAttr *attr;

  attr = wall->attribs;
  z = wall->nVerts;
  do {
    attr->r = 0.0;
    attr->g = 0.0;
    attr->b = 0.0;
    attr++;
    z--;
  } while (z != 0);
}
*/
void lightWall(Wall *wall, Vector *light, unsigned int *pMap, unsigned int **binfo) {
  Vector u;
  int bumpx, bumpy;
/*  int z;
  Vector *v;
  WVertexAttr *attr;
  float l, h;

  v = wall->verts;             //optimize  bei bump-mapping farbe weg!
  attr = wall->attribs;
  z = wall->nVerts;
  do {
    vecSub(&u, v, light);
    l = vecAbs(&u);
    h = 255.0 - l*16.0;
    if (h > 0.0) {
//      if (h > 150.0) h = 150.0;
      if (h > 100.0) h = 100.0;
      //h *= -vecMul(&u, &wall->n)/l;
      attr->r += h*color->x;
      attr->g += h*color->y;
      attr->b += h*color->z;
    }
    v++;
    attr++;
    z--;
  } while (z != 0);
*/
  vecSub3(&u, light, &wall->p);
  bumpx = (int) floor(vecMul(&u, &wall->u)*wall->xscale);
  bumpy = (int) floor(vecMul(&u, &wall->v)*wall->yscale);
  if (bumpx < 0) bumpx = 0; else if (bumpx > 63) bumpx = 63;
  if (bumpy < 0) bumpy = 0; else if (bumpy > 63) bumpy = 63;
  *binfo = &pMap[bumpx | (bumpy << 8)];
}

int wFaceZclip(WFace *face, Vector *verts, WVertexAttr *attribs, ClipInfo *info) {
  int nidx[5];
  ParamPoint *dp;
  Vector *v, *nv;
  WVertexAttr *a, *na;
  int z, count, vis, nvis;
  float r;

  if (recursionLevel & 1) {
    nidx[0] = face->idx[3];
    nidx[1] = face->idx[2];
    nidx[2] = face->idx[1];
    nidx[3] = face->idx[0];
  } else {
    nidx[0] = face->idx[0];
    nidx[1] = face->idx[1];
    nidx[2] = face->idx[2];
    nidx[3] = face->idx[3];
  }
  nidx[4] = nidx[0];

  dp = info->start;

  //project & z-clip
  z = 0;
  count = 0;
  v = &verts[nidx[0]];
  a = &attribs[nidx[0]];
  vis = (v->z >= 1.0);
  do {
    if (vis) {
      //point in front of projection plane (visible)
//      dp->x = mxres*(1.0 + v->x/v->z);
//      dp->y = myres*(1.0 - v->y/v->z);
      dp->x = xAdd + v->x/v->z;
      dp->y = yAdd + yMul*v->y/v->z;
      dp->z = v->z;
      dp->param[0] = a->u;
      dp->param[1] = a->v;
//      dp->param[2] = a->r;  //optimize bei bumpmapping weg
//      dp->param[3] = a->g;
//      dp->param[4] = a->b;
      count++;
      dp = (ParamPoint *) ((char *) dp + info->size);
    }
    z++;
    nv = &verts[nidx[z]];
    na = &attribs[nidx[z]];
    nvis = (nv->z >= 1.0);
    if (vis != nvis) {
      r = (1.0 - v->z)/(nv->z - v->z);

//      dp->x = mxres*(1.0 + v->x + r*(nv->x - v->x));
//      dp->y = myres*(1.0 - v->y - r*(nv->y - v->y));
      dp->x = xAdd + v->x + r*(nv->x - v->x);
      dp->y = yAdd + yMul*(v->y + r*(nv->y - v->y));
      dp->z = 1.0;
      dp->param[0] = a->u + r*(na->u - a->u);
      dp->param[1] = a->v + r*(na->v - a->v);
//      dp->param[2] = a->r + r*(na->r - a->r);
//      dp->param[3] = a->g + r*(na->g - a->g);
//      dp->param[4] = a->b + r*(na->b - a->b);
      count++;
      dp = (ParamPoint *) ((char *) dp + info->size);
    }
    v = nv;
    a = na;
    vis = nvis;
  } while (z != 4);
  info->end = dp;
  return count >= 3;
}


//--- cubes
Face *edgeFaces(Face *face, int n1, int step1, int n2, int step2) {
  int z;
  for (z = 0; z < points -1; z++) {
    face->c = n2 + (z +1)*step2;
    face->b = n2 + z*step2;
    face->a = n1 + z*step1;
    face++;
    face->c = n1 + z*step1;
    face->b = n1 + (z +1)*step1;
    face->a = n2 + (z +1)*step2;
    face++;
  }
  return face;
}

void makeAlpha(float *alpha) {
  int y, x, z;
  float pts, sy, sx, a;

  pts = points -1;
  for (y = 0; y < points; y++) {
    sy = sin((y -0)*3.141/pts)*1.2;
    if (sy < 0.0) sy = 0.0;
    for (x = 0; x < points; x++) {
      sx = sin((x -0)*3.141/pts)*1.2;
      if (sx < 0.0) sx = 0.0;
      a = 1.0 - sx*sy;
      if (a < 0.3) a = 0.3;
      a *= 255.0*256.0;
      for (z = 0; z < 6; z++) alpha[z*points*points] = a;
      alpha++;
    }
  }
}

void updateGeometry(Object *obj, float t);
void initCube(Object *obj/*, float xoffs, float yoffs, float zoffs, float vz*/) {
  int z, old, i, oldi, x, y;
  Mass *cube;
  CEdge *edge;
  CFace *cFace;
  Vector *vertex, *normal;
  Vector v2, g;
  float f;
  Mapping *m;
  Face *face;

  //set location of masses
  obj->nMasses = 8;
  obj->masses = cube = (Mass *) getmem(sizeof(Mass)*obj->nMasses);
  for (z = 0; z < obj->nMasses; z++) {
    cube[z].p.x = ((z & 1) ? -1.0 : 1.0);
    cube[z].p.y = ((z & 2) ? -1.0 : 1.0);
    cube[z].p.z = ((z & 4) ? -1.0 : 1.0);
  }


  //build collision cFaces and edges
  obj->nCEdges = 12;
  obj->cEdges = edge = (CEdge *) getmem(sizeof(CEdge)*12);
  obj->nCFaces = 12;
  obj->cFaces = cFace = (CFace *) getmem(sizeof(CFace)*12);
  old = 0;
  z = 1;
  while (z < 8) { //z = 1, 2, 4, 7
    oldi = 4;
    for (i = 1; i <= 4; i <<= 1) { //i = 1, 2, 4
      edge->ma = &cube[z];//.p;
      edge->mb = &cube[z ^ i];//.p;
      edge++;

      cFace->ma = &cube[z];
      cFace->mb = &cube[z ^ oldi];
      cFace->mc = &cube[z ^ i];
      oldi = i;
      cFace++;
    }
    i = old + 1;
    old = z;
    z += i;
  }

  //build visible geometry
  //mem for vertices and normals
  obj->nVerts = 6*points*points;
  obj->verts = (Vector *) getmem(obj->nVerts*sizeof(Vector));
  obj->normals = (Vector *) getmem(obj->nVerts*sizeof(Vector));

  //build one side of a cube
  vertex = cubeVerts;
  for (y = 0; y < points; y++) {
    for (x = 0; x < points; x++) {
      vecAssign4(vertex,
        2.0*x/(float) (points -1) -1.0, 2.0*y/(float) (points -1) -1.0, 1.2);
      vertex++;
    }
  }
  //approximate to function
  for (z = 0; z < 5; z++) {
    vertex = cubeVerts;
    for (y = 0; y < points; y++) {
      for (x = 0; x < points; x++) {
        //function f(v) = x^4 + y^4 + z^4 - x^2 - y^2 - z^2
        //gradient(f) = nabla(f)
        //for x component: gx = 4*x^3 - 2*x = (x*2.0*x*x - x)*2.0
        //newton approximation vn+1 = vn - gradient(f(v))*f(v)/gradient(f(v))^2
        vecAssign(&v2, vertex);
        vecMulC(&v2, &v2);

        f = vecMul(&v2, &v2) - v2.x - v2.y - v2.z;

        vecMul3(&g, vertex, 2.0);
        vecMulC(&g, &v2);
        vecSub(&g, vertex);
        vecMulf(&g, 2.0);

        vecMulf(&g, f/vecMul(&g, &g));
        vecSub(vertex, &g);

        vertex++;
      }
    }
  }
  //build template vertices and normals
  vertex = cubeVerts;
  normal = cubeNormals;
  for (y = 0; y < points; y++) {
    for (x = 0; x < points; x++) {
      //calc normal from gradient
      vecMul3(&g, vertex, 2.0);
      vecMulC(&g, vertex);
      vecMulC(&g, vertex);
      vecSub(&g, vertex);
     g.x *= 0.5;
     g.y *= 0.5;
      vecNorm(&g);
      normal->x = (+g.x + g.y)*0.25;
      normal->y = (-g.y + g.x)*0.25;
      normal->z = g.z;

      //adjust ranges (-1.0 .. 1.0 -> 0.0 .. 1.0)
      vertex->x = (vertex->x + 1.0)*0.5;
      vertex->y = (vertex->y + 1.0)*0.5;
      vertex->z -= 1.0;
//     vertex->z *= 1.6;

      vertex++;
      normal++;
    }
  }

  updateGeometry(obj, 0.0);

  //texture mapping
  vertex = obj->verts;
  m = obj->mapping = (Mapping *) getmem(obj->nVerts*sizeof(Mapping));
  for (z = 0; z < obj->nVerts; z++) {
    m->u = ((vertex->x + vertex->z)*50.0 + 127.5)*256.0; //param scale
    m->v = ((vertex->y + vertex->z)*50.0 + 127.5)*256.0;
    vertex++;
    m++;
  }

  //alpha channel
  obj->alpha = (float *) getmem(obj->nVerts*sizeof(float));
  makeAlpha(obj->alpha);


  //mem for faces
  obj->nFaces = 6*2*(points -1)*(points -1) + 12*2*(points -1) + 8;
  face = obj->faces = (Face *) getmem(obj->nFaces*sizeof(Face));

  //build faces on sides
  for (z = 0; z < 6; z++) {
    for (y = 0; y < points -1; y++) {
      for (x = 0; x < points -1; x++) {
        i = (z*points + y)*points + x;
        face->c = i + points + 1;
        face->b = i + 1;
        face->a = i;
        face->light = 1;
        face++;
        face->c = i;
        face->b = i + points;
        face->a = i + points + 1;
        face->light = 1;
        face++;
      }
    }
  }
  //build faces on edges (12 edges)
  face = edgeFaces(face, 0*points*points, 1, 1*points*points, points);
  face = edgeFaces(face, 1*points*points, 1, 2*points*points, points);
  face = edgeFaces(face, 2*points*points, 1, 0*points*points, points);

  face = edgeFaces(face, 4*points*points -1, -1, (0*points + points -1)*points, 1);
  face = edgeFaces(face, 5*points*points + points -1, points, 1*points*points -1, -points);
  face = edgeFaces(face, 6*points*points -1, -1, (1*points + points -1)*points, 1);
  face = edgeFaces(face, 1*points*points + points -1, points, 5*points*points -1, -points);
  face = edgeFaces(face, 3*points*points -1, -1, (4*points + points -1)*points, 1);
  face = edgeFaces(face, 3*points*points + points -1, points, 3*points*points -1, -points);

  face = edgeFaces(face, 3*points*points, 1, 4*points*points, points);
  face = edgeFaces(face, 4*points*points, 1, 5*points*points, points);
  face = edgeFaces(face, 5*points*points, 1, 3*points*points, points);

  //buld faces on corners (8)
  face->c = 0*points*points;
  face->b = 1*points*points;
  face->a = 2*points*points;
  face++;
  face->a = 2*points*points + points -1;
  face->b = (0*points + points -1)*points;
  face->c = 4*points*points -1;
  face++;
  face->c = 0*points*points + points -1;
  face->b = 6*points*points -1;
  face->a = (1*points + points -1)*points;
  face++;
  face->c = 5*points*points + points -1;
  face->b = 1*points*points -1;
  face->a = (3*points + points -1)*points;
  face++;
  face->c = 1*points*points + points -1;
  face->b = 5*points*points -1;
  face->a = (2*points + points -1)*points;
  face++;
  face->c = 3*points*points + points -1;
  face->b = 3*points*points -1;
  face->a = (4*points + points -1)*points;
  face++;
  face->c = 4*points*points + points -1;
  face->b = 2*points*points -1;
  face->a = (5*points + points -1)*points;
  face++;
  face->c = 3*points*points;
  face->b = 4*points*points;
  face->a = 5*points*points;

  //set location
/*  cube = obj->masses;
  for (z = 0; z < obj->nMasses; z++) {
    cube[z].p.x += xoffs;
    cube[z].p.y += yoffs;
    cube[z].p.z += zoffs;
    cube[z].v.z = vz;
  }*/
}

void setCubeLocation(Object *obj, double xoffs, double yoffs, double zoffs) {
  int z;
  Mass *m;

  m = obj->masses;
  for (z = 0; z < 8; z++) {
    m->p.x = ((z & 1) ? -1.0 : 1.0) + xoffs;
    m->p.y = ((z & 2) ? -1.0 : 1.0) + yoffs;
    m->p.z = ((z & 4) ? -1.0 : 1.0) + zoffs;
    m->v.x = 0.0;
    m->v.y = 0.0;
    m->v.z = 0.0;//vz;
    m++;
  }
}

//--------
void moveObject(Object *obj, int gravity) {
  int z;
  Mass *m;

  //init bounding box
  obj->bound1.x = 1e10;
  obj->bound1.y = 1e10;
  obj->bound1.z = 1e10;
  obj->bound2.x = -1e10;
  obj->bound2.y = -1e10;
  obj->bound2.z = -1e10;
  z = obj->nMasses;
  m = obj->masses;
  do {
    dvecAdd(&m->p, &m->v); //move
    if (gravity) m->v.z -= 0.005; //gravity
    //update bounding box
    if (m->p.x < obj->bound1.x) obj->bound1.x = m->p.x;
    if (m->p.y < obj->bound1.y) obj->bound1.y = m->p.y;
    if (m->p.z < obj->bound1.z) obj->bound1.z = m->p.z;
    if (m->p.x > obj->bound2.x) obj->bound2.x = m->p.x;
    if (m->p.y > obj->bound2.y) obj->bound2.y = m->p.y;
    if (m->p.z > obj->bound2.z) obj->bound2.z = m->p.z;
    m++;
    z--;
  } while (z != 0);
  obj->bound1.x -= 0.5;
  obj->bound1.y -= 0.5;
  obj->bound1.z -= 0.5;
  obj->bound2.x += 0.5;
  obj->bound2.y += 0.5;
  obj->bound2.z += 0.5;
  obj->center.x = (obj->bound1.x + obj->bound2.x)*0.5;
  obj->center.y = (obj->bound1.y + obj->bound2.y)*0.5;
  obj->center.z = (obj->bound1.z + obj->bound2.z)*0.5;
//  vecMul3(&obj->center, &obj->bound1, 0.5);
//  vecMulAdd(&obj->center, &obj->bound2, 0.5);
}

/*
void moveMasses(int z, Mass *m) {
//  if (z <= 0) return;
  do {
    dvecAdd(&m->p, &m->v);
    m->v.z -= 0.005; //gravity
    m++;
    z--;
  } while (z != 0);
}
*/

/*
void cubeForce(Mass *m1, Mass *m2, float distance) {
  Vector d;

  vecSub(&d, &m2->p, &m1->p);
  vecMul(&d, (1.0 - distance/vecAbs(&d))*0.2);
  vecAdd(&m1->F, &d);
//  vecSub(&m2->F, &d);
}
*/

//imaginary spring between two masses
void spring(Mass *m1, Mass *m2, double length) {
  DVector d;

  dvecSub3(&d, &m2->p, &m1->p);
  dvecMulAdd(&m1->v, &d, (1.0 - length/dvecAbs(&d))*0.2);
}

//reduce velocity due to friction between two masses
void friction(Mass *m1, Mass *m2) {
  DVector d, vd1, vd2;
  float l;

  dvecSub3(&d, &m2->p, &m1->p);
  l = dvecMul(&d, &d);

  dvecMul3(&vd1, &d, dvecMul(&d, &m1->v)/l);
  dvecMul3(&vd2, &d, dvecMul(&d, &m2->v)/l);
  dvecSub(&vd1, &vd2);
  dvecMulAdd(&m1->v, &vd1, (0.95 - 1.0)/2.0);
}

void cubeStep(Object *obj) {
  int z, i;
  Mass *cube, *m1;

  cube = obj->masses;

  for (z = 0; z < 8; z++) {
    m1 = &cube[z];
  //  cubeForce(m1, &cube[z ^ 7], 3.4641);
    for (i = 1; i <= 4; i <<= 1) { //1, 2, 4
      friction(m1, &cube[z ^ i]);
      friction(m1, &cube[z ^ i ^ 7]);
      spring(m1, &cube[z ^ i], 2.0);
      spring(m1, &cube[z ^ i ^ 7], 2.8284);
    }
  }
}



/*void cubeStep(Object *obj) {
  float m, g, t;
  int z, i;
  Mass *cube, *m1;

  cube = obj->masses;
  m = 1.0;
  g = -0.005;
//  delta = 0.999;
  t = 1.0;

  for (z = 0; z < 8; z++) {
    m1 = &cube[z];
    m1->F.x = 0.0;
    m1->F.y = 0.0;
    m1->F.z = 0.0;
  }
  for (z = 0; z < 8; z++) {
    m1 = &cube[z];
    for (i = 1; i <= 4; i <<= 1) {
      cubeForce(m1, &cube[z ^ i], 2.0);
      cubeForce(m1, &cube[z ^ i ^ 7], 2.8284);

      cubeDelta(m1, &cube[z ^ i]);
      cubeDelta(m1, &cube[z ^ i ^ 7]);
    }
  //  cubeForce(m1, &cube[z ^ 7], 3.4641);
  }

  for (z = 0; z < 8; z++) {
    m1 = &cube[z];

    vecMul(&m1->F, t/m);
    vecAdd(&m1->v, &m1->F);
    m1->v.z += g*t;
  }
} */


/*
  given : face verts a, b, c and point p (all 2D)
  problem : find the weights of a, b and c to get p
  wa*a + wb*b + wc*c = p
  wa + wb + wc = 1
  matix expression:

      (ax  bx  cx)
  A = (ay  by  cy)
      ( 1   1   1)

    (wa)   (px)      (wa)        (px)
  A*(wb) = (py)  =>  (wb) = A^-1*(py)
    (wc)   ( 1)      (wc)        ( 1)

  if another object hits a face, the weights can be used to calculate the
  force distribution across the three masses at the face verts.
*/

void updateCFace(CFace *face) {
  DVector u, v;
  DVector *a, *b, *c;
  double max, f, det;
  int i;

  //face verts
  a = &face->ma->p;
  b = &face->mb->p;
  c = &face->mc->p;

  //calc face normal
  dvecSub3(&u, b, a);
  dvecSub3(&v, c, a);
  dvecCross(&face->n, &v, &u);
  dvecNorm(&face->n);

  //find projection plane for weight calculation
  max = fabs(face->n.x);
  i = 0;
  f = fabs(face->n.y);
  if (f > max) {
    i = 1;
    max = f;
  }
  if (fabs(face->n.z) > max) i = 2;
  face->orientation = i;
  if (i == 2) {         //x y
    //determinant
    det = a->x*b->y + b->x*c->y + c->x*a->y - b->y*c->x - c->y*a->x - a->y*b->x;
    //calc inverse weight matrix for collision
    dvecAssign4(&face->M1, b->y - c->y, c->x - b->x, b->x*c->y - b->y*c->x);
    dvecAssign4(&face->M2, c->y - a->y, a->x - c->x, c->x*a->y - c->y*a->x);
    dvecAssign4(&face->M3, a->y - b->y, b->x - a->x, a->x*b->y - a->y*b->x);
  } else if (i == 0) {  //y z
    det = a->y*b->z + b->y*c->z + c->y*a->z - b->z*c->y - c->z*a->y - a->z*b->y;
    dvecAssign4(&face->M1, b->z - c->z, c->y - b->y, b->y*c->z - b->z*c->y);
    dvecAssign4(&face->M2, c->z - a->z, a->y - c->y, c->y*a->z - c->z*a->y);
    dvecAssign4(&face->M3, a->z - b->z, b->y - a->y, a->y*b->z - a->z*b->y);
  } else {              //z x
    det = a->z*b->x + b->z*c->x + c->z*a->x - b->x*c->z - c->x*a->z - a->x*b->z;
    dvecAssign4(&face->M1, b->x - c->x, c->z - b->z, b->z*c->x - b->x*c->z);
    dvecAssign4(&face->M2, c->x - a->x, a->z - c->z, c->z*a->x - c->x*a->z);
    dvecAssign4(&face->M3, a->x - b->x, b->z - a->z, a->z*b->x - a->x*b->z);
  }
  //finish inverse calculation
  det = 1.0/det;
  dvecMulf(&face->M1, det);
  dvecMulf(&face->M2, det);
  dvecMulf(&face->M3, det);
}

void updateObject(Object *obj) {
  int z;
  CFace *face;

  z = obj->nCFaces;
  face = obj->cFaces;
  do {
    updateCFace(face);
    face++;
    z--;
  } while (z > 0);
}

void collision(Object *obj1, Object *obj2) {
  int faces, edges;
  CFace *face;
  CEdge *edge;
  DVector u, v;
  double f, r;
  Point p;
  double wa, wb, wc, vd;



  faces = obj1->nCFaces;
  face = obj1->cFaces;
  do {
    edges = obj2->nCEdges;
    edge = obj2->cEdges;
    do {
      //u = direction vector of edge
      dvecSub3(&u, &edge->mb->p, &edge->ma->p);
      //find crossing of face and edge (the hitpoint)
      //r = (p - a)*n/(u*n), p: point in face-plane, n: face-normal, a: point in edge
      f = dvecMul(&u, &face->n);
      if (fabs(f) > 1e-5) {
        dvecSub3(&v, &face->ma->p, &edge->ma->p);
        r = dvecMul(&v, &face->n)/f;
        if (r >= 0.0 && r <= 1.0) {
          if (face->orientation == 2) {        //x y
            p.x = edge->ma->p.x + r*u.x;
            p.y = edge->ma->p.y + r*u.y;
          } else if (face->orientation == 0) { //y z
            p.x = edge->ma->p.y + r*u.y;
            p.y = edge->ma->p.z + r*u.z;
          } else {                             //z x
            p.x = edge->ma->p.z + r*u.z;
            p.y = edge->ma->p.x + r*u.x;
          }
          //calc weights for hitpoint
          wa = face->M1.x*p.x + face->M1.y*p.y + face->M1.z;
          wb = face->M2.x*p.x + face->M2.y*p.y + face->M2.z;
          wc = face->M3.x*p.x + face->M3.y*p.y + face->M3.z;
          if (wa >= 0.0 && wb >= 0.0 && wc >= 0.0) { //hit inside face
            //difference of face hitpoint velocity and edge hitpoint velocity,
            //measured in the direction of the face normal
            vd = dvecMul(&face->n, &face->ma->v)*wa
               + dvecMul(&face->n, &face->mb->v)*wb
               + dvecMul(&face->n, &face->mc->v)*wc
               - dvecMul(&face->n, &edge->ma->v)*(1.0 - r)
               - dvecMul(&face->n, &edge->mb->v)*r;
            if (vd > 0.0) { //positive: calculate bounce off
              dvecMul3(&v, &face->n, -vd*wa);
              dvecAdd(&face->ma->v, &v);
              dvecMul3(&v, &face->n, -vd*wb);
              dvecAdd(&face->mb->v, &v);
              dvecMul3(&v, &face->n, -vd*wc);
              dvecAdd(&face->mc->v, &v);

              dvecMul3(&v, &face->n, vd*(1.0 - r));
              dvecAdd(&edge->ma->v, &v);
              dvecMul3(&v, &face->n, vd*r);
              dvecAdd(&edge->mb->v, &v);
            }
          }
        }
      }
      edge++;
      edges--;
    } while (edges != 0);
    face++;
    faces--;
  } while (faces != 0);
}

void pCollision(int particles, Particle *particle, Object *obj) {
  int faces;
  CFace *face;
  DVector u;
  double d, vn;//d2;
  Point p;
  double wa, wb, wc, vd;

  do {
    if (particle->state == 0) goto skip;
    //bounding box check
    dvecAdd3(&u, &particle->m.p, &particle->m.v);
    if (obj->bound1.x < obj->bound2.x) {
      if (u.x < obj->bound1.x || u.x > obj->bound2.x) goto skip;
    } else {
      if (u.x < obj->bound2.x || u.x > obj->bound1.x) goto skip;
    }
    if (obj->bound1.y < obj->bound2.y) {
      if (u.y < obj->bound1.y || u.y > obj->bound2.y) goto skip;
    } else {
      if (u.y < obj->bound2.y || u.y > obj->bound1.y) goto skip;
    }
    if (obj->bound1.z < obj->bound2.z) {
      if (u.z < obj->bound1.z || u.z > obj->bound2.z) goto skip;
    } else {
      if (u.z < obj->bound2.z || u.z > obj->bound1.z) goto skip;
    }

    faces = obj->nCFaces;
    face = obj->cFaces;
    do {
      //calc distance between face and particle
      dvecSub3(&u, &particle->m.p, &face->ma->p);
      d = dvecMul(&u, &face->n);
      vn = dvecMul(&particle->m.v, &face->n);
//      d2 = d + vecMul(&particle->m.v, &face->n);
//printf("d %7.2f d %7.2f\n", d, d2);
      if (d + vn < 0.1 && d > -0.5) { //(d2 < 0.1 && d > -0.5) {
        if (face->orientation == 2) {        //x y
          p.x = particle->m.p.x - d*face->n.x;
          p.y = particle->m.p.y - d*face->n.y;
        } else if (face->orientation == 0) { //y z
          p.x = particle->m.p.y - d*face->n.y;
          p.y = particle->m.p.z - d*face->n.z;
        } else {                             //z x
          p.x = particle->m.p.z - d*face->n.z;
          p.y = particle->m.p.x - d*face->n.x;
        }
        //calc weights for hitpoint
        wa = face->M1.x*p.x + face->M1.y*p.y + face->M1.z;
        wb = face->M2.x*p.x + face->M2.y*p.y + face->M2.z;
        wc = face->M3.x*p.x + face->M3.y*p.y + face->M3.z;
        if (wa >= 0.0 && wb >= 0.0 && wc >= 0.0) { //hit inside face
//printf("wa %7.2f wb %7.2f wc %7.2f\n", wa, wb, wc);
          //difference of face hitpoint velocity and edge hitpoint velocity,
          //measured in the direction of the face normal
          vd = dvecMul(&face->n, &face->ma->v)*wa
             + dvecMul(&face->n, &face->mb->v)*wb
             + dvecMul(&face->n, &face->mc->v)*wc
             - vn;//vecMul(&face->n, &particle->m.v);
          if (vd > 0.0) { //positive: calculate bounce off
//printf("collision %7.3f\n", vd);
//            vecMul3(&u, &face->n, vd*1.6);
//            vecAdd(&particle->m.v, &u);
            dvecMulAdd(&particle->m.v, &face->n, vd*1.6);
          }
        }
      }
      face++;
      faces--;
    } while (faces != 0);
//printf("\n");
  skip:
    particle++;
    particles--;
  } while (particles != 0);
}

void mCollision(int masses, Mass *m, Object *obj) { //cloth
  int faces;
  CFace *face;
  DVector u;
  double d, vn;
  Point p;
  double wa, wb, wc, vd;

  do {
    //bounding box check
    dvecAdd3(&u, &m->p, &m->v);
    if (obj->bound1.x < obj->bound2.x) {
      if (u.x < obj->bound1.x || u.x > obj->bound2.x) goto skip;
    } else {
      if (u.x < obj->bound2.x || u.x > obj->bound1.x) goto skip;
    }
    if (obj->bound1.y < obj->bound2.y) {
      if (u.y < obj->bound1.y || u.y > obj->bound2.y) goto skip;
    } else {
      if (u.y < obj->bound2.y || u.y > obj->bound1.y) goto skip;
    }
    if (obj->bound1.z < obj->bound2.z) {
      if (u.z < obj->bound1.z || u.z > obj->bound2.z) goto skip;
    } else {
      if (u.z < obj->bound2.z || u.z > obj->bound1.z) goto skip;
    }

    faces = obj->nCFaces;
    face = obj->cFaces;
    do {
      //calc distance between face and particle
      dvecSub3(&u, &m->p, &face->ma->p);
      d = dvecMul(&u, &face->n);
      vn = dvecMul(&m->v, &face->n); //velocity in normal direction
//printf("d %7.2f d %7.2f\n", d, d2);
      if (d + vn < 0.5 && d > -0.5) { //(d + vn < 0.5 && d > -0.5) {
        if (face->orientation == 2) {        //x y
          p.x = m->p.x - d*face->n.x;
          p.y = m->p.y - d*face->n.y;
        } else if (face->orientation == 0) { //y z
          p.x = m->p.y - d*face->n.y;
          p.y = m->p.z - d*face->n.z;
        } else {                             //z x
          p.x = m->p.z - d*face->n.z;
          p.y = m->p.x - d*face->n.x;
        }
        //calc weights for hitpoint
        wa = face->M1.x*p.x + face->M1.y*p.y + face->M1.z;
        wb = face->M2.x*p.x + face->M2.y*p.y + face->M2.z;
        wc = face->M3.x*p.x + face->M3.y*p.y + face->M3.z;
        if (wa >= 0.0 && wb >= 0.0 && wc >= 0.0) { //hit inside face
          //remove normal velocity component, lead to surface
//          vecMulAdd(&m->v, &face->n, d - vn);

          vd = dvecMul(&face->n, &face->ma->v)*wa
             + dvecMul(&face->n, &face->mb->v)*wb
             + dvecMul(&face->n, &face->mc->v)*wc
             - vn;//vecMul(&face->n, &particle->m.v);
          if (vd > 0.0) { //positive: calculate bounce off
//printf("collision %7.3f\n", vd);
//            vecMul3(&u, &face->n, vd*1.6);
//            vecAdd(&particle->m.v, &u);
            dvecMulAdd(&m->v, &face->n, vd + 0.05);
          }

//          vecMulAdd(&m->v, &face->n, -vn);
//          vecMulAdd(&m->p, &face->n, 0.5 - d);
        }
      }
      face++;
      faces--;
    } while (faces != 0);
  skip:
    m++;
    masses--;
  } while (masses != 0);
}

void bounce(int z, Mass *m) {
//  if (z <= 0) return;
  do {
/*
    if (m->p.y > 0.9) {
      if (m->p.x - (m->p.z - 16.0) < 0.0 && m->p.z < 16.0) {
        m->p.z = 16.0;
        m->v.z = 0.0;
      }
      if (m->p.x - (m->p.z - 16.0) > 0.0 && m->p.x < 0.0) {
        m->p.x = 0.0;
        m->v.x = 0.0;
      }
    }
*/
    if (m->p.z + m->v.z < 0.0) {
      m->v.z = 0.0 - m->p.z;
      m->v.x *= 0.3;
      m->v.y *= 0.3;
    }
    if (m->p.x + m->v.x < -10.0) {
      m->v.x = -10.0 - m->p.x;
      m->v.y *= 0.3;
      m->v.z *= 0.3;
    }
    if (m->p.x + m->v.x > 10.0) {
      m->v.x = 10.0 - m->p.x;
      m->v.y *= 0.3;
      m->v.z *= 0.3;
    }
    if (m->p.y + m->v.y < -10.0) {
      m->v.y = -10.0 - m->p.y;
      m->v.x *= 0.3;
      m->v.z *= 0.3;
    }
    if (m->p.y + m->v.y > 10.0) {
      m->v.y = 10.0 - m->p.y;
      m->v.x *= 0.3;
      m->v.z *= 0.3;
    }
    m++;
    z--;
  } while (z != 0);
}
/*
void moveParticles(int z, Particle *p) {
//  if (z <= 0) return;
  do {
    vecAdd(&p->m.p, &p->m.v);
    p->m.v.z -= 0.005; //gravity
    p++;
    z--;
  } while (z != 0);
}
*/
float random2(int a, int b) {
  int i;
  //  i = (a*b) % 219;
  //  i = (i ^ a ^ b) % 111;
  i = a*b;
  i = i ^ (i >> 8) ^ (i >> 16);
  i = i % 111;
  return (i - 55)/3000.0;
//  return ((i & 0xFF) - 128)/128.0 ;
}

void bounceParticles(int z, Particle *p, int countpos) {
  Vector pSourcePos;
  int active, newCount;

//printf("countpos %d\n", countpos);
//  if (z <= 0) return;
  //position track of particle source
  doTrack3(&pSourceTrack, countpos, &pSourcePos);
  active = doSwitchTrack(&pSourceActiveTrack, countpos);
  newCount = 1;
  do {
    if (active && p->state == 0 && newCount > 0) {
      //generate particle
      p->state = 1;
      p->m.p.x = pSourcePos.x;// + random2(countpos, newCount + 3);
      p->m.p.y = pSourcePos.y;// + random2(countpos, newCount + 7);
      p->m.p.z = pSourcePos.z;// + random2(countpos, newCount + 11);
      p->m.v.x = random2(countpos, newCount + 3);
      p->m.v.y = random2(countpos, newCount + 7);
      p->m.v.z = random2(countpos, newCount + 11);//(newCount - 2)*0.05;
      newCount--;
    }
    if (p->state != 0) {
      dvecAdd(&p->m.p, &p->m.v);
      p->m.v.z -= 0.005; //gravity


      if (p->m.p.z < 0.0 && p->m.v.z < 0.0) {
        p->m.v.z *= -0.6;
        p->m.v.x *= 0.9;
        p->m.v.y *= 0.9;
        p->state++;
        if (p->state == 4) p->state = 0;
      }
      if (p->m.p.x < -10.0 && p->m.v.x < 0.0) {
        p->m.v.x *= -0.6;
        p->m.v.y *= 0.9;
        p->m.v.z *= 0.9;
      }
      if (p->m.p.x > 10.0 && p->m.v.x > 0.0) {
        p->m.v.x *= -0.6;
        p->m.v.y *= 0.9;
        p->m.v.z *= 0.9;
      }
      if (p->m.p.y < -10.0 && p->m.v.y < 0.0) {
        p->m.v.y *= -0.6;
        p->m.v.x *= 0.9;
        p->m.v.z *= 0.9;
      }
      if (p->m.p.y > 10.0 && p->m.v.y > 0.0) {
        p->m.v.y *= -0.6;
        p->m.v.x *= 0.9;
        p->m.v.z *= 0.9;
      }
    }
    p++;
    z--;
  } while (z != 0);
}

void cubeGeometry(Vector *vertex, Vector *normal, Vector *a, Vector *b,
  Vector *c, Vector *d) {
  int x, y;
  Vector *cVertex, *cNormal;
  Vector da, cb, n, r, l, v;

  vecSub3(&da, d, a); //expected length of da and cb : 2*sqrt(2)
  vecSub3(&cb, c, b);
  vecCross(&n, &da, &cb);
  vecMulf(&n, 0.125);
// vecPrint(&da);
// vecPrint(&cb);
// vecPrint(&n);
  cVertex = cubeVerts;
  cNormal = cubeNormals;
  for (y = 0; y < points; y++) {
    for (x = 0; x < points; x++) {
      //r = a + (profile->y)*(b - a);
      //l = c + (profile->y)*(d - c);
      //x = r + (profile->x)*(l - r);

      vecSub3(&r, b, a);
      vecMulf(&r, cVertex->y);
      vecAdd(&r, a);
      vecSub3(&l, d, c);
      vecMulf(&l, cVertex->y);
      vecAdd(&l, c);

      vecSub3(&v, &l, &r);
      vecMulf(&v, cVertex->x);
      vecAdd(&v, &r);

      vecMul3(vertex, &n, cVertex->z);
      vecAdd(vertex, &v);
      vecMul3(normal, &da, cNormal->x);
      vecMulAdd(normal, &cb, cNormal->y);
      vecMulAdd(normal, &n, cNormal->z);
//printf("n-len %7.3f ", vecAbs(normal));
//vecPrint(normal);

      cVertex++;
      cNormal++;
      vertex++;
      normal++;
    }
  }
}

void updateGeometry(Object *obj, float t) {
  int z;
  Mass *m;
  Vector u;

  z = obj->nMasses;
  m = obj->masses;
  do {
    m->tp.x = (float) m->p.x + t*(float) m->v.x;
    m->tp.y = (float) m->p.y + t*(float) m->v.y;
    m->tp.z = (float) m->p.z + t*(float) m->v.z;
//    vecMul3(&m->tp, &m->v, t);
//    vecAdd(&m->tp, &m->p);
    m++;
    z--;
  } while (z != 0);

  //shrink geometry a bit
  z = 3;
  do {
    vecSub3(&u, &obj->masses[z ^ 7].tp, &obj->masses[z].tp);
    vecMulAdd(&obj->masses[z].tp, &u, 0.05);
    vecMulAdd(&obj->masses[z ^ 7].tp, &u, -0.05);
    z--;
  } while (z >= 0);

  cubeGeometry(obj->verts + 0*points*points, obj->normals + 0*points*points,
    &obj->masses[0].tp,
    &obj->masses[1].tp,
    &obj->masses[2].tp,
    &obj->masses[3].tp);

  cubeGeometry(obj->verts + 1*points*points, obj->normals + 1*points*points,
    &obj->masses[0].tp,
    &obj->masses[2].tp,
    &obj->masses[4].tp,
    &obj->masses[6].tp);

  cubeGeometry(obj->verts + 2*points*points, obj->normals + 2*points*points,
    &obj->masses[0].tp,
    &obj->masses[4].tp,
    &obj->masses[1].tp,
    &obj->masses[5].tp);

  cubeGeometry(obj->verts + 3*points*points, obj->normals + 3*points*points,
    &obj->masses[7].tp,
    &obj->masses[3].tp,
    &obj->masses[5].tp,
    &obj->masses[1].tp);

  cubeGeometry(obj->verts + 4*points*points, obj->normals + 4*points*points,
    &obj->masses[7].tp,
    &obj->masses[5].tp,
    &obj->masses[6].tp,
    &obj->masses[4].tp);

  cubeGeometry(obj->verts + 5*points*points, obj->normals + 5*points*points,
    &obj->masses[7].tp,
    &obj->masses[6].tp,
    &obj->masses[3].tp,
    &obj->masses[2].tp);
}


void updateParticle(int z, Particle *particle, float t) {
  do {
    particle->m.tp.x = (float) particle->m.p.x + t*(float) particle->m.v.x;
    particle->m.tp.y = (float) particle->m.p.y + t*(float) particle->m.v.y;
    particle->m.tp.z = (float) particle->m.p.z + t*(float) particle->m.v.z;
//    vecMul3(&particle->m.tp, &particle->m.v, t);
//    vecAdd(&particle->m.tp, &particle->m.p);
    particle++;
    z--;
  } while (z != 0);
}


//--- Cloth
#define clothd1 0.3
#define clothd2 0.6
#define clothd3 0.8485

void initCloth(Cloth *c) {
  int x, y, cp2, yc;
  Mapping *ma;
  Face *face;

  //masses
  c->nMasses = clothPoints*clothPoints;
  c->masses = (Mass *) getmem(c->nMasses*sizeof(Mass));
//  c->center = &c->masses[clothPoints*(clothPoints/2) + clothPoints/2].tp;

  //verts, normals
  cp2 = clothPoints -2;
  c->nVerts = cp2*cp2;
  c->verts = (Vector *) getmem(c->nVerts*sizeof(Vector));
  c->normals = (Vector *) getmem(c->nVerts*sizeof(Vector));
  //mapping
  ma = c->mapping = (Mapping *) getmem(c->nVerts*sizeof(Mapping));
  for (y = 0; y < cp2; y++) {
    for (x = 0; x < cp2; x++) {
      ma->u = 128.0*256.0 + 127.0*256.0*x/(float) (cp2 -1);
      ma->v = 127.0*256.0*y/(float) (cp2 -1);
      ma++;
    }
  }

  //faces
  c->nFaces = (clothPoints -3)*(clothPoints -3)*2;
  face = c->faces = (Face *) getmem(c->nFaces*sizeof(Face));
  for (y = 0; y < cp2 -1; y++) {
    yc = y*cp2;
    for (x = 0; x < cp2 -1; x++) {
      face->a = yc + x;
      face->b = yc + cp2 + x;
      face->c = yc + cp2 + (x +1);
      face++;
      face->c = yc + x;
      face->b = yc + (x +1);
      face->a = yc + cp2 + (x +1);
      face++;
    }
  }
}

void setClothLocation(Cloth *c, double xoffs, double yoffs, double zoffs) {
  int x, y;
  Mass *m;

  m = c->masses;
  for (y = 0; y < clothPoints; y++) {
    for (x = 0; x < clothPoints; x++) {
      m->p.x = (x - clothPoints/2)*clothd1 + xoffs;
      m->p.y = (y - clothPoints/2)*clothd1 + yoffs;
      m->p.z = zoffs;
      m->v.x = 0.0;
      m->v.y = 0.0;
      m->v.z = 0.0;
      m++;
    }
  }
}

void spring2(Mass *m1, Mass *m2, double length) {
  DVector d;

  dvecSub3(&d, &m2->p, &m1->p);
  dvecMulf(&d, (1.0 - length/dvecAbs(&d))*0.2);
  dvecAdd(&m1->v, &d);
  dvecSub(&m2->v, &d);
}

//reduce velocity due to friction between two masses
void friction2(Mass *m1, Mass *m2) {
  DVector d, vd1, vd2;
  double l;

  dvecSub3(&d, &m2->p, &m1->p);
  l = dvecMul(&d, &d);

  dvecMul3(&vd1, &d, dvecMul(&d, &m1->v)/l);
  dvecMul3(&vd2, &d, dvecMul(&d, &m2->v)/l);
  dvecSub(&vd1, &vd2);
  dvecMulf(&vd1, (0.3 - 1.0)/2.0);
  dvecAdd(&m1->v, &vd1);
  dvecSub(&m2->v, &vd1);
}

void clothStep(Cloth *c) {
  int z, x, y;
  Mass *m, *m1, *m2;

  m = c->masses;
  for (z = 0; z < c->nMasses; z++) {
    dvecAdd(&m->p, &m->v); //move
    m->v.z -= 0.005; //gravity
    m++;
  }

/*
  int min, max;
  min = clothPoints/2 -2;
  max = clothPoints/2 +1;
  for (y = 0; y < clothPoints; y++) {
    for (x = 0; x < clothPoints; x++) {
      vecAdd(&m->p, &m->v); //move
      if ((x < min || x > max) || (y < min || y > max)) {
        m->v.z -= 0.005; //gravity
      } else {
        m->p.z = 10.0;
        m->v.z = 0.0;
      }
      m++;
    }
  }
*/
  for (y = 1; y < clothPoints -1; y++) {
    for (x = 1; x < clothPoints -1; x++) {
      m1 = &c->masses[y*clothPoints + x];
      m2 = &c->masses[y*clothPoints + x +1];

      spring2(m1, m2, clothd1);
      friction2(m1, m2);
      m2 = &c->masses[y*clothPoints + x -1];
      spring2(m1, m2, clothd1);
      friction2(m1, m2);
      m2 = &c->masses[(y +1)*clothPoints + x];
      spring2(m1, m2, clothd1);
      friction2(m1, m2);
      m2 = &c->masses[(y -1)*clothPoints + x];
      spring2(m1, m2, clothd1);
      friction2(m1, m2);

      m1 = &c->masses[(y -1)*clothPoints + x -1];
      m2 = &c->masses[(y +1)*clothPoints + x +1];
      spring2(m1, m2, clothd3);
      friction2(m1, m2);
      m1 = &c->masses[(y -1)*clothPoints + x +1];
      m2 = &c->masses[(y +1)*clothPoints + x -1];
      spring2(m1, m2, clothd3);
      friction2(m1, m2);

      m1 = &c->masses[y*clothPoints + x -1];
      m2 = &c->masses[y*clothPoints + x +1];
      spring2(m1, m2, clothd2);
      friction2(m1, m2);
      m1 = &c->masses[(y -1)*clothPoints + x];
      m2 = &c->masses[(y +1)*clothPoints + x];
      spring2(m1, m2, clothd2);
      friction2(m1, m2);
    }
  }

  m = c->masses;
  for (z = 0; z < c->nMasses; z++) {
    if (m->p.z < 0.0) {
      m->p.z = 0.0;
      m->v.x = 0.0;
      m->v.y = 0.0;
      m->v.z = 0.0;
    }
    m++;
  }
}

void updateCloth(Cloth *c, float t) {
  int z;
  Mass *m;
  int x, y, yc;
  Vector *vertex, *n;
  Vector u, v;

  z = c->nMasses;
  m = c->masses;
  do {
    m->tp.x = (float) m->p.x + t*(float) m->v.x;
    m->tp.y = (float) m->p.y + t*(float) m->v.y;
    m->tp.z = (float) m->p.z + t*(float) m->v.z;
//    vecMul3(&m->tp, &m->v, t);
//    vecAdd(&m->tp, &m->p);
    m++;
    z--;
  } while (z != 0);
  m = &c->masses[clothPoints*(clothPoints/2) + clothPoints/2];
  c->center.x = m->tp.x;
  c->center.y = m->tp.y;
  c->center.z = m->tp.z;

  m = c->masses;
  vertex = c->verts;
  n = c->normals;
  yc = 0;
  for (y = 1; y < clothPoints -1; y++) {
    yc += clothPoints;
    for (x = 1; x < clothPoints -1; x++) {
      vecAssign(vertex, &m[yc + x].tp);
      vecSub3(&u, &m[yc - clothPoints + x].tp, &m[yc + clothPoints + x].tp);
      vecSub3(&v, &m[yc + x +1].tp, &m[yc + x -1].tp);
      vecCross(n, &u, &v);
      vecNorm(n);
      vertex++;
      n++;
    }
  }
}

//---
int sFaceZclip(Face *face, Vector *verts, ClipInfo *info) {
  int idx[4];
  ParamPoint *dp;
  Vector *v, *nv;
  int z, count, vis, nvis;
  float r;

  idx[0] = idx[3] = face->a;
  idx[1] = face->b;
  idx[2] = face->c;

  dp = info->start;

  //project & z-clip
  z = 0;
  count = 0;
  v = &verts[idx[0]];
  vis = (v->z >= 1.0);
  do {
    if (vis) {
      //point in front of projection plane (visible)
//      dp->x = mxres*(1.0 + v->x/v->z);
//      dp->y = myres*(1.0 - v->y/v->z);
      dp->x = xAdd + v->x/v->z;
      dp->y = yAdd + yMul*v->y/v->z;
      count++;
      dp = (ParamPoint *) ((char *) dp + sizeof(ParamPoint) );
    }
    z++;
    nv = &verts[idx[z]];
    nvis = (nv->z >= 1.0);
    if (vis != nvis) {
      r = (1.0 - v->z)/(nv->z - v->z);

//      dp->x = mxres*(1.0 + v->x + r*(nv->x - v->x));
//      dp->y = myres*(1.0 - v->y - r*(nv->y - v->y));
      dp->x = xAdd + v->x + r*(nv->x - v->x);
      dp->y = yAdd + yMul*(v->y + r*(nv->y - v->y));
      count++;
      dp = (ParamPoint *) ((char *) dp + sizeof(ParamPoint) );
    }
    v = nv;
    vis = nvis;
  } while (z < 3);
  info->end = dp;
  return count >= 3;
}

void setPicControl(int countpos) { //0 - 116
  int *pc, *tc;
  int i;

  i = countpos + 7;
  pc = picControl;
  *pc = 0;
  if (countpos > 10 && countpos < 110) {
    *pc = ((i >> 5) & 7) +1;
  }
  pc++;
  i += 33;
  *pc = 0;
  if (countpos > 25 && countpos < 100) {
    *pc = ((i >> 4) & 7) +1;
  }
  pc++;
  i = -i;
  *pc = 0;
  if (countpos > 40 && countpos < 80) {
    *pc = ((i >> 4) & 7) +1;
  }
  pc++;
  i += 20;
  *pc = 0;
  if (countpos > 20 && countpos < 105) {
    *pc = ((i >> 4) & 7) +1;
  }
  pc++;
  i += 33;
  *pc = 0;
  if (countpos > 5 && countpos < 114) {
    *pc = ((i >> 5) & 7) +1;
  }

  tc = textControl;
  *tc = !(i & 16) + 1;
  tc++;
  *tc = 0;
  tc++;
  *tc = (countpos >> 3) & 3;
  if (*tc) *tc = 3;
  tc++;
  *tc = 0;
  tc++;
  i += 13;
  *tc = !(i & 16) + 1;
}

void step() {
  int s = 256;
  int timepos;
  float t;

  timepos = countpos*s;
  while (timecount >= timepos) {
    //frame*500/256
    if (countpos >= 469 && countpos < 586) {  //240 - 300
      renderFlag = 1;
      setPicControl(countpos - 469);
    }
    if (countpos == 586) { //300
      renderFlag = 0;
//      setCubeLocation(&objects[0], 4.0, 1.0, 5.0, 0.0);
      setCubeLocation(&objects[0], 4.8, 1.0, 5.0);
      setCubeLocation(&objects[1], 5.0, 0.0, 10.0);
      cubeGravity = 0;
    }
    if (countpos == 566 +78) { //330  //290
      clothVisible = 1;
    }
    if (countpos == 580 +78) { //337  //297
      startLockObj = &objects[1];
    }
    if (countpos == 585 +78) { //340  //300
      cubeGravity = 1;
    }
    if (countpos == 625 +78) { //360  //320
      startLockObj = 0L;
    }
    if (countpos == 938) { //480
      renderFlag = 2;
    }
    if (countpos == 990) renderFlag++;
    if (countpos >= 1055) { //540
      endFlag = 1;
    }
    //objects
    moveObject(&objects[0], 1); //move, gravity, bounding box
    cubeStep(&objects[0]); //springs and friction between masses
    updateObject(&objects[0]); //update collision parameters

    moveObject(&objects[1], cubeGravity);
    cubeStep(&objects[1]);
    updateObject(&objects[1]);

    collision(&objects[0], &objects[1]);
    collision(&objects[1], &objects[0]);

    bounce(objects[0].nMasses, objects[0].masses);
    bounce(objects[1].nMasses, objects[1].masses);

    //particles
    bounceParticles(nParticles, particles, timepos);
//    pCollision(nParticles, particles, &objects[0]);
    pCollision(nParticles, particles, &objects[1]);
//    bounceParticles(nParticles, particles);

    if (clothVisible) {
      clothStep(&cloth);
      mCollision(cloth.nMasses, cloth.masses, &objects[1]);
    }
    countpos++;
    timepos += 256;
  }
  t = (s + timecount - timepos)/(float) s;
  updateGeometry(&objects[0], t);
  updateGeometry(&objects[1], t);
  updateParticle(nParticles, particles, t);
  if (clothVisible) updateCloth(&cloth, t);
}

//--- shadow and light
void lightVerts(int z, Vector *vertex, Vector *normal, Vector *light, Wall *walls, LightInfo *info) {
  Vector *svert, *lvert;
  char *tag;
  Vector u, p_a;
  float m, f, r, rmin;
  int i;
  Wall *w;

  //get mem
  svert = info->sVerts = (Vector *) globalHeap;
  lvert = info->lVerts = svert + z;;
  tag = info->vTags = (char *) (lvert + z);
  globalHeap = (void *) (tag + (z +3 & ~3));

  do {
//   vecMul(&u, normal, 0.5);
//   vecAdd3(lvert, vertex, &u);
//   *tag = 1;
    vecSub3(&u, vertex, light);
    m = vecMul(normal, &u);
    if ((*tag = (m < 0.0))) { //is point visible from light?
      //shadow
      rmin = 1e10;
      w = walls;
      i = 6;
      do {
        f = vecMul(&u, &w->n);
        if (f < -1e-10) {
          vecSub3(&p_a, &w->p, vertex);
          r = vecMul(&p_a, &w->n)/f;
          if (r < rmin) {
            rmin = r;
          }
        }
        w++;
        i--;
      } while (i != 0);
      svert->x = vertex->x + rmin*u.x;
      svert->y = vertex->y + rmin*u.y;
      svert->z = vertex->z + rmin*u.z;

      //light
      //mirror u with normal
      vecMulAdd(&u, normal, -2.0*m);
 // printf("len %7.3f\n", vecAbs(normal));
      rmin = 1e10;
      w = walls;
      i = 6;
      do {
        f = vecMul(&u, &w->n);
        if (f < -1e-10) {
          vecSub3(&p_a, &w->p, vertex);
          r = vecMul(&p_a, &w->n)/f;
          if (r < rmin) {
            rmin = r;
          }
        }
        w++;
        i--;
      } while (i != 0);
      lvert->x = vertex->x + rmin*u.x;
      lvert->y = vertex->y + rmin*u.y;
      lvert->z = vertex->z + rmin*u.z;
    }
    vertex++;
    normal++;
    svert++;
    lvert++;
    tag++;
    z--;
  } while (z != 0);
}

void lightIntensities(int z, Face *face, LightInfo *info) {
  char *tag;
  float *intensity;
  Vector *a;
  Vector u, v, n;

  //get mem
  tag = info->fTags = (char *) globalHeap;
  intensity = info->lIntensities = (float *) (tag + (z +3 & ~3));

  do {
    if ((*tag = (info->vTags[face->a] && info->vTags[face->b]
      && info->vTags[face->c]))) {

      if (face->light) {
        *tag = 2;
        a = &info->lVerts[face->a];
        vecSub3(&u, &info->lVerts[face->b], a);
        vecSub3(&v, &info->lVerts[face->c], a);
        vecCross(&n, &u, &v);
//        *intensity = 100.0*2.0/(2.0 + vecAbs(&n)); //optimize : n*n?
//        *intensity = 120.0*16.0/(16.0 + vecMul(&n, &n));
        *intensity = 50.0*8.0/(8.0 + vecAbs(&n)); //optimize : n*n?
//        *intensity = 50.0;
      }
    }
    face++;
    intensity++;
    tag++;
    z--;
  } while (z != 0);
  globalHeap = (void *) intensity;
}

//-- transform vertices to camera coordinate system
Vector *xformVerts(int z, Vector *vertex, Camera *camera) {
  Vector *r;
  Vector *start;
  Vector u;

  start = (Vector *) globalHeap;
  r = start;
  do {
//    vecSub3(&u, vertex, &camera->p);   //optimize
    u.x = vertex->x - camera->p.x;
    u.y = vertex->y - camera->p.y;
    u.z = vertex->z - camera->p.z;
//    r->x = vecMul(&u, &camera->lx);
    r->x = u.x*camera->lx.x + u.y*camera->lx.y + u.z*camera->lx.z;
//    r->y = vecMul(&u, &camera->ly);
    r->y = u.x*camera->ly.x + u.y*camera->ly.y + u.z*camera->ly.z;
//    r->z = vecMul(&u, &camera->lz);
    r->z = u.x*camera->lz.x + u.y*camera->lz.y + u.z*camera->lz.z;
    vertex++;
    r++;
    z--;
  } while (z != 0);
  globalHeap = (void *) r;
  return start;
}

Vector *xformTaggedVerts(int z, Vector *vertex, char *tag, Camera *camera) {
  Vector *r;
  Vector *start;
  Vector u;

  start = (Vector *) globalHeap;
  r = start;
  do {
    if (*tag) {
//      vecSub3(&u, vertex, &camera->p);   //optimize
      u.x = vertex->x - camera->p.x;
      u.y = vertex->y - camera->p.y;
      u.z = vertex->z - camera->p.z;
//      r->x = vecMul(&u, &camera->lx);
      r->x = u.x*camera->lx.x + u.y*camera->lx.y + u.z*camera->lx.z;
//      r->y = vecMul(&u, &camera->ly);
      r->y = u.x*camera->ly.x + u.y*camera->ly.y + u.z*camera->ly.z;
//      r->z = vecMul(&u, &camera->lz);
      r->z = u.x*camera->lz.x + u.y*camera->lz.y + u.z*camera->lz.z;
    }
    vertex++;
    tag++;
    r++;
    z--;
  } while (z != 0);
  globalHeap = (void *) r;
  return start;
}

Vector *xformMasses(int z, Mass *mass, Camera *camera) {
  Vector *r;
  Vector *start;
  Vector u;

  r = start = (Vector *) globalHeap;
  do {
//    vecSub3(&u, &mass->tp, &camera->p);   //optimize
    u.x = mass->tp.x - camera->p.x;
    u.y = mass->tp.y - camera->p.y;
    u.z = mass->tp.z - camera->p.z;
//    r->x = vecMul(&u, &camera->lx);
    r->x = u.x*camera->lx.x + u.y*camera->lx.y + u.z*camera->lx.z;
//    r->y = vecMul(&u, &camera->ly);
    r->y = u.x*camera->ly.x + u.y*camera->ly.y + u.z*camera->ly.z;
//    r->z = vecMul(&u, &camera->lz);
    r->z = u.x*camera->lz.x + u.y*camera->lz.y + u.z*camera->lz.z;
    mass++;
    r++;
    z--;
  } while (z != 0);
  globalHeap = (void *) r;
  return start;
}

//--- phong

void phongMapping(int z, Vector *normal, Vector *camPos, Vector *objPos,
  Vector *lightPos, PhongInfo *pinfo) {

 Vector lightDir;
  Vector lx, ly, lz;
  char *tag;
  Mapping *m;

 vecSub3(&lightDir, lightPos, objPos);
 vecNorm(&lightDir);

  vecSub3(&lz, camPos, objPos);
  vecCross(&ly, &lz, &lightDir);
  vecMulf(&ly, 127.0/vecAbs(&ly));
  vecNorm(&lz);
  vecAdd(&lz, &lightDir);
  vecNorm(&lz);
  vecCross(&lx, &lz, &ly);
/*vecPrint(&lx);
vecPrint(&ly);
vecPrint(&lz);
printf("\n");*/
  //get mem
  tag = pinfo->vTags = (char *) globalHeap;
  m = pinfo->mapping = (Mapping *) (tag + (z +3 & ~3));

  do {
//    if ((*tag = (vecMul(normal, &lz) > 0.0))) {
//      m->u = (vecMul(normal, &lx) + 128.0)*256.0; //param scale
//      m->v = (vecMul(normal, &ly) + 128.0)*256.0;
//    }
    if ((*tag = (normal->x*lz.x + normal->y*lz.y + normal->z*lz.z > 0.0))) {
      m->u = (normal->x*lx.x + normal->y*lx.y + normal->z*lx.z + 128.0)*256.0;
      m->v = (normal->x*ly.x + normal->y*ly.y + normal->z*ly.z + 128.0)*256.0;
    }
    normal++;
    tag++;
    m++;
    z--;
  } while (z != 0);
  globalHeap = (void *) m;
}


//--- clip

//Point testClip[] = {
//  {2.0, 2.0}, {100.0, 2.0}, {100.0, 100.0}, {2.0, 100.0}};
//RgbPoint testRgb[] = {
//  {22.0, 22.0, 0, 0, 0}, {80.0, 22.0, 255, 0, 0}, {80.0, 80.5, 0, 255, 0}, {22.0, 80.5, 0, 0, 255}};



int clipArbitrary(ParamPoint *cpStart, ParamPoint *cpEnd, ClipInfo *info) {
  ParamPoint *cp;
  float x, y, ux, uy;
  ParamPoint *sp, *np, *dp;
  int clip;
  int z, count;
  float r;

  cp = cpStart;
  do {
    x = cp->x;
    y = cp->y;
    cp++;
    if (cp == cpEnd) cp = cpStart;
    ux = cp->x - x;
    uy = cp->y - y;

    sp = info->start;
    clip = 0;
    count = 0;
    do {
      sp->dist = (sp->x - x)*uy - (sp->y - y)*ux;
      clip += sp->clip = (sp->dist > 0.0);
      count++;
      sp = (ParamPoint *) ((char *) sp + info->size);
    } while (sp != info->end);
    if (clip == count) return 0;
    if (clip > 0) {
      sp = info->start;
      dp = info->end;
      do {
        if (!sp->clip) {
          dp->x = sp->x;
          dp->y = sp->y;
          for (z = 0; z < info->nParams; z++) {
            dp->param[z] = sp->param[z];
          }
          dp = (ParamPoint *) ((char *) dp + info->size);
        }
        np = (ParamPoint *) ((char *) sp + info->size);
        if (np == info->end) np = info->start;
        if (sp->clip != np->clip) {
          r = sp->dist/(sp->dist - np->dist);
          dp->x = sp->x + r*(np->x - sp->x);
          dp->y = sp->y + r*(np->y - sp->y);
          for (z = 0; z < info->nParams; z++) {
            dp->param[z] = sp->param[z] + r*(np->param[z] - sp->param[z]);
          }
          dp = (ParamPoint *) ((char *) dp + info->size);
        }
        sp = np;
      } while (sp != info->start);
      info->start = info->end;
      info->end = dp;
    }
  } while (cp != cpStart);
  return 1;
}

int clipArbitrary2(ParamPoint *cpStart, ParamPoint *cpEnd, ClipInfo *info) {
  ParamPoint *cp;
  float x, y, ux, uy;
  ParamPoint *sp, *np, *dp;
  int clip;
  int z, count;
  float r;

  cp = cpStart;
  do {
    x = cp->x;
    y = cp->y;
    cp++;
    if (cp == cpEnd) cp = cpStart;
    ux = cp->x - x;
    uy = cp->y - y;

    sp = info->start;
    clip = 0;
    count = 0;
    do {
      sp->dist = (sp->x - x)*uy - (sp->y - y)*ux;
      clip += sp->clip = (sp->dist > 0.0);
      count++;
      sp = (ParamPoint *) ((char *) sp + info->size);
    } while (sp != info->end);
    if (clip == count) return 0;
    if (clip > 0) {
      sp = info->start;
      dp = info->end;
      do {
        if (!sp->clip) {
          dp->x = sp->x;
          dp->y = sp->y;
         dp->z = sp->z;
          for (z = 0; z < info->nParams; z++) {
            dp->param[z] = sp->param[z];
          }
          dp = (ParamPoint *) ((char *) dp + info->size);
        }
        np = (ParamPoint *) ((char *) sp + info->size);
        if (np == info->end) np = info->start;
        if (sp->clip != np->clip) {
          //clip on edge
          r = sp->dist/(sp->dist - np->dist);

          dp->x = sp->x + r*(np->x - sp->x);
          dp->y = sp->y + r*(np->y - sp->y);
//float f = r;
          r = sp->dist*sp->z;
          r /= r - np->dist*np->z;
//printf("sp->z %7.3f, np->z %7.3f, r1 %7.3f, r2 %7.3f\n",sp->z, np->z, f, r);
         dp->z = sp->z + r*(np->z - sp->z);
          for (z = 0; z < info->nParams; z++) {
            dp->param[z] = sp->param[z] + r*(np->param[z] - sp->param[z]);
          }
          dp = (ParamPoint *) ((char *) dp + info->size);
        }
        sp = np;
      } while (sp != info->start);
      info->start = info->end;
      info->end = dp;
    }
  } while (cp != cpStart);
  return 1;
}

//--- visualization



void inSort(List **sortList, List *element, float dist) {

  sortList += (int) floor(dist*25.6) & sortListLen -1;
  element->next = *sortList;
  *sortList = element;
}

void paintObjectFace(List *element, ParamPoint* clipStart, ParamPoint *clipEnd) {
  SortFace *sortFace;
  Face *face;
  int z, i, xor;
  int numPhong;
  PhongInfo *pinfo;
  PhongInfo *pInfos[numLights];
  unsigned int *pMaps[numLights];
  ClipInfo cinfo;
  ParamPoint *dp;
  int idx[3];
  Vector *v;
  Mapping *m;
  float x[3], y[3];

  sortFace = (SortFace *) element;
  face = sortFace->face;

  xor = (recursionLevel & 1) ? 0 : 2;

  v = sortFace->v[0 ^ xor];
  x[0] = xAdd + v->x/v->z;
  y[0] = yAdd + yMul*v->y/v->z;
  v = sortFace->v[1];
  x[1] = xAdd + v->x/v->z;
  y[1] = yAdd + yMul*v->y/v->z;
  v = sortFace->v[2 ^ xor];
  x[2] = xAdd + v->x/v->z;
  y[2] = yAdd + yMul*v->y/v->z;

  if ((x[0] - x[1])*(y[2] - y[0]) < (y[0] - y[1])*(x[2] - x[0])) return;

  idx[0 ^ xor] = face->a;
  idx[1]       = face->b;
  idx[2 ^ xor] = face->c;

  pinfo = sortFace->pinfo;
  numPhong = 0;
  for (z = 0; z < numLights; z++) {
    if (pinfo->vTags[face->a] &&  pinfo->vTags[face->b]
      && pinfo->vTags[face->c]) {

      pInfos[numPhong] = pinfo;
      pMaps[numPhong] = phongMaps[z];
      numPhong++;
    }
    pinfo++;
  }

  cinfo.nParams = 3 + numPhong*2; //u, v, alpha, numPhong*(phong u, v)
  cinfo.size = sizeof(ParamPoint) + cinfo.nParams*4;
  dp = cinfo.start = (ParamPoint *) globalHeap;
  i = 2;
  do {
    dp->x = x[i];
    dp->y = y[i];
    m = &sortFace->mapping[idx[i]];
    dp->param[0] = m->u;
    dp->param[1] = m->v;
    dp->param[2] = sortFace->alpha[idx[i]];
    for (z = 0; z < numPhong; z++) {
      m = &pInfos[z]->mapping[idx[i]];
      dp->param[3 + z*2] = m->u;
      dp->param[4 + z*2] = m->v;
    }
    dp = (ParamPoint *) ((char *) dp + cinfo.size);
    i--;
  } while (i >= 0);
  cinfo.end = dp;

  if (clipArbitrary(clipStart, clipEnd, &cinfo)) {
    txtAlphaPhongPolygon(cinfo.start, cinfo.end,
      cinfo.nParams -1, cinfo.size, cubeMap, pMaps);
  }
}

void paintClothFace(List *element, ParamPoint* clipStart, ParamPoint *clipEnd) {
  SortFace *sortFace;
  Face *face;
  int z, xor;
  int numPhong;
  PhongInfo *pinfo;
  PhongInfo *pInfos[numLights];
  unsigned int *pMaps[numLights];
  ClipInfo cinfo;
  ParamPoint *dp;
  int idx[3];
  Vector *v;
  Mapping *m;


  sortFace = (SortFace *) element;
  face = sortFace->face;

  pinfo = sortFace->pinfo;
  numPhong = 0;
  for (z = 0; z < numLights; z++) {
    if (pinfo->vTags[face->a] &&  pinfo->vTags[face->b]
      && pinfo->vTags[face->c]) {

      pInfos[numPhong] = pinfo;
      pMaps[numPhong] = phongMaps[z];
      numPhong++;
    }
    pinfo++;
  }

  cinfo.nParams = 2 + numPhong*2; //u, v, numPhong*(phong u, v)
  cinfo.size = sizeof(ParamPoint) + cinfo.nParams*4;
  dp = cinfo.start = (ParamPoint *) globalHeap;

  xor = (recursionLevel & 1) ? 2 : 0;
  idx[0 ^ xor] = face->a;
  idx[1]       = face->b;
  idx[2 ^ xor] = face->c;
  //vertex a
  v = sortFace->v[0 ^ xor];
//  dp->x = mxres*(1.0 + v->x/v->z);
//  dp->y = myres*(1.0 - v->y/v->z);
  dp->x = xAdd + v->x/v->z;
  dp->y = yAdd + yMul*v->y/v->z;
  m = &sortFace->mapping[idx[0]];
  dp->param[0] = m->u;
  dp->param[1] = m->v;
  for (z = 0; z < numPhong; z++) {
    m = &pInfos[z]->mapping[idx[0]];
    dp->param[2 + z*2] = m->u;
    dp->param[3 + z*2] = m->v;
  }
  dp = (ParamPoint *) ((char *) dp + cinfo.size);
  //vertex b
  v = sortFace->v[1];
//  dp->x = mxres*(1.0 + v->x/v->z);
//  dp->y = myres*(1.0 - v->y/v->z);
  dp->x = xAdd + v->x/v->z;
  dp->y = yAdd + yMul*v->y/v->z;
  m = &sortFace->mapping[idx[1]];
  dp->param[0] = m->u;
  dp->param[1] = m->v;
  for (z = 0; z < numPhong; z++) {
    m = &pInfos[z]->mapping[idx[1]];
    dp->param[2 + z*2] = m->u;
    dp->param[3 + z*2] = m->v;
  }
  dp = (ParamPoint *) ((char *) dp + cinfo.size);
  //vertex c
  v = sortFace->v[2 ^ xor];
//  dp->x = mxres*(1.0 + v->x/v->z);
//  dp->y = myres*(1.0 - v->y/v->z);
  dp->x = xAdd + v->x/v->z;
  dp->y = yAdd + yMul*v->y/v->z;
  m = &sortFace->mapping[idx[2]];
  dp->param[0] = m->u;
  dp->param[1] = m->v;
  for (z = 0; z < numPhong; z++) {
    m = &pInfos[z]->mapping[idx[2]];
    dp->param[2 + z*2] = m->u;
    dp->param[3 + z*2] = m->v;
  }
  cinfo.end = (ParamPoint *) ((char *) dp + cinfo.size);

  if (clipArbitrary(clipStart, clipEnd, &cinfo)) {
    txtPhongPolygon(cinfo.start, cinfo.end,
      cinfo.nParams -1, cinfo.size, particleMap, pMaps);
  }
}


void renderR(Camera *camera, float minDist,
  ParamPoint *clipStart, ParamPoint *clipEnd, Object *lockObj);

void paintMirror(List *element, ParamPoint *clipStart, ParamPoint *clipEnd) {
  void *oldHeap;
  SortMirror *sortMirror;
  ClipInfo cinfo;
  ParamPoint *dp;
  int z, xor;
  Vector *v;

//printf("paintMirror\n");
  oldHeap = globalHeap;

  sortMirror = (SortMirror *) element;

  cinfo.nParams = 0;
  cinfo.size = sizeof(ParamPoint);
  dp = cinfo.start = (ParamPoint *) globalHeap;

  z = 3;
  xor = (recursionLevel & 1) ? 3 : 0;
  do {
    v = sortMirror->v[z ^ xor];
//    dp->x = mxres*(1.0 + v->x/v->z);
//    dp->y = myres*(1.0 - v->y/v->z);
    dp->x = xAdd + v->x/v->z;
    dp->y = yAdd + yMul*v->y/v->z;
    dp++;
    z--;
  } while (z >= 0);
  cinfo.end = dp;
  if (clipArbitrary(clipStart, clipEnd, &cinfo)) {
    globalHeap = (void *) cinfo.end;
// clearPolygon(cinfo.start, cinfo.end);
    recursionLevel++;
    renderR(&sortMirror->camera, sortMirror->minDist, cinfo.start, cinfo.end,
      sortMirror->obj);
    recursionLevel--;
  }
  globalHeap = oldHeap;
}

void addMirror(List **sortList, Camera *camera, float minDist, Object *obj,
  Vector *a, Vector *b, Vector *c, Vector *d,
  Vector *ta, Vector *tb, Vector *tc, Vector *td) {

  float newMinDist, maxDist;
  SortMirror *sortMirror;
  Vector da, cb, n, u;

  newMinDist = ta->z;
  if (tb->z < newMinDist) newMinDist = tb->z;
  if (tc->z < newMinDist) newMinDist = tc->z;
  if (td->z < newMinDist) newMinDist = td->z;
  if (newMinDist < minDist) return;

  //get mem
  sortMirror = (SortMirror *) globalHeap;
  globalHeap = (void *) (sortMirror +1);

  //calc normal
  //order of verts: a b
  //                c d
  vecSub3(&da, d, a); //expected length of da and cb : 2*sqrt(2)
  vecSub3(&cb, c, b);
  vecCross(&n, &da, &cb);
  vecNorm(&n);
//  vecMul(&n, 0.125);
//  if (vecMul(&n, &camera->lz) >= 0.0) return;
  vecSub3(&u, a, &camera->p);
  if (vecMul(&n, &u) >= 0.0) return;

  //copy camera
  vecAssign(&sortMirror->camera.p, &camera->p);
  vecAssign(&sortMirror->camera.lx, &camera->lx);
  vecAssign(&sortMirror->camera.ly, &camera->ly);
  vecAssign(&sortMirror->camera.lz, &camera->lz);

  //mirror new camera
  vecSub3(&u, &camera->p, a);
  vecMulAdd(&sortMirror->camera.p, &n, -2.0*vecMul(&u, &n));
  vecMulAdd(&sortMirror->camera.lx, &n, -2.0*vecMul(&camera->lx, &n));
  vecMulAdd(&sortMirror->camera.ly, &n, -2.0*vecMul(&camera->ly, &n));
  vecMulAdd(&sortMirror->camera.lz, &n, -2.0*vecMul(&camera->lz, &n));

  sortMirror->minDist = newMinDist;

  ((List *) sortMirror)->paint = paintMirror;
  sortMirror->v[0] = ta;
  sortMirror->v[1] = tb;
  sortMirror->v[2] = td;
  sortMirror->v[3] = tc;
  sortMirror->obj = obj;
  //insert into sortlist with maximum distance
  maxDist = ta->z;
  if (tb->z > maxDist) maxDist = tb->z;
  if (tc->z > maxDist) maxDist = tc->z;
  if (td->z > maxDist) maxDist = td->z;
//printf("inSort\n");
  inSort(sortList, (List *) sortMirror, maxDist + 0.1);
}

void paintParticle(List *element, ParamPoint *clipStart, ParamPoint *clipEnd) {
  SortParticle *sortParticle;
  ClipInfo cinfo;
  ParamPoint *dp;
  float l, u, v, x, y;

//if (recursionLevel > 0) return;
  sortParticle = (SortParticle *) element;

  l = sqrt(sortParticle->u*sortParticle->u + sortParticle->v*sortParticle->v);
  if (l > 1e-10) {
    l = camMul/(l*80.0);
    u = sortParticle->u*l;
    v = sortParticle->v*l;
  } else {
    u = camMul/80.0;
    v = 0.0;
  }

  cinfo.nParams = 2;
  cinfo.size = sizeof(ParamPoint) + 2*4;
  dp = cinfo.start = (ParamPoint *) globalHeap;

  x = sortParticle->x - sortParticle->u - u;
  y = sortParticle->y - sortParticle->v - v;

  dp->x = xAdd +      (x + v);
  dp->y = yAdd + yMul*(y - u);
  dp->param[0] = 0.0;
  dp->param[1] = 0.0;
  dp = (ParamPoint *) ((char *) dp + cinfo.size);

  dp->x = xAdd +      (x - v);
  dp->y = yAdd + yMul*(y + u);
  dp->param[0] = 127.0*256.0; //param scale
  dp->param[1] = 0.0;
  dp = (ParamPoint *) ((char *) dp + cinfo.size);

  x = sortParticle->x + sortParticle->u/*1.5*/ + u;
  y = sortParticle->y + sortParticle->v/*1.5*/ + v;

  dp->x = xAdd +      (x - v);
  dp->y = yAdd + yMul*(y + u);
  dp->param[0] = 127.0*256.0; //param scale
  dp->param[1] = 127.0*256.0; //param scale
  dp = (ParamPoint *) ((char *) dp + cinfo.size);

  dp->x = xAdd +      (x + v);
  dp->y = yAdd + yMul*(y - u);
  dp->param[0] = 0.0;
  dp->param[1] = 127.0*256.0; //param scale

  cinfo.end = (ParamPoint *) ((char *) dp + cinfo.size);

  if (clipArbitrary(clipStart, clipEnd, &cinfo)) {
    txtAddPolygon(cinfo.start, cinfo.end, particleMap);
  }
}

void paintLight(List *element, ParamPoint *clipStart, ParamPoint *clipEnd) {
  SortLight *sortLight;
  ClipInfo cinfo;
  ParamPoint *dp;
  float size;

  sortLight = (SortLight *) element;

  cinfo.nParams = 2;
  cinfo.size = sizeof(ParamPoint) + 2*4;
  dp = cinfo.start = (ParamPoint *) globalHeap;

  size = camMul*0.07;
  dp->x = xAdd +      (sortLight->x - size);
  dp->y = yAdd + yMul*(sortLight->y - size);
//  dp->param[0] = sortLight->mapx;
//  dp->param[1] = sortLight->mapy;
  dp->param[0] = 68.0*256.0;
  dp->param[1] = 68.0*256.0;
  dp = (ParamPoint *) ((char *) dp + cinfo.size);

  dp->x = xAdd +      (sortLight->x - size);
  dp->y = yAdd + yMul*(sortLight->y + size);
//  dp->param[0] = sortLight->mapx + 127.0*256.0; //param scale
//  dp->param[1] = sortLight->mapy;
  dp->param[0] = 188.0*256.0;
  dp->param[1] = 68.0*256.0;
  dp = (ParamPoint *) ((char *) dp + cinfo.size);

  dp->x = xAdd +      (sortLight->x + size);
  dp->y = yAdd + yMul*(sortLight->y + size);
//  dp->param[0] = sortLight->mapx + 127.0*256.0; //param scale
//  dp->param[1] = sortLight->mapy + 127.0*256.0; //param scale
  dp->param[0] = 188.0*256.0;
  dp->param[1] = 188.0*256.0;
  dp = (ParamPoint *) ((char *) dp + cinfo.size);

  dp->x = xAdd +      (sortLight->x + size);
  dp->y = yAdd + yMul*(sortLight->y - size);
//  dp->param[0] = sortLight->mapx;
//  dp->param[1] = sortLight->mapy + 127.0*256.0; //param scale
  dp->param[0] = 68.0*256.0;
  dp->param[1] = 188.0*256.0;

  cinfo.end = (ParamPoint *) ((char *) dp + cinfo.size);

  if (clipArbitrary(clipStart, clipEnd, &cinfo)) {
    txtAddPolygon(cinfo.start, cinfo.end, sortLight->map/*particleMap*/);
  }
}

//--- render
void renderR(Camera *camera, float minDist,
  ParamPoint *clipStart, ParamPoint *clipEnd, Object *lockObj) {

  int o, z, l;
  Wall *wall;
  Object *obj;
  LightInfo *linfo;

  void *oldHeap;
  Vector *verts;
  ClipInfo cinfo;
  Face *face;
  WFace *wface;
  float *intensity;
  Vector u;
  List **sortList;
//  PhongInfo *pinfo;
  SortFace *sortFace;
  Vector *vertex;
  Particle *particle;
  SortParticle *sortParticle;
  float dist;
  SortLight *sortLight;
  List *element;


  //paint walls
  cinfo.nParams = 2;
  cinfo.size = sizeof(ParamPoint) + 2*4;
  for (wall = walls; wall < &walls[6]; wall++) {
    oldHeap = globalHeap;
    verts = xformVerts(wall->nVerts, wall->verts, camera);
    wface = wall->faces;
    for (z = 0; z < wall->nFaces; z++) {
      cinfo.start = (ParamPoint *) globalHeap;
      if (wFaceZclip(wface, verts, wall->attribs, &cinfo)) {
        if (clipArbitrary2(clipStart, clipEnd, &cinfo)) {
      /*  ParamPoint *sp = cinfo.start;
          do {
            ParamPoint *np = (ParamPoint *) ((char *) sp + cinfo.size);
            if (np == cinfo.end) np = cinfo.start;
            paint.drawLine((int) sp->x, (int) sp->y, (int) np->x, (int) np->y);
            sp = np;
          } while (sp != cinfo.start);*/
//          txtRgbPolygon(cinfo.start, cinfo.end, /*phongMaps[1]*/map1);
//printf("bumppoly!!!\n");
          bumpPolygon(cinfo.start, cinfo.end, bumpMap, wall->binfo);
        }
      }
      wface++;
    }
    globalHeap = oldHeap;
  }

//  if (recursionLevel > 1) return;

  //paint shadow
  cinfo.nParams = 0;
  cinfo.size = sizeof(ParamPoint);
if (recursionLevel == 0) {
  for (o = 0; o < nObjects; o++) {
    obj = &objects[o];
    linfo = obj->linfo;
    for (l = 0; l < numLights; l++) {
      oldHeap = globalHeap;
      verts = xformTaggedVerts(obj->nVerts, linfo->sVerts, linfo->vTags, camera);
      face = obj->faces;
      for (z = 0; z < obj->nFaces; z++) {
        if (linfo->fTags[z]) {
          cinfo.start = (ParamPoint *) globalHeap;
          if (sFaceZclip(face, verts, &cinfo)) {
            if (clipArbitrary(clipStart, clipEnd, &cinfo)) {
              shadowPolygon(cinfo.start, cinfo.end);
            }
          }
        }
        face++;
      }
      globalHeap = oldHeap;
      linfo++;
    }
  }
}
  //paint light
  //cinfo: 0 parameter
  for (o = 0; o < nObjects; o++) {
    obj = &objects[o];
    linfo = obj->linfo;
    for (l = 0; l < numLights; l++) {
      oldHeap = globalHeap;
      verts = xformTaggedVerts(obj->nVerts, linfo->lVerts, linfo->vTags, camera);
      face = obj->faces;
      intensity = linfo->lIntensities;
      for (z = 0; z < obj->nFaces; z++) {
        if (linfo->fTags[z] == 2) {
          cinfo.start = (ParamPoint *) globalHeap;
          if (sFaceZclip(face, verts, &cinfo)) {
            if (clipArbitrary(clipStart, clipEnd, &cinfo)) {
//              ParamPoint *sp = cinfo.start;
//              do {
//                ParamPoint *np = (ParamPoint *) ((char *) sp + cinfo.size);
//                if (np == cinfo.end) np = cinfo.start;
//                paint.drawLine((int) sp->x, (int) sp->y, (int) np->x, (int) np->y);
//                sp = np;
//              } while (sp != cinfo.start);
              vecMul3(&u, &lightColor[l], *intensity);
              lightPolygon(cinfo.start, cinfo.end, &u);
            }
          }
        }
        face++;
        intensity++;
      }
      globalHeap = oldHeap;
      linfo++;
    }
  }

//  if (recursionLevel > 0) return;

  //get sortlist
  sortList = (List **) globalHeap;
  globalHeap = (void *) (sortList + sortListLen);
  bzero(sortList, sortListLen*sizeof(List *));

  //phong shade object, transform object verts and insert into sorting list
  for (o = 0; o < nObjects; o++) {
    obj = &objects[o];
    if (obj == lockObj) continue;
/*
    linfo = obj->linfo;
    pinfo = (PhongInfo *) globalHeap;
    globalHeap = (void *) (pinfo + numLights);
    for (l = 0; l < numLights; l++) {
      //make tagged phong mapping
      phongMapping(obj->nVerts, obj->normals, &camera->p, &obj->center,
        &linfo->lightDir, &pinfo[l]);

      linfo++;
    }
*/
    //tramsform object verts
    verts = xformVerts(obj->nVerts, obj->verts, camera);

    //insert faces into sorting list
    face = obj->faces;
    sortFace = (SortFace *) globalHeap;
    for (z = 0; z < obj->nFaces; z++) {
      vertex = &verts[face->a];
      if (vertex->z > minDist) {
        sortFace->v[0] = vertex;
        sortFace->v[1] = &verts[face->b];
        sortFace->v[2] = &verts[face->c];
        ((List *) sortFace)->paint = paintObjectFace;
        sortFace->mapping = obj->mapping;
        sortFace->alpha = obj->alpha;
        sortFace->face = face;
        sortFace->pinfo = obj->pinfo;
        inSort(sortList, (List *) sortFace, vertex->z);
        sortFace++;
      }
      face++;
    }
    globalHeap = (void *) sortFace;
  if (recursionLevel == 0) {
    //mirrors
    verts = xformMasses(obj->nMasses, obj->masses, camera);
    addMirror(sortList, camera, minDist, obj,
      &obj->masses[0].tp,
      &obj->masses[1].tp,
      &obj->masses[2].tp,
      &obj->masses[3].tp,
      &verts[0],
      &verts[1],
      &verts[2],
      &verts[3]);
    addMirror(sortList, camera, minDist, obj,
      &obj->masses[0].tp,
      &obj->masses[2].tp,
      &obj->masses[4].tp,
      &obj->masses[6].tp,
      &verts[0],
      &verts[2],
      &verts[4],
      &verts[6]);
    addMirror(sortList, camera, minDist, obj,
      &obj->masses[0].tp,
      &obj->masses[4].tp,
      &obj->masses[1].tp,
      &obj->masses[5].tp,
      &verts[0],
      &verts[4],
      &verts[1],
      &verts[5]);
    addMirror(sortList, camera, minDist, obj,
      &obj->masses[7].tp,
      &obj->masses[3].tp,
      &obj->masses[5].tp,
      &obj->masses[1].tp,
      &verts[7],
      &verts[3],
      &verts[5],
      &verts[1]);
    addMirror(sortList, camera, minDist, obj,
      &obj->masses[7].tp,
      &obj->masses[5].tp,
      &obj->masses[6].tp,
      &obj->masses[4].tp,
      &verts[7],
      &verts[5],
      &verts[6],
      &verts[4]);
    addMirror(sortList, camera, minDist, obj,
      &obj->masses[7].tp,
      &obj->masses[6].tp,
      &obj->masses[3].tp,
      &obj->masses[2].tp,
      &verts[7],
      &verts[6],
      &verts[3],
      &verts[2]);
  }//if
  }

  //insert particles
  particle = particles;
  sortParticle = (SortParticle *) globalHeap;
  for (z = 0; z < nParticles; z++) {
    if (particle->state != 0) {
      vecSub3(&u, &particle->m.tp, &camera->p);
      dist = vecMul(&u, &camera->lz);
      if (dist > minDist) {
        ((List *) sortParticle)->paint = paintParticle;
        sortParticle->x = vecMul(&u, &camera->lx)/dist; //optimize
        sortParticle->y = vecMul(&u, &camera->ly)/dist;
        sortParticle->u = dnvecMul(&particle->m.v, &camera->lx)/dist;
        sortParticle->v = dnvecMul(&particle->m.v, &camera->ly)/dist;
        inSort(sortList, (List*) sortParticle, dist);
        sortParticle++;
      }
    }
    particle++;
  }
  globalHeap = (void *) sortParticle;

  //insert light sources
  sortLight = (SortLight *) globalHeap;
  for (l = 0; l < numLights; l++) {
    vecSub3(&u, &lightPos[l], &camera->p);
    dist = vecMul(&u, &camera->lz);
    if (dist > minDist) {
      ((List *) sortLight)->paint = paintLight;
      sortLight->x = vecMul(&u, &camera->lx)/dist;
      sortLight->y = vecMul(&u, &camera->ly)/dist;
      sortLight->map = phongMaps[l];
      sortLight->mapx = 0.0;
      sortLight->mapy = 0.0;
      inSort(sortList, (List*) sortLight, dist);
      sortLight++;
    }
  }
  globalHeap = (void *) sortLight;

  //insert cloth
  if (clothVisible) {
    //tramsform cloth verts
    verts = xformVerts(cloth.nVerts, cloth.verts, camera);
    //insert faces into sorting list
    face = cloth.faces;
    sortFace = (SortFace *) globalHeap;
    for (z = 0; z < cloth.nFaces; z++) {
      vertex = &verts[face->a];
      if (vertex->z > minDist) {
        sortFace->v[0] = vertex;
        sortFace->v[1] = &verts[face->b];
        sortFace->v[2] = &verts[face->c];
        ((List *) sortFace)->paint = paintClothFace;
        sortFace->mapping = cloth.mapping;
        sortFace->face = face;
        sortFace->pinfo = cloth.pinfo;
        inSort(sortList, (List *) sortFace, vertex->z);
        sortFace++;
      }
      face++;
    }
    globalHeap = (void *) sortFace;
  }


  //paint sorted stuff
  z = sortListLen;
  sortList += sortListLen;
  do {
    sortList--;
    element = *sortList;
    while (element) {
      element->paint(element, clipStart, clipEnd);
      element = element->next;
    }
    z--;
  } while (z != 0);

}

typedef struct {
  int x;
  int y;
  int w;
  int h;
  int ow;
  int oh;
} Rectangle;

void setRect(Rectangle *r, int x, int y, int w, int h) {
  r->ow = w;
  r->oh = h;
  r->w = (x + w)*xres/320;
  r->h = (y + h)*yres/240;
  r->x = x*xres/320;
  r->y = y*yres/240;
  r->w -= r->x;
  r->h -= r->y;
}

void paintPlain(Rectangle *r, unsigned int col) {
  unsigned int *screen;
  int h, z;

  if (r->w == 0 || r->h == 0) return;
  screen = &screenBuf[r->x + r->w + xres*r->y];
  h = r->h;
  do {
    z = -r->w;
    do {
      screen[z] = col;
      z++;
    } while (z != 0);
    screen += xres;
    h--;
  } while (h != 0);
}

void paintPic(Rectangle *r, unsigned char *pic, unsigned int *tab) {
  unsigned int *screen;
  unsigned char *picl;
  int h, z, uAdd, vAdd, u, v, p;

  if (r->w == 0 || r->h == 0) return;
  screen = &screenBuf[r->x + r->w + xres*r->y];
  uAdd = (r->ow << 8)/r->w;
  vAdd = (r->oh << 8)/r->h;
  v = 0;
  h = r->h;
  do {
    picl = &pic[(v >> 8)*r->ow];
    z = -r->w;
    u = 0;
    do {
      p = picl[u >> 8];
      if (p) screen[z] = tab[p];
      u += uAdd;
      z++;
    } while (z != 0);
    screen += xres;
    v += vAdd;
    h--;
  } while (h != 0);
}

void paintText(int x, int y, unsigned short *text, unsigned int *tab) {
  int offset, width;
  Rectangle r;

  while (*text != 65535) {
    offset = *text;
    text++;
    width = *text;
    text++;
    setRect(&r, x, y, width, 14);
    paintPic(&r, &font[offset*14], tab);
    x += width +2;
  }
}

void pictures() {
  int x, y, z;
  Rectangle r;
  unsigned int *picTab;

  x = 0;
  y = 70;
  for (z = 0; z < 5; z++) {
    setRect(&r, x, y, 64, 64);
    picTab = picTabs[z % 3];
    switch (picControl[z]) {
      case 1:
      case 2:
      case 3:
      case 4:
        paintPlain(&r, picTab[100]);
        break;
      case 5:
        paintPic(&r, p1, picTab);
        break;
      case 6:
        paintPic(&r, p2, picTab);
        break;
      case 7:
        paintPic(&r, p3, picTab);
        break;
    }
//    picTab = picTabs[(z + (picControl[z] ? 1 : 0)) % 3];
    picTab = picTabs[picControl[z] ? ((z & 4) >> 2) +1 : z % 3];
    switch (textControl[z]) {
      case 1:
        paintText(x +2, y + 64 - 14, text0, picTab);
        break;
      case 2:
        paintText(x +2, y + 64 - 14, text1, picTab);
        break;
//      case 3:
//        paintText(x +2, y + 64 - 42, text2, picTab);
//        paintText(x +2, y + 64 - 28, text3, picTab);
//        paintText(x +2, y + 64 - 14, text4, picTab);
//        break;
    }
    x += 64;
  }
}

void pictures2() {
  Rectangle r;

  paintText(100, 70     , text2, picTabs[0]);
  paintText( 90, 70 + 14, text3, picTabs[2]);
  paintText(105, 70 + 28, text4, picTabs[1]);
  if (renderFlag > 2) {
    setRect(&r, 150, 75, 64, 83);
    paintPic(&r, laby, picTabs[1]);
  }
}

void render2(Camera *camera, ParamPoint *clipStart, ParamPoint *clipEnd) {
  Particle *particle;
  SortParticle sortParticle;
  int z;
  Vector u;
  float dist;

  //insert particles
  particle = particles;
  for (z = 0; z < nParticles; z++) {
    if (particle->state != 0) {
      vecSub3(&u, &particle->m.tp, &camera->p);
      dist = vecMul(&u, &camera->lz);
      if (dist > 1.0) {
        sortParticle.x = vecMul(&u, &camera->lx)/dist; //optimize
        sortParticle.y = vecMul(&u, &camera->ly)/dist;
        sortParticle.u = dnvecMul(&particle->m.v, &camera->lx)/dist;
        sortParticle.v = dnvecMul(&particle->m.v, &camera->ly)/dist;
        paintParticle((List *) &sortParticle, clipStart, clipEnd);
      }
    }
    particle++;
  }
}

void render() {
  void *oldHeap;
  Vector camPos;
  Vector camTarget;
//  Vector lightPos[numLights]; -> global
//  float camRoll = 0.0;
//  float camFOV = 1.0;
  Camera camera;
  int o, l;
  Wall *wall;
  Object *obj;
  LightInfo *linfo;
  PhongInfo *pinfo;

  oldHeap = globalHeap;

  //tracks
  doTrack3(&camPosTrack, timecount, &camPos);
  doTrack3(&camTargetTrack, timecount, &camTarget);
  for (l = 0; l < numLights; l++) {
    doTrack3(&lightTrack[l], timecount, &lightPos[l]);
  }

  //camera
//  makeCamera(&camera, &pCamLight[0], &pCamLight[1], camRoll, camFOV);
  makeCamera(&camera, &camPos, &camTarget, camRoll, camFOV);

  if (renderFlag == 0) {
    //render-init walls
  //  makeWalls();
  //  vecAssign(&cam, &camera.p);
  //  vecMulAdd(&cam, &camera.lz, 2.0);
    for (wall = walls; wall < &walls[6]; wall++) {
  //    makeWall(wall, &cam);
  //    clearWall(wall);
      for (l = 0; l < numLights; l++) {
        lightWall(wall, &lightPos[l], miniPhongMaps[l], &wall->binfo[l]);
      }
    }

    //render-init objects
    for (o = 0; o < nObjects; o++) {
      obj = &objects[o];
      linfo = obj->linfo;
      pinfo = obj->pinfo;
      for (l = 0; l < numLights; l++) {
        //make tagged shadow and light verts
        lightVerts(obj->nVerts, obj->verts, obj->normals, &lightPos[l]/*pCamLight[2+l]*/, walls, linfo);
        //tag faces and calc light polygon areas
        lightIntensities(obj->nFaces, obj->faces, linfo);

        //direction vector to light
  //      vecSub3(&linfo->lightDir, &lightPos[l], &obj->center);
  //      vecNorm(&linfo->lightDir);

        phongMapping(obj->nVerts, obj->normals, &camera.p, &obj->center,
          &lightPos[l], pinfo);


        linfo++;
        pinfo++;
      }
    }

    //render-init cloth
    if (clothVisible) {
      pinfo = cloth.pinfo;
      for (l = 0; l < numLights; l++) {
        //make phong mapping
        phongMapping(cloth.nVerts, cloth.normals, &camera.p, &cloth.center,
          &lightPos[l], pinfo);
        pinfo++;
      }
    }

    //enter render recursion
    recursionLevel = 0;
    renderR(&camera, 1.0, clipArea, &clipArea[4], startLockObj);
  } else {
    bzero(screenBuf,xres*yres*4);
    if (renderFlag == 1) pictures(); else pictures2();
    render2(&camera, clipArea, &clipArea[4]);
  }
  globalHeap = oldHeap;
}


//--- init
void initScreen(int x, int y) {
  int z;
  float mxres, myres;
//  float *divTab;

  clipArea[0].x = +0.5 +1.0;
  clipArea[0].y = +0.5 +1.0;
  clipArea[1].x = x +0.5 -1.0;
  clipArea[1].y = +0.5 +1.0;
  clipArea[2].x = x +0.5 -1.0;
  clipArea[2].y = y +0.5 -1.0;
  clipArea[3].x = +0.5 +1.0;
  clipArea[3].y = y +0.5 -1.0;
  xres = x;
  yres = y;

  //x' = xres/2.0*(1.0 + x/z);
  //y' = yres/2.0*(1.0 - y/z);
  mxres = (float) x/2.0;
  myres = (float) y/2.0;
  xAdd = camMul = mxres;
  yMul = -xyratio*myres/mxres;
  yAdd = myres;

  divTab = (float *) getmem((x +1)*sizeof(float));
  for (z = 2; z <= x; z++) divTab[z] = 1.0/(float) z;
  screenBuf = (unsigned int *) getmem(x*y*sizeof(int));
//  polySetGlobal(x, screenBuf, divTab);
}


void initWalls() {
  Vector p, u1, u2, u3;

  vecAssign4(&p, -10.0, -10.0, 0.0);
  vecAssign4(&u1, 20.0, 0.0, 0.0);
  vecAssign4(&u2, 0.0, 20.0, 0.0);
  vecAssign4(&u3, 0.0, 0.0, 30.0);
  initWall(&walls[0], &p, &u1, &u2, 8);
// vecPrint(&walls[0].n);
  initWall(&walls[1], &p, &u2, &u3, 4);
// vecPrint(&walls[1].n);
  initWall(&walls[2], &p, &u3, &u1, 4);
// vecPrint(&walls[2].n);
  vecAssign4(&p, 10.0, 10.0, 30.0);
  vecNeg(&u1);
  vecNeg(&u2);
  vecNeg(&u3);
  initWall(&walls[3], &p, &u2, &u1, 3);
// vecPrint(&walls[3].n);
  initWall(&walls[4], &p, &u3, &u2, 4);
// vecPrint(&walls[4].n);
  initWall(&walls[5], &p, &u1, &u3, 4);
// vecPrint(&walls[5].n);
}

void initObjects() {
//  int z;


  nObjects = 2;
  initCube(&objects[0]);
  initCube(&objects[1]);

  nParticles = 60;
  particles = (Particle *) getmem(nParticles*sizeof(Particle));
//  particles->p.x = 4.0;
//  particles->p.y = 1.0;
//  particles->p.z = 30.0;
/*  for (z = 0; z < nParticles; z++) {
    particles[z].p.x = -4.0 + random()/16.0;
    particles[z].p.y = -8.0 + random()/16.0;
    particles[z].p.z = 20.0 + random()/16.0;
  }*/

  initCloth(&cloth);
}

void setLocations() {
  countpos = 0;

  setCubeLocation(&objects[0], 4.0, 1.0, 19.0);
  setCubeLocation(&objects[1], 5.0, 0.0, 24.0);
//  initCube(&objects[0], 4.0, 1.0, 15.0, 0.1);
//  initCube(&objects[1], 5.0, 0.0, 20.0, 0.1);
  cubeGravity = 1;
  startLockObj = 0L;

  setClothLocation(&cloth, 4.7, -0.3, 12.0); //4.6, 0.0
  clothVisible = 0;
  renderFlag = 0;
}

void initPicTab(unsigned int *tab, int rmax, int gmax, int bmax) {
  int z, r, g, b;

  for (z = 0; z < 256; z++) {
    r = rmax*z >> 8;
    g = gmax*z >> 8;
    b = bmax*z >> 8;
    tab[z] = r | g << 11 | b << 22;
  }
}

void initIntro(int xres, int yres) {
  int z;

  initmem(4000000 + xres*(yres +1)*4);
  initScreen(xres, yres);

  initPicTab(picTabs[0],  36,  41, 251);
  initPicTab(picTabs[1], 256, 163,   0);
  initPicTab(picTabs[2],  28, 208,  28);

  genMaps();
  for (z = 0; z < numLights; z++) {
    phongMaps[z] = genPhongMap(&lightColor[z]);
    miniPhongMaps[z] = scaleMap(phongMaps[z], 128);
  }
  particleMap = genParticleMap();

  initTracks();
  initWalls();
  initObjects();
  setLocations();
}
