/* title.c - 
   Copyright (C) 1999 Tijs van Bakel.
   Tijs van Bakel <smoke@casema.net>, 
 
 This file is part of the bizarre99 linux invitation intro.

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <math.h>
#include <stdlib.h>

#include "types.h"
#include "image.h"
#include "modplay.h"
#include "graphics.h"
#include "title.h"
#include "screen.h"
#include "framecount.h"

#define TITLE_FADEIN 0
#define TITLE_FADEOUT 1
#define TITLE_IDLE 2
#define TITLE_QUIT 3

#define FIREFLY_AMOUNT 9
#define FIREFLY_MAXSIZE 40

/* note that the palette is organized in a way to
   help the flare effect. */

struct flare {
    int width, height;
    int* addTable;
};

/* this renders a ball of light, to use as a map for
   the fireflies */

void initFlare(struct flare* f, uint32 diameter)
{
    int x,y;
    int addValue;
    float x0,y0;
    
    f->width = diameter;
    f->height = diameter;
    
    f->addTable = malloc( f->width * f->height * sizeof(int) );

    for (y = 0; y < f->height; y++)
    {
        y0 = (y - (f->height/2))+0.5;
        for (x = 0; x < f->width; x++)
        {
            x0 = (x - (f->width/2))+0.5;
            addValue = (diameter/2) - sqrt(x0*x0+y0*y0);
            if (addValue < 0)
                addValue = 0;
            f->addTable[ y * f->width + x ] = addValue;
        }
    }
}

void drawFlare(struct image* destImage, uint32 destx, uint32 desty, float intensity, struct flare* f)
{
    int x,y;
    int destOfs, flareOfs;
    uint8 p;

    destx -= f->width/2;
    desty -= f->height/2;
    
    destOfs = destx + destImage->width * desty;
    flareOfs = 0;
    
    for (y = 0; y < f->height; y++)
    {
        for (x = 0; x < f->width; x++)
        {
            p = destImage->buffer[destOfs+x];
            if (p != 0)
                p += f->addTable[flareOfs];
            destImage->buffer[destOfs+x] = p;
            flareOfs++;
        }
        destOfs += destImage->width;
    }
}

void doTitleScreen( struct screen* s, struct graphics* g, struct music* m )
{
    int i;
    int color;
    int idleTime;
    int state;
    struct palette_color destPal[256];
    float fadePerunage = 0.0;
    int flyPos[FIREFLY_AMOUNT];
    struct flare f[FIREFLY_MAXSIZE];
    struct frame_count fc_global, fc_firefly;
    double firefly_frame = 0.0;
    double frame;
    int flies;
    
            /* precalc the fireflie */
    for (i = 0; i < FIREFLY_MAXSIZE; i++)
        initFlare(&(f[i]), i+5);
    
    for (i = 0; i < FIREFLY_AMOUNT; i++)
        flyPos[i] = (FIREFLY_AMOUNT-1) * 20.0;

            /* set up palette for the titlescreen */
    for ( color = 0; color < 256; color++ ) 
    {
        destPal[color].r = g->titleImage.palette[color].r;
        destPal[color].g = g->titleImage.palette[color].g;
        destPal[color].b = g->titleImage.palette[color].b;

        screen_set_color (s, color, 0,0,0);
    }

    video_update_palette ( s, 0, 255 );
    video_update_buffer ( s );

    drawImage( &g->titleImage, 0, 0, 320, 200, 0, 0 );

    idleTime = 0;

    init_frame_counter ( &fc_global, SCR_REFRESH_RATE );
    init_frame_counter ( &fc_firefly, SCR_REFRESH_RATE );
    
    state = TITLE_FADEIN;
    while (state != TITLE_QUIT) {

        wait_next_frame ( &fc_global );
        drawImage( &g->titleImage, 0, 0, 320, 200, 0, 0 );

                /* if i can find the time, the flies will have to go
                   to a point outside the logo when fading out */
        for( i = 0; i < FIREFLY_AMOUNT; i++)
        {
            drawFlare ( s->image,
                        160 + sin ( ( flyPos[ i ] + 600.0 ) / 50.0 ) * 120.0, /* x */
                        100 + cos ( ( flyPos[ i ] + 600.0 ) / 30.0 ) * 60.0, /* y */
                        1.0,
                        &f[ 30 - i * 3 ] /* size */ );
        }

        if ( ( state == TITLE_IDLE ) || ( state == TITLE_FADEOUT ) )
        {
            update_frame_counter ( &fc_firefly );
            firefly_frame = get_frame_counter ( &fc_firefly );

            flies = ( firefly_frame / 20.0 );
            flies = ( flies > FIREFLY_AMOUNT ) ? FIREFLY_AMOUNT : flies;
            for (i = FIREFLY_AMOUNT-flies; i < FIREFLY_AMOUNT; i++)
            {
                flyPos[i] = firefly_frame + 20.0 * i;
            }
        }

        switch ( state ) {
                case TITLE_FADEIN:

                    frame = get_frame_counter ( &fc_global );
                    fadePerunage = frame * 0.02;

                    if ( frame > 900.0 )
                    {
                        state = TITLE_IDLE;
                        reset_frame_counter ( &fc_firefly );
                    }
                    
                    if (fadePerunage > 1.0)
                    {
                        fadePerunage = 1.0;
                    }
                    screen_fade( s, destPal, fadePerunage );

                    break;

                case TITLE_IDLE:

                    if (firefly_frame > 600.0) /* i've spent more than this on creating it */
                    {
                        reset_frame_counter ( &fc_global );
                        state = TITLE_FADEOUT;
                    }
                    break;

                case TITLE_FADEOUT:

                    fadePerunage = 1.0 - get_frame_counter ( &fc_global ) * 0.02;
                    if (fadePerunage < 0.0)
                    {
                        state = TITLE_QUIT;
                        fadePerunage = 0.0;
                    }
                    screen_fade ( s, destPal, fadePerunage );
                    
                    break;
        }                        

        switch ( video_poll_input(s) ) {
                case INPUT_CONTINUE:
                case INPUT_QUIT:
                    state = TITLE_FADEOUT;
                    break;
        }

        modplay_update( m );

        video_update_palette ( s, 0, 255 );
        video_update_buffer ( s );
    }
}
