/* interface.c - main reader interface
   Copyright (C) 1999 Tijs van Bakel.
   Tijs van Bakel <smoke@casema.net>, 
 
 This file is part of the bizarre99 linux invitation intro.

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <math.h>

#include "interface.h"
#include "image.h"
#include "font.h"
#include "graphics.h"
#include "framecount.h"
#include "modplay.h"

/* global state (InterFace State Global) */
#define GLOBALSTATE_FADEIN   0
#define GLOBALSTATE_FADEOUT  1
#define GLOBALSTATE_LOGO_IN  2
#define GLOBALSTATE_LOGO_OUT 3
#define GLOBALSTATE_IDLE     4
#define GLOBALSTATE_QUIT     5

/* page state (InterFace State Page) */
#define PAGESTATE_STARTUP    0
#define PAGESTATE_MOVE_TO    1
#define PAGESTATE_MOVE_FROM  2
#define PAGESTATE_ERASE      3
#define PAGESTATE_WRITE      4
#define PAGESTATE_IDLE       5

#define GLOBAL_FADESTEP  0.02
#define GLOBAL_LOGO_STEP 0.03
#define GLOBAL_COVERSTEP 0.01

#define PAGE_FADESTEP_WRITE 0.005
#define PAGE_FADESTEP_ERASE 0.01
#define PAGE_FADESTEP_WRITE_FAST 0.1
#define PAGE_FADESTEP_ERASE_FAST 0.1
#define PAGE_FADESTEP_MOVE_TO 0.01
#define PAGE_FADESTEP_MOVE_FROM 0.01

struct state {

    int globalState; /* fadein,out, logoin,out, idle, quit */
    
    float globalFadePerunage;
    float logoFadePerunage;
    float coverPerunage;

    
    int page; /* current page that's being read or faded to */
    int nextpage;
    int prevpage;
    
    int pageState; /* fade in or out ? */

    float pageWriteAmount; /* -1.0 -> 0.0 from empty to full
                              0.0 -> 1.0 from full to empty, when it
                              reaches 1.0 then it is reset to 0.0 */
    float pageMovePerunage; /* 0.0 = old offset, 1.0 = new page reached,
                               if the next page is reached number is reset to 0.0 */ 
    
    struct frame_count fc_writer;
    
};

int getPageXOfs(int page)
{
    if ((page % 2) == 0)
        return 0;
    else
        return 319;
}

int getPageYOfs(int page)
{
    if (((page/2) % 2) == 0)
        return 0;
    else
        return 199;
}

void blendBackground( struct image* i, uint32 srcx, uint32 srcy, float brightness )
{
    uint32 x,y;
    uint32 destwidth,destheight;
    uint32 row;
    
    uint8 *from, *to;

    x = WINDOW_X0+2;
    y = WINDOW_Y0+2;

    srcx += WINDOW_X0+2;
    srcy += WINDOW_Y0+2;

    destwidth = WINDOW_X1 - WINDOW_X0 - 4;
    destheight = WINDOW_Y1 - WINDOW_Y0 - 4;
    
    to = i->dest->buffer + x + y*i->dest->width;
    from = i->buffer + srcx + srcy * i->width;

    for ( row = 0; row < destheight; row++ )
    {
        memcpy ( to, from, destwidth );
        to += i->dest->width; 
        from += i->width;
    }
}


/* amount is in [-1.0,..,1.0] where
   -1.0 means no text yet
   0.0 means all text
   1.0 means all text gone */
/* draw a string, clipped to the window borders */

void writeString( struct image* image, struct font* f, uint32 destx, uint32 desty, char* text, float fromPerunage, float toPerunage)
{
    int i, length;
    int x,y;
    int visibleStart, visibleEnd;

    length = strlen(text);

    y = desty + WINDOW_Y0;
    x = destx + WINDOW_X0;

    visibleStart = length * fromPerunage;
    visibleEnd = length * toPerunage;

    for (i = 0; i < length; i++)
    {
        if ((i >= visibleStart) && (i < visibleEnd))
            drawTransparentChar( image, f, x, y, text[i]);
        x += f->width;
    }
}

void writePage( struct image* image, struct font* f, int page, struct text* t, float amount )
{
    int i;
    int lines;
    int x,y;
    int startline,endline;
    float lineamount;
    
    lines = t->writerText[page].lines;
    y = (WINDOW_Y1 - WINDOW_Y0)/2 - ((lines+1) * f->height)/2 - 2;

    if (amount <= 0.0)
    {
        startline = 0;
        endline = ((float) lines) * (1.0+amount);
        for (i = startline; i < endline+1; i++)
        {
            if (i == endline)
            {
                lineamount = (1.0+amount - (float) endline/(float) lines)*lines;
            }
            else
                lineamount = 1.0;

            x = (WINDOW_X1-WINDOW_X0)/2 - (t->writerText[page].cols[i] * f->width)/2;
            writeString(image, f, x, y + 9*i, t->writerText[page].buffer[i], 0.0, lineamount);
        }
    }
    else
    {
        startline = (lines) * amount;
        endline = lines;
        for (i = startline; i < endline; i++)
        {
            if ( i == startline )
                lineamount =  (amount - (float) startline/(float) lines)*(float) lines;
            else
                lineamount = 0.0;
            
            x = (WINDOW_X1-WINDOW_X0)/2 - (t->writerText[page].cols[i] * f->width)/2;
            writeString(image, f, x, y + 9*i, t->writerText[page].buffer[i], lineamount, 1.0);
        }
    }
}

void updateBackground( struct image* i, struct state* s, uint32 xofs, uint32 yofs )
{
    int sw,sh; /* screen width and height */

    sw = i->dest->width;
    sh = i->dest->height;

    drawImage( i,
               LOGO_WIDTH, SCROLL_HEIGHT,
               WINDOW_X0-LOGO_WIDTH, sh-SCROLL_HEIGHT,
               xofs+LOGO_WIDTH, yofs+SCROLL_HEIGHT );
    drawImage( i,
               WINDOW_X0, SCROLL_HEIGHT,
               WINDOW_X1-WINDOW_X0, WINDOW_Y0-SCROLL_HEIGHT,
               xofs+WINDOW_X0, yofs+SCROLL_HEIGHT );
    drawImage( i,
               WINDOW_X1, SCROLL_HEIGHT,
               sw - WINDOW_X1, sh-SCROLL_HEIGHT,
               xofs+WINDOW_X1, yofs+SCROLL_HEIGHT );
    drawImage( i,
               WINDOW_X0, WINDOW_Y1,
               (WINDOW_X1 - WINDOW_X0), sh - WINDOW_Y1,
               xofs+WINDOW_X0, yofs+WINDOW_Y1 );
}

        /* the part of the logo that's over the scroller */
void updateLogoHack( struct image* i, float perunage )
{
    drawImage( i,
               0, 0,
               LOGO_WIDTH * perunage, SCROLL_HEIGHT,
               LOGO_WIDTH * (1-perunage), 0 );
}

void updateLogo( struct image* i, float perunage )
{
    updateLogoHack(i,perunage); 
    drawImage( i,
               0, SCROLL_HEIGHT,
               LOGO_WIDTH * perunage, LOGO_HEIGHT-SCROLL_HEIGHT,
               LOGO_WIDTH * (1-perunage), SCROLL_HEIGHT );
    image_clear_part( i, LOGO_WIDTH*perunage, 0, (1-perunage)*LOGO_WIDTH, LOGO_HEIGHT);
}

void updatePage( struct state* s, struct graphics* g, struct screen* scrn, struct text* t )
{
    uint32 xofs, yofs;
    float movePerunage;
    float fadePerunage;

    movePerunage = 0.5*( sin( ( s->pageMovePerunage - 0.5)*M_PI ) + 1.0 );
    
    xofs = (1 - movePerunage) * getPageXOfs( s->page ) + movePerunage * getPageXOfs( s->nextpage );
    yofs = (1 - movePerunage) * getPageYOfs( s->page ) + movePerunage * getPageYOfs( s->nextpage );

    update_frame_counter ( &s->fc_writer );
    
    switch ( s->pageState ) {
            case PAGESTATE_ERASE:
                s->pageWriteAmount = 0.0 + get_frame_counter ( &s->fc_writer ) * PAGE_FADESTEP_ERASE;
                if (s->pageWriteAmount >= 1.0) {
                    s->pageWriteAmount = 1.0;
                    s->pageMovePerunage = 0.0;
                    s->pageState = PAGESTATE_MOVE_FROM;
                    reset_frame_counter ( &s->fc_writer );
                }
                break;
            case PAGESTATE_MOVE_FROM:
                        /* fade out the text */
                s->pageMovePerunage = 0.0 + get_frame_counter ( &s->fc_writer ) * PAGE_FADESTEP_MOVE_FROM;
                if (s->pageMovePerunage >= 0.5) {
                    s->pageMovePerunage = 0.5;
                    s->pageState = PAGESTATE_MOVE_TO;
                    reset_frame_counter ( &s->fc_writer );
                }
                break;
            case PAGESTATE_MOVE_TO:
                        /* fade in the text */
                s->pageMovePerunage = 0.5 + get_frame_counter ( &s->fc_writer ) * PAGE_FADESTEP_MOVE_TO;
                if (s->pageMovePerunage >= 1.0) {
                    s->page = s->nextpage;
                    s->pageState = PAGESTATE_WRITE;
                    s->pageMovePerunage = 0.0;
                    s->pageWriteAmount = -1.0;
                    reset_frame_counter ( &s->fc_writer );
                }
                break;
            case PAGESTATE_WRITE:
                s->pageWriteAmount = -1.0 + get_frame_counter ( &s->fc_writer ) * PAGE_FADESTEP_WRITE;
                if (s->pageWriteAmount >= 0.0) {
                    s->pageWriteAmount = 0.0;
                    s->pageMovePerunage = 0.0;

                    s->pageState = PAGESTATE_IDLE;
                    reset_frame_counter ( &s->fc_writer );
                }
                break;
            case PAGESTATE_IDLE:
                fadePerunage = 0.5;
    }                

            /* the blended part of the background image
               (center, beneath the text) */
    blendBackground( &g->blended_background_image, xofs, yofs, fadePerunage );

            /* borders of the background image */
    updateBackground( &g->backgroundImage, s, xofs, yofs );
    
            /* write the text */
    writePage( scrn->image, &g->writerFont, s->page, t, s->pageWriteAmount );
}

void speedupWriting(struct state* s)
{
    switch (s->pageState) {
            case PAGESTATE_WRITE:
                inc_frame_counter ( &s->fc_writer );
                break;
            case PAGESTATE_ERASE:
                inc_frame_counter ( &s->fc_writer );
                break;
    }
}

void incPage(struct state* s, struct text* t)
{
    if (s->pageState != PAGESTATE_IDLE)
    {
        speedupWriting(s);
        return;
    }
    if (s->page < (t->pages-1)) {
        s->nextpage = s->page + 1;
        s->prevpage = s->page;
        
        s->pageState = PAGESTATE_ERASE;
        reset_frame_counter ( &s->fc_writer );
        s->pageWriteAmount = 0.0;
    }
}

void decPage(struct state* s, struct text* t)
{
    if (s->pageState != PAGESTATE_IDLE)
    {
        speedupWriting(s);
        return;
    }
    if (s->page > 0) {
        s->nextpage = s->page - 1;
        s->prevpage = s->page;
        
        s->pageState = PAGESTATE_ERASE;
        reset_frame_counter ( &s->fc_writer );
        s->pageWriteAmount = 0.0;
    }
}

void plotCover(struct image* i, float perunage)
{
    int x0,x1,y0,y1;
    int xa,xb,ya,yb;

    x0 = LOGO_WIDTH;
    x1 = SCR_WIDTH;
    y0 = SCROLL_HEIGHT;
    y1 = SCR_HEIGHT;

    xa = x0 + (perunage * (x1-x0)/2.0);
    xb = x1 - (perunage * (x1-x0)/2.0);
    ya = y0 + (perunage * (y1-y0)/2.0);
    yb = y1 - (perunage * (y1-y0)/2.0);
    
    image_clear_part (i, x0,y0, xa-x0, y1-y0);
    image_clear_part (i, xa,y0, xb-xa, ya-y0);
    image_clear_part (i, xb,y0, x1-xb, y1-y0);
    image_clear_part (i, xa,yb, xb-xa, y1-yb);
}

void uncover(struct state* s, struct graphics* g, struct screen* scrn, struct text* t)
{
    if (s->coverPerunage < 0.0)
        s->coverPerunage = 0.0;
    if (s->coverPerunage > 1.0)
        s->coverPerunage = 1.0;

    drawImage( &g->textframeImage,
               WINDOW_X0, WINDOW_Y0,
               (WINDOW_X1-WINDOW_X0), (WINDOW_Y1-WINDOW_Y0), 0, 0 );

    updatePage( s, g, scrn, t ); 

    plotCover(&g->backgroundImage,s->coverPerunage);
}

void doInterface( struct screen* scrn, struct graphics* g, struct music* m, struct text* t )
{
    struct state s;
    int color;
    struct palette_color destPal[256];
    struct frame_count fc, fc_cover, fc_scroller;
    double scroller_frame = 0.0;

    screen_clear (scrn);
    
            /* set up palette */
    for ( color = 0; color < 256; color++ ) 
    {
        destPal[color].r = g->backgroundImage.palette[color].r;
        destPal[color].g = g->backgroundImage.palette[color].g;
        destPal[color].b = g->backgroundImage.palette[color].b;

        screen_set_color (scrn, color, 0,0,0);
    }
    
    s.pageState = PAGESTATE_STARTUP;
    s.globalFadePerunage = 0.0;
    s.logoFadePerunage = 0.0;
    s.coverPerunage = 1.0;
    s.pageWriteAmount = -1.0; /* no text visible yet */
    s.globalState = GLOBALSTATE_FADEIN;
    
    s.page = 0;
    s.nextpage = 0;
    s.prevpage = 0;

            /* enter the biggy loop */
    init_frame_counter ( &fc, SCR_REFRESH_RATE );
    init_frame_counter ( &fc_cover, SCR_REFRESH_RATE );
    init_frame_counter ( &s.fc_writer, SCR_REFRESH_RATE );
    init_frame_counter ( &fc_scroller, SCR_REFRESH_RATE );

    while (s.globalState != GLOBALSTATE_QUIT ) {

        wait_next_frame ( &fc );

	if ( scrn->wait_retrace )
	  scroller_frame += 1.0;
	else
	  scroller_frame = update_frame_counter ( &fc_scroller );
        
        switch ( s.globalState )
        {
                case GLOBALSTATE_IDLE:
                            /* this is the default behaviour,
                               when no global fades occur */
                    updatePage( &s, g, scrn, t );
                    break;
                case GLOBALSTATE_FADEIN:
                    
                    s.globalFadePerunage = update_frame_counter ( &fc ) * GLOBAL_FADESTEP;
                    if (s.globalFadePerunage >= 1.0)
                    {
                        s.globalFadePerunage = 1.0;
                        if ( get_frame_counter ( &fc ) * GLOBAL_FADESTEP >= 2.0 )
                        {
                            s.globalState = GLOBALSTATE_LOGO_IN;
                            reset_frame_counter ( &fc );
                        }
                    }
                    
                    s.coverPerunage = 1.0 - update_frame_counter ( &fc_cover ) * GLOBAL_COVERSTEP;
                    uncover(&s,g,scrn,t);
                    
                    screen_fade( scrn, destPal, s.globalFadePerunage);

                    break;
                case GLOBALSTATE_LOGO_IN:
                    
                    s.logoFadePerunage = update_frame_counter ( &fc ) * GLOBAL_LOGO_STEP;
                    
                    if (s.logoFadePerunage >= 1.0)
                    {
                        s.logoFadePerunage = 1.0;
                        if ( get_frame_counter ( &fc ) * GLOBAL_LOGO_STEP >= 2.0 )
                        {
                            reset_frame_counter ( &s.fc_writer );
                            s.globalState = GLOBALSTATE_IDLE;
                            s.pageState = PAGESTATE_WRITE;
                            reset_frame_counter ( &fc );
                        }
                    }

                    s.coverPerunage = 1.0 - update_frame_counter ( &fc_cover ) * GLOBAL_COVERSTEP;
                    uncover(&s,g,scrn,t);
                    updateLogo( &g->logoImage, s.logoFadePerunage );
                    
                    break;
                case GLOBALSTATE_LOGO_OUT:

                    s.logoFadePerunage = 1.0 - update_frame_counter ( &fc ) * GLOBAL_LOGO_STEP;
                    if (s.logoFadePerunage <= 0.0)
                    {
                        s.logoFadePerunage = 0.0;
                        reset_frame_counter ( &fc );
                        s.globalState = GLOBALSTATE_FADEOUT;
                    }

                    updateLogo( &g->logoImage, s.logoFadePerunage );

                    break;
                case GLOBALSTATE_FADEOUT:

                    s.globalFadePerunage = 1.0 - update_frame_counter ( &fc ) * GLOBAL_FADESTEP;
                    if (s.globalFadePerunage <= 0.0)
                    {
                        s.globalFadePerunage = 0.0;
                        s.globalState = GLOBALSTATE_QUIT;
                    }

                    modplay_set_volume ( m, s.globalFadePerunage * 128.0 ); 
                    screen_fade ( scrn, destPal, s.globalFadePerunage );
                    
                    break;
        }

        switch ( video_poll_input(scrn) ) {
                case INPUT_UP:
                case INPUT_LEFT:
                case INPUT_BACK:
                    decPage( &s, t );
                    break;
                case INPUT_CONTINUE:
                case INPUT_DOWN:
                case INPUT_RIGHT:
                    incPage( &s, t );
                    break;
                case INPUT_QUIT:
                    if ( s.globalState == GLOBALSTATE_IDLE )
                    {
                        s.pageState = PAGESTATE_ERASE;
                        s.globalState = GLOBALSTATE_LOGO_OUT;
                        reset_frame_counter ( &fc );
                    }
                    break;
        }

        updateScroller( &g->scroll, scroller_frame );
        updateLogoHack( &g->logoImage, s.logoFadePerunage );

        video_update_palette( scrn, 0, 255 );
        video_update_buffer( scrn );

        modplay_update ( m );
    }
}

