/*
 * Copyright (c) 2011, Harley Laue
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *    * Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    * Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    * Neither the name of the <organization> nor the
 *      names of its contributors may be used to endorse or promote products
 *      derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/**
 * This is the low level PSG functions for interfacing with the SN79489/SN76496.
 */

#ifndef __SN79489_H__
#define __SN79489_H__
#include <stdint.h>

/**
 * This is just a plain wrapper around out(PSG, dat) with dat being a uint8_t
 */
#define sn79489_register_write(dat) out(PSG, dat)

/**
 * These are quite literal representations of the upper 4 bits for the latch
 * byte
 */
#define SN79489_CHANNEL0 0x00
#define SN79489_CHANNEL1 0x20
#define SN79489_CHANNEL2 0x40
#define SN79489_CHANNEL3 0x60

/**
 * This latches the volume register and fills in the dat
 * channel: one of the SN79489_CHANNEL defines
 * dat: 0-0xF signifies the low 4 bits of the register,
 *       with the highest discarded for the 3-bit noise.
 */
void sn79489_latch_volume(uint8_t channel, uint8_t dat);
/**
 * This latches the tone or noise register and fills in some or all of the dat
 * channel: 0-3
 * dat: 0xF is the high 6 bits for the tone register or the low 3-4 bits with
 * the rest discarded
 */
void sn79489_latch_tone(uint8_t channel, uint8_t dat);

/**
 * This writes dat to the latched register's dat
 * dat: 0-0x3F
 */
void sn79489_dat_write(uint8_t dat);

/**
 * If the above seem wasteful for having a channel that's only 2 bits, the below
 * can be used if desired, but require more intimate knowledge of the dat being
 * written. I.e, the user must know what each bit means.
 */
/**
 * This latches a specified channel, type, and dat. It has almost no math when
 * compared to the above functions.
 *
 * dat: Xcctdddd
 * with X being ignored, cc is the channel, t i the type (0:tone, 1:vol), and
 * d is the dat
 */
void sn79489_latch(uint8_t dat);

#endif

