#include <stdlib.h>
#include <sms.h>

unsigned char pal1[] = {0x00, 0x01, 0x02, 0x03, 0x10, 0x20, 0x30, 0x2B,
				0x15, 0x35, 0x1D, 0x3D, 0x00, 0x15, 0x2A, 0x3F};

unsigned char pal2[] = {0x00, 0x01, 0x02, 0x03, 0x10, 0x20, 0x30, 0x2B,
				0x15, 0x35, 0x1D, 0x3D, 0x00, 0x15, 0x2A, 0x3F};

extern unsigned char pong_graphics[];

typedef struct {
	int x;
	int y;
	int size;
	int height;
	int base_spr;
	int base_tile;
	int spd_x;
	int spd_y;
	int top_spd;
	int accel;
	int friction;
} paddle_rec;

typedef struct {
	int x;
	int y;
	int base_spr;
	int base_tile;
	int spd_x;
	int spd_y;
	int base_spd;
	int top_spd;
	int spin;
	int spin_friction;
} ball_rec;

typedef struct {
	int top;
	int bottom;
	int left;
	int right;
	int	score0;
	int score1;
} playfield_rec;

int sig_shr(int number, int shift) {
	if (number < 0) {
		return -((-number) >> shift);
	} else {
		return number >> shift;
	}
}

void draw_score_digits(int x, int y, int n) {
	unsigned int buffer[2];
	buffer[0] = (n / 10) + 1;
	buffer[1] = (n % 10) + 1;
	set_bkg_map(buffer, x, y, 2, 1);
}

void draw_score(playfield_rec *playfield) {
	draw_score_digits(4, 0, playfield->score0);
	draw_score_digits(28, 0, playfield->score1);
}

void draw_paddle(paddle_rec *paddle) {
	int x = (paddle->x >> 4);
	int y = (paddle->y >> 4);
	int spr = paddle->base_spr;
	int tile = paddle->base_tile;
	int i;

	set_sprite(spr, x, y, tile);
	y += 8; spr++; tile++;
	for (i = paddle->size; i; i--, y += 8, spr++) {
		set_sprite(spr, x, y, tile);
	}
	tile++;
	set_sprite(spr, x, y, tile);
}

void handle_paddle_joypad(int joy, paddle_rec *paddle) {
	if (joy & JOY_UP) {
		paddle->spd_y -= paddle->accel;
		if (paddle->spd_y < -paddle->top_spd) {
			paddle->spd_y = -paddle->top_spd;
		}
	}
	if (joy & JOY_DOWN) {
		paddle->spd_y += paddle->accel;
		if (paddle->spd_y > paddle->top_spd) {
			paddle->spd_y = paddle->top_spd;
		}
	}
}

void handle_paddle_physics(paddle_rec *paddle, playfield_rec *playfield) {
	paddle->x += paddle->spd_x;
	paddle->y += paddle->spd_y;

	if (paddle->x < playfield->left) {
		paddle->x = playfield->left;
		paddle->spd_x = -paddle->spd_x;
	}
	if (paddle->x > playfield->right-8) {
		paddle->x = playfield->right-8;
		paddle->spd_x = -paddle->spd_x;
	}

	if (paddle->y < playfield->top) {
		paddle->y = playfield->top;
		paddle->spd_y = -paddle->spd_y;
	}
	if (paddle->y > playfield->bottom-paddle->height) {
		paddle->y = playfield->bottom-paddle->height;
		paddle->spd_y = -paddle->spd_y;
	}

	if (paddle->spd_x > 0) {
		if (paddle->spd_x > paddle->friction) {
			paddle->spd_x -= paddle->friction;
		} else {
			paddle->spd_x = 0;
		}
	} else {
		if (paddle->spd_x < -paddle->friction) {
			paddle->spd_x += paddle->friction;
		} else {
			paddle->spd_x = 0;
		}
	}

	if (paddle->spd_y > 0) {
		if (paddle->spd_y > paddle->friction) {
			paddle->spd_y -= paddle->friction;
		} else {
			paddle->spd_y = 0;
		}
	} else {
		if (paddle->spd_y < -paddle->friction) {
			paddle->spd_y += paddle->friction;
		} else {
			paddle->spd_y = 0;
		}
	}
}

void draw_ball(ball_rec *ball) {
	int x = (ball->x >> 4);
	int y = (ball->y >> 4);
	int spr = ball->base_spr;
	int tile = ball->base_tile;

	set_sprite(spr, x, y, tile);
}

void deploy_ball(ball_rec *ball) {
	ball->x = 128 << 4;
	ball->y = 32 << 4;
	ball->base_spr = 12;
	ball->base_tile = 14;
	if (rand() & 0x01) {
		ball->spd_x = 0x20;
	} else {
		ball->spd_x = -0x20;
	}
	ball->spd_y = (rand() & 0x3F) - 0x20;
	ball->base_spd = 0x20;
	ball->top_spd = 0x60;
	ball->spin = 0;
	ball->spin_friction = 0x01;
}

void handle_ball_physics(ball_rec *ball, playfield_rec *playfield, paddle_rec paddles[], int paddle_count) {
	int i;
	int collided;
	paddle_rec *paddle;

	ball->x += ball->spd_x;
	ball->y += ball->spd_y;

	ball->spd_y -= sig_shr(ball->spin, 3);
	if (ball->spin > 0) {
		if (ball->spin > ball->spin_friction) {
			ball->spin -= ball->spin_friction;
		} else {
			ball->spin = 0;
		}
	} else {
		if (ball->spin < -ball->spin_friction) {
			ball->spin += ball->spin_friction;
		} else {
			ball->spin = 0;
		}
	}

	if (ball->spd_x < 0) {
		if (ball->spd_x < -ball->top_spd) {
			ball->spd_x = -ball->top_spd;
		}
	} else {
		if (ball->spd_x > ball->top_spd) {
			ball->spd_x = ball->top_spd;
		}
	}

	if (ball->x < playfield->left) {
		ball->x = playfield->left;
		ball->spd_x = -ball->spd_x;
		playfield->score0++;
		deploy_ball(ball);
		draw_score(playfield);
	}
	if (ball->x > playfield->right-8) {
		ball->x = playfield->right-8;
		ball->spd_x = -ball->spd_x;
		playfield->score1++;
		deploy_ball(ball);
		draw_score(playfield);
	}

	if (ball->spd_y < 0) {
		if (ball->spd_y < -ball->top_spd) {
			ball->spd_y = -ball->top_spd;
		}
	} else {
		if (ball->spd_y > ball->top_spd) {
			ball->spd_y = ball->top_spd;
		}
	}

	if (ball->y < playfield->top) {
		ball->y = playfield->top;
		ball->spd_y = -ball->spd_y;
	}
	if (ball->y > playfield->bottom-8) {
		ball->y = playfield->bottom-8;
		ball->spd_y = -ball->spd_y;
	}

	for (i = paddle_count, paddle = paddles; i; i--, paddle++) {
		if ((ball->y > paddle->y - 0x80) && (ball->y < paddle->y + paddle->height)) {
			collided = 0;
			if (ball->spd_x < 0) {
				if ((ball->x < paddle->x + 0x80) && (ball->x > paddle->x)) {
					ball->x = paddle->x + 0x80;
					ball->spd_x = -ball->spd_x;
					ball->spin = paddle->spd_y + sig_shr(ball->spd_y, 1);
					collided = 1;
				}
			} else {
				if ((ball->x < paddle->x) && (ball->x > paddle->x - 0x80)) {
					ball->x = paddle->x - 0x80;
					ball->spd_x = -ball->spd_x;
					ball->spin = paddle->spd_y + sig_shr(ball->spd_y, 1);
					collided = 1;
				}
			}
			if (collided) {
				if (ball->y < paddle->y) {
					ball->spd_y -= 0x100;
				} else if (ball->y > paddle->y + paddle->height - 0x80) {
					ball->spd_y += 0x100;
				}
			}
		}
	}
}

void main() {
	int joy1, joy2;
	int i;
	playfield_rec playfield;
	paddle_rec paddles[2], *paddle;
	ball_rec ball;

	playfield.top = 16 << 4;
	playfield.bottom = 184 << 4;
	playfield.left = 0 << 4;
	playfield.right = 255 << 4;
	playfield.score0 = 0;
	playfield.score1 = 0;

	paddle = paddles;
	paddle->x = 8 << 4;
	paddle->y = 32 << 4;
	paddle->size = 2;
	paddle->height = (paddle->size + 2) << 7;
	paddle->base_spr = 0;
	paddle->base_tile = 11;
	paddle->spd_x = 0;
	paddle->spd_y = 0;
	paddle->top_spd = 0x40;
	paddle->accel = 0x09;
	paddle->friction = 0x04;

	paddle++;
	paddle->x = 240 << 4;
	paddle->y = 32 << 4;
	paddle->size = 2;
	paddle->height = (paddle->size + 2) << 7;
	paddle->base_spr = 6;
	paddle->base_tile = 11;
	paddle->spd_x = 0;
	paddle->spd_y = 0;
	paddle->top_spd = 0x40;
	paddle->accel = 0x09;
	paddle->friction = 0x04;

	deploy_ball(&ball);

	set_vdp_reg(VDP_REG_FLAGS1, VDP_REG_FLAGS1_SCREEN);
	load_tiles(pong_graphics, 1, 14, 4);
	load_palette(pal1, 0, 16);
	load_palette(pal2, 16, 16);

	draw_score(&playfield);

	for (;;) {
		paddle = paddles;
		joy1 = read_joypad1();
		handle_paddle_joypad(joy1, paddle);
		handle_paddle_physics(paddle, &playfield);

		paddle++;
		joy2 = read_joypad2();
		handle_paddle_joypad(joy2, paddle);
		handle_paddle_physics(paddle, &playfield);

		handle_ball_physics(&ball, &playfield, paddles, 2);

		wait_vblank_noint();
		for (i = 0, paddle = paddles; i != 2; i++, paddle++) {
			draw_paddle(paddle);
		}
		draw_ball(&ball);
	}
}
