! cpu.s
! shuboy cpu emulator
! sh2 assembly version
! /mic 2009

.global _cpu_execute
.global _cpu_reset
.global _cpu_rst


! Register usage:
! 
!  R4:  return address for instructions
!  R5:  scratch register not destroyed by external functions
!  R6:  pointer to the read_byte function
!  R7:  A
!  R8:  F
!  R9:  address mask used by some of the memory access functions
!  R10: PC
!  R11: SP
!  R12: cycles
!  R13: emulation status flags
!  R14: pointer to IOREGS
!
!  GBR: register list pointer (B, C, D, E, H, L)
!
! R4-R6,R14 can be used for other purposes after _cpu_execute has finished,
! but R7-R9,R10-R13,GBR need to be preserved at all times.


! ###########################################################################################################


.include "shuboy.inc"


! Indices for the register list pointed to by GBR
.equ REG_B, 0
.equ REG_C, 1
.equ REG_D, 2
.equ REG_E, 3
.equ REG_H, 4
.equ REG_L, 5


! Bitmasks for the Z80 flag register
.equ FLAG_Z, 0x80
.equ FLAG_N, 0x40
.equ FLAG_H, 0x20
.equ FLAG_C, 0x10


! ###########################################################################################################


.macro ADD_CYCLES cycnum
	add 	#\cycnum,r12
.endm


! Move the Z80 program counter <dist> bytes
.macro MOVE_PC dist 
	add 	#\dist,r10
.endm


! Move the Z80 stack pointer <dist> bytes
.macro MOVE_SP dist 
	add 	#\dist,r11
.endm


! Jump to _mem_read_byte (located in memory.s)
.macro READ_BYTE
	jsr 	@r6
.endm


! Used for debugging purposes
.macro ILLEGAL_OP
	mov	r0,r13
	mov.l	ROM_\@,r14
	\@: 
		bra	\@b
		mov	#0x22,r5
	.align 2
	ROM_\@:	.long _ROM0
.endm


! The reason for using JMP instructions to jump to / return from
! the instruction handlers instead of JSR/RTS is to avoid the need
! of saving and restoring PR, since many of the instruction handlers
! will JSR to some memory access routine.
.macro RETURN
	jmp	@r4
.endm


! ###########################################################################################################


! Move a Z80 register to another Z80 register
.macro LD_REG8_REG8 dest src
	mov.b	@(\src,gbr),r0
 	MOVE_PC	1
 	mov.b	r0,@(\dest,gbr)
 	RETURN
 	ADD_CYCLES 4
.endm


! Move a Z80 register to Z80 register A
.macro LD_A_REG8 reg
 	mov.b	@(\reg,gbr),r0
	MOVE_PC	1
 	extu.b	r0,r7
 	RETURN
 	ADD_CYCLES 4
.endm


! Move Z80 register A to another Z80 register
.macro LD_REG8_A reg
	mov	r7,r0
	MOVE_PC	1
 	mov.b	r0,@(\reg,gbr)
 	RETURN
 	ADD_CYCLES 4
.endm
 

! Move the byte pointed to by Z80 register pair HL to another Z80 register 
.macro LD_REG8_ATHL reg rd8
	mov.w	@(REG_H,gbr),r0
 	READ_BYTE
 	extu.w	r0,r0	!  delayed branch slot
 	MOVE_PC	1
 	mov.b	r0,@(\reg,gbr)
 	RETURN
 	ADD_CYCLES 8
.endm


! Move a Z80 register to the byte pointed to by Z80 register pair HL
.macro LD_ATHL_REG8 reg wr8
	mov.b	@(\reg,gbr),r0
 	mov.l	\wr8,r2
 	extu.b	r0,r1
	mov.w	@(REG_H,gbr),r0
 	jsr	@r2
 	extu.w	r0,r0	! delayed branch slot
 	MOVE_PC	1
 	RETURN
 	ADD_CYCLES 8
.endm

 
 ! Move an immediate byte to a Z80 register
.macro LD_IMM8 reg rd8
 	MOVE_PC 1
 	mov	r10,r0
	READ_BYTE
 	MOVE_PC 1
 	mov.b	r0,@(\reg,gbr)
 	RETURN
 	ADD_CYCLES 8
.endm
 
 
 ! Move an immediate word to a Z80 register pair
.macro LD_IMM16 hireg loreg rd8
 	MOVE_PC 1
 	mov	r10,r0
	READ_BYTE
 	MOVE_PC 1
 	mov.b	r0,@(\loreg,gbr)
 	mov	r10,r0
 	READ_BYTE
 	MOVE_PC 1
 	mov.b	r0,@(\hireg,gbr)
 	RETURN
 	ADD_CYCLES 12
.endm


! Push a Z80 register pair onto the Z80 stack
.macro PUSH_REG16 hireg loreg wr8
	MOVE_SP	-1
	mov.b	@(\hireg,gbr),r0
	mov.l	\wr8,r5
	extu.b	r0,r1
	jsr	@r5
	extu.w	r11,r0	! delayed branch slot
	MOVE_SP	-1
	mov.b	@(\loreg,gbr),r0
!	mov.l	\wr8,r2
	extu.b	r0,r1
	jsr	@r5
	extu.w	r11,r0	! delayed branch slot
	MOVE_PC	1
	RETURN
	ADD_CYCLES 16
.endm


! Pop a Z80 register pair from the Z80 stack
.macro POP_REG16 hireg loreg rd8
	READ_BYTE
	extu.w	r11,r0	! delayed branch slot
	MOVE_SP	1
	mov.b	r0,@(\loreg,gbr)
	READ_BYTE
	extu.w	r11,r0	! delayed branch slot
	MOVE_SP	1
	mov.b	r0,@(\hireg,gbr)
	MOVE_PC	1
	RETURN
	ADD_CYCLES 12
.endm

                          
! ###########################################################################################################


! Increase the value of a Z80 register by 1
.macro INC_REG8 reg
	mov.b	@(\reg,gbr),r0
	mov	r0,r1
	add	#1,r0
	extu.b 	r0,r0
	tst	r0,r0
	movt	r2
	mov.b	r0,@(\reg,gbr)
	shll8	r2
	xor	r1,r0	

	shlr	r2
	shll2	r0
	and	#FLAG_H,r0
	or	r8,r0
	and	#FLAG_C+FLAG_H,r0
	or	r2,r0
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4
.endm


! Decrease the value of a Z80 register by 1
.macro DEC_REG8 reg
	mov.b	@(\reg,gbr),r0
	mov	r0,r1
	mov 	#FLAG_C,r2
	add 	#-1,r0
	and 	r2,r8
	extu.b 	r0,r0
	cmp/eq 	#0,r0
	movt 	r2
	shll8 	r2
	mov.b	r0,@(\reg,gbr)
	shlr 	r2	! F.Zero
	xor 	r1,r0
	or 	r2,r8
	shll2 	r0
	and 	#FLAG_H,r0
	or	#FLAG_N,r0
	MOVE_PC	1
	or 	r0,r8
	RETURN
	ADD_CYCLES 4
.endm


! Add a Z80 register to Z80 register A
! The H-flag is calculated by taking (A^operand^result) and
! using bit 3 of that value.
.macro ADD_A_REG8 reg
	mov.b	@(\reg,gbr),r0
	extu.b	r0,r1
	extu.b	r7,r0		! r0 = A
	add	r1,r0		! r0 = A + reg
	extu.b	r0,r0
	cmp/eq	#0,r0
	movt 	r8
	shll8 	r8
	xor 	r7,r1		! r1 = reg^A
	shlr 	r8		! Zero
	cmp/hi	r0,r7
	bf/s	NC\@
	mov	r0,r7		! A = A + reg	
	add	#FLAG_C,r8	
NC\@:
	xor	r1,r0		! r0 = result^reg^A
	shll2	r0		! move bit 3 into bit 5
	and	#FLAG_H,r0
	or	r0,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4
.endm


! Add a Z80 register plus the Z80 carry to Z80 register A
.macro ADC_A_REG8 reg
	mov.b	@(\reg,gbr),r0
	extu.b	r0,r1
	extu.b	r8,r0
	xor 	#FLAG_C,r0
	tst 	#FLAG_C,r0
	extu.b	r7,r0		! r0 = A
	addc 	r1,r0
	mov 	r0,r2
	extu.b	r0,r0
	cmp/eq	#0,r0
	movt 	r8
	shll8 	r8
	xor 	r7,r1
	shlr 	r8		! Zero
	mov 	r0,r7
	xor 	r0,r1
	extu.b 	r1,r0
	shll	r0
	and	#FLAG_H,r0
	or	r0,r8
	mov 	r2,r0
	shlr2 	r0
	shlr2 	r0
	and 	#FLAG_C,r0
	or 	r0,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4
.endm


! Subtract a Z80 register from Z80 register A
! The H-flag is calculated by taking (A^operand^result) and
! using bit 4 of that value.
.macro SUB_A_REG8 reg
	mov.b	@(\reg,gbr),r0
	extu.b	r0,r1
	extu.b	r7,r0		! r0 = A
	sub	r1,r0		! r0 = A - reg
	extu.b 	r0,r0
	cmp/eq	#0,r0
	movt	r8
	shll8	r8
	shlr	r8
	cmp/hs	r1,r7
	xor	r7,r1
	bt/s	NC\@
	mov	r0,r7
	add	#FLAG_C,r8	! Set F.Carry if A<reg
NC\@:
	xor	r1,r0
	shll	r0
	and	#FLAG_H,r0
	or	r0,r8
	add	#FLAG_N,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4
.endm


.macro SBC_A_REG8 reg
	mov.b	@(\reg,gbr),r0
	extu.b	r0,r1
	extu.b	r8,r0
	xor	#FLAG_C,r0	! tst sets T if (op1&op2)==0, so invert the bit to test for
	tst	#FLAG_C,r0
	extu.b	r7,r0		! r0 = A
	subc	r1,r0		! r0 = A - reg - F.Carry
	cmp/pz	r0
	bt/s	NC\@
	mov	#FLAG_N,r8
	add	#FLAG_Z,r8
NC\@:
	extu.b	r0,r0
	cmp/eq	#0,r0
	bf/s	\@f
	xor	r7,r1
	add	#FLAG_Z,r8	
\@:
	extu.b	r0,r7		
	xor	r1,r0
	shll2	r0
	and	#FLAG_H,r0
	or	r0,r8
	extu.b	r8,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4
.endm


.macro CP_A_REG8 reg
	mov.b	@(\reg,gbr),r0
	extu.b	r0,r1
	extu.b	r7,r0
	sub	r1,r0
	cmp/eq 	r1,r7
	movt	r8
	shll8	r8
	shlr	r8		! Zero
	cmp/hs	r1,r7
	bt/s	\@f
	add	#FLAG_N,r8
	add	#FLAG_C,r8
\@:
	xor 	r7,r1
	xor 	r1,r0
	shll	r0
	and	#FLAG_H,r0
	or	r0,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4
.endm


.macro ADD_REG16_REG16 hidest lodest hisrc losrc
	mov.w	@(\hisrc,gbr),r0
	extu.w	r0,r1
	mov.w	@(\hidest,gbr),r0
	extu.w	r0,r2
	add	r1,r2
	mov	#FLAG_Z,r5
	and	r5,r8	
	xor 	r0,r1
	xor 	r2,r1
	extu.w	r2,r0
	mov.w	r0,@(\hidest,gbr)
	shlr8 	r2
	shlr8 	r1
	shlr2 	r2
	shll2 	r1
	shlr2 	r2
	mov 	r2,r0
	and 	#FLAG_C,r0
	or 	r0,r8
	mov 	r1,r0
	and 	#FLAG_H,r0
	or 	r0,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
.endm

! ###########################################################################################################


! Do a bitwise AND between Z80 register A and another Z80 register
.macro AND_A_REG8 reg
	mov.b	@(\reg,gbr),r0
	and	r0,r7
	tst 	r7,r7
	movt	r8
	shll8 	r8
	MOVE_PC 1
	shlr 	r8		! Zero
	add 	#FLAG_H,r8	! AND sets F.Halfcarry	
	RETURN
	ADD_CYCLES 4
.endm


! Do a bitwise OR between Z80 register A and another Z80 register
.macro OR_A_REG8 reg
	mov.b	@(\reg,gbr),r0
	or	r0,r7
	extu.b	r7,r7		! mov.b sign-extends, so make sure bits 8-31 are cleared
	tst 	r7,r7
	movt 	r8
	shll8 	r8
	MOVE_PC 1
	shlr 	r8		! Zero
	RETURN
	ADD_CYCLES 4
.endm


! Do a bitwise XOR between Z80 register A and another Z80 register
.macro XOR_A_REG8 reg
	mov.b	@(\reg,gbr),r0
	xor	r0,r7
	mov	#0,r8
	extu.b	r7,r0	
	cmp/eq	#0,r0
	bf/s	\@f
	MOVE_PC	1
	add	#FLAG_Z,r8
\@:
	extu.b	r7,r7		! mov.b sign-extends, so make sure bits 8-31 are cleared	
	extu.b	r8,r8
	RETURN
	ADD_CYCLES 4
.endm
	 
							 
! ###########################################################################################################

.macro RL_REG8 reg
	mov.b	@(\reg,gbr),r0
	extu.b	r0,r1
	extu.b	r8,r0
	xor	#FLAG_C,r0
	tst	#FLAG_C,r0	! T=(F.Carry?1:0)
	rotcl	r1		! r1 = (reg<<1)|F.Carry
	mov	r1,r0
	mov.b	r0,@(\reg,gbr)
	extu.b	r0,r1
	tst	r1,r1
	movt	r8
	shll8	r8
	MOVE_PC	1
	shlr	r8		! Zero
	shlr2	r0
	shlr2	r0		! Move the 8th bit into bit 4
	and	#FLAG_C,r0
	or	r0,r8		! Carry
	RETURN
	ADD_CYCLES 8	
.endm


.macro RLC_REG8 reg
	mov.b	@(\reg,gbr),r0
	extu.b	r0,r0
	shll	r0
	swap.b	r0,r1		! Move bit 8 of r0 (Carry) into bit 0 of r1
	or	r1,r0
	mov.b	r0,@(\reg,gbr)
	extu.b	r0,r1		
	and	#1,r0		! Carry
	tst	r1,r1
	movt	r8
	shll2 	r0	
	shll8	r8
	MOVE_PC	1
	shll2 	r0
	shlr	r8		! Zero
	or	r0,r8
	RETURN
	ADD_CYCLES 8
.endm


.macro RRC_REG8 reg
	mov.b	@(\reg,gbr),r0
	extu.b	r0,r1
	shll8	r1
	extu.b	r0,r0
	shlr	r1
	shlr	r0
	or	r1,r0		! r0 = (reg>>1)|(reg<<7)
	movt	r8
	mov.b	r0,@(\reg,gbr)
	shll2	r8
	shll2	r8		! Carry
	extu.b	r0,r0
	tst	r0,r0
	movt	r1
	MOVE_PC	1
	shll8	r1
	shlr	r1
	or	r1,r8		! Zero
	RETURN
	ADD_CYCLES 8
.endm
	

.macro RR_REG8 reg
	mov.b	@(\reg,gbr),r0
	extu.b	r0,r1
	extu.b	r8,r0
	and	#FLAG_C,r0
	shll2	r0
	shll	r0
	shlr	r1	
	movt	r8
	shll2	r8
	shll2	r8		! Carry
	or	r1,r0		! r0 = (reg>>1)|(OldCarry<<7)
	mov.b	r0,@(\reg,gbr)
	MOVE_PC 1
	tst	r0,r0
	movt	r1
	shll8	r1
	shlr	r1
	or	r1,r8		! Zero
	RETURN
	ADD_CYCLES 8
.endm
          
                     
.macro SLA_REG8 reg
	mov.b	@(\reg,gbr),r0
	shll	r0
	mov.b	r0,@(\reg,gbr)
	extu.b	r0,r1
	tst	r1,r1
	movt	r8
	shll8	r8
	MOVE_PC	1
	shlr	r8		! Zero
	shlr2	r0
	shlr2	r0
	and	#FLAG_C,r0
	or	r0,r8		! Carry
	RETURN
	ADD_CYCLES 8
.endm


.macro SRA_REG8 reg
	mov.b	@(\reg,gbr),r0
	shlr	r0		! r0 was sign-extended by mov.b
	mov.b	r0,@(\reg,gbr)
	movt	r8
	shll2	r8
	shll2	r8		! Carry
	extu.b	r0,r0
	tst	r0,r0
	movt	r1
	MOVE_PC	1
	shll8	r1
	shlr	r1
	or	r1,r8		! Zero
	RETURN
	ADD_CYCLES 8
.endm


.macro SRL_REG8 reg
	mov.b	@(\reg,gbr),r0
	extu.b 	r0,r0
	shlr	r0
	mov.b	r0,@(\reg,gbr)
	movt	r8
	shll2	r8
	shll2	r8		! Carry
	extu.b	r0,r0
	tst	r0,r0
	movt	r1
	MOVE_PC	1
	shll8	r1
	shlr	r1
	or	r1,r8		! Zero
	RETURN
	ADD_CYCLES 8
.endm


.macro SWAP_REG8 reg
	mov.b	@(\reg,gbr),r0
	mov	r0,r1
	shlr2	r0
	shll2	r1
	shlr2	r0
	shll2	r1
	extu.b	r1,r1
	and	#0x0F,r0
	or	r1,r0		! r0 = ((reg>>4)&0x0F)|(reg<<4)
	MOVE_PC 1
	mov.b	r0,@(\reg,gbr)
	tst	r0,r0
	movt	r8
	shll8	r8
	shlr	r8		! Zero
	RETURN
	ADD_CYCLES 8
.endm

					  
! ###########################################################################################################
                     

.macro BIT_REG8 reg mask
	extu.b	r8,r0
	and	#FLAG_C,r0
	or	#FLAG_H,r0	! r0 = (F & FLAG_C) | FLAG_H
	mov	r0,r8	
	mov.b	@(\reg,gbr),r0
	extu.b	r0,r0
	tst	#\mask,r0
	movt	r1
	shll8	r1
	shlr	r1		! Zero
	or	r1,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
.endm


.macro BIT_A mask
	extu.b	r8,r0
	and	#FLAG_C,r0
	or	#FLAG_H,r0
	mov	r0,r8
	extu.b	r7,r0
	tst	#\mask,r0
	movt	r1
	shll8	r1
	shlr	r1
	or	r1,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
.endm


.macro BIT_ATHL mask
	extu.b	r8,r0
	and	#FLAG_C,r0
	or	#FLAG_H,r0
	mov	r0,r8
	mov.w	@(REG_H,gbr),r0
	READ_BYTE
	extu.w	r0,r0
	tst	#\mask,r0
	movt	r1
	shll8	r1
	shlr	r1
	or	r1,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 12
.endm


.macro RES_REG8 reg mask
	mov.b	@(\reg,gbr),r0
	and	#\mask,r0
	MOVE_PC	1
	mov.b	r0,@(\reg,gbr)
	RETURN
	ADD_CYCLES 8
.endm


.macro RES_A mask
	mov	#\mask,r0
	MOVE_PC	1
	and	r0,r7
	RETURN
	ADD_CYCLES 8
.endm


.macro RES_ATHL mask wr8
	mov.w	@(REG_H,gbr),r0
	extu.w	r0,r5
	READ_BYTE
	mov	r5,r0
	and	#\mask,r0
	MOVE_PC	1
	mov.l	\wr8,r2
	mov	r0,r1
	jsr	@r2
	mov	r5,r0
	RETURN
	ADD_CYCLES 16
.endm


.macro SET_REG8 reg mask
	mov.b	@(\reg,gbr),r0
	or	#\mask,r0
	MOVE_PC	1
	mov.b	r0,@(\reg,gbr)
	RETURN
	ADD_CYCLES 8
.endm


.macro SET_A mask
	mov	#\mask,r0
	MOVE_PC	1
	or	r0,r7
	extu.b	r7,r7
	RETURN
	ADD_CYCLES 8
.endm


.macro SET_ATHL mask wr8
	mov.w	@(REG_H,gbr),r0
	extu.w	r0,r5
	READ_BYTE
	mov	r5,r0
	or	#\mask,r0
	MOVE_PC	1
	mov.l	\wr8,r2
	extu.b	r0,r1
	jsr	@r2
	mov	r5,r0	
	RETURN
	ADD_CYCLES 16
.endm


! ###########################################################################################################


.macro JR_COND flg branch
	mov	#\flg,r0
	extu.b	r0,r0
	tst	r0,r8
	\branch/s \@f
	MOVE_PC 2
	mov	r10,r0
	add	#-1,r0		! Reads from [oldPC+1], since we've already moved PC 2 bytes forward
	READ_BYTE
	extu.w	r0,r0
	ADD_CYCLES 4
	exts.b	r0,r0
	add	r0,r10
\@:
	RETURN
	ADD_CYCLES 8
.endm


.macro JP_COND flg branch rd8
	mov	#\flg,r0
	extu.b	r0,r0
	tst	r0,r8
	\branch	\@f
	extu.w	r10,r0
	add	#1,r0
	READ_BYTE
	MOVE_PC	2
	mov	r0,r5
	extu.w	r10,r0
	READ_BYTE
	extu.b	r5,r10
	shll8	r0
	or	r0,r10
	RETURN
	ADD_CYCLES 16
\@:
	MOVE_PC	3
	RETURN
	ADD_CYCLES 12
.endm


.macro CALL_COND flg branch rd8 wr8
	mov	#\flg,r0
	extu.b	r0,r0
	tst	r0,r8
	\branch	\@f
	extu.w	r10,r5
	MOVE_SP -2
	add	#3,r5
	mov	r5,r1
	mov.l	\wr8,r2
	jsr	@r2		! Push PC (low)
	extu.w	r11,r0		
	mov	r5,r1
	mov.l	\wr8,r2
	shlr8	r1
	extu.w	r11,r0
	jsr	@r2		! Push PC (high)
	add	#1,r0
	extu.w	r10,r0
	add	#1,r0
	READ_BYTE
	MOVE_PC	2
	mov	r0,r2
	extu.w	r10,r0
	READ_BYTE
	extu.b	r2,r10
	shll8	r0
	or	r0,r10
	RETURN
	ADD_CYCLES 24
\@:
	MOVE_PC	3
	RETURN
	ADD_CYCLES 12
.endm


.macro RET_COND flg branch rd8
	mov	#\flg,r0
	extu.b	r0,r0
	tst	r0,r8
	\branch	\@f
	extu.w	r11,r0
	READ_BYTE
	MOVE_SP	1
	mov	r0,r10
	extu.w	r11,r0
	READ_BYTE
	MOVE_SP	1
	shll8	r0
	or	r0,r10
	RETURN
	ADD_CYCLES 20
\@:
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
.endm


.macro RST addr wr8 halted
	MOVE_PC 1
	mov 	#0xFF-CPU_HALTED,r0
	and 	r0,r13
	MOVE_SP -2
	mov	r10,r1
	mov.l	\wr8,r2
	jsr	@r2		! Push PC (low)
	extu.w	r11,r0		
	mov	r10,r1
	mov.l	\wr8,r2
	shlr8	r1
	extu.w	r11,r0
	jsr	@r2		! Push PC (high)
	add	#1,r0
	mov	#\addr,r10
	RETURN
	ADD_CYCLES 16
.endm


! ###########################################################################################################

.section .data

.align 1

! NOP
op_00:
	MOVE_PC 1
	RETURN
	ADD_CYCLES 4

	
! LD BC,nn
op_01:
	LD_IMM16 REG_B,REG_C,_read_byte1
	

! LD (BC),A
op_02:
	mov.w	@(REG_B,gbr),r0
	mov	r7,r1
	mov.l	_write_byte1,r2
	jsr	@r2
	extu.w	r0,r0
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
	
			
! INC BC
op_03:
	mov.w	@(REG_B,gbr),r0
	add	#1,r0
	mov.w	r0,@(REG_B,gbr)
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
	
					 
! INC B
op_04:
	INC_REG8 REG_B
	

! DEC B
op_05:
	DEC_REG8 REG_B
	

! RLCA
op_07:
	mov	r7,r0
	mov	#0x80,r1
	extu.b	r1,r1
	xor	r1,r0
	tst	r1,r0		! T should now be set if bit 7 of A was set
	mov	#0,r8
	bf/s	_op_07_NC
	rotcl	r7		! A = (A<<1)|A.7
	mov	#FLAG_C,r8
_op_07_NC:
	extu.b	r7,r7
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4
	

! LD (nn),SP
op_08:
	MOVE_PC 1
 	mov	r10,r0
	READ_BYTE
 	MOVE_PC 1
 	mov	r0,r5	! Save low byte in r5
 	mov	r10,r0
 	READ_BYTE
 	MOVE_PC 1
 	mov.l	_write_byte1,r2
 	shll8	r0
	mov	r11,r1
	or	r0,r5
 	jsr	@r2	! Write low byte of SP at [nn]
  	mov	r5,r0
 	mov.l	_write_byte1,r2
	mov	r11,r1
	add	#1,r5
	shlr8	r1
	jsr	@r2	! Write high byte of SP at [nn+1]
	extu.w	r5,r0
 	RETURN
 	ADD_CYCLES 20
 	

! ADD HL,BC
op_09:
	ADD_REG16_REG16 REG_H,REG_L,REG_B,REG_C


! LD A,(BC)
op_0A:
	mov.w	@(REG_B,gbr),r0
	READ_BYTE
	extu.w	r0,r0
	MOVE_PC	1
	extu.b	r0,r7
	RETURN
	ADD_CYCLES 8
	

! DEC BC
op_0B:
	mov.w	@(REG_B,gbr),r0
	add	#-1,r0
	mov.w	r0,@(REG_B,gbr)
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8

	
! INC C
op_0C:
	INC_REG8 REG_C

	
! DEC C
op_0D:
	DEC_REG8 REG_C


! LD C,n
op_0E:
	LD_IMM8 REG_C,_read_byte1


! RRCA
op_0F:
	extu.b	r7,r7
	extu.b	r7,r1
	shll8	r1
	shlr	r1
	shlr	r7
	or	r1,r7	! A = (A>>1)|(A<<7)
	movt	r8
	shll2	r8
	shll2	r8	! Carry
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4
	

.align 2
_read_byte1:
	.long _read_byte
_write_byte1:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte - __MEM_COPY_TO_IWRAM_START__
	

! ###########################################################################################################

! STOP
op_10:
	mov 	#CPU_STOPPED,r0
	or 	r0,r13
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4


! LD DE,nn
op_11:
	LD_IMM16 REG_D,REG_E,_read_byte2
	

! op_12 and op_13 are in the __CPU_COPY_TO_IWRAM area


! INC D
op_14:
	INC_REG8 REG_D
	
	
! DEC D
op_15:
	DEC_REG8 REG_D


! LD D,n
op_16:
	LD_IMM8 REG_D,_read_byte2
	
				
! RLA
op_17:
	extu.b	r7,r1
	extu.b	r8,r0
	xor	#FLAG_C,r0
	tst	#FLAG_C,r0	! T=(F.Carry?1:0)
	rotcl	r1
	mov	r1,r0
	extu.b	r1,r7
	MOVE_PC	1
	shlr2	r0
	shlr2	r0
	and	#FLAG_C,r0
	mov	r0,r8		! Carry
	RETURN
	ADD_CYCLES 4	
	

! op_18 is in the __CPU_COPY_TO_IWRAM area


! ADD HL,DE
op_19:
	ADD_REG16_REG16 REG_H,REG_L,REG_D,REG_E

			
! LD A,(DE)
op_1A:
	mov.w	@(REG_D,gbr),r0
	READ_BYTE
	extu.w	r0,r0
	MOVE_PC	1
	extu.b	r0,r7
	RETURN
	ADD_CYCLES 8
	

! DEC DE
op_1B:
	mov.w	@(REG_D,gbr),r0
	add	#-1,r0
	mov.w	r0,@(REG_D,gbr)
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
	
! INC E
op_1C:
	INC_REG8 REG_E

	
! DEC E
op_1D:
	DEC_REG8 REG_E


! LD E,n
op_1E:
	LD_IMM8 REG_E,_read_byte2


! RRA
op_1F:
	extu.b	r7,r1
	extu.b	r8,r0
	and	#FLAG_C,r0
	shll2	r0
	shll	r0
	shlr	r1
	movt	r8
	shll2	r8
	or	r1,r0	! r0 = (A>>1)|(OldCarry<<7)
	shll2	r8	! Carry
	extu.b	r0,r7
	MOVE_PC 1
	RETURN
	ADD_CYCLES 4


.align 2
_read_byte2:
	.long _read_byte
_write_byte2:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte - __MEM_COPY_TO_IWRAM_START__
	
! ###########################################################################################################


! op_20 is in the __CPU_COPY_TO_IWRAM area


! LD HL,nn
op_21:
	LD_IMM16 REG_H,REG_L,_read_byte3


! LD (HL+),A
op_22:
	mov.w	@(REG_H,gbr),r0
	mov	r0,r5
	add	#1,r5
	mov	r7,r1
	mov.l	_write_byte3,r2
	jsr	@r2
	extu.w	r0,r0
	mov	r5,r0
	MOVE_PC	1
	mov.w	r0,@(REG_H,gbr)
	RETURN
	ADD_CYCLES 8
				
				
! INC HL
op_23:
	mov.w	@(REG_H,gbr),r0
	add	#1,r0
	mov.w	r0,@(REG_H,gbr)
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
	

! INC H
op_24:
	INC_REG8 REG_H

	
! DEC H
op_25:
	DEC_REG8 REG_H


! LD H,n
op_26:
	LD_IMM8 REG_H,_read_byte3

   				
! DAA
op_27:
	extu.b	r8,r0
	extu.b	r7,r1
	tst	#FLAG_N,r0
	bf	_op_27_N
	tst	#FLAG_H,r0	
	bf	_op_27_addlo
	mov	r1,r0
	and	#0x0F,r0
	mov	#9,r2
	cmp/hi	r2,r0
	bf/s	_op_27_checkhi
	extu.b	r8,r0
_op_27_addlo:
	add	#6,r1
_op_27_checkhi:
	tst	#FLAG_C,r0
	bf	_op_27_addhi
	mov	#0x9F,r2
	extu.b 	r2,r2
	cmp/hi	r2,r1
	bf	_op_27_addsub_done
_op_27_addhi:	
	bra	_op_27_addsub_done
	add	#0x60,r1
_op_27_N:	
	tst	#FLAG_H,r0
	bt	_op_27_N_checkhi
	add	#-6,r1
	extu.b	r1,r1
_op_27_N_checkhi:
	tst	#FLAG_C,r0
	bt	_op_27_addsub_done
	mov	#0x60,r2
	sub	r2,r1
_op_27_addsub_done:
	and	#FLAG_N+FLAG_C,r0
	extu.b	r0,r8
	swap.b	r1,r0
	and	#1,r0
	shll2	r0
	shll2	r0	! Carry
	or	r0,r8
	extu.b	r1,r7
	tst	r7,r7
	movt	r0
	shll8	r0
	shlr	r0	! Zero
	or	r0,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4


! op_28 is in the __CPU_COPY_TO_IWRAM area


! ADD HL,HL
op_29:
	ADD_REG16_REG16 REG_H,REG_L,REG_H,REG_L


! DEC HL
op_2B:
	mov.w	@(REG_H,gbr),r0
	add	#-1,r0
	mov.w	r0,@(REG_H,gbr)
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
	

! INC L
op_2C:
	INC_REG8 REG_L

	
! DEC L
op_2D:
	DEC_REG8 REG_L


! LD L,n
op_2E:
	LD_IMM8 REG_L,_read_byte3
	

! CPL
op_2F:
	mov	#0xFF,r0
	extu.b	r0,r0
	xor	r0,r7
	mov	#FLAG_H+FLAG_N,r0
	or	r0,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4

				
.align 2
_read_byte3:
	.long _read_byte
_write_byte3:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte - __MEM_COPY_TO_IWRAM_START__
	
! ###########################################################################################################

! JR NC,n
op_30:
	JR_COND	FLAG_C,bf


! LD SP,nn
op_31:
	MOVE_PC 1
	mov	r10,r0
	READ_BYTE
	MOVE_PC 1
	mov	r0,r11
	mov	r10,r0
	READ_BYTE
	MOVE_PC 1
	shll8	r0
	or	r0,r11
	RETURN
	ADD_CYCLES 12
	

! LD (HL-),A
op_32:
	mov.w	@(REG_H,gbr),r0
	mov	r0,r5
	add	#-1,r5
	mov	r7,r1
	mov.l	_write_byte4,r2
	jsr	@r2
	extu.w	r0,r0
	mov	r5,r0
	MOVE_PC	1
	mov.w	r0,@(REG_H,gbr)
	RETURN
	ADD_CYCLES 8
	

! INC SP
op_33:
	MOVE_SP 1
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
	

! INC (HL)
op_34:
	mov.w	@(REG_H,gbr),r0
	extu.w	r0,r5
	READ_BYTE
	mov	r5,r0
	mov	r0,r1
	mov 	#FLAG_C,r2
	add 	#1,r1
	and 	r2,r8
	extu.b 	r1,r1
	tst	r1,r1
	movt 	r2
	shll8 	r2
	shlr 	r2	! Zero
	xor 	r1,r0
	or 	r2,r8
	shll2 	r0
	and 	#FLAG_H,r0
	MOVE_PC	1
	mov.l	_write_byte4,r2
	or 	r0,r8
	jsr	@r2
	mov	r5,r0
	RETURN
	ADD_CYCLES 12
	

! DEC (HL)
op_35:
	mov.w	@(REG_H,gbr),r0
	extu.w	r0,r5
	READ_BYTE
	mov	r5,r0
	mov	r0,r1
	mov 	#FLAG_C,r2
	add 	#-1,r1
	and 	r2,r8
	extu.b 	r1,r1
	tst	r1,r1
	movt 	r2
	shll8 	r2
	extu.b	r7,r7	
	shlr 	r2	! Zero
	xor 	r1,r0
	or 	r2,r8
	shll2 	r0
	and 	#FLAG_H,r0
	or	#FLAG_N,r0
	MOVE_PC	1
	mov.l	_write_byte4,r2
	or 	r0,r8
	jsr	@r2
	mov	r5,r0
	RETURN
	ADD_CYCLES 12
	

! LD (HL),n
op_36:
	MOVE_PC 1
	READ_BYTE
	extu.w	r10,r0
	extu.b	r0,r1
	mov.w	@(REG_H,gbr),r0
	mov.l	_write_byte4,r2
	extu.w	r0,r0
	jsr	@r2
	MOVE_PC	1
	RETURN
	ADD_CYCLES 12
	

! SCF
op_37:
	mov	r8,r0
	and	#FLAG_Z,r0
	or	#FLAG_C,r0
	MOVE_PC	1
	extu.b	r0,r8
	RETURN
	ADD_CYCLES 4
	

! JR C,n
op_38:
	JR_COND	FLAG_C,bt
				

! ADD HL,SP
op_39:
	extu.w	r11,r1
	mov.w	@(REG_H,gbr),r0
	extu.w	r0,r2
	extu.b	r0,r0
	extu.b	r1,r5
	add	r1,r2
	add	r5,r0
	shlr2	r0
	shlr2	r0
	mov	#FLAG_Z,r5
	and	#FLAG_C,r0
	and	r5,r8	
	or	r0,r8
	extu.w	r2,r0
	mov.w	r0,@(REG_H,gbr)
	xor	r1,r0
	swap.b	r0,r0
	shll2	r0
	and	#FLAG_H,r0
	or	r0,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
	

! LD A,(HL-)
op_3A:
	mov.w	@(REG_H,gbr),r0
	mov	r0,r5
	add	#-1,r5	
	READ_BYTE
	extu.w	r0,r0
	extu.b	r0,r7
	MOVE_PC	1
	mov	r5,r0
	mov.w	r0,@(REG_H,gbr)
	RETURN
	ADD_CYCLES 8
	

! DEC SP
op_3B:
	MOVE_SP -1
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8


! op_3C, op_3D and op_3E are in the __CPU_COPY_TO_IWRAM area


! CCF
op_3F:
	mov	#FLAG_C,r0
	mov	r0,r1
	or	#FLAG_Z,r0
	and	r0,r8
	MOVE_PC	1
	xor	r1,r8
	extu.b	r8,r8
	RETURN
	ADD_CYCLES 4
				
				
.align 2
_read_byte4:
	.long _read_byte
_write_byte4:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte - __MEM_COPY_TO_IWRAM_START__
	
! ###########################################################################################################


! LD B,B
op_40:
	LD_REG8_REG8 REG_B,REG_B

! LD B,C
op_41:
	LD_REG8_REG8 REG_B,REG_C

! LD B,D
op_42:
	LD_REG8_REG8 REG_B,REG_D

! LD B,E
op_43:
	LD_REG8_REG8 REG_B,REG_E

! LD B,H
op_44:
	LD_REG8_REG8 REG_B,REG_H

! LD B,L
op_45:
	LD_REG8_REG8 REG_B,REG_L

! LD B,(HL)
op_46:
	LD_REG8_ATHL REG_B,_read_byte5


! LD C,B
op_48:
	LD_REG8_REG8 REG_C,REG_B

! LD C,C
op_49:
	LD_REG8_REG8 REG_C,REG_C

! LD C,D
op_4A:
	LD_REG8_REG8 REG_C,REG_D

! LD C,E
op_4B:
	LD_REG8_REG8 REG_C,REG_E

! LD C,H
op_4C:
	LD_REG8_REG8 REG_C,REG_H

! LD C,L
op_4D:
	LD_REG8_REG8 REG_C,REG_L

! LD C,(HL)
op_4E:
	LD_REG8_ATHL REG_C,_read_byte5

! LD C,A
op_4F:
	LD_REG8_A REG_C
	
.align 2
_read_byte5:
	.long _read_byte
_write_byte5:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte
	
! ###########################################################################################################


! LD D,B
op_50:
	LD_REG8_REG8 REG_D,REG_B

! LD D,C
op_51:
	LD_REG8_REG8 REG_D,REG_C

! LD D,D
op_52:
	LD_REG8_REG8 REG_D,REG_D

! LD D,E
op_53:
	LD_REG8_REG8 REG_D,REG_E

! LD D,H
op_54:
	LD_REG8_REG8 REG_D,REG_H

! LD D,L
op_55:
	LD_REG8_REG8 REG_D,REG_L

! LD D,(HL)
op_56:
	LD_REG8_ATHL REG_D,_read_byte6

! LD D,A
op_57:
	LD_REG8_A REG_D


! LD E,B
op_58:
	LD_REG8_REG8 REG_E,REG_B

! LD E,C
op_59:
	LD_REG8_REG8 REG_E,REG_C

! LD E,D
op_5A:
	LD_REG8_REG8 REG_E,REG_D

! LD E,E
op_5B:
	LD_REG8_REG8 REG_E,REG_E

! LD E,H
op_5C:
	LD_REG8_REG8 REG_E,REG_H

! LD E,L
op_5D:
	LD_REG8_REG8 REG_E,REG_L

! LD E,(HL)
op_5E:
	LD_REG8_ATHL REG_E,_read_byte6

! LD E,A
op_5F:
	LD_REG8_A REG_E

	
.align 2
_read_byte6:
	.long _read_byte
_write_byte6:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte - __MEM_COPY_TO_IWRAM_START__
	
! ###########################################################################################################


! LD H,B
op_60:
	LD_REG8_REG8 REG_H,REG_B

! LD H,C
op_61:
	LD_REG8_REG8 REG_H,REG_C

! LD H,D
op_62:
	LD_REG8_REG8 REG_H,REG_D

! LD H,E
op_63:
	LD_REG8_REG8 REG_H,REG_E

! LD H,H
op_64:
	LD_REG8_REG8 REG_H,REG_H

! LD H,L
op_65:
	LD_REG8_REG8 REG_H,REG_L

! LD H,(HL)
op_66:
	LD_REG8_ATHL REG_H,_read_byte7

! LD H,A
op_67:
	LD_REG8_A REG_H


! LD L,B
op_68:
	LD_REG8_REG8 REG_L,REG_B

! LD L,C
op_69:
	LD_REG8_REG8 REG_L,REG_C

! LD L,D
op_6A:
	LD_REG8_REG8 REG_L,REG_D

! LD L,E
op_6B:
	LD_REG8_REG8 REG_L,REG_E

! LD L,H
op_6C:
	LD_REG8_REG8 REG_L,REG_H

! LD L,L
op_6D:
	LD_REG8_REG8 REG_L,REG_L

! LD L,(HL)
op_6E:
	LD_REG8_ATHL REG_L,_read_byte7

! LD L,A
op_6F:
	LD_REG8_A REG_L

	
.align 2
_read_byte7:
	.long _read_byte
_write_byte7:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte - __MEM_COPY_TO_IWRAM_START__
	
! ###########################################################################################################

! LD (HL),B
op_70:
	LD_ATHL_REG8 REG_B,_write_byte8

! LD (HL),C
op_71:
	LD_ATHL_REG8 REG_C,_write_byte8

! LD (HL),D
op_72:
	LD_ATHL_REG8 REG_D,_write_byte8

! LD (HL),E
op_73:
	LD_ATHL_REG8 REG_E,_write_byte8

! LD (HL),H
op_74:
	LD_ATHL_REG8 REG_H,_write_byte8

! LD (HL),L
op_75:
	LD_ATHL_REG8 REG_L,_write_byte8

! HALT
op_76:
	mov 	#CPU_HALTED,r0
	or 	r0,r13
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4

 ! LD A,B
 op_78:
 	LD_A_REG8 REG_B
 
 ! LD A,C
 op_79:
 	LD_A_REG8 REG_C
 
 ! LD A,D
 op_7A:
 	LD_A_REG8 REG_D
 
 ! LD A,E
 op_7B:
 	LD_A_REG8 REG_E
 
 ! LD A,H
 op_7C:
 	LD_A_REG8 REG_H
 
 ! LD A,L
 op_7D:
 	LD_A_REG8 REG_L
 
  
 ! LD A,A
 op_7F:
	MOVE_PC	1
 	rts
 	ADD_CYCLES 4	
	
.align 2
_read_byte8:
	.long _read_byte
_write_byte8:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte - __MEM_COPY_TO_IWRAM_START__
	
! ###########################################################################################################

! ADD A,B
op_80:
	ADD_A_REG8 REG_B

! ADD A,C
op_81:
	ADD_A_REG8 REG_C

! ADD A,D
op_82:
	ADD_A_REG8 REG_D

! ADD A,E
op_83:
	ADD_A_REG8 REG_E

! ADD A,H
op_84:
	ADD_A_REG8 REG_H

! ADD A,L
op_85:
	ADD_A_REG8 REG_L

! ADD A,(HL)
op_86:
	mov.w	@(REG_H,gbr),r0
	READ_BYTE
	extu.w	r0,r0
	extu.b	r0,r1
	extu.b	r7,r0		! r0 = A
	add	r1,r0		! r0 = A + reg
	extu.b	r0,r0
	cmp/eq	#0,r0
	movt 	r8
	shll8 	r8
	xor 	r7,r1
	shlr 	r8		! Zero
	cmp/hi	r0,r7
	bf/s	_op_86_NC
	mov	r0,r7		! A = A + reg	
	add	#FLAG_C,r8	
_op_86_NC:
	xor	r1,r0
	shll2	r0
	and	#FLAG_H,r0
	or	r0,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
	
! ADD A,A
op_87:
	extu.b	r7,r0		! r0 = A
	mov	r0,r1
	add	r1,r0		! r0 = A + reg
	extu.b 	r0,r0
	tst 	r0,r0
	movt 	r8
	shll8 	r8
	shlr 	r8		! Zero
	cmp/hi	r0,r1
	bf	_op_87_NC
	add	#FLAG_C,r8	! Set F.Carry if (A+reg)<A
_op_87_NC:
	extu.b	r0,r7		! A = A+reg
	! The calculation (A^operand^result) can be simplified to
	! just (result) in this case, since A==operand and
	! A^operand therefore is zero.
	shll2	r0
	and	#FLAG_H,r0
	or	r0,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4
	

! ADC A,B
op_88:
	ADC_A_REG8 REG_B

! ADC A,C
op_89:
	ADC_A_REG8 REG_C

! ADC A,D
op_8A:
	ADC_A_REG8 REG_D

! ADC A,E
op_8B:
	ADC_A_REG8 REG_E

! ADC A,H
op_8C:
	ADC_A_REG8 REG_H

! ADC A,L
op_8D:
	ADC_A_REG8 REG_L

! ADC A,(HL)
op_8E:
	mov.w	@(REG_H,gbr),r0
	READ_BYTE
	extu.w	r0,r0
	extu.b	r0,r1
	extu.b	r8,r0
	xor	#FLAG_C,r0	! tst sets T if (op1&op2)==0, so invert the bit to test for
	tst	#FLAG_C,r0
	extu.b	r7,r0		! r0 = A
	addc	r1,r0		! r0 = A + reg + F.Carry
	mov r0,r2
	extu.b 	r0,r0
	cmp/eq	#0,r0
	movt r8
	shll8 r8
	shlr r8			! Zero
	extu.b	r0,r7		
	xor	r7,r1
	xor	r1,r0
	shll2	r0
	and	#FLAG_H,r0
	or	r0,r8
	shlr2 r2
	shlr2 r2
	mov r2,r0
	and #FLAG_C,r0
	or r0,r8		! Carry
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8

! ADC A,A
op_8F:
	extu.b	r7,r1
	extu.b	r8,r0
	xor	#FLAG_C,r0	! tst sets T if (op1&op2)==0, so invert the bit to test for
	tst	#FLAG_C,r0
	extu.b	r7,r0		! r0 = A
	addc	r1,r0		! r0 = A + reg + F.Carry
	mov 	r0,r2
	extu.b 	r0,r0
	cmp/eq	#0,r0
	movt 	r8
	shll8 	r8
	shlr 	r8
	shlr2 	r2
	xor 	r7,r1
	shlr2 	r2
	extu.b	r0,r7		
	xor	r1,r0
	shll	r0
	and	#FLAG_H,r0
	or	r0,r8
	mov 	r2,r0
	and 	#FLAG_C,r0
	or 	r0,r8
!	extu.b	r8,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4

.align 2
_read_byte9:
	.long _read_byte
_write_byte9:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte - __MEM_COPY_TO_IWRAM_START__
	
! ###########################################################################################################

! SUB A,B
op_90:
	SUB_A_REG8 REG_B

! SUB A,C
op_91:
	SUB_A_REG8 REG_C

! SUB A,D
op_92:
	SUB_A_REG8 REG_D

! SUB A,E
op_93:
	SUB_A_REG8 REG_E

! SUB A,H
op_94:
	SUB_A_REG8 REG_H

! SUB A,L
op_95:
	SUB_A_REG8 REG_L

! SUB A,(HL)
op_96:
	mov.w	@(REG_H,gbr),r0
	READ_BYTE
	extu.w	r0,r0
	extu.b	r0,r1
	extu.b	r7,r0		! r0 = A
	sub	r1,r0		! r0 = A - reg
	extu.b 	r0,r0
	cmp/eq	#0,r0
	bf/s	_op_96_NZ
	mov	#FLAG_N,r8	! clear F in the delayed branch slot
	add	#FLAG_Z,r8	! Set F.Zero if (A+reg)==0
_op_96_NZ:
	cmp/hs	r1,r7
	bt/s	_op_96_NC
	xor	r7,r1
	add	#FLAG_C,r8	! Set F.Carry if A<reg
_op_96_NC:
	extu.b	r0,r7		! A = A-reg
	xor	r1,r0
	shll	r0
	and	#FLAG_H,r0
	or	r0,r8
	extu.b	r8,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8

!SUB A,A
op_97:
	mov	#FLAG_N+FLAG_Z,r8	! clear F in the delayed branch slot
	mov	#0,r7
	extu.b	r8,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4
	

! SBC A,B
op_98:
	SBC_A_REG8 REG_B

! SBC A,C
op_99:
	SBC_A_REG8 REG_C

! SBC A,D
op_9A:
	SBC_A_REG8 REG_D

! SBC A,E
op_9B:
	SBC_A_REG8 REG_E

! SBC A,H
op_9C:
	SBC_A_REG8 REG_H

! SBC A,L
op_9D:
	SBC_A_REG8 REG_L

! SBC A,(HL)
op_9E:
	mov.w	@(REG_H,gbr),r0
	READ_BYTE
	extu.w	r0,r0
	extu.b	r0,r1
	extu.b	r8,r0
	xor	#FLAG_C,r0	! tst sets T if (op1&op2)==0, so invert the bit to test for
	tst	#FLAG_C,r0
	extu.b	r7,r0		! r0 = A
	subc	r1,r0		! r0 = A - [HL] - F.Carry
extu.b r0,r0
	cmp/eq	#0,r0
	bf/s	_op_9E_NZ
	mov	#FLAG_N,r8	! clear F in the delayed branch slot
	add	#FLAG_Z,r8	
_op_9E_NZ:
	cmp/hs	r1,r7
	bt	_op_9E_NC
	add	#FLAG_C,r8	
_op_9E_NC:
	extu.b	r0,r7		
	xor	r1,r0
	shll2	r0
	and	#FLAG_H,r0
	or	r0,r8
	extu.b	r8,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8

! SBC A,A
op_9F:
	extu.b	r7,r1
	extu.b	r8,r0
	xor	#FLAG_C,r0	! tst sets T if (op1&op2)==0, so invert the bit to test for
	tst	#FLAG_C,r0
	extu.b	r7,r0		! r0 = A
	subc	r1,r0		! r0 = A - A - F.Carry
extu.b r0,r0
	cmp/eq	#0,r0
	bf/s	_op_9F_NZ
	mov	#FLAG_N,r8	! clear F in the delayed branch slot
	add	#FLAG_Z,r8	
_op_9F_NZ:
	cmp/hs	r1,r7
	bt	_op_9F_NC
	add	#FLAG_C,r8	
_op_9F_NC:
	extu.b	r0,r7		
	xor	r1,r0
	shll2	r0
	and	#FLAG_H,r0
	or	r0,r8
	extu.b	r8,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4
	

.align 2
_read_byte10:
	.long _read_byte
_write_byte10:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte - __MEM_COPY_TO_IWRAM_START__
	
! ###########################################################################################################

! AND A,B
op_A0:
	AND_A_REG8 REG_B

! AND A,C
op_A1:
	AND_A_REG8 REG_C

! AND A,D
op_A2:
	AND_A_REG8 REG_D

! AND A,E
op_A3:
	AND_A_REG8 REG_E

! AND A,H
op_A4:
	AND_A_REG8 REG_H

! AND A,L
op_A5:
	AND_A_REG8 REG_L

! AND A,(HL)
op_A6:
	mov.w	@(REG_H,gbr),r0
	READ_BYTE
	extu.w	r0,r0
	and	r0,r7
	tst 	r7,r7
	movt 	r8
	shll8 	r8
	MOVE_PC 1
	shlr 	r8	! Zero
	add 	#FLAG_H,r8
	RETURN
	ADD_CYCLES 8


! XOR A,B
op_A8:
	XOR_A_REG8 REG_B

! XOR A,C
op_A9:
	XOR_A_REG8 REG_C

! XOR A,D
op_AA:
	XOR_A_REG8 REG_D

! XOR A,E
op_AB:
	XOR_A_REG8 REG_E

! XOR A,H
op_AC:
	XOR_A_REG8 REG_H

! XOR A,L
op_AD:
	XOR_A_REG8 REG_L

! XOR A,(HL)
op_AE:
	mov.w	@(REG_H,gbr),r0
	READ_BYTE
	extu.w	r0,r0		! delayed branch slot
	xor	r0,r7
	tst 	r7,r7
	movt 	r8
	shll8 	r8
	MOVE_PC 1
	shlr 	r8		! Zero
	RETURN
	ADD_CYCLES 8
	

! XOR A,A
op_AF:
	xor	r7,r7
	mov	#FLAG_Z,r8
	MOVE_PC 1
	extu.b	r8,r8		
	RETURN
	ADD_CYCLES 4

.align 2
_read_byte11:
	.long _read_byte
_write_byte11:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte - __MEM_COPY_TO_IWRAM_START__
	
	
! ###########################################################################################################


! OR A,B
op_B0:
	OR_A_REG8 REG_B

! OR A,C
op_B1:
	OR_A_REG8 REG_C

! OR A,D
op_B2:
	OR_A_REG8 REG_D

! OR A,E
op_B3:
	OR_A_REG8 REG_E

! OR A,H
op_B4:
	OR_A_REG8 REG_H

! OR A,L
op_B5:
	OR_A_REG8 REG_L

! OR A,(HL)
op_B6:
	mov.w	@(REG_H,gbr),r0
	READ_BYTE
	extu.w	r0,r0
	or	r0,r7
	tst 	r7,r7
	movt 	r8
	shll8 	r8
	MOVE_PC 1
	shlr 	r8	! Zero
	RETURN
	ADD_CYCLES 8

! OR A,A
op_B7:
	tst 	r7,r7
	movt 	r8
	shll8 	r8
	MOVE_PC 1
	shlr 	r8	! Zero
	RETURN
	ADD_CYCLES 4




! CP A,E 
op_BB:
	CP_A_REG8 REG_E

! CP A,H 
op_BC:
	CP_A_REG8 REG_H

! CP A,L 
op_BD:
	CP_A_REG8 REG_L

! CP A,(HL)
op_BE:
	mov.w	@(REG_H,gbr),r0
	READ_BYTE
	extu.w	r0,r0
	extu.b	r0,r1
	extu.b	r7,r0		! r0 = A
	sub	r1,r0		! r0 = A - [HL]
	extu.b 	r0,r0
	tst 	r0,r0
	movt 	r8
	shll8 	r8
	shlr 	r8		! F.Zero
	cmp/hs	r1,r7
	bt/s	_op_BE_NC
	xor	r7,r1
	add	#FLAG_C,r8	! Set F.Carry if A<reg
_op_BE_NC:
	xor	r1,r0
	shll	r0
	and	#FLAG_H,r0
	or	r0,r8
	add	#FLAG_N,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8

! CP A,A 
op_BF:
	mov	#FLAG_N+FLAG_Z,r8
	extu.b	r8,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4
	
	
.align 2
_read_byte12:
	.long _read_byte
_write_byte12:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte - __MEM_COPY_TO_IWRAM_START__
	
! ###########################################################################################################


! RET NZ
op_C0:
	RET_COND FLAG_Z,bf,_read_byte13
	

! POP BC
op_C1:
	POP_REG16 REG_B,REG_C,_read_byte13


! JP NZ,nn
op_C2:
	JP_COND FLAG_Z,bf,_read_byte13
	


! CALL NZ,nn
op_C4:
	CALL_COND FLAG_Z,bf,_read_byte13,_write_byte13


! PUSH BC
op_C5:
	PUSH_REG16 REG_B,REG_C,_write_byte13


! ADD A,n
op_C6:
	MOVE_PC	1
	READ_BYTE
	extu.w	r10,r0
	extu.b	r0,r1
	extu.b	r7,r0		! r0 = A
	add	r1,r0		! r0 = A + n
	extu.b	r0,r0
	tst 	r0,r0
	movt 	r8
	shll8 	r8
	shlr 	r8		! F.Zero
	cmp/hi	r0,r7
	bf/s	_op_C6_NC
	xor	r7,r1
	add	#FLAG_C,r8	! Set F.Carry if (A+n)<A
_op_C6_NC:
	extu.b	r0,r7		! A = A+n
	xor	r1,r0
	shll2	r0
	and	#FLAG_H,r0
	or	r0,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
	

! RST 0x00
op_C7:
	RST 0x00,_write_byte13,_cpu_halted13

! RET Z
op_C8:
	RET_COND FLAG_Z,bt,_read_byte13


! RET
op_C9:
	extu.w	r11,r0
	READ_BYTE
	MOVE_SP	1
	mov	r0,r10
	extu.w	r11,r0
	READ_BYTE
	MOVE_SP	1
	shll8	r0
	or	r0,r10
	RETURN
	ADD_CYCLES 16
	

! JP Z,nn
op_CA:
	JP_COND FLAG_Z,bt,_read_byte13


! CALL Z,nn
op_CC:
	CALL_COND FLAG_Z,bt,_read_byte13,_write_byte13


! CALL nn
op_CD:
	MOVE_SP -2
	extu.w	r10,r5
	add	#3,r5
	mov	r5,r1
	mov.l	_write_byte13,r2
	jsr	@r2		! Push PC (low)
	extu.w	r11,r0		
	mov	r5,r1
	mov.l	_write_byte13,r2
	extu.w	r11,r0
	shlr8	r1
	jsr	@r2		! Push PC (high)
	add	#1,r0
	extu.w	r10,r0
	READ_BYTE
	add	#1,r0
	mov	r0,r5
	extu.w	r10,r0
	add	#2,r0
	READ_BYTE
	extu.b	r5,r10
	shll8	r0
	or	r0,r10
	RETURN
	ADD_CYCLES 24


! ADC A,n
op_CE:
	MOVE_PC	1
	READ_BYTE
	extu.w	r10,r0
	extu.b	r0,r1
	extu.b	r8,r0
	xor	#FLAG_C,r0	! tst sets T if (op1&op2)==0, so invert the bit to test for
	tst	#FLAG_C,r0
	extu.b	r7,r0		! r0 = A
	addc	r1,r0		! r0 = A + reg + F.Carry
	mov 	r0,r2
	extu.b	r0,r0
	cmp/eq	#0,r0
	movt 	r8
	shll8 	r8
	shlr 	r8		! Zero
	xor	r7,r1
	shlr2	r2
	extu.b	r0,r7		
	xor	r1,r0
	shlr2	r2
	shll	r0
	and	#FLAG_H,r0
	or	r0,r8
	mov	r2,r0
	and	#FLAG_C,r0
	or	r0,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
	

			
! RST 0x08
op_CF:
	RST 0x08,_write_byte13,_cpu_halted13
			
.align 2
_adrlim2: .long 0x2000
_cpu_halted13:
!	.long cpu_halted
_read_byte13:
	.long _read_byte
_write_byte13:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte - __MEM_COPY_TO_IWRAM_START__

! ###########################################################################################################

! RET NC
op_D0:
	RET_COND FLAG_C,bf,_read_byte14
	

! POP DE
op_D1:
	POP_REG16 REG_D,REG_E,_read_byte14


! JP NC,nn
op_D2:
	JP_COND FLAG_C,bf,_read_byte14
	

op_D3:
	ILLEGAL_OP
	
	
! CALL NC,nn
op_D4:
	CALL_COND FLAG_C,bf,_read_byte14,_write_byte14


! PUSH DE
op_D5:
	PUSH_REG16 REG_D,REG_E,_write_byte14


! SUB A,n
op_D6:
	MOVE_PC	1
	READ_BYTE
	extu.w	r10,r0
	extu.b	r0,r1
	extu.b	r7,r0		! r0 = A
	sub	r1,r0		! r0 = A - n
	extu.b 	r0,r0
	cmp/eq	#0,r0
	bf/s	_op_D6_NZ
	mov	#FLAG_N,r8	! clear F in the delayed branch slot
	add	#FLAG_Z,r8	! Set F.Zero if (A-n)==0
_op_D6_NZ:
	cmp/hs	r1,r7
	bt/s	_op_D6_NC
	xor	r7,r1
	add	#FLAG_C,r8	! Set F.Carry if A<n
_op_D6_NC:
	extu.b	r0,r7		! A = A-n
	xor	r1,r0
	shll	r0
	and	#FLAG_H,r0
	or	r0,r8
	extu.b	r8,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
	

! RST 0x10
op_D7:
	RST 0x10,_write_byte14,_cpu_halted14
	

! RET C
op_D8:
	RET_COND FLAG_C,bt,_read_byte14


! RETI
op_D9:
	mov	#CPU_EI_PENDING,r0
	or	r0,r13
	extu.w	r11,r0
	READ_BYTE
	MOVE_SP	1
	mov	r0,r10
	extu.w	r11,r0
	READ_BYTE
	MOVE_SP	1
	shll8	r0
	or	r0,r10
	RETURN
	ADD_CYCLES 16


! JP C,nn
op_DA:
	JP_COND FLAG_C,bt,_read_byte14

op_DB:
	ILLEGAL_OP

! CALL C,nn
op_DC:
	CALL_COND FLAG_C,bt,_read_byte14,_write_byte14
	
op_DD:
	ILLEGAL_OP
	
! SBC A,n
op_DE:
	MOVE_PC	1
	READ_BYTE
	extu.w	r10,r0
	extu.b	r0,r1
	extu.b	r8,r0
	xor	#FLAG_C,r0	! tst sets T if (op1&op2)==0, so invert the bit to test for
	tst	#FLAG_C,r0
	extu.b	r7,r0		! r0 = A
	subc	r1,r0		! r0 = A - n - F.Carry
extu.b r0,r0
	cmp/eq	#0,r0
	bf/s	_op_DE_NZ
	mov	#FLAG_N,r8	! clear F in the delayed branch slot
	add	#FLAG_Z,r8	
_op_DE_NZ:
	cmp/hs	r1,r7
	bt	_op_DE_NC
	add	#FLAG_C,r8	
_op_DE_NC:
	extu.b	r0,r7		
	xor	r1,r0
	shll2	r0
	and	#FLAG_H,r0
	or	r0,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
	

! RST 0x18
op_DF:
	RST 0x18,_write_byte14,_cpu_halted14			
			
.align 2
_cpu_halted14:
!	.long cpu_halted
_read_byte14:
	.long _read_byte
_write_byte14:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte - __MEM_COPY_TO_IWRAM_START__

! ###########################################################################################################

			
! POP HL
op_E1:
	POP_REG16 REG_H,REG_L,_read_byte15
	

! LD (C),A
op_E2:
	mov.b	@(REG_C,gbr),r0
	mov	#0xFF,r1
	extu.b	r0,r0
	extu.b	r1,r1
	shll8	r1
	mov.l	_write_byte15,r2
	or	r1,r0
	jsr	@r2
	extu.b	r7,r1
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
	

op_E3:
op_E4:
	ILLEGAL_OP
	
			
! PUSH HL
op_E5:
	PUSH_REG16 REG_H,REG_L,_write_byte15


! AND A,n
op_E6:
	MOVE_PC	1
	READ_BYTE
	extu.w	r10,r0
	and	r0,r7
	extu.b	r7,r7
	tst	r7,r7
	movt	r8
	shll8	r8
	shlr	r8	! Zero
	add	#FLAG_H,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
		
			
! RST 0x20
op_E7:
	RST 0x20,_write_byte15,_cpu_halted15			


! ADD SP,n
op_E8:
	MOVE_PC	1
	READ_BYTE
	extu.w	r10,r0
	exts.b	r0,r0
	add	r0,r11
	mov	#FLAG_C+FLAG_H,r1
	and	r1,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 16
	

! JP (HL)
op_E9:
	mov.w	@(REG_H,gbr),r0
	extu.w	r0,r10
	RETURN
	ADD_CYCLES 4


! LD (nn),A
op_EA:
!RETURN
!nop
	MOVE_PC 1
 	mov	r10,r0
	READ_BYTE
 	MOVE_PC 1
 	mov	r0,r5	! Save low byte in r5
 	mov	r10,r0
 	READ_BYTE
 	MOVE_PC 1
 	mov.l	_write_byte15,r2
 	shll8	r0
	mov	r7,r1
 	jsr	@r2
  	or	r5,r0
 	RETURN
 	ADD_CYCLES 16
 	

op_EB:
op_EC:
op_ED:
	ILLEGAL_OP
	

! XOR A,n
op_EE:
	MOVE_PC	1
	READ_BYTE
	extu.w	r10,r0	! delayed branch slot
	xor	r0,r7
	tst 	r7,r7
	movt 	r8
	shll8 	r8
	MOVE_PC 1
	shlr 	r8	! Zero
	RETURN
	ADD_CYCLES 8

			
! RST 0x28
op_EF:
	RST 0x28,_write_byte15,_cpu_halted15			

.align 2
_cpu_halted15:
!	.long cpu_halted
_read_byte15:
	.long _read_byte
_write_byte15:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte - __MEM_COPY_TO_IWRAM_START__
	
! ###########################################################################################################


! POP AF
op_F1:
	READ_BYTE
	extu.w	r11,r0
	extu.b	r0,r8
	MOVE_SP	1
	READ_BYTE
	extu.w	r11,r0
	extu.b	r0,r7
	MOVE_SP	1
	MOVE_PC	1
	RETURN
	ADD_CYCLES 12
	

! LD A,(C)
op_F2:
	mov.b	@(REG_C,gbr),r0
	mov	#0xFF,r1
	extu.b	r0,r0
	extu.b	r1,r1
	shll8	r1
	READ_BYTE
	or	r1,r0
	extu.b	r0,r7
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8

			
! DI
op_F3:
	mov	#CPU_DI_PENDING,r0
	or	r0,r13
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4


op_F4:
	ILLEGAL_OP
	

! PUSH AF
op_F5:
	MOVE_SP	-1
	mov.l	_write_byte16,r2
	extu.b	r7,r1
	jsr	@r2
	extu.w	r11,r0
	MOVE_SP	-1
	mov.l	_write_byte16,r2
!	extu.b	r8,r1
extu.b r8,r0
and #0xF0,r0
extu.b r0,r1
	jsr	@r2
	extu.w	r11,r0
	MOVE_PC	1
	RETURN
	ADD_CYCLES 16


! OR A,n
op_F6:
	MOVE_PC	1
	READ_BYTE
	extu.w	r10,r0
	or	r0,r7
	mov	#0,r8
	extu.b	r7,r0
	cmp/eq	#0,r0
	bf/s	_op_F6_NZ
	MOVE_PC	1
	add	#FLAG_Z,r8
_op_F6_NZ:
	extu.b	r8,r8
	RETURN
	ADD_CYCLES 8
	


! RST 0x30
op_F7:
	RST 0x30,_write_byte16,_cpu_halted16			


! LD HL,SP+n
op_F8:
	MOVE_PC	1
	READ_BYTE
	extu.w	r10,r0
	exts.b	r0,r0
	add	r11,r0
	mov.b	r0,@(REG_L,gbr)
	mov	#FLAG_C+FLAG_H,r1
	shlr8	r0
	and	r1,r8
	mov.b	r0,@(REG_H,gbr)
	MOVE_PC	1
	RETURN
	ADD_CYCLES 12


! LD SP,HL
op_F9:
	mov.w	@(REG_H,gbr),r0
	extu.w	r0,r11
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
	

! LD A,(nn)
op_FA:
	MOVE_PC	1
	READ_BYTE
	extu.w	r10,r0
	extu.b	r0,r5
	MOVE_PC	1
	READ_BYTE
	extu.w	r10,r0
	extu.b	r0,r0
	shll8	r0
	READ_BYTE
	or	r5,r0
	extu.b	r0,r7
	MOVE_PC	1
	RETURN
	ADD_CYCLES 16
	
				
! EI
op_FB:
	mov	#CPU_EI_PENDING,r0
	or	r0,r13
	MOVE_PC	1
	RETURN
	ADD_CYCLES 4

op_FC:
op_FD:
	ILLEGAL_OP
	

	
! RST 0x38
op_FF:
	RST 0x38,_write_byte16,_cpu_halted16			
			
.align 2
_cpu_halted16:
!	.long cpu_halted
_read_byte16:
	.long _read_byte
_write_byte16:
	.long MEM_CACHE_COPY_ADR + 0x34 !_mem_write_byte - __MEM_COPY_TO_IWRAM_START__
	
! ###########################################################################################################

			
				
! RLC r
op_CB_00:
	RLC_REG8 REG_B
op_CB_01:
	RLC_REG8 REG_C
op_CB_02:
	RLC_REG8 REG_D
op_CB_03:
	RLC_REG8 REG_E
op_CB_04:
	RLC_REG8 REG_H
op_CB_05:
	RLC_REG8 REG_L
op_CB_06:
	mov.w	@(REG_H,gbr),r0
	extu.w	r0,r5
	READ_BYTE
	mov	r5,r0
	extu.b	r0,r0
	shll	r0
	swap.b	r0,r1	! Move bit 8 of r0 (Carry) into bit 0 of r1
	or	r1,r0
	extu.b	r0,r3		
	extu.b	r0,r1
	tst	r1,r1
	movt	r8
	shll8	r8
	MOVE_PC	1
	shlr	r8	! Zero
	shlr2	r0
	shlr2	r0
	and	#FLAG_C,r0
	or	r0,r8	! Carry
	mov.l	_write_byteCB0,r2
	mov	r5,r0
	jsr	@r2
	mov	r3,r1
	RETURN
	ADD_CYCLES 16

op_CB_07:
	extu.b	r7,r0
	shll	r0
	swap.b	r0,r1	! Move bit 8 of r0 (Carry) into bit 0 of r1
	or	r1,r0
	extu.b	r0,r7
	extu.b	r0,r1
	tst	r1,r1
	movt	r8
	shll8	r8
	MOVE_PC	1
	shlr	r8	! Zero
	shlr2	r0
	shlr2	r0
	and	#FLAG_C,r0
	or	r0,r8	! Carry
	RETURN
	ADD_CYCLES 8


! RRC r
op_CB_08:
	RRC_REG8 REG_B
op_CB_09:
	RRC_REG8 REG_C
op_CB_0A:
	RRC_REG8 REG_D
op_CB_0B:
	RRC_REG8 REG_E
op_CB_0C:
	RRC_REG8 REG_H
op_CB_0D:
	RRC_REG8 REG_L
op_CB_0E:
	mov.w	@(REG_H,gbr),r0
	extu.w	r0,r5
	READ_BYTE
	mov	r5,r0
	extu.b	r0,r1
	shll8	r1
	shlr	r1
	shlr	r0
	or	r1,r0	! r0 = (reg>>1)|(reg<<7)
	movt	r8
	extu.b	r0,r3
	shll2	r8
	shll2	r8	! Carry
	extu.b	r0,r0
	tst	r0,r0
	movt	r1
	MOVE_PC	1
	shll8	r1
	shlr	r1
	or	r1,r8	! Zero
	mov.l	_write_byteCB0,r2
	mov	r5,r0
	jsr	@r2
	mov	r3,r1
	RETURN
	ADD_CYCLES 16

op_CB_0F:
	extu.b	r7,r0
	extu.b	r0,r1
	shll8	r1
	shlr	r1
	shlr	r0
	or	r1,r0	! r0 = (reg>>1)|(reg<<7)
	movt	r8
	extu.b	r0,r7
	shll2	r8
	shll2	r8	! Carry
	extu.b	r0,r0
	tst	r0,r0
	movt	r1
	MOVE_PC	1
	shll8	r1
	shlr	r1
	or	r1,r8	! Zero
	RETURN
	ADD_CYCLES 8

.align 2
_write_byteCB0:
	.long _mem_write_byte
	
! ###########################################################################################################

! RL r
op_CB_10:
	RL_REG8 REG_B
op_CB_11:
	RL_REG8 REG_C
op_CB_12:
	RL_REG8 REG_D
op_CB_13:
	RL_REG8 REG_E
op_CB_14:
	RL_REG8 REG_H
op_CB_15:
	RL_REG8 REG_L
op_CB_16:
	mov.w	@(REG_H,gbr),r0
	extu.w	r0,r5
	READ_BYTE
	mov	r5,r0
	extu.b	r0,r1
	extu.b	r8,r0
	xor	#FLAG_C,r0
	tst	#FLAG_C,r0	! T=(F.Carry?1:0)
	rotcl	r1
	mov	r1,r0
	extu.b	r0,r3
	extu.b	r0,r1
	tst	r1,r1
	movt	r8
	shll8	r8
	MOVE_PC	1
	shlr	r8	! Zero
	shlr2	r0
	shlr2	r0
	and	#FLAG_C,r0
	or	r0,r8	! Carry
	mov.l	_write_byteCB1,r2
	mov	r5,r0
	jsr	@r2
	mov	r3,r1
	RETURN
	ADD_CYCLES 16	
op_CB_17:
	extu.b	r7,r1
	extu.b	r8,r0
	xor	#FLAG_C,r0
	tst	#FLAG_C,r0	! T=(F.Carry?1:0)
	rotcl	r1
	mov	r1,r0
	extu.b	r0,r7
	extu.b	r0,r1
	tst	r1,r1
	movt	r8
	shll8	r8
	MOVE_PC	1
	shlr	r8	! Zero
	shlr2	r0
	shlr2	r0
	and	#FLAG_C,r0
	or	r0,r8	! Carry
	RETURN
	ADD_CYCLES 8	


! RR r
op_CB_18:
	RR_REG8 REG_B
op_CB_19:
	RR_REG8 REG_C
op_CB_1A:
	RR_REG8 REG_D
op_CB_1B:
	RR_REG8 REG_E
op_CB_1C:
	RR_REG8 REG_H
op_CB_1D:
	RR_REG8 REG_L
op_CB_1E:
	mov.w	@(REG_H,gbr),r0
	extu.w	r0,r5
	READ_BYTE
	mov	r5,r0
	extu.b	r0,r1
	extu.b	r8,r0
	and	#FLAG_C,r0
	shll2	r0
	shll	r0
	shlr	r1
	movt	r8
	shll2	r8
	shll2	r8	! Carry
	or	r1,r0
	extu.b	r0,r3
	MOVE_PC 1
	tst	r0,r0
	movt	r1
	shll8	r1
	shlr	r1
	or	r1,r8	! Zero
	mov.l	_write_byteCB1,r2
	mov	r5,r0
	jsr	@r2
	mov	r3,r1
	RETURN
	ADD_CYCLES 16

op_CB_1F:
	extu.b	r7,r1
	extu.b	r8,r0
	and	#FLAG_C,r0
	shll2	r0
	shll	r0
	shlr	r1
	movt	r8
	shll2	r8
	shll2	r8	! Carry
	or	r1,r0
	extu.b	r0,r7
	MOVE_PC 1
	tst	r0,r0
	movt	r1
	shll8	r1
	shlr	r1
	or	r1,r8	! Zero
	RETURN
	ADD_CYCLES 8
	

.align 2
_write_byteCB1:
	.long _mem_write_byte
	
! ###########################################################################################################

! SLA r
op_CB_20:
	SLA_REG8 REG_B
op_CB_21:
	SLA_REG8 REG_C
op_CB_22:
	SLA_REG8 REG_D
op_CB_23:
	SLA_REG8 REG_E
op_CB_24:
	SLA_REG8 REG_H
op_CB_25:
	SLA_REG8 REG_L
op_CB_26:
	mov.w	@(REG_H,gbr),r0
	extu.w	r0,r5
	READ_BYTE
	mov	r5,r0
	shll	r0
	extu.b	r0,r3
	extu.b	r0,r1
	tst	r1,r1
	movt	r8
	shll8	r8
	MOVE_PC	1
	shlr	r8	! Zero
	shlr2	r0
	shlr2	r0
	and	#FLAG_C,r0
	or	r0,r8	! Carry
	mov.l	_write_byteCB2,r2
	mov	r5,r0
	jsr	@r2
	mov	r3,r1
	RETURN
	ADD_CYCLES 16
op_CB_27:
	shll	r7
	mov	r7,r0
	extu.b	r7,r1
	tst	r1,r1
	movt	r8
	shll8	r8
	MOVE_PC	1
	shlr	r8	! Zero
	shlr2	r0
	shlr2	r0
	and	#FLAG_C,r0
	or	r0,r8	! Carry
	RETURN
	ADD_CYCLES 8

! SRA r
op_CB_28:
	SRA_REG8 REG_B
op_CB_29:
	SRA_REG8 REG_C
op_CB_2A:
	SRA_REG8 REG_D
op_CB_2B:
	SRA_REG8 REG_E
op_CB_2C:
	SRA_REG8 REG_H
op_CB_2D:
	SRA_REG8 REG_L
op_CB_2E:
	mov.w	@(REG_H,gbr),r0
	extu.w	r0,r5
	READ_BYTE
	mov	r5,r0
	exts.b	r0,r0
	shlr	r0
	movt	r8
	extu.b	r0,r3
	shll2	r8
	shll2	r8	! Carry
	tst	r3,r3
	movt	r1
	MOVE_PC	1
	shll8	r1
	shlr	r1
	or	r1,r8	! Zero
	mov.l	_write_byteCB2,r2
	mov	r5,r0
	jsr	@r2
	mov	r3,r1
	RETURN
	ADD_CYCLES 8

! SRA A
op_CB_2F:
	exts.b	r7,r7	
	shlr	r7
	movt	r8
	shll2	r8
	shll2	r8	! Carry
	extu.b	r7,r7
	tst	r7,r7
	movt	r1
	MOVE_PC	1
	shll8	r1
	shlr	r1
	or	r1,r8	! Zero
	RETURN
	ADD_CYCLES 8

.align 2
_write_byteCB2:
	.long _mem_write_byte
	
! ###########################################################################################################

! SWAP r
op_CB_30:
	SWAP_REG8 REG_B
op_CB_31:
	SWAP_REG8 REG_C
op_CB_32:
	SWAP_REG8 REG_D
op_CB_33:
	SWAP_REG8 REG_E
op_CB_34:
	SWAP_REG8 REG_H
op_CB_35:
	SWAP_REG8 REG_L
op_CB_36:
	mov.w	@(REG_H,gbr),r0
	extu.w	r0,r5
	READ_BYTE
	mov	r5,r0
	mov	r0,r1
	shlr2	r0
	shlr2	r0
	shll2	r1
	shll2	r1
	extu.b	r1,r1
	and	#0x0F,r0
	or	r1,r0
	MOVE_PC 1
	extu.b	r0,r3
	tst	r0,r0
	movt	r8
	shll8	r8
	shlr	r8	! Zero
	mov.l	_write_byteCB3,r2
	mov	r5,r0
	jsr	@r2
	mov	r3,r1
	RETURN
	ADD_CYCLES 16

op_CB_37:
	extu.b	r7,r0
	mov	r0,r1
	shlr2	r0
	shlr2	r0
	shll2	r1
	shll2	r1
	extu.b	r1,r1
	and	#0x0F,r0
	or	r1,r0
	MOVE_PC 1
	extu.b	r0,r7
	tst	r0,r0
	movt	r8
	shll8	r8
	shlr	r8	! Zero
	RETURN
	ADD_CYCLES 8

! SRL r
op_CB_38:
	SRL_REG8 REG_B
op_CB_39:
	SRL_REG8 REG_C
op_CB_3A:
	SRL_REG8 REG_D
op_CB_3B:
	SRL_REG8 REG_E
op_CB_3C:
	SRL_REG8 REG_H
op_CB_3D:
	SRL_REG8 REG_L
op_CB_3E:
	mov.w	@(REG_H,gbr),r0
	extu.w	r0,r5
	READ_BYTE
	mov	r5,r0
	shlr	r0
	extu.b	r0,r3
	movt	r8
	shll2	r8
	shll2	r8	! Carry
	extu.b	r0,r0
	tst	r0,r0
	movt	r1
	MOVE_PC	1
	shll8	r1
	shlr	r1
	or	r1,r8	! Zero
	mov.l	_write_byteCB3,r2
	mov	r5,r0
	jsr	@r2
	mov	r3,r1
	RETURN
	ADD_CYCLES 16

op_CB_3F:
	shlr	r7
	movt	r8
	shll2	r8
	shll2	r8	! Carry
	extu.b	r7,r7
	tst	r7,r7
	movt	r1
	MOVE_PC	1
	shll8	r1
	shlr	r1
	or	r1,r8	! Zero
	RETURN
	ADD_CYCLES 8

.align 2
_write_byteCB3:
	.long _mem_write_byte
	
! ###########################################################################################################

! BIT 0,r
op_CB_40:
	BIT_REG8 REG_B,0x01
op_CB_41:
	BIT_REG8 REG_C,0x01
op_CB_42:
	BIT_REG8 REG_D,0x01
op_CB_43:
	BIT_REG8 REG_E,0x01
op_CB_44:
	BIT_REG8 REG_H,0x01
op_CB_45:
	BIT_REG8 REG_L,0x01
op_CB_46:
	BIT_ATHL 0x01
op_CB_47:
	BIT_A 0x01

! BIT 1,r
op_CB_48:
	BIT_REG8 REG_B,0x02
op_CB_49:
	BIT_REG8 REG_C,0x02
op_CB_4A:
	BIT_REG8 REG_D,0x02
op_CB_4B:
	BIT_REG8 REG_E,0x02
op_CB_4C:
	BIT_REG8 REG_H,0x02
op_CB_4D:
	BIT_REG8 REG_L,0x02
op_CB_4E:
	BIT_ATHL 0x02
op_CB_4F:
	BIT_A 0x02	

! ###########################################################################################################


! BIT 2,r
op_CB_50:
	BIT_REG8 REG_B,0x04
op_CB_51:
	BIT_REG8 REG_C,0x04
op_CB_52:
	BIT_REG8 REG_D,0x04
op_CB_53:
	BIT_REG8 REG_E,0x04
op_CB_54:
	BIT_REG8 REG_H,0x04
op_CB_55:
	BIT_REG8 REG_L,0x04
op_CB_56:
	BIT_ATHL 0x04
op_CB_57:
	BIT_A 0x04

! BIT 3,r
op_CB_58:
	BIT_REG8 REG_B,0x08
op_CB_59:
	BIT_REG8 REG_C,0x08
op_CB_5A:
	BIT_REG8 REG_D,0x08
op_CB_5B:
	BIT_REG8 REG_E,0x08
op_CB_5C:
	BIT_REG8 REG_H,0x08
op_CB_5D:
	BIT_REG8 REG_L,0x08
op_CB_5E:
	BIT_ATHL 0x08
op_CB_5F:
	BIT_A 0x08	
	
! ###########################################################################################################

! BIT 4,r
op_CB_60:
	BIT_REG8 REG_B,0x10
op_CB_61:
	BIT_REG8 REG_C,0x10
op_CB_62:
	BIT_REG8 REG_D,0x10
op_CB_63:
	BIT_REG8 REG_E,0x10
op_CB_64:
	BIT_REG8 REG_H,0x10
op_CB_65:
	BIT_REG8 REG_L,0x10
op_CB_66:
	BIT_ATHL 0x10
op_CB_67:
	BIT_A 0x10

! BIT 5,r
op_CB_68:
	BIT_REG8 REG_B,0x20
op_CB_69:
	BIT_REG8 REG_C,0x20
op_CB_6A:
	BIT_REG8 REG_D,0x20
op_CB_6B:
	BIT_REG8 REG_E,0x20
op_CB_6C:
	BIT_REG8 REG_H,0x20
op_CB_6D:
	BIT_REG8 REG_L,0x20
op_CB_6E:
	BIT_ATHL 0x20
op_CB_6F:
	BIT_A 0x20	

! ###########################################################################################################

! BIT 6,r
op_CB_70:
	BIT_REG8 REG_B,0x40
op_CB_71:
	BIT_REG8 REG_C,0x40
op_CB_72:
	BIT_REG8 REG_D,0x40
op_CB_73:
	BIT_REG8 REG_E,0x40
op_CB_74:
	BIT_REG8 REG_H,0x40
op_CB_75:
	BIT_REG8 REG_L,0x40
op_CB_76:
	BIT_ATHL 0x40
op_CB_77:
	BIT_A 0x40

! BIT 7,r
op_CB_78:
	BIT_REG8 REG_B,0x80
op_CB_79:
	BIT_REG8 REG_C,0x80
op_CB_7A:
	BIT_REG8 REG_D,0x80
op_CB_7B:
	BIT_REG8 REG_E,0x80
op_CB_7C:
	BIT_REG8 REG_H,0x80
op_CB_7D:
	BIT_REG8 REG_L,0x80
op_CB_7E:
	BIT_ATHL 0x80
op_CB_7F:
	BIT_A 0x80	

! ###########################################################################################################

! RES 0,r
op_CB_80:
	RES_REG8 REG_B,0xFE
op_CB_81:
	RES_REG8 REG_C,0xFE
op_CB_82:
	RES_REG8 REG_D,0xFE
op_CB_83:
	RES_REG8 REG_E,0xFE
op_CB_84:
	RES_REG8 REG_H,0xFE
op_CB_85:
	RES_REG8 REG_L,0xFE
op_CB_86:
	RES_ATHL 0xFE,_write_byteCB8
op_CB_87:
	RES_A 0xFE

! RES 1,r
op_CB_88:
	RES_REG8 REG_B,0xFD
op_CB_89:
	RES_REG8 REG_C,0xFD
op_CB_8A:
	RES_REG8 REG_D,0xFD
op_CB_8B:
	RES_REG8 REG_E,0xFD
op_CB_8C:
	RES_REG8 REG_H,0xFD
op_CB_8D:
	RES_REG8 REG_L,0xFD
op_CB_8E:
	RES_ATHL 0xFD,_write_byteCB8
op_CB_8F:
	RES_A 0xFD

.align 2
_write_byteCB8:
	.long _mem_write_byte
	
! ###########################################################################################################

! RES 2,r
op_CB_90:
	RES_REG8 REG_B,0xFB
op_CB_91:
	RES_REG8 REG_C,0xFB
op_CB_92:
	RES_REG8 REG_D,0xFB
op_CB_93:
	RES_REG8 REG_E,0xFB
op_CB_94:
	RES_REG8 REG_H,0xFB
op_CB_95:
	RES_REG8 REG_L,0xFB
op_CB_96:
	RES_ATHL 0xFB,_write_byteCB9
op_CB_97:
	RES_A 0xFB

! RES 3,r
op_CB_98:
	RES_REG8 REG_B,0xF7
op_CB_99:
	RES_REG8 REG_C,0xF7
op_CB_9A:
	RES_REG8 REG_D,0xF7
op_CB_9B:
	RES_REG8 REG_E,0xF7
op_CB_9C:
	RES_REG8 REG_H,0xF7
op_CB_9D:
	RES_REG8 REG_L,0xF7
op_CB_9E:
	RES_ATHL 0xF7,_write_byteCB9
op_CB_9F:
	RES_A 0xF7

.align 2
_write_byteCB9:
	.long _mem_write_byte
	
! ###########################################################################################################

! RES 4,r
op_CB_A0:
	RES_REG8 REG_B,0xEF
op_CB_A1:
	RES_REG8 REG_C,0xEF
op_CB_A2:
	RES_REG8 REG_D,0xEF
op_CB_A3:
	RES_REG8 REG_E,0xEF
op_CB_A4:
	RES_REG8 REG_H,0xEF
op_CB_A5:
	RES_REG8 REG_L,0xEF
op_CB_A6:
	RES_ATHL 0xEF,_write_byteCBA
op_CB_A7:
	RES_A 0xEF

! RES 5,r
op_CB_A8:
	RES_REG8 REG_B,0xDF
op_CB_A9:
	RES_REG8 REG_C,0xDF
op_CB_AA:
	RES_REG8 REG_D,0xDF
op_CB_AB:
	RES_REG8 REG_E,0xDF
op_CB_AC:
	RES_REG8 REG_H,0xDF
op_CB_AD:
	RES_REG8 REG_L,0xDF
op_CB_AE:
	RES_ATHL 0xDF,_write_byteCBA
op_CB_AF:
	RES_A 0xDF

.align 2
_write_byteCBA:
	.long _mem_write_byte
	
! ###########################################################################################################

! RES 6,r
op_CB_B0:
	RES_REG8 REG_B,0xBF
op_CB_B1:
	RES_REG8 REG_C,0xBF
op_CB_B2:
	RES_REG8 REG_D,0xBF
op_CB_B3:
	RES_REG8 REG_E,0xBF
op_CB_B4:
	RES_REG8 REG_H,0xBF
op_CB_B5:
	RES_REG8 REG_L,0xBF
op_CB_B6:
	RES_ATHL 0xBF,_write_byteCBB
op_CB_B7:
	RES_A 0xBF

! RES 7,r
op_CB_B8:
	RES_REG8 REG_B,0x7F
op_CB_B9:
	RES_REG8 REG_C,0x7F
op_CB_BA:
	RES_REG8 REG_D,0x7F
op_CB_BB:
	RES_REG8 REG_E,0x7F
op_CB_BC:
	RES_REG8 REG_H,0x7F
op_CB_BD:
	RES_REG8 REG_L,0x7F
op_CB_BE:
	RES_ATHL 0x7F,_write_byteCBB
op_CB_BF:
	RES_A 0x7F

.align 2
_write_byteCBB:
	.long _mem_write_byte
	
! ###########################################################################################################

! SET 0,r
op_CB_C0:
	SET_REG8 REG_B,0x01
op_CB_C1:
	SET_REG8 REG_C,0x01
op_CB_C2:
	SET_REG8 REG_D,0x01
op_CB_C3:
	SET_REG8 REG_E,0x01
op_CB_C4:
	SET_REG8 REG_H,0x01
op_CB_C5:
	SET_REG8 REG_L,0x01
op_CB_C6:
	SET_ATHL 0x01,_write_byteCBC
op_CB_C7:
	SET_A 0x01

! SET 1,r
op_CB_C8:
	SET_REG8 REG_B,0x02
op_CB_C9:
	SET_REG8 REG_C,0x02
op_CB_CA:
	SET_REG8 REG_D,0x02
op_CB_CB:
	SET_REG8 REG_E,0x02
op_CB_CC:
	SET_REG8 REG_H,0x02
op_CB_CD:
	SET_REG8 REG_L,0x02
op_CB_CE:
	SET_ATHL 0x02,_write_byteCBC
op_CB_CF:
	SET_A 0x02

.align 2
_write_byteCBC:
	.long _mem_write_byte
	
! ###########################################################################################################

! SET 2,r
op_CB_D0:
	SET_REG8 REG_B,0x04
op_CB_D1:
	SET_REG8 REG_C,0x04
op_CB_D2:
	SET_REG8 REG_D,0x04
op_CB_D3:
	SET_REG8 REG_E,0x04
op_CB_D4:
	SET_REG8 REG_H,0x04
op_CB_D5:
	SET_REG8 REG_L,0x04
op_CB_D6:
	SET_ATHL 0x04,_write_byteCBD
op_CB_D7:
	SET_A 0x04

! SET 3,r
op_CB_D8:
	SET_REG8 REG_B,0x08
op_CB_D9:
	SET_REG8 REG_C,0x08
op_CB_DA:
	SET_REG8 REG_D,0x08
op_CB_DB:
	SET_REG8 REG_E,0x08
op_CB_DC:
	SET_REG8 REG_H,0x08
op_CB_DD:
	SET_REG8 REG_L,0x08
op_CB_DE:
	SET_ATHL 0x08,_write_byteCBD
op_CB_DF:
	SET_A 0x08

.align 2
_write_byteCBD:
	.long _mem_write_byte
	
! ###########################################################################################################

! SET 4,r
op_CB_E0:
	SET_REG8 REG_B,0x10
op_CB_E1:
	SET_REG8 REG_C,0x10
op_CB_E2:
	SET_REG8 REG_D,0x10
op_CB_E3:
	SET_REG8 REG_E,0x10
op_CB_E4:
	SET_REG8 REG_H,0x10
op_CB_E5:
	SET_REG8 REG_L,0x10
op_CB_E6:
	SET_ATHL 0x10,_write_byteCBE
op_CB_E7:
	SET_A 0x10

! SET 5,r
op_CB_E8:
	SET_REG8 REG_B,0x20
op_CB_E9:
	SET_REG8 REG_C,0x20
op_CB_EA:
	SET_REG8 REG_D,0x20
op_CB_EB:
	SET_REG8 REG_E,0x20
op_CB_EC:
	SET_REG8 REG_H,0x20
op_CB_ED:
	SET_REG8 REG_L,0x20
op_CB_EE:
	SET_ATHL 0x20,_write_byteCBE
op_CB_EF:
	SET_A 0x20

.align 2
_write_byteCBE:
	.long _mem_write_byte
	
! ###########################################################################################################

! SET 6,r
op_CB_F0:
	SET_REG8 REG_B,0x40
op_CB_F1:
	SET_REG8 REG_C,0x40
op_CB_F2:
	SET_REG8 REG_D,0x40
op_CB_F3:
	SET_REG8 REG_E,0x40
op_CB_F4:
	SET_REG8 REG_H,0x40
op_CB_F5:
	SET_REG8 REG_L,0x40
op_CB_F6:
	SET_ATHL 0x40,_write_byteCBF
op_CB_F7:
	SET_A 0x40

! SET 7,r
op_CB_F8:
	SET_REG8 REG_B,0x80
op_CB_F9:
	SET_REG8 REG_C,0x80
op_CB_FA:
	SET_REG8 REG_D,0x80
op_CB_FB:
	SET_REG8 REG_E,0x80
op_CB_FC:
	SET_REG8 REG_H,0x80
op_CB_FD:
	SET_REG8 REG_L,0x80
op_CB_FE:
	SET_ATHL 0x80,_write_byteCBF
op_CB_FF:
	SET_A 0x80

.align 2
_write_byteCBF:
	.long _mem_write_byte
	
! ###########################################################################################################

.align 2

! Jump table for CB prefix instruction handlers
opcode_cb_table:
	.long op_CB_00,op_CB_01,op_CB_02,op_CB_03,op_CB_04,op_CB_05,op_CB_06,op_CB_07
	.long op_CB_08,op_CB_09,op_CB_0A,op_CB_0B,op_CB_0C,op_CB_0D,op_CB_0E,op_CB_0F
	.long op_CB_10,op_CB_11,op_CB_12,op_CB_13,op_CB_14,op_CB_15,op_CB_16,op_CB_17
	.long op_CB_18,op_CB_19,op_CB_1A,op_CB_1B,op_CB_1C,op_CB_1D,op_CB_1E,op_CB_1F
	.long op_CB_20,op_CB_21,op_CB_22,op_CB_23,op_CB_24,op_CB_25,op_CB_26,op_CB_27
	.long op_CB_28,op_CB_29,op_CB_2A,op_CB_2B,op_CB_2C,op_CB_2D,op_CB_2E,op_CB_2F
	.long op_CB_30,op_CB_31,op_CB_32,op_CB_33,op_CB_34,op_CB_35,op_CB_36,op_CB_37
	.long op_CB_38,op_CB_39,op_CB_3A,op_CB_3B,op_CB_3C,op_CB_3D,op_CB_3E,op_CB_3F
	.long op_CB_40,op_CB_41,op_CB_42,op_CB_43,op_CB_44,op_CB_45,op_CB_46,op_CB_47
	.long op_CB_48,op_CB_49,op_CB_4A,op_CB_4B,op_CB_4C,op_CB_4D,op_CB_4E,op_CB_4F
	.long op_CB_50,op_CB_51,op_CB_52,op_CB_53,op_CB_54,op_CB_55,op_CB_56,op_CB_57
	.long op_CB_58,op_CB_59,op_CB_5A,op_CB_5B,op_CB_5C,op_CB_5D,op_CB_5E,op_CB_5F
	.long op_CB_60,op_CB_61,op_CB_62,op_CB_63,op_CB_64,op_CB_65,op_CB_66,op_CB_67
	.long op_CB_68,op_CB_69,op_CB_6A,op_CB_6B,op_CB_6C,op_CB_6D,op_CB_6E,op_CB_6F
	.long op_CB_70,op_CB_71,op_CB_72,op_CB_73,op_CB_74,op_CB_75,op_CB_76,op_CB_77
	.long op_CB_78,op_CB_79,op_CB_7A,op_CB_7B,op_CB_7C,op_CB_7D,op_CB_7E,op_CB_7F
	.long op_CB_80,op_CB_81,op_CB_82,op_CB_83,op_CB_84,op_CB_85,op_CB_86,op_CB_87
	.long op_CB_88,op_CB_89,op_CB_8A,op_CB_8B,op_CB_8C,op_CB_8D,op_CB_8E,op_CB_8F
	.long op_CB_90,op_CB_91,op_CB_92,op_CB_93,op_CB_94,op_CB_95,op_CB_96,op_CB_97
	.long op_CB_98,op_CB_99,op_CB_9A,op_CB_9B,op_CB_9C,op_CB_9D,op_CB_9E,op_CB_9F
	.long op_CB_A0,op_CB_A1,op_CB_A2,op_CB_A3,op_CB_A4,op_CB_A5,op_CB_A6,op_CB_A7
	.long op_CB_A8,op_CB_A9,op_CB_AA,op_CB_AB,op_CB_AC,op_CB_AD,op_CB_AE,op_CB_AF
	.long op_CB_B0,op_CB_B1,op_CB_B2,op_CB_B3,op_CB_B4,op_CB_B5,op_CB_B6,op_CB_B7
	.long op_CB_B8,op_CB_B9,op_CB_BA,op_CB_BB,op_CB_BC,op_CB_BD,op_CB_BE,op_CB_BF
	.long op_CB_C0,op_CB_C1,op_CB_C2,op_CB_C3,op_CB_C4,op_CB_C5,op_CB_C6,op_CB_C7
	.long op_CB_C8,op_CB_C9,op_CB_CA,op_CB_CB,op_CB_CC,op_CB_CD,op_CB_CE,op_CB_CF
	.long op_CB_D0,op_CB_D1,op_CB_D2,op_CB_D3,op_CB_D4,op_CB_D5,op_CB_D6,op_CB_D7
	.long op_CB_D8,op_CB_D9,op_CB_DA,op_CB_DB,op_CB_DC,op_CB_DD,op_CB_DE,op_CB_DF
	.long op_CB_E0,op_CB_E1,op_CB_E2,op_CB_E3,op_CB_E4,op_CB_E5,op_CB_E6,op_CB_E7
	.long op_CB_E8,op_CB_E9,op_CB_EA,op_CB_EB,op_CB_EC,op_CB_ED,op_CB_EE,op_CB_EF
	.long op_CB_F0,op_CB_F1,op_CB_F2,op_CB_F3,op_CB_F4,op_CB_F5,op_CB_F6,op_CB_F7
	.long op_CB_F8,op_CB_F9,op_CB_FA,op_CB_FB,op_CB_FC,op_CB_FD,op_CB_FE,op_CB_FF


! ###########################################################################################################


_read_byte:
_write_byte:
	rts

! ###########################################################################################################

.align 1	
_cpu_rst:
	sts.l	pr,@-r15
	mov	r0,r5
	mov 	#0xFF-CPU_HALTED,r0
	and 	r0,r13
	MOVE_SP -2
	mov	r10,r1
	mov.l	write_byte_func,r2
	jsr	@r2		! Push PC (low)
	extu.w	r11,r0		
	mov	r10,r1
	mov.l	write_byte_func,r2
	shlr8	r1
	extu.w	r11,r0
	add	#1,r0
	jsr	@r2		! Push PC (high)
	nop
	lds.l	@r15+,pr
	nop
	rts
	mov	r5,r10


.align 2
write_byte_func:
	.long _mem_write_byte

! ###########################################################################################################

.global divcnt
.global opcode_table

.align 2

! Jump table for instruction handlers
opcode_table:
	.long op_00,op_01,op_02,op_03,op_04,op_05,CPU_CACHE_COPY_ADR + op_06 - __CPU_COPY_TO_IWRAM_START__,op_07
	.long op_08,op_09,op_0A,op_0B,op_0C,op_0D,op_0E,op_0F
	.long op_10,op_11,CPU_CACHE_COPY_ADR + op_12 - __CPU_COPY_TO_IWRAM_START__,CPU_CACHE_COPY_ADR + op_13 - __CPU_COPY_TO_IWRAM_START__,op_14,op_15,op_16,op_17
	.long CPU_CACHE_COPY_ADR + op_18 - __CPU_COPY_TO_IWRAM_START__,op_19,op_1A,op_1B,op_1C,op_1D,op_1E,op_1F
	.long CPU_CACHE_COPY_ADR + op_20 - __CPU_COPY_TO_IWRAM_START__,op_21,op_22,op_23,op_24,op_25,op_26,op_27
	.long CPU_CACHE_COPY_ADR + op_28 - __CPU_COPY_TO_IWRAM_START__,op_29,CPU_CACHE_COPY_ADR + op_2A - __CPU_COPY_TO_IWRAM_START__,op_2B,op_2C,op_2D,op_2E,op_2F
	.long op_30,op_31,op_32,op_33,op_34,op_35,op_36,op_37
	.long op_38,op_39,op_3A,op_3B,CPU_CACHE_COPY_ADR + op_3C - __CPU_COPY_TO_IWRAM_START__,CPU_CACHE_COPY_ADR + op_3D - __CPU_COPY_TO_IWRAM_START__,CPU_CACHE_COPY_ADR + op_3E - __CPU_COPY_TO_IWRAM_START__,op_3F
	.long op_40,op_41,op_42,op_43,op_44,op_45,op_46,CPU_CACHE_COPY_ADR + op_47 - __CPU_COPY_TO_IWRAM_START__
	.long op_48,op_49,op_4A,op_4B,op_4C,op_4D,op_4E,op_4F
	.long op_50,op_51,op_52,op_53,op_54,op_55,op_56,op_57
	.long op_58,op_59,op_5A,op_5B,op_5C,op_5D,op_5E,op_5F
	.long op_60,op_61,op_62,op_63,op_64,op_65,op_66,op_67
	.long op_68,op_69,op_6A,op_6B,op_6C,op_6D,op_6E,op_6F
	.long op_70,op_71,op_72,op_73,op_74,op_75,op_76,CPU_CACHE_COPY_ADR + op_77 - __CPU_COPY_TO_IWRAM_START__
	.long op_78,op_79,op_7A,op_7B,op_7C,op_7D,CPU_CACHE_COPY_ADR + op_7E - __CPU_COPY_TO_IWRAM_START__,op_7F
	.long op_80,op_81,op_82,op_83,op_84,op_85,op_86,op_87
	.long op_88,op_89,op_8A,op_8B,op_8C,op_8D,op_8E,op_8F
	.long op_90,op_91,op_92,op_93,op_94,op_95,op_96,op_97
	.long op_98,op_99,op_9A,op_9B,op_9C,op_9D,op_9E,op_9F
	.long op_A0,op_A1,op_A2,op_A3,op_A4,op_A5,op_A6,CPU_CACHE_COPY_ADR + op_A7 - __CPU_COPY_TO_IWRAM_START__
	.long op_A8,op_A9,op_AA,op_AB,op_AC,op_AD,op_AE,op_AF
	.long op_B0,op_B1,op_B2,op_B3,op_B4,op_B5,op_B6,op_B7
	.long CPU_CACHE_COPY_ADR + op_B8 - __CPU_COPY_TO_IWRAM_START__,CPU_CACHE_COPY_ADR + op_B9 - __CPU_COPY_TO_IWRAM_START__,CPU_CACHE_COPY_ADR + op_BA - __CPU_COPY_TO_IWRAM_START__,op_BB,op_BC,op_BD,op_BE,op_BF
	.long op_C0,op_C1,op_C2,CPU_CACHE_COPY_ADR + op_C3 - __CPU_COPY_TO_IWRAM_START__,op_C4,op_C5,op_C6,op_C7
	.long op_C8,op_C9,op_CA,CPU_CACHE_COPY_ADR + op_CB - __CPU_COPY_TO_IWRAM_START__,op_CC,op_CD,op_CE,op_CF
	.long op_D0,op_D1,op_D2,op_D3,op_D4,op_D5,op_D6,op_D7
	.long op_D8,op_D9,op_DA,op_DB,op_DC,op_DD,op_DE,op_DF
	.long op_E0,op_E1,op_E2,op_E3,op_E4,op_E5,op_E6,op_E7
	.long op_E8,op_E9,op_EA,op_EB,op_EC,op_ED,op_EE,op_EF
	.long CPU_CACHE_COPY_ADR + op_F0 - __CPU_COPY_TO_IWRAM_START__,op_F1,op_F2,op_F3,op_F4,op_F5,op_F6,op_F7
	.long op_F8,op_F9,op_FA,op_FB,op_FC,op_FD,CPU_CACHE_COPY_ADR + op_FE - __CPU_COPY_TO_IWRAM_START__,op_FF


.global __CPU_COPY_TO_IWRAM_START__

__CPU_COPY_TO_IWRAM_START__:

divcnt: .long 0
regs:	.byte 0,0,0,0,0,0,0,0

! +0x0C
_cpu_execute:
	sts.l	pr,@-r15
	mov.l	read_byte_func,r6
	mov.l	_cpu_max_cycles20,r1
	mov.l	r0,@r1

_ce_start_loop:	
	mov.l	_adrmask,r9
	mov.l	__cpu_instr_done,r4

_cpu_execute_loop:
	mova	cpu_old_cycles,r0
	mov.l	r12,@r0

	extu.b 	r13,r0
	tst 	#CPU_STOPPED+CPU_HALTED,r0
	bt/s 	_ce_fetch_opcode
	extu.w	r10,r0
	bra 	_ce_update_div
	ADD_CYCLES 48
_ce_fetch_opcode:
	mov	r0,r3	
	mov.l	rd8_tbl20,r2
	shlr8	r0
	shlr2	r0
	and 	#0x3C,r0
	mov.l	@(r0,r2),r1
	jsr	@r1
	mov	r3,r0
	mov.l 	_opcode_table20,r1
	shll2	r0
	mov.l	@(r0,r1),r5	! Get the address of the instruction handler
	jmp	@r5		! ..and branch to it
	nop
! +0x42	
_cpu_instr_done:

_ce_update_div:
	mov.l	_IOREGS20,r14

	! REG_DIV is updated at 16384Hz, i.e. 1/256th of the CPU frequency
	! So a byte is decremented once per clock cycle, and REG_DIV is
	! updated when the byte wraps around at zero.
	mov.l	_divcnt20,r0
	mov.b	@r0,r1
	mov	r0,r5
	mov.l	cpu_old_cycles,r2 
	mov	r12,r3
	extu.b	r1,r0
	sub	r2,r3
	sub	r3,r0
	cmp/pz	r0
	bt/s	_ce_no_div_of
	mov.b	r0,@r5
	mov.b	@(4,r14),r0	! REG_DIV
	add	#1,r0
	mov.b	r0,@(4,r14)	! REG_DIV
_ce_no_div_of:

	mov.b	@(0x07,r14),r0	! REG_TAC
	tst	#4,r0
	bt	_ce_timer_inactive
	mov.l	_timercnt20,r0
	mov.l	@r0,r1
	sub	r3,r1
	cmp/pz	r1
	bt/s	_ce_timer_inactive
	mov.l	r1,@r0
	mov.b	@(5,r14),r0	! REG_TIMA
	add	#1,r0
	extu.b	r0,r0
	cmp/eq	#0,r0
	bf/s	_ce_no_tima_reload
	mov.b	r0,@(5,r14)	! REG_TIMA
	mov.b	@(6,r14),r0	! REG_TMA
	mov.b	r0,@(5,r14)	! REG_TIMA
	mov.b	@(0x0F,r14),r0	! REG_IF
	or	#4,r0
	mov.b	r0,@(0x0F,r14)	! REG_IF
	mov	#CPU_IME,r0
	tst	r0,r13
	bt	_ce_no_tima_reload
	mov	#0xFF,r0
	extu.b	r0,r0
	mov.b	@(r0,r14),r1	! REG_IE
	extu.b	r1,r0
	tst	#4,r0
	bt	_ce_no_tima_reload
	mov.b	@(0x0F,r14),r0	! REG_IF
	and	#0xFB,r0
	mov.b	r0,@(0x0F,r14)	! REG_IF
	mov.l	_cpu_rst20,r2
	jsr	@r2
	mov	#0x50,r0
	mov	#0xFF-CPU_IME,r1
	and	r1,r13
_ce_no_tima_reload:
	mov.l	_timerld20,r0
	mov.l	_timercnt20,r1
	mov.l	@r0,r2
	mov.l	@r1,r3
	add	r2,r3
	mov.l	r3,@r1
_ce_timer_inactive:

	mov	#CPU_DI_PENDING,r0
	tst	r0,r13
	bt	_ce_no_pending_di
	mov	#0xFF-(CPU_IME+CPU_DI_PENDING),r1
	and	r1,r13
_ce_no_pending_di:
	mov	#CPU_EI_PENDING,r0
	tst	r0,r13
	bt	_ce_no_pending_ei
	mov	#0xFF-CPU_EI_PENDING,r1
	and	r1,r13
	mov	#CPU_IME,r1
	or	r1,r13
_ce_no_pending_ei:
	mov.l	cpu_max_cycles,r0
	cmp/hs	r0,r12
	bt	_ce_done
	bra	_cpu_execute_loop
	nop
_ce_done:

	mov	#CPU_IME,r0
	tst	r0,r13
	bt	_ce_no_irq
	mov	#0xFF,r0
	extu.b	r0,r0
	mov.b	@(r0,r14),r1	! REG_IE
	mov.b	@(0x0F,r14),r0	! REG_IF
	mov	r0,r2
	and	r1,r0
	cmp/eq	#0,r0
	bt	_ce_no_irq
	tst	#1,r0
	bt	_ce_no_vbi
	mov	r2,r0
	and	#0xFE,r0
	mov.b	r0,@(0x0F,r14)	! REG_IF
	mov.l	_cpu_rst20,r2
	jsr	@r2
	mov	#0x40,r0
	mov	#0xFF-CPU_IME,r1
	bra	_ce_no_irq
	and	r1,r13
_ce_no_vbi:
	tst	#2,r0
	bt	_ce_no_stati
	mov	r2,r0
	and	#0xFD,r0
	mov.b	r0,@(0x0F,r14)	! REG_IF
	mov.l	_cpu_rst20,r2
	jsr	@r2
	mov	#0x48,r0
	mov	#0xFF-CPU_IME,r1
	bra	_ce_no_irq
	and	r1,r13
_ce_no_stati:
	tst	#4,r0
	bt	_ce_no_timeri
	mov	r2,r0
	and	#0xFB,r0
	mov.b	r0,@(0x0F,r14)	! REG_IF
	mov.l	_cpu_rst20,r2
	jsr	@r2
	mov	#0x50,r0
	mov	#0xFF-CPU_IME,r1
	bra	_ce_no_irq
	and	r1,r13
_ce_no_timeri:
	tst	#0x10,r0
	bt	_ce_no_irq
	mov	r2,r0
	and	#0xEF,r0
	mov.b	r0,@(0x0F,r14)	! REG_IF
	mov.l	_cpu_rst20,r2
	jsr	@r2
	mov	#0x60,r0
	mov	#0xFF-CPU_IME,r1
	and	r1,r13
_ce_no_irq:	

	lds.l	@r15+,pr
	nop
	rts
	nop


.align 2
! + 14C
cpu_old_cycles:
	.long 0
cpu_max_cycles:
	.long 0
_adrmask:
	.long 0x1FFF
_opcode_table20:
	.long opcode_table
__cpu_instr_done:
	.long CPU_CACHE_COPY_ADR + _cpu_instr_done - __CPU_COPY_TO_IWRAM_START__
_IOREGS20:
	.long IOREGS + UNCACHED_ADDRESS
_cpu_max_cycles20:
	.long CPU_CACHE_COPY_ADR  + cpu_max_cycles - __CPU_COPY_TO_IWRAM_START__
_cpu_rst20:
	.long _cpu_rst
_divcnt20:
	.long CPU_CACHE_COPY_ADR + divcnt - __CPU_COPY_TO_IWRAM_START__
_timercnt20:
	.long MEM_CACHE_COPY_ADR + 0x10C !timercnt
_timerld20:
	.long MEM_CACHE_COPY_ADR + 0x110 !timerld
read_byte_func:
	.long MEM_CACHE_COPY_ADR + 0x24 !_mem_read_byte
rd8_tbl20: .long MEM_CACHE_COPY_ADR + 0x4C !read_byte_ftbl


! +0x180
! JR NZ,n
op_20:
	JR_COND	FLAG_Z,bf
! +0x1B8	
! JR Z,n	
op_28:
	JR_COND	FLAG_Z,bt

! +0x1F0
! LDH A,(n)
op_F0:
	MOVE_PC	1
	READ_BYTE
	extu.w	r10,r0
	mov	#0xFF,r1
	shll8	r1
	or	r1,r0
	READ_BYTE
	extu.w	r0,r0
	extu.b	r0,r7
	MOVE_PC	1
	RETURN
	ADD_CYCLES 12

! +0x1FC
! CP n
op_FE:
	MOVE_PC	1
	READ_BYTE
	extu.w	r10,r0
	extu.b 	r7,r2
	sub 	r0,r2
	cmp/eq 	r0,r7
	movt	r8
	shll8	r8
	shlr	r8		! Zero
	cmp/hs	r0,r7
	bt/s	_op_FE_NC
	xor	r7,r0
	add	#FLAG_C,r8	! Set F.Carry if A<n
_op_FE_NC:
	xor	r2,r0
	shll	r0
	and	#FLAG_H,r0
	or	r0,r8
	add	#FLAG_N,r8
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8

! +0x226
! LDH (n),A
op_E0:
	MOVE_PC	1
	READ_BYTE
	extu.w	r10,r0
	mov	#0xFF,r1
	shll8	r1
	or	r1,r0
	mov.l	_write_byte20,r2
	extu.w	r0,r0
	jsr	@r2
	extu.b	r7,r1
	MOVE_PC	1
	RETURN
	ADD_CYCLES 12

! +0x240
! LD (DE),A
op_12:
	mov.w	@(REG_D,gbr),r0
	mov	r7,r1
	mov.l	_write_byte20,r2
	jsr	@r2
	extu.w	r0,r0
	MOVE_PC	1
	RETURN
	ADD_CYCLES 8
	
! +0x250
! INC DE
op_13:
	mov.w	@(REG_D,gbr),r0
	add	#1,r0
	MOVE_PC	1
	mov.w	r0,@(REG_D,gbr)
	RETURN
	ADD_CYCLES 8
	
! +0x25C
_write_byte20:
	.long MEM_CACHE_COPY_ADR + 0x34

! +0x260
! JR n
op_18:
	MOVE_PC 2
	mov	r10,r0
	add	#-1,r0		! Reads from [oldPC+1], since we've already moved PC 2 bytes forward
	READ_BYTE
	extu.w	r0,r0
	exts.b	r0,r0
	add	r0,r10
	RETURN
	ADD_CYCLES 12

! +0x272
! INC A
op_3C:
	mov	r7,r1
	mov 	#FLAG_C,r2
	add 	#1,r7
	and 	r2,r8
	extu.b 	r7,r0
	cmp/eq 	#0,r0
	movt 	r2
	shll8 	r2
	extu.b	r7,r7
	shlr 	r2		! Zero
	xor 	r1,r0
	or 	r2,r8
	shll2 	r0
	and 	#FLAG_H,r0
	MOVE_PC	1
	or 	r0,r8
	RETURN
	ADD_CYCLES 4

! +0x296
! DEC A
op_3D:
	mov	r7,r1
	mov 	#FLAG_C,r2
	add 	#-1,r7
	and 	r2,r8
	extu.b 	r7,r0
	cmp/eq 	#0,r0
	movt 	r2
	shll8 	r2
	extu.b	r7,r7	
	shlr 	r2		! Zero
	xor 	r1,r0
	or 	r2,r8
	shll2 	r0
	and 	#FLAG_H,r0
	or	#FLAG_N,r0
	MOVE_PC	1
	or 	r0,r8
	RETURN
	ADD_CYCLES 4

! +0x2BC
! LD A,n
op_3E:
	MOVE_PC 1
	mov	r10,r0
	READ_BYTE
	MOVE_PC 1
	extu.b	r0,r7
	RETURN
	ADD_CYCLES 8	

! +0x2CA
!AND A,A
op_A7:
	tst 	r7,r7
	movt 	r8
	shll8 	r8
	MOVE_PC 1
	shlr 	r8	! Zero
	add	#FLAG_H,r8
	RETURN
	ADD_CYCLES 4

! +0x2DA
! LD (HL),A
op_77:
	mov.w	@(REG_H,gbr),r0
 	mov.l	_write_byte21,r2
	extu.b	r7,r1
 	jsr	@r2
 	extu.w	r0,r0
 	MOVE_PC	1
 	RETURN
 	ADD_CYCLES 8
 	
! +0x2EA
! LD A,(HL)
 op_7E:
 	mov.w	@(REG_H,gbr),r0
  	READ_BYTE
  	extu.w	r0,r0	! delayed branch slot
  	MOVE_PC	1
  	extu.b	r0,r7
  	RETURN
 	ADD_CYCLES 8

! +0x2F8
_write_byte21:
	.long MEM_CACHE_COPY_ADR + 0x34

! +0x2FC
! JP nn
op_C3:
	extu.w	r10,r0
	add	#1,r0
	READ_BYTE
	MOVE_PC	2
	mov	r0,r5
	extu.w	r10,r0
	READ_BYTE
	extu.b	r5,r10
	shll8	r0
	or	r0,r10
	RETURN
	ADD_CYCLES 16

! +0x314
! LD A,(HL+)
op_2A:
	mov.w	@(REG_H,gbr),r0
	mov	r0,r5
	add	#1,r5	
	READ_BYTE
	extu.w	r0,r0
	extu.b	r0,r7
	mov	r5,r0
	MOVE_PC	1
	mov.w	r0,@(REG_H,gbr)
	RETURN
	ADD_CYCLES 8

! +0x32A
! CB prexif				
op_CB:
	MOVE_PC 1
	READ_BYTE
	extu.w	r10,r0
	mov.l	_opcode_cb_table13,r1
	extu.b	r0,r0
	shll2	r0
	mov.l	@(r0,r1),r2
	jmp	@r2
	nop

! +0x33C
_opcode_cb_table13:
	.long opcode_cb_table

! +0x340
! LD B,n
op_06:
	LD_IMM8 REG_B,_read_byte1

! LD B,A
op_47:
	LD_REG8_A REG_B

! CP A,B 
op_B8:
	CP_A_REG8 REG_B

! CP A,C 
op_B9:
	CP_A_REG8 REG_C

! CP A,D 
op_BA:
	CP_A_REG8 REG_D
	
__CPU_COPY_TO_IWRAM_END__:



	
! ###########################################################################################################

!.section .data

_cpu_reset:
	mov.l	_cache_start,r0
	mov.l	_block_start,r1
	mov.l	_block_size,r2
_cr_copy_to_cache:
	mov.b	@r1,r3
	add	#1,r1
	mov.b	r3,@r0
	dt	r2
	bf/s	_cr_copy_to_cache
	add	#1,r0
	
	mov.l	_regs,r0
	ldc	r0,gbr

	mov	#0x01,r7	! A
	mov	#0xB0,r8	! F
	extu.b	r8,r8
	mov	#0x00,r0
	mov.b	r0,@(REG_B,gbr)
	mov.b	r0,@(REG_D,gbr)
	mov	#0x01,r0
	mov.b	r0,@(REG_H,gbr)
	mov	#0x13,r0
	mov.b	r0,@(REG_C,gbr)
	mov	#0xD8,r0
	mov.b	r0,@(REG_E,gbr)
	mov	#0x4D,r0
	mov.b	r0,@(REG_L,gbr)
	
	mov	#1,r10
	mov	#0xFE,r11
	shll8	r10		! PC = 0x0100
	extu.w	r11,r11		! SP = 0xFFFE

	! Clear halted, stopped, pendingDI and pendingEI, IME
	mov	#0,r13
	
	rts
	nop

.align 2
_cache_start:
	.long CPU_CACHE_COPY_ADR
_block_start:
	.long __CPU_COPY_TO_IWRAM_START__
_block_size:
	.long __CPU_COPY_TO_IWRAM_END__ - __CPU_COPY_TO_IWRAM_START__
_regs:	.long CPU_CACHE_COPY_ADR + regs - __CPU_COPY_TO_IWRAM_START__

! ###########################################################################################################


	
	