;***************************************************************************************************************************
;*
;*   Project	: '720 Sprites' - An example showing that 720 sprites on a screen is possible.
;*
;*   Module	: 720Sprites.asm
;*
;*   Function	: The demo code.
;*
;*   Comment	: My screen column width is 125.
;*
;*   Tab Stops	: 20 30 70
;*                 |         |                                       |
;*
;*   (Most of it was) Written by Bob Koon (bob@onemanband.com)
;*
;***************************************************************************************************************************

	INCLUDE	"Hardware.inc"
	INCLUDE	"StarDataTest.asm"


;***************************************************************************************************************************
;*	externally referenced symbols
;***************************************************************************************************************************


;***************************************************************************************************************************
;*	equates
;***************************************************************************************************************************

HIGHMEM_VARSIZE	EQU	1	; size (in bytes) of the variable block in high mem (see below)

NUM_STARS	EQU	10

OAMZ1_ADDR	EQU	BASE_OAMRAM

SPRITEDMA_ADDR	EQU	BASE_HIGHRAM + 16	;HIGHMEM_VARSIZE

LCDC_INCREMENT	EQU	2

; various user/state flags
USER_F_VBLANKEND	EQU	%00000010	; set when vblank is done
USER_F_ISCGB	EQU	%00000001	; set when running on a CGB


;***************************************************************************************************************************
;*	variables
;***************************************************************************************************************************

	SECTION	"UserVars",BSS

StarCoordsZ1	DS	((NUM_STARS * 4) * 4)	; y,x,z,attr
StarCoordsZ1E	DS	((NUM_STARS * 4) * 4)	; y,x,z,attr

	; variables located in high memory
	RSSET	BASE_HIGHRAM
UserFlags	RB	1	; various user/state flags (see list above)
ScanLine	RB	1


;***************************************************************************************************************************
;*	cartridge header
;***************************************************************************************************************************

;	SECTION	"Org $00",HOME[$00]
	; $0000 - Restart $00 address
;RST_00:	jp	<somewhere>

;	SECTION	"Org $08",HOME[$08]
	; $0008 - Restart $08 address
;RST_08:	jp	<somewhere>

;	SECTION	"Org $10",HOME[$10]
	; $0010 - Restart $10 address
;RST_10:	jp	<somewhere>

;	SECTION	"Org $18",HOME[$18]
	; $0018 - Restart $18 address
;RST_18:	jp	<somewhere>

;	SECTION	"Org $20",HOME[$20]
	; $0020 - Restart $20 address
;RST_20:	jp	<somewhere>

;	SECTION	"Org $28",HOME[$28]
	; $0028 - Restart $28 address
;RST_28:	jp	<somewhere>

;	SECTION	"Org $30",HOME[$30]
	; $0030 - Restart $30 address
;RST_30:	jp	<somewhere>

;	SECTION	"Org $38",HOME[$38]
	; $0038 - Restart $38 address
;RST_38:	jp	<somewhere>

	SECTION	"Org $40",HOME[$40]
	; $0040 - V-Blank interrupt start address
	jp	IRQ_VBlank

	SECTION	"Org $48",HOME[$48]
	; $0048 - LCDC Status interrupt start address
	jp	IRQ_LCDC

;	SECTION	"Org $50",HOME[$50]
	; $0050 - Timer Overflow interrupt start address
;	jp	IRQ_Timer

;	SECTION	"Org $58",HOME[$58]
	; $0058 - Serial Transfer Completion interrupt start address
;	jp	IRQ_Serial

;	SECTION	"Org $60",HOME[$60]
	; $0060 - Joypad interrupt start address
;	jp	IRQ_Joypad


	SECTION	"Start",HOME[$100]
	; $0100-$0103 (Game code start)
	nop
	jp	Start

	; $0104-$0133 (Nintendo logo - do _not_ modify the logo data here or the CGB will freeze)
	DB	$CE,$ED,$66,$66,$CC,$0D,$00,$0B,$03,$73,$00,$83,$00,$0C,$00,$0D
	DB	$00,$08,$11,$1F,$88,$89,$00,$0E,$DC,$CC,$6E,$E6,$DD,$DD,$D9,$99
	DB	$BB,$BB,$67,$63,$6E,$0E,$EC,$CC,$DD,$DC,$99,$9F,$BB,$B9,$33,$3E

	; $0134-$013E (Game title - up to 11 upper case ASCII characters; pad with $00)
	DB	"720 SPRITES"
		;0123456789A

	; $013F-$0142 (Product code - 4 ASCII characters)
	DB	"    "
		;0123

	; $0143 (Color GameBoy compatability code)
	DB	$C0	; $C0 - Dedicated Color GameBoy cartridge
			; $80 - Color GameBoy compatible cartridge
			; $00 - Not a Color GameBoy cartridge (all other values)

	; $0144 (High-nibble of license code - normally $00 if $014B != $33)
	DB	$00

	; $0145 (Low-nibble of license code - normally $00 if $014B != $33)
	DB	$00

	; $0146 (GameBoy/Super GameBoy indicator)
	DB	$00	; $00 - GameBoy
			; $03 - Super GameBoy

	; $0147 (Cartridge type - all Color GameBoy cartridges are at least $19)
	DB	$00	; $00 - ROM only
			; $01 - ROM + MBC1
			; $02 - ROM + MBC1 + RAM
			; $03 - ROM + MBC1 + RAM + Battery
			; $05 - ROM + MBC2
			; $06 - ROM + MBC2 + Battery
			; $08 - ROM + RAM
			; $09 - ROM + RAM + Battery
			; $0B - ROM + MMMO1
			; $0C - ROM + MMMO1 + SRAM
			; $0D - ROM + MMMO1 + SRAM + Battery
			; $0F - ROM + MBC3 + Timer + Battery
			; $10 - ROM + MBC3 + Timer + RAM + Battery
			; $11 - ROM + MBC3
			; $12 - ROM + MBC3 + RAM
			; $13 - ROM + MBC3 + RAM + Battery
			; $19 - ROM + MBC5
			; $1A - ROM + MBC5 + RAM
			; $1B - ROM + MBC5 + RAM + Battery
			; $1C - ROM + MBC5 + Rumble
			; $1D - ROM + MBC5 + Rumble + SRAM
			; $1E - ROM + MBC5 + Rumble + SRAM + Battery
			; $1F - Pocket Camera
			; $FD - Bandai TAMA5
			; $FE - Hudson HuC-3
			; $FF - Hudson HuC-1

	; $0148 (ROM size)
	DB	$00	; $00 - 256Kbit =  32Kbyte =   2 banks
			; $01 - 512Kbit =  64Kbyte =   4 banks
			; $02 -   1Mbit = 128Kbyte =   8 banks
			; $03 -   2Mbit = 256Kbyte =  16 banks
			; $04 -   4Mbit = 512Kbyte =  32 banks
			; $05 -   8Mbit =   1Mbyte =  64 banks
			; $06 -  16Mbit =   2Mbyte = 128 banks
			; $52 -   9Mbit = 1.1Mbyte =  72 banks
			; $53 -  10Mbit = 1.2Mbyte =  80 banks
			; $54 -  12Mbit = 1.5Mbyte =  96 banks

	; $0149 (RAM size)
	DB	$00	; $00 - None
			; $01 -  16Kbit =   2Kbyte =  1 bank
			; $02 -  64Kbit =   8Kbyte =  1 bank
			; $03 - 128Kbit =  32Kbyte =  4 banks
			; $04 -   1Mbit = 128Kbyte = 16 banks

	; $014A (Destination code)
	DB	$01	; $00 - Japanese
			; $01 - All others

	; $014B (Licensee code - this _must_ be $33)
	DB	$33	; $33 - Check $0144/$0145 for Licensee code.

	; $014C (Mask ROM version - handled by a post-linking tool)
	DB	$00

	; $014D (Complement check - handled by a post-linking tool)
	DB	$00

	; $014E-$014F (Cartridge checksum - handled by a post-linking tool)
	DW	$00


;***************************************************************************************************************************
;*	let the games...BEGIN!
;***************************************************************************************************************************

Start::
	di

	ld	sp,$E000	; set up a stack

	; if running on a CGB, set the user flag
	and	a
	cp	CPU_GBC
	ld	a,0
	jr	nz,.setuserflags
	ld	a,USER_F_ISCGB
.setuserflags
	ld	[UserFlags],a

	call	TurnOffDisplay

	; clear LCD control registers
	xor	a
	ld	[REG_IE],a
	ld	[REG_IF],a
	ld	[REG_SCX],a
	ld	[REG_SCY],a
	ld	[REG_STAT],a

	; if running on CGB, set the CPU to double speed (the CGB is single speed on boot up)...
	ld	a,[UserFlags]
	and	USER_F_ISCGB
	jr	z,.clearram

	call	ToggleCPUSpeed

	; ...and clear the CGB-specific registers
	xor	a
	ld	[REG_VBK],a	; set VRAM Bank(0)
	ld	[REG_SVBK],a	; set internal RAM Bank(0)
	ld	[REG_RP],a	; clear IR port

	; clear $C000-$DFFF (internal RAM)
.clearram:
	xor	a
	ld	hl,$DFFF
	ld	c,$20
	ld	b,0
.clear1:
	ld	[hl-],a
	dec	b
	jr	nz,.clear1
	dec	c
	jr	nz,.clear1

	; clear $FE00-$FEFF (OAM [sprite attribute] RAM)
	ld	hl,$FEFF
	ld	b,0
.clear2:
	ld	[hl-],a
	dec	b
	jr	nz,.clear2

	; clear $FF80-$FFFF (internal [high] RAM)
	ld	hl,$FFFF
	ld	b,$80 - HIGHMEM_VARSIZE	; also clear the variables (except UserFlags)
.clear3:
	ld	[hl-],a
	dec	b
	jr	nz,.clear3

	; clear the bg maps
	ld	hl,BASE_MAPRAM9800
	call	ClearAllBGAttrMap
	ld	hl,BASE_MAPRAM9C00
	call	ClearAllBGAttrMap

	ld	hl,BASE_MAPRAM9800
	call	ClearAllBGMap
	ld	hl,BASE_MAPRAM9C00
	call	ClearAllBGMap

	; set up the default palettes
	ld	a,$E4
	ld	[REG_BGP],a
	ld	[REG_OBP0],a
	ld	[REG_OBP1],a

	; set up the custom palettes
	ld	a,[UserFlags]
	and	USER_F_ISCGB
	jr	z,.setvram

	ld	hl,BGPaletteData
	call	SetBGPalettes
	ld	hl,OBJPaletteData
	call	SetOBJPalettes

	; copy the graphic tile data to VRAM ($8000:Bank 0)
.setvram:
	ld	bc,TileGraphicData
	ld	hl,BASE_VRAM
	ld	de,TileGraphicDataEnd - TileGraphicData
	call	CopyToVRAM

	; create random star coordinates and copy them to shadow memory
	ld	a,[UserFlags]
	and	USER_F_ISCGB
	jr	z,.printnoncgbmsg

	call	CreateSpriteCoords
	ld	hl,StarCoordsZ1E
	ld	de,StarCoordsZ1
	call	AssignSpritePos

	; copy sprite DMA routine to HIRAM
	ld	hl,SpriteDMAFunc
	ld	de,SPRITEDMA_ADDR
	ld	b,SpriteDMAFuncEnd - SpriteDMAFunc
.loadhigh:
	ld	a,[hl+]
	ld	[de],a
	inc	de
	dec	b
	jr	nz,.loadhigh

	jp	.initdisplay

	; print a special message for non-CGB users
.printnoncgbmsg:
	ld	bc,NonCGBString0
	ld	hl,$9800
	ld	de,$0701	; d = y coord, e = x coord
	call	PrintText
	ld	bc,NonCGBString1
	ld	hl,$9800
	ld	de,$0803	; d = y coord, e = x coord
	call	PrintText
	ld	bc,NonCGBString2
	ld	hl,$9800
	ld	de,$0903	; d = y coord, e = x coord
	call	PrintText

	; initialize the display
.initdisplay:
	xor	a
	ld	[REG_NR52],a	; turn sound off
	ld	[REG_WY],a	; window ypos = 0
	ld	a,$07
	ld	[REG_WX],a	; window xpos = 8

	; turn the display on
	ld	a,REG_LCDC_F_ON | REG_LCDC_F_BG8000 | REG_LCDC_F_BG9800 | REG_LCDC_F_OBJON | REG_LCDC_F_BGON
	ld	[REG_LCDC],a
	xor	a
	ld	[REG_IF],a

	ld	a,[UserFlags]
	and	USER_F_ISCGB
	jr	z,.setvblankonly
	ld	a,REG_IE_F_LCDC | REG_IE_F_VBLANK
	jp	.setie
.setvblankonly:
	ld	a,REG_IE_F_VBLANK
.setie:
	ld	[REG_IE],a

	ei

.mainloop:
	ld	a,[UserFlags]
	and	USER_F_ISCGB
	jr	z,.donothing

	; do stuff here
	; do stuff here
	; do stuff here

	jp	.waitvblank

	; save the batteries on the old GB
.donothing:
	halt
	nop

	; wait for a V-Blank
.waitvblank:
	ld	a,[UserFlags]
	and	USER_F_VBLANKEND	; clobbers all other bits but bit 1
	jr	z,.waitvblank

	ld	a,[UserFlags]	; see comment above
	xor	USER_F_VBLANKEND
	ld	[UserFlags],a

	jp	.mainloop


;***************************************************************************************************************************
;*	Support routines
;***************************************************************************************************************************

IRQ_VBlank::
	push	af
	push	bc
	push	de
	push	hl

	; do nothing if not a CGB
	ld	a,[UserFlags]
	and	USER_F_ISCGB
	jr	z,.vblankend

	ld	de,StarCoordsZ1E
	ld	hl,StarCoordsZ1
	call	AssignSpritePos

	call	SPRITEDMA_ADDR	; update the OAM RAM

	; set up the LCDC interrupt
	ld	a,LCDC_INCREMENT	; set line at which lcdc interrupt occurs
	dec	a
	ld	[ScanLine],a
	ld	[REG_LYC],a
	ld	a,REG_STAT_F_LYC | REG_STAT_F_LYCF
	ld	[REG_STAT],a

	; move the sprite positions for the next LCDC interrupt
	ld	hl,StarCoordsZ1
	ld	b,LCDC_INCREMENT
	ld	c,NUM_STARS
	ld	de,4
.adjypos:
	ld	a,[hl]
	add	a,b
	ld	[hl],a
	add	hl,de

	dec	c
	jr	nz,.adjypos

	; set the V-Blank end flag
	ld	a,[UserFlags]
	or	USER_F_VBLANKEND
	ld	[UserFlags],a

.vblankend:
	pop	hl
	pop	de
	pop	bc
	pop	af

	reti


IRQ_LCDC::
	push	af
	push	bc
	push	de
	push	hl

	call	SPRITEDMA_ADDR	; update the OAM RAM

	; set the next LCDC interrupt
	ld	b,LCDC_INCREMENT
	ld	a,[ScanLine]
	add	a,b
	cp	145
	jr	z,.lcdcend
	ld	[ScanLine], a
	ld	[REG_LYC],a

	; move the sprite positions for the next LCDC interrupt
	ld	hl,StarCoordsZ1
	ld	b,LCDC_INCREMENT
	ld	c,NUM_STARS
	ld	de,4
.adjypos:
	ld	a,[hl]
	add	a,b
	ld	[hl],a
	add	hl,de

	dec	c
	jr	nz,.adjypos

.lcdcend:
	pop	hl
	pop	de
	pop	bc
	pop	af

	reti


CreateSpriteCoords::
	ld	c,NUM_STARS
	ld	hl,StarCoordsZ1
.loop:
	; y coord
	ld	a,16
	ld	[hl+],a
	; x coord
	ld	a,88
	add	a,c
	add	a,c
	ld	[hl+],a
	; tile number
	ld	a,1
	ld	[hl+],a
	inc	hl	; skip attribute (already 0)
	dec	c
	jr	nz,.loop

	ret


AssignSpritePos::
	ld	b,NUM_STARS * 4
.loop:
	ld	a,[de]
	inc	de
	ld	[hl+],a

	dec	b
	jr	nz,.loop

	ret


SpriteDMAFunc::
	push	af
	ld	a,$C0
	ld	[REG_DMA],a
	; wait 160 ms
	ld	a,40
.oamloop:
	dec	a
	jr	nz,.oamloop
	pop	af

	ret
SpriteDMAFuncEnd::


ToggleCPUSpeed::
	di

	ld	hl,REG_IE
	ld	a,[hl]
	push	af

	; disable interrupts
	xor	a
	ld	[hl],a
	ld	[REG_IF],a

	ld	a,REG_P1_F_5 | REG_P1_F_4
	ld	[REG_P1],a

	ld	a,1
	ld	[REG_KEY1],a

	stop

	pop	af
	ld	[hl],a

	ei

	ret


TurnOffDisplay::
	ld	hl,REG_LCDC
	bit	7,[hl]	; is the LCD already off?
	ret	z	; yes, exit

	; disable the V-Blank interrupt if enabled
	ld	a,[REG_IE]
	push	af
	res	0,a
	ld	[REG_IE],a

	; wait for the next V-Blank
.waitvbl:
	ld	a,[REG_LY]
	cp	145
	jr	nz,.waitvbl

	; turn off the screen
	res	7,[hl]

	; restore the state of the V-Blank interrupt
	pop	af
	ld	[REG_IE],a

	ret


ClearAllBGAttrMap::
	ld	a,1
	ld	[REG_VBK],a	; set bank 1 (for attributes)

	xor	a
	ld	de,(MAP_BLOCKWIDTH - SCREEN_BLOCKWIDTH)
.memc0:
	ld	c,MAP_BLOCKHEIGHT
.memc1:
	ld	b,MAP_BLOCKWIDTH
.memc2:
	ld	[hl+],a

	dec	b
	jr	nz,.memc2

	add	hl,de

	dec	c
	jr	nz,.memc1

	xor	a
	ld	[REG_VBK],a	; reset back to bank 0

	ret


ClearAllBGMap::
	xor	a
	ld	e,SCREEN_BLOCKHEIGHT

.clearloop1:
	ld	d,MAP_BLOCKWIDTH
.clearloop2:
	ld	[hl+],a
	dec	d
	jr	nz,.clearloop2
	dec	e
	jr	nz,.clearloop1

	ret


; *** copy data to vram ***
; Entry: BC = Pointer to BG tile data
;        HL = Pointer to VRAM destination
;        DE = Number of bytes to copy
CopyToVRAM::
.copyloop:
	; wait for OAM RAM to become available for use
	ld	a,[REG_STAT]
	and	REG_STAT_F_OAM
	jr	nz,.copyloop

	ld	a,[bc]
	ld	[hl+],a
	inc	bc
	dec	de
	ld	a,d
	or	e
	jr	nz,.copyloop

	ret


; *** Set background palettes ***
; Entry: HL = Pointer to BG palettes
SetBGPalettes::
	ld	a,$80
	ld	[REG_BCPS],a

	ld	bc,$4069	; b = 64, c = $69
.loop:
	ld	a,[hl+]
	ld	[c],a
	dec	b
	jr	nz,.loop

	ret


; *** Set Object palettes ***
; Entry: HL = Pointer to OBJ (sprite) palettes
SetOBJPalettes::
	ld	a,$80
	ld	[REG_OCPS],a

	ld	bc,$406B	; b = 64, c = $6B
.loop:
	ld	a,[hl+]
	ld	[c],a
	dec	b
	jr	nz,.loop

	ret


; Entry: HL = Pointer to BG map destination
;        BC = Pointer to NULL-terminated string
;         D = Y coordinate (in tiles)
;         E = X coordinate (in tiles)
PrintText::
	; calculate the destination start address
	push	bc
	ld	bc,MAP_BLOCKWIDTH

	ld	a,d
.yloop
	cp	0
	jr	z,.addx
	add	hl,bc	; add a whole row
	dec	a
	jp	.yloop	

.addx
	ld	b,0
	ld	c,e
	add	hl,bc	; add x coordinate
	pop	bc

.textloop:
	ld	a,[bc]
	cp	0
	jr	z,.end	; found the NULL terminator, exit
	ld	[hl+],a
	inc	bc
	jp	.textloop

.end:
	ret
