//-------------------------------------------------------------------------------------
//
// Copyright 2009 Intel Corporation
// All Rights Reserved
//
// Permission is granted to use, copy, distribute and prepare derivative works of this
// software for any purpose and without fee, provided, that the above copyright notice
// and this statement appear in all copies.  Intel makes no representations about the
// suitability of this software for any purpose.  THIS SOFTWARE IS PROVIDED "AS IS."
// INTEL SPECIFICALLY DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, AND ALL LIABILITY,
// INCLUDING CONSEQUENTIAL AND OTHER INDIRECT DAMAGES, FOR THE USE OF THIS SOFTWARE,
// INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PROPRIETARY RIGHTS, AND INCLUDING THE
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  Intel does not
// assume any responsibility for any errors which may appear in this software nor any
// responsibility to update it.
//

#ifndef _NULSTEIN_ENTITY_H_
#define _NULSTEIN_ENTITY_H_

#include "Misc/Defines.h"

class CFrame;
class CEntitySlot;
class CDisplayListAdder;
class CDisplayItemKey;

#define MAX_DEPENDENTCOUNT 8

//________________________________________________________________________________
#define ENTITY_NOPREUPDATE			0x00000001
#define ENTITY_NOUPDATE_internal	0x00000002
#define ENTITY_NOUPDATE				( ENTITY_NOPREUPDATE | ENTITY_NOUPDATE_internal )
#define ENTITY_NODRAW				0x00000004

#define ENTITY_DEPENDENT			0x00000008  /* needs other entity(-ies) to update first */ 

#define ENTITY_GFX_REGISTERED		0x80000000	/* for Gfx where each entity is shadowed	*/ 

//________________________________________________________________________________
class CEntity
{
public:
	CEntity(CEntitySlot* pSlot);
	virtual ~CEntity();
	
	bool UpdateDependsOn(CEntity* pOther);	/* must only be called during PreUpdate phase */ 

public:
	virtual bool PreUpdate(CFrame* pFrame, float dt)	=0;
	virtual bool Update(CFrame* pFrame, float dt)		=0;
	
	virtual bool Draw(CDisplayListAdder* pDL, uint32_t EntityKey, int iView)=0;
	
	virtual bool Render(CGfx* pGfx, const CDisplayItemKey &Key, uint32_t EntityKey, ULONG Param) =0;

public:
	CEntitySlot* m_pSlot;
};

//________________________________________________________________________________
class CACHELINE_ALIGNED CEntitySlot
{
public:
	void Init(CEntity* pEntity);
	
public:
	/* operations */ 
	bool PreUpdate(CFrame* pFrame, float dt)
	{
		m_PredecessorCount = 0;
		
		if (m_Flags & ENTITY_NOPREUPDATE)
			return true;
			
		return m_pEntity->PreUpdate(pFrame, dt);
	}
	
	bool Update(CFrame* pFrame, float dt)
	{
		/* dependent entities updated by their last running parent */ 
		if (m_Flags & ENTITY_DEPENDENT) 
			return true;
		
		return _Update(pFrame, dt);
	}
	
	bool _Update(CFrame* pFrame, float dt)
	{
		bool bOk;
		
		/* update this entity */ 	
		if (m_Flags & ENTITY_NOUPDATE_internal)
			 bOk = true;
		else bOk = m_pEntity->Update(pFrame, dt);
		
		/* update dependents */ 
		if (m_DependentCount)
			UpdateDependents(pFrame, dt);
		
		return bOk;
	}
	
	/* flags */ 
	void SetFlag(DWORD f)
	{
		m_Flags |= f;
	}

	void ResetFlag(DWORD f)
	{
		m_Flags &= ~f;
	}
	
public: /* internal use */ 	
	void UpdateDependents(CFrame* pFrame, float dt);
	
public:
	volatile LONG	m_DependentCount;				 /* NOTE: far away from other volatile to avoid false sharing */ 
	CEntitySlot*	m_Dependent[MAX_DEPENDENTCOUNT]; /* entities depending on us  */ 
	volatile LONG	m_PredecessorCount;				 /* entities we wait for	  */ 

	CEntity*		m_pEntity;
	DWORD			m_Flags;	
};

#endif // _NULSTEIN_ENTITY_H_
