//-------------------------------------------------------------------------------------
//
// Copyright 2009 Intel Corporation
// All Rights Reserved
//
// Permission is granted to use, copy, distribute and prepare derivative works of this
// software for any purpose and without fee, provided, that the above copyright notice
// and this statement appear in all copies.  Intel makes no representations about the
// suitability of this software for any purpose.  THIS SOFTWARE IS PROVIDED "AS IS."
// INTEL SPECIFICALLY DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, AND ALL LIABILITY,
// INCLUDING CONSEQUENTIAL AND OTHER INDIRECT DAMAGES, FOR THE USE OF THIS SOFTWARE,
// INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PROPRIETARY RIGHTS, AND INCLUDING THE
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  Intel does not
// assume any responsibility for any errors which may appear in this software nor any
// responsibility to update it.
//

#include "ReturnValues.h"
#include "DisplayItem.h"

#pragma pack(push, 1)
struct sKey_Opaque
{
	uint8_t		LowZ;
	uint16_t	HighZ;
	uint8_t		MtlPass;

	uint16_t	Mtl;
	uint16_t	BaseKey;
};

struct sKey_Alpha
{
	uint8_t		MtlPass;
	uint16_t	Mtl;
	uint8_t		LowZ;

	uint16_t	HighZ;
	uint16_t	BaseKey;
};

struct sKey_Command
{
	uint32_t	Param;

	uint16_t	Priority;
	uint16_t	BaseKey;
};

union uKey
{
	uint64_t		Value;
	sKey_Opaque		Opaque;
	sKey_Alpha		Alpha;
	sKey_Command	Command;
};

#pragma pack(pop)



//________________________________________________________________________________
void CDisplayItemKey::MakeOutOfRange()
{
	m_iRenderTarget		= -1;
	m_iLayer			= -1;
	m_iViewport			= -1;
	m_iViewportLayer	= -1;
	m_AlphaBlend		= -1; 
	m_isCmd				= -1;
	
	m_CmdPriority		= -1;
	m_CmdParam			= -1;
	
	m_MtlId				= -1;
	m_MtlPass			= -1;
	m_Depth				= -1.0f; 
}

void CDisplayItemKey::Clear()
{
	m_iRenderTarget		= 0;
	m_iLayer			= 0;
	m_iViewport			= 0;
	m_iViewportLayer	= 0;
	m_AlphaBlend		= 0; 
	m_isCmd				= 0;
	
	m_CmdPriority		= 0;
	m_CmdParam			= 0;
	
	m_MtlId				= 0;
	m_MtlPass			= 0;
	m_Depth				= 0.0f; 
}

CDisplayItemKey::operator uint64_t()
{
	uKey		A;
	uint16_t	BaseKey;
	
	BaseKey	=	(m_iRenderTarget	<< 11)	/* 5 bits (32)   */ 
			|	(m_iLayer			<<  9)	/* 2 bits ( 4)   */ 
			|	(m_iViewport		<<  4)	/* 5 bits (32)   */ 
			|	(m_iViewportLayer	<<  2);	/* 2 bits ( 4)   */ 
											/* 1 bit = alpha */ 
											/* 1 bit = !cmd   */ 
	if (m_AlphaBlend) 
		BaseKey |= 2;

	if (m_isCmd)
	{
		A.Command.BaseKey	= BaseKey;
		A.Command.Priority	= m_CmdPriority;
		A.Command.Param		= m_CmdParam;
	}
	else
	{	
		BaseKey |= 1;

		if (m_AlphaBlend) 
		{	/* translucent, sort back to front and by material */ 
			int z = int( (1.0f-m_Depth) * ((1<<24)-1) );
			
			A.Alpha.BaseKey		= BaseKey;
			A.Alpha.HighZ		= z >> 8;
			A.Alpha.LowZ		= z & 0xFF;
			A.Alpha.Mtl			= m_MtlId;
			A.Alpha.MtlPass		= m_MtlPass;
		}
		else
		{	/* opaque, sort by material and front to back */ 
			int z = int( (m_Depth) * ((1<<24)-1) );

			A.Opaque.BaseKey	= BaseKey;
			A.Opaque.HighZ		= z >> 8;
			A.Opaque.LowZ		= z & 0xFF;
			A.Opaque.Mtl		= m_MtlId;
			A.Opaque.MtlPass	= m_MtlPass;
		}
	}
	
	return A.Value;
}

void CDisplayItemKey::Decode(uint64_t Key)
{
	uKey	A;
	int		z;
	
	A.Value = Key;
	
	m_iRenderTarget	= (A.Opaque.BaseKey >> 11) & 0x001F;
	m_iLayer		= (A.Opaque.BaseKey >>  9) & 0x0003;
	m_iViewport		= (A.Opaque.BaseKey >>  4) & 0x001F;
	m_iViewportLayer= (A.Opaque.BaseKey >>  2) & 0x0003;
	
	m_AlphaBlend	=     A.Opaque.BaseKey & 2;
	m_isCmd			= 0==(A.Opaque.BaseKey & 1);
	
	if (m_isCmd)
	{
		m_CmdPriority = A.Command.Priority;
		m_CmdParam	= A.Command.Param;
		
		m_MtlId		= 0;
		m_MtlPass	= 0;
		m_Depth		= 0.0f;
	}
	else
	{
		m_CmdPriority	= 0;
		m_CmdParam		= 0;
		
		if (m_AlphaBlend)
		{	/* alpha blended */ 
			z			= (A.Alpha.HighZ << 8) | (A.Alpha.LowZ);
			m_MtlId		=  A.Alpha.Mtl;
			m_MtlPass	=  A.Alpha.MtlPass;
			
			m_Depth	= 1.0f - (z / float((1<<24)-1));
		}
		else
		{	/* opaque */ 
			z			= (A.Opaque.HighZ << 8) | (A.Opaque.LowZ);
			m_MtlId		= A.Opaque.Mtl;
			m_MtlPass	= A.Opaque.MtlPass;

			m_Depth	= z / float((1<<24)-1);
		}
	}
}
