//-------------------------------------------------------------------------------------
//
// Copyright 2009 Intel Corporation
// All Rights Reserved
//
// Permission is granted to use, copy, distribute and prepare derivative works of this
// software for any purpose and without fee, provided, that the above copyright notice
// and this statement appear in all copies.  Intel makes no representations about the
// suitability of this software for any purpose.  THIS SOFTWARE IS PROVIDED "AS IS."
// INTEL SPECIFICALLY DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, AND ALL LIABILITY,
// INCLUDING CONSEQUENTIAL AND OTHER INDIRECT DAMAGES, FOR THE USE OF THIS SOFTWARE,
// INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PROPRIETARY RIGHTS, AND INCLUDING THE
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  Intel does not
// assume any responsibility for any errors which may appear in this software nor any
// responsibility to update it.
//

#ifndef _NULSTEIN_GFXMATH_H_
#define _NULSTEIN_GFXMATH_H_

#include <math.h>
#include <string.h>

#ifndef _M_X64
 #include <emmintrin.h>
 #include "AMath/AMaths.h"
#endif

#ifdef _M_X64
	inline float Pow(float x, float y)					{return powf(x,y);}
	inline float rsqrt(float a)							{return 1.0f/sqrtf(a);}
	inline float Cos(float a)							{return cosf(a);}
	inline float Tan(float a)							{return tanf(a);}
	inline void	 SinCos(float a, float *pC, float *pS)	{*pC=cosf(a); *pS=sinf(a);}
#else
	inline
	float Pow(float x, float y)
	{
		float	a;

		_mm_store_ss(	&a, 
						am_pow_ess(
							_mm_load_ss(&x),
							_mm_load_ss(&y) ) );

		return a;
	}

	inline 
	float rsqrt(float a)
	{
		_mm_store_ss(	&a, 
						_mm_rsqrt_ps( _mm_load1_ps(&a)) );
		return a;
	}
	
	inline
	float Cos(float a)
	{	
		float c;
		
		_mm_store_ss(	&c, 
						am_cos_ess( _mm_load_ss(&a) ) );
		return c;
	}
	
	inline
	float Tan(float a)
	{	
		float c;
		
		_mm_store_ss(	&c, 
						am_tan_ess( _mm_load_ss(&a) ) );
		return c;
	}
	
	inline void SinCos(float a, float *pC, float *pS)	
	{	
		__m128 mmAlpha, mmSin, mmCos;

		mmAlpha = _mm_load_ss(&a);
		am_sincos_ess(mmAlpha , &mmSin, &mmCos);
		_mm_store_ss(pC, mmCos);
		_mm_store_ss(pS, mmSin);
	};

#endif

//________________________________________________________________________________
class CVector3
{
public:
	CVector3(){}
	
	CVector3(const CVector3& V)				{ Set(V); }
	CVector3(float _x, float _y, float _z)	{ Set(_x,_y,_z); }

public:	
	void Set(const CVector3& V)
	{
		x=V.x; y=V.y; z=V.z;
	}

	void Set(float _x, float _y, float _z)
	{ 
		x=_x; y=_y; z=_z;
	}
	
	const CVector3 &operator += (const CVector3 &V);
	const CVector3 &operator -= (const CVector3 &V);
	
public:
	float x, y, z;
};

float	 Dot(const CVector3 &A, const CVector3 &B);
CVector3 Cross(const CVector3 &A, const CVector3 &B);
CVector3 Normalize(const CVector3 &V);
float	 SquareNorm(const CVector3 &V);

CVector3 operator *(const CVector3 &V, float k);
CVector3 operator +(const CVector3 &A, const CVector3 &B);
CVector3 operator -(const CVector3 &A, const CVector3 &B);

//________________________________________________________________________________
class CVector4
{
public:
	CVector4(){}
	
	CVector4(const CVector4& V)							{ Set(V);				}
	CVector4(const CVector3& V, float _w)				{ Set(V.x,V.y,V.z, _w);	}
	CVector4(float _x, float _y, float _z, float _w)	{ Set(_x, _y, _z, _w);	}

public:	
	void Set(const CVector4& V)
	{
		x=V.x; y=V.y; z=V.z; w=V.w; 
	}

	void Set(const CVector3& V, float _w)
	{
		x=V.x; y=V.y; z=V.z; w=_w; 
	}

	void Set(float _x, float _y, float _z, float _w)
	{ 
		x=_x; y=_y; z=_z; w=_w; 
	}
	
public:
	float x, y, z, w;
};

CVector4 operator *(const CVector4 &V, float k);

class CRGBA:public CVector4
{
public:
	CRGBA(int r, int g, int b):CVector4(r/255.0f, g/255.0f, b/255.0f, 1.0f) {}
	CRGBA(int r, int g, int b, int a):CVector4(r/255.0f, g/255.0f, b/255.0f, a/255.0f) {}
};

//________________________________________________________________________________
class CRotation
{
public:
	CRotation(){}
	
	CRotation(const CRotation &R)					{ Set(R);			}
	CRotation(const CVector3 &Axis, float angle)	{ Set(Axis, angle); }

public:
	void Set(const CRotation &R)
	{
		Axis = R.Axis; Angle = R.Angle;
	}

	void Set(const CVector3 &_Axis, float _Angle)
	{
		Axis  = _Axis;
		Angle = _Angle;
	}

public:
	CVector3	Axis;
	float		Angle;
};

//________________________________________________________________________________
class CTransform
{
public:
	CRotation	Rot;
	CVector3	Pos;
	float		Scl;
};

//________________________________________________________________________________
#define MTXSET(M, _m11,_m12,_m13,_m14, _m21,_m22,_m23,_m24, _m31,_m32,_m33,_m34, _m41,_m42,_m43,_m44 ) do{ \
		(M).m11=(_m11); (M).m12=(_m12); (M).m13=(_m13); (M).m14=(_m14);	\
		(M).m21=(_m21); (M).m22=(_m22); (M).m23=(_m23); (M).m24=(_m24);	\
		(M).m31=(_m31); (M).m32=(_m32); (M).m33=(_m33); (M).m34=(_m34);	\
		(M).m41=(_m41); (M).m42=(_m42); (M).m43=(_m43); (M).m44=(_m44);	\
	} while(0)

#define MTXSETthis(_m11,_m12,_m13,_m14, _m21,_m22,_m23,_m24, _m31,_m32,_m33,_m34, _m41,_m42,_m43,_m44 ) do{ \
		m11=(_m11); m12=(_m12); m13=(_m13); m14=(_m14);	\
		m21=(_m21); m22=(_m22); m23=(_m23); m24=(_m24);	\
		m31=(_m31); m32=(_m32); m33=(_m33); m34=(_m34);	\
		m41=(_m41); m42=(_m42); m43=(_m43); m44=(_m44);	\
	} while(0)

class CMatrix 
{
public:
	CMatrix(){}
	
	CMatrix(const CMatrix &X)	{ Set(X); }
	CMatrix(const CTransform &T) { Set(T); }

public:
	void SetIdentity()
	{
		     m12= m13= m14=
		m21=      m23= m24=
		m31= m32=      m34=
		m41= m42= m43=      0.0f; 
		
		m11= m22= m33= m44= 1.0f;
	}

	void Set(const CMatrix &X)
	{
		memcpy(&m11, &X.m11, 16*sizeof(float));
	}
	
	void Set(const float M[16])
	{
		memcpy(&m11, M, 16*sizeof(float));
	}
	
	void Set( const CTransform &T);

public:
	float m11, m12, m13, m14;
	float m21, m22, m23, m24;
	float m31, m32, m33, m34;
	float m41, m42, m43, m44;
};


const CMatrix& MtxMul(CMatrix* pAtoC, const CMatrix& AtoB, const CMatrix& BtoC);
const CMatrix& Transpose(CMatrix* pDest, const CMatrix &Src);
const CMatrix& MtxLookat(CMatrix* pDest, const CVector3 &Eye, const CVector3 &At, const CVector3 &Up);
const CMatrix& MtxPerspective(CMatrix* pDest, float yFOV, float Aspect, float Near, float Far);

const CVector3& MtxMulPoint(CVector3* pRes, const CVector3& A, const CMatrix& M);
const CVector3& MtxMulVector(CVector3* pRes, const CVector3& IAn, const CMatrix& M);
const CVector4& MtxMulPoint(CVector4* pRes, const CVector3& A, const CMatrix& M);
const CVector4& MtxMulVector(CVector4* pRes, const CVector3& IAn, const CMatrix& M);

const CVector4& MtxTransposeMulPoint(CVector4* pRes, const CVector3& A, const CMatrix& M);

CMatrix operator *(const CMatrix& AtoB, const CMatrix& BtoC);

//________________________________________________________________________________
#include "GfxMath.inl"

#endif //_NULSTEIN_GFXMATH_H_