// Copyright (C) 1999-2000 Id Software, Inc.
//

#include "g_local.h"


/*
===============
G_DamageFeedback

Called just before a snapshot is sent to the given player.
Totals up all damage and generates both the player_state_t
damage values to that client for pain blends and kicks, and
global pain sound events for all clients.
===============
*/
void P_DamageFeedback( gentity_t *player ) {
	gclient_t	*client;
	float	count;
	vec3_t	angles;

	client = player->client;
	if ( client->ps.pm_type == PM_DEAD ) {
		return;
	}

	// total points of damage shot at the player this frame
	count = client->damage_blood + client->damage_armor;
	if ( count == 0 ) {
		return;		// didn't take any damage
	}

	if ( count > 255 ) {
		count = 255;
	}

	// send the information to the client

	// world damage (falling, slime, etc) uses a special code
	// to make the blend blob centered instead of positional
	if ( client->damage_fromWorld ) {
		client->ps.damagePitch = 255;
		client->ps.damageYaw = 255;

		client->damage_fromWorld = qfalse;
	} else {
		vectoangles( client->damage_from, angles );
		client->ps.damagePitch = angles[PITCH]/360.0 * 256;
		client->ps.damageYaw = angles[YAW]/360.0 * 256;
	}

	// play an apropriate pain sound
	if ( (level.time > player->pain_debounce_time) && !(player->flags & FL_GODMODE) ) {
		player->pain_debounce_time = level.time + 700;		
		G_AddEvent( player, EV_PAIN, player->health );
		client->ps.damageEvent++;
	}


	client->ps.damageCount = count;

	//
	// clear totals
	//
	client->damage_blood = 0;
	client->damage_armor = 0;
	client->damage_knockback = 0;
}



/*
=============
P_WorldEffects

Check for lava / slime contents and drowning
=============
*/
void P_WorldEffects( gentity_t *ent ) {
	qboolean	envirosuit;
	int			waterlevel;

	if ( ent->client->noclip ) {
		ent->client->airOutTime = level.time + 12000;	// don't need air
		return;
	}

	waterlevel = ent->waterlevel;

	envirosuit = ent->client->ps.powerups[PW_BATTLESUIT] > level.time;

	//
	// check for drowning
	//
	if ( waterlevel == 3 ) {
		// envirosuit give air
		if ( envirosuit ) {
			ent->client->airOutTime = level.time + 10000;
		}

		// if out of air, start drowning
		if ( ent->client->airOutTime < level.time) {
			// drown!
			ent->client->airOutTime += 1000;
			if ( ent->health > 0 ) {
				// take more damage the longer underwater
				ent->damage += 2;
				if (ent->damage > 15)
					ent->damage = 15;

				// play a gurp sound instead of a normal pain sound
				if (ent->health <= ent->damage) {
					G_Sound(ent, CHAN_VOICE, G_SoundIndex("*drown.wav"));
				} else if (rand()&1) {
					G_Sound(ent, CHAN_VOICE, G_SoundIndex("sound/player/gurp1.wav"));
				} else {
					G_Sound(ent, CHAN_VOICE, G_SoundIndex("sound/player/gurp2.wav"));
				}

				// don't play a normal pain sound
				ent->pain_debounce_time = level.time + 200;

				G_Damage (ent, NULL, NULL, NULL, NULL, 
					ent->damage, DAMAGE_NO_ARMOR | DAMAGE_NO_BLEEDING, MOD_WATER);
			}
		}
	} else {
		ent->client->airOutTime = level.time + 12000;
		ent->damage = 2;
	}

	//
	// check for sizzle damage (move to pmove?)
	//
	if (waterlevel && 
		(ent->watertype&(CONTENTS_LAVA|CONTENTS_SLIME)) ) {
		if (ent->health > 0
			&& ent->pain_debounce_time <= level.time	) {

			if ( envirosuit ) {
				G_AddEvent( ent, EV_POWERUP_BATTLESUIT, 0 );
			} else { // ## Hentai ## - these have been sped up and toned down, damage per second is roughly the same
				if (ent->watertype & CONTENTS_SLIME) {
					G_Damage (ent, NULL, NULL, NULL, NULL, 
						3 * waterlevel, 0, MOD_SLIME);
					ent->pain_debounce_time = level.time + 200;
				}
				if (ent->watertype & CONTENTS_LAVA) {
					G_Damage (ent, NULL, NULL, NULL, NULL, 
						5 * waterlevel, DAMAGE_NO_BLEEDING, MOD_LAVA);
					ent->pain_debounce_time = level.time + 100;
				}

				
				
			}
		}
	}
}



/*
===============
G_SetClientSound
===============
*/
void G_SetClientSound( gentity_t *ent ) {
	if (ent->waterlevel && (ent->watertype&(CONTENTS_LAVA|CONTENTS_SLIME)) )
		ent->s.loopSound = level.snd_fry;
	else
		ent->s.loopSound = 0;
}



//==============================================================

/*
==============
ClientImpacts
==============
*/
void ClientImpacts( gentity_t *ent, pmove_t *pm ) {
	int		i, j;
	trace_t	trace;
	gentity_t	*other;

	memset( &trace, 0, sizeof( trace ) );
	for (i=0 ; i<pm->numtouch ; i++) {
		for (j=0 ; j<i ; j++) {
			if (pm->touchents[j] == pm->touchents[i] ) {
				break;
			}
		}
		if (j != i) {
			continue;	// duplicated
		}
		other = &g_entities[ pm->touchents[i] ];

		if ( ( ent->r.svFlags & SVF_BOT ) && ( ent->touch ) ) {
			ent->touch( ent, other, &trace );
		}

		if ( !other->touch ) {
			continue;
		}

		other->touch( other, ent, &trace );
	}

}

/*
============
G_TouchTriggers

Find all trigger entities that ent's current position touches.
Spectators will only interact with teleporters.
============
*/
void	G_TouchTriggers( gentity_t *ent ) {
	int			i, num;
	int			touch[MAX_GENTITIES];
	gentity_t	*hit;
	trace_t		trace;
	vec3_t		mins, maxs;
	static vec3_t	range = { 40, 40, 52 };

	if ( !ent->client ) {
		return;
	}

	// dead clients don't activate triggers!
	if ( ent->client->ps.stats[STAT_HEALTH] <= 0 ) {
		return;
	}

	VectorSubtract( ent->client->ps.origin, range, mins );
	VectorAdd( ent->client->ps.origin, range, maxs );

	num = trap_EntitiesInBox( mins, maxs, touch, MAX_GENTITIES );

	// can't use ent->absmin, because that has a one unit pad
	VectorAdd( ent->client->ps.origin, ent->r.mins, mins );
	VectorAdd( ent->client->ps.origin, ent->r.maxs, maxs );

	for ( i=0 ; i<num ; i++ ) {
		hit = &g_entities[touch[i]];

		if ( !hit->touch && !ent->touch ) {
			continue;
		}
		if ( !( hit->r.contents & CONTENTS_TRIGGER ) ) {
			continue;
		}

		// ignore most entities if a spectator
		if ( ent->client->sess.sessionTeam == TEAM_SPECTATOR ) {
			if ( hit->s.eType != ET_TELEPORT_TRIGGER &&
				// this is ugly but adding a new ET_? type will
				// most likely cause network incompatibilities
				hit->touch != Touch_DoorTrigger) {
				continue;
			}
		}

		// use seperate code for determining if an item is picked up
		// so you don't have to actually contact its bounding box
		if ( hit->s.eType == ET_ITEM ) {
			if ( !BG_PlayerTouchesItem( &ent->client->ps, &hit->s, level.time ) ) {
				continue;
			}
		} else {
			if ( !trap_EntityContact( mins, maxs, hit ) ) {
				continue;
			}
		}

		memset( &trace, 0, sizeof(trace) );

		if ( hit->touch ) {
			hit->touch (hit, ent, &trace);
		}

		if ( ( ent->r.svFlags & SVF_BOT ) && ( ent->touch ) ) {
			ent->touch( ent, hit, &trace );
		}
	}
}

/*
=================
SpectatorThink
=================
*/
void SpectatorThink( gentity_t *ent, usercmd_t *ucmd ) {
	pmove_t	pm;
	gclient_t	*client;

	client = ent->client;

	if ( client->sess.spectatorState != SPECTATOR_FOLLOW ) {
		client->ps.pm_type = PM_SPECTATOR;
		client->ps.speed = 400;	// faster than normal

		// set up for pmove
		memset (&pm, 0, sizeof(pm));
		pm.ps = &client->ps;
		pm.cmd = *ucmd;
		pm.tracemask = MASK_PLAYERSOLID & ~CONTENTS_BODY;	// spectators can fly through bodies
		pm.trace = trap_Trace;
		pm.pointcontents = trap_PointContents;

		// perform a pmove
		Pmove (&pm);

		// save results of pmove
		VectorCopy( client->ps.origin, ent->s.origin );

		G_TouchTriggers( ent );
		trap_UnlinkEntity( ent );
	}

	client->oldbuttons = client->buttons;
	client->buttons = ucmd->buttons;

	// attack button cycles through spectators
	if ( ( client->buttons & BUTTON_ATTACK ) && ! ( client->oldbuttons & BUTTON_ATTACK ) ) {
		Cmd_FollowCycle_f( ent, 1 );
	}
}



/*
=================
ClientInactivityTimer

Returns qfalse if the client is dropped
=================
*/
qboolean ClientInactivityTimer( gclient_t *client ) {
	if ( ! g_inactivity.integer ) {
		// give everyone some time, so if the operator sets g_inactivity during
		// gameplay, everyone isn't kicked
		client->inactivityTime = level.time + 60 * 1000;
		client->inactivityWarning = qfalse;
	} else if ( client->pers.cmd.forwardmove || 
		client->pers.cmd.rightmove || 
		client->pers.cmd.upmove ||
		(client->pers.cmd.buttons & BUTTON_ATTACK) ) {
		client->inactivityTime = level.time + g_inactivity.integer * 1000;
		client->inactivityWarning = qfalse;
	} else if ( !client->pers.localClient ) {
		if ( level.time > client->inactivityTime ) {
			trap_DropClient( client - level.clients, "Dropped due to inactivity" );
			return qfalse;
		}
		if ( level.time > client->inactivityTime - 10000 && !client->inactivityWarning ) {
			client->inactivityWarning = qtrue;
			trap_SendServerCommand( client - level.clients, "cp \"Ten seconds until inactivity drop!\n\"" );
		}
	}
	return qtrue;
}

/*
==================
ClientTimerActions

Actions that happen once a second
==================
*/
void ClientTimerActions( gentity_t *ent, int msec ) {
	gclient_t *client;

	client = ent->client;
	client->timeResidual += msec;

	while ( client->timeResidual >= 1000 ) {
		client->timeResidual -= 1000;

		// regenerate
		if ( client->ps.powerups[PW_REGEN] ) {
			if ( ent->health < client->ps.stats[STAT_MAX_HEALTH]) {
				ent->health += 15;
				if ( ent->health > client->ps.stats[STAT_MAX_HEALTH] * 1.1 ) {
					ent->health = client->ps.stats[STAT_MAX_HEALTH] * 1.1;
				}
				G_AddEvent( ent, EV_POWERUP_REGEN, 0 );
			} else if ( ent->health < client->ps.stats[STAT_MAX_HEALTH] * 2) {
				ent->health += 5;
				if ( ent->health > client->ps.stats[STAT_MAX_HEALTH] * 2 ) {
					ent->health = client->ps.stats[STAT_MAX_HEALTH] * 2;
				}
				G_AddEvent( ent, EV_POWERUP_REGEN, 0 );
			}
		} else {
			// count down health when over max
			if ( ent->health > client->ps.stats[STAT_MAX_HEALTH] ) {
				ent->health--;
			}
		}

		// count down armor when over max
		if ( client->ps.stats[STAT_ARMOR] > client->ps.stats[STAT_MAX_HEALTH] ) {
			client->ps.stats[STAT_ARMOR]--;
		}

	}
}

/*
====================
ClientIntermissionThink
====================
*/
void ClientIntermissionThink( gclient_t *client ) {
	client->ps.eFlags &= ~EF_TALK;
	client->ps.eFlags &= ~EF_FIRING;

	// the level will exit when everyone wants to or after timeouts

	// swap and latch button actions
	client->oldbuttons = client->buttons;
	client->buttons = client->pers.cmd.buttons;
	if ( client->buttons & ( BUTTON_ATTACK | BUTTON_USE_HOLDABLE ) & ( client->oldbuttons ^ client->buttons ) ) {
		client->readyToExit ^= 1;
	}
}


/*
================
ClientEvents

Events will be passed on to the clients for presentation,
but any server game effects are handled here
================
*/
void ClientEvents( gentity_t *ent, int oldEventSequence ) {
	int		i;
	int		event;
	gclient_t *client;
	int		damage;
	vec3_t	dir;
	vec3_t	origin, angles;
//	qboolean	fired;
	gitem_t *item;
	gentity_t *drop;

	client = ent->client;

	if ( oldEventSequence < client->ps.eventSequence - MAX_PS_EVENTS ) {
		oldEventSequence = client->ps.eventSequence - MAX_PS_EVENTS;
	}
	for ( i = oldEventSequence ; i < client->ps.eventSequence ; i++ ) {
		event = client->ps.events[ i & (MAX_PS_EVENTS-1) ];

		switch ( event ) {
		case EV_FALL_SHORT:
			if(ent->client->bleeding <= 0 && ent->client->ps.stats[STAT_LEG_DAMAGE] <= 0)
				break;
		case EV_FALL_MEDIUM:
		case EV_FALL_FAR:
			if ( ent->s.eType != ET_PLAYER ) {
				break;		// not in the player model
			}
			if ( g_dmflags.integer & DF_NO_FALLING ) {
				break;
			}
			if ( event == EV_FALL_FAR ) {
				damage = 10;
			} else if ( event == EV_FALL_MEDIUM ) {
				damage = 5;
			}
			else
			{
				damage = 2;
			}
			VectorSet (dir, 0, 0, 1);
			ent->pain_debounce_time = level.time + 200;	// no normal pain sound
			G_Damage (ent, NULL, NULL, NULL, NULL, damage, DAMAGE_NO_BLEEDING, MOD_FALLING);
			if(ent->client->bleeding || ent->client->ps.stats[STAT_LEG_DAMAGE])
				G_Damage (ent, ent, ent->enemy, NULL, NULL, (int) ((ent->client->bleeding + ent->client->ps.stats[STAT_LEG_DAMAGE] * 1.0 / ent->client->pers.maxHealth) * damage), DAMAGE_NO_ARMOR, MOD_BLEED);
			break;

		case EV_FIRE_WEAPON:
			FireWeapon( ent );
			break;

		case EV_USE_ITEM1:		// teleporter
			// drop flags in CTF
			item = NULL;

			if ( ent->client->ps.powerups[ PW_REDFLAG ] ) {
				item = BG_FindItemForPowerup( PW_REDFLAG );
				i = PW_REDFLAG;
			} else if ( ent->client->ps.powerups[ PW_BLUEFLAG ] ) {
				item = BG_FindItemForPowerup( PW_BLUEFLAG );
				i = PW_BLUEFLAG;
			}

			if ( item ) {
				drop = Drop_Item( ent, item, 0 );
				// decide how many seconds it has left
				drop->count = ( ent->client->ps.powerups[ i ] - level.time ) / 1000;
				if ( drop->count < 1 ) {
					drop->count = 1;
				}

				ent->client->ps.powerups[ i ] = 0;
			}

			SelectSpawnPoint( ent->client->ps.origin, origin, angles );
			TeleportPlayer( ent, origin, angles );
			break;

		case EV_USE_ITEM2:		// medkit
			ent->health = ent->client->ps.stats[STAT_MAX_HEALTH];
			ent->client->bleeding = 0;
			ent->client->bleed_damage = 0;
			ent->client->bleedtime = level.time;
			break;

		default:
			break;
		}
	}

}

void BotTestSolid(vec3_t origin);

/*
==============
ClientThink

This will be called once for each client frame, which will
usually be a couple times for each server frame on fast clients.

If "g_syncronousClients 1" is set, this will be called exactly
once for each server frame, which makes for smooth demo recording.
==============
*/
void TossClientWeapon(gentity_t *self, int weapon);


static float G_PlayerAngles(gclient_t *client, vec3_t muzzle, vec3_t headpoint ) {
	playerState_t *ps = &client->ps;
	clientPersistant_t *pers = &client->pers;
	vec3_t		legsAngles, torsoAngles, headAngles;
	//vec3_t		legs[3], torso[3], head[3];
	vec3_t		waistpoint, neckpoint, gunpoint;
	vec3_t		forward, right, up;
	float		dest, *tempv;
	static	int	movementOffsets[8] = { 0, 22, 45, -22, 0, 22, -45, -22 };
	vec3_t		velocity;
	float		speed;
	int			dir;
	
	if((ps->eFlags & EF_DEAD) || (pers->playerclass < 0) || (pers->playerclass >= p_classes))
	{
		muzzle[0] = 443556;
		return 0; // dead players don't matter.
	}
	
	

	VectorCopy( ps->viewangles, headAngles );
	headAngles[YAW] = AngleMod( headAngles[YAW] );
	VectorClear( legsAngles );
	VectorClear( torsoAngles );

	// --------- yaw -------------

	// allow yaw to drift a bit
	if ( ( ps->legsAnim & ~ANIM_TOGGLEBIT ) != LEGS_IDLE 
		|| ( ps->torsoAnim & ~ANIM_TOGGLEBIT ) != TORSO_STAND  ) {
		// if not standing still, always point all in the same direction

	}

	// adjust legs for movement dir
	if ( ps->eFlags & EF_DEAD ) {
		// don't let dead bodies twitch
		dir = 0;
	} 
	else 
	{
		dir = ps->movementDir & 7;
	}
	legsAngles[YAW] = headAngles[YAW] + movementOffsets[ dir ];
	torsoAngles[YAW] = headAngles[YAW] + 0.25 * movementOffsets[ dir ];


	// --------- pitch -------------

	// only show a fraction of the pitch angle in the torso
	if ( headAngles[PITCH] > 180 ) {
		dest = (-360 + headAngles[PITCH]) * 0.75;
	} else {
		dest = headAngles[PITCH] * 0.75;
	}
	torsoAngles[PITCH] = dest;

	// --------- roll -------------


	// lean towards the direction of travel
	
	VectorCopy( ps->velocity, velocity );
	speed = VectorNormalize( velocity );
	if ( speed ) {
		vec3_t	axis[3];
		float	side;

		speed *= 0.05;

		AnglesToAxis( legsAngles, axis );
		side = speed * DotProduct( velocity, axis[1] );
		legsAngles[ROLL] -= side;

		side = speed * DotProduct( velocity, axis[0] );
		legsAngles[PITCH] += side;
	}

	
	
	if(ps->pm_flags & PMF_DUCKED)
		tempv = p_class[pers->playerclass].crouch;
	else if(ps->groundEntityNum == ENTITYNUM_NONE)
		tempv = p_class[pers->playerclass].jump;
	else
		tempv = p_class[pers->playerclass].stand;
	AngleVectors(legsAngles, forward, right, up);

	waistpoint[0] = tempv[0] * forward[0] + tempv[1] * right[0] + tempv[2] * up[0];
	waistpoint[1] = tempv[0] * forward[1] + tempv[1] * right[1] + tempv[2] * up[1];
	waistpoint[2] = tempv[0] * forward[2] + tempv[1] * right[2] + tempv[2] * up[2];
	
	AngleVectors(torsoAngles, forward, right, up);

	tempv = p_class[pers->playerclass].head;
	neckpoint[0] = waistpoint[0] + tempv[0] * forward[0] + tempv[1] * right[0] + tempv[2] * up[0];
	neckpoint[1] = waistpoint[1] + tempv[0] * forward[1] + tempv[1] * right[1] + tempv[2] * up[1];
	neckpoint[2] = waistpoint[2] + tempv[0] * forward[2] + tempv[1] * right[2] + tempv[2] * up[2];
	
	tempv = p_class[pers->playerclass].gun;
	gunpoint[0] = waistpoint[0] + tempv[0] * forward[0] + tempv[1] * right[0] + tempv[2] * up[0];
	gunpoint[1] = waistpoint[1] + tempv[0] * forward[1] + tempv[1] * right[1] + tempv[2] * up[1];
	gunpoint[2] = waistpoint[2] + tempv[0] * forward[2] + tempv[1] * right[2] + tempv[2] * up[2];
	
	AngleVectors(headAngles, forward, right, up);
	
	if(client->ps.stats[STAT_EXTENDED_INFO] & EXT_WALLCLIMBER)
	{
	}
	else
	{
		headpoint[0] = neckpoint[0] + up[0] * 4;
		headpoint[1] = neckpoint[1] + up[1] * 4;
		headpoint[2] = neckpoint[2] + up[2] * 4;
	}
	muzzle[0] = gunpoint[0] + forward[0] * 8;
	muzzle[1] = gunpoint[1] + forward[1] * 8;
	muzzle[2] = gunpoint[2] + forward[2] * 8;
	
	//G_Printf("legyaw: %f torsoyaw: %f headyaw: %f\n", legsAngles[YAW], torsoAngles[YAW], headAngles[YAW]);
	return waistpoint[2];
}

void G_ExplodeMissile( gentity_t *ent );

void ClientThink_real( gentity_t *ent ) {
	gclient_t	*client;
	pmove_t		pm;
	vec3_t		oldOrigin;
	int			oldEventSequence;
	int			msec;
	usercmd_t	*ucmd;
	float jumpheight = 64.0;

	client = ent->client;
	if(!g_allowArmor.integer)
		client->ps.stats[STAT_ARMOR] = 0;
	
	// mark the time, so the connection sprite can be removed
	ucmd = &ent->client->pers.cmd;

	// sanity check the command time to prevent speedup cheating
	if ( ucmd->serverTime > level.time + 200 ) {
		ucmd->serverTime = level.time + 200;
//		G_Printf("serverTime <<<<<\n" );
	}
	if ( ucmd->serverTime < level.time - 1000 ) {
		ucmd->serverTime = level.time - 1000;
//		G_Printf("serverTime >>>>>\n" );
	} 

	msec = ucmd->serverTime - client->ps.commandTime;
	// following others may result in bad times, but we still want
	// to check for follow toggles
	if ( msec < 1 && client->sess.spectatorState != SPECTATOR_FOLLOW ) {
		return;
	}
	if ( msec > 200 ) {
		msec = 200;
	}

	//
	// check for exiting intermission
	//
	if ( level.intermissiontime ) {
		ClientIntermissionThink( client );
		return;
	}

	// spectators don't do much
	if ( client->sess.sessionTeam == TEAM_SPECTATOR ) {
		if ( client->sess.spectatorState == SPECTATOR_SCOREBOARD ) {
			return;
		}
		SpectatorThink( ent, ucmd );
		return;
	}

	// check for inactivity timer, but never drop the local client of a non-dedicated server
	if ( !ClientInactivityTimer( client ) ) {
		return;
	}

	// clear the rewards if time
	
	if(ent->s.eFlags & EF_DEAD)
	{
		if(g_LudicrousGibs.integer)
			G_AddEvent( ent, EV_GIB_PLAYER, ent->enemy->s.number );

		client->ps.eFlags &= ~(EF_AWARD_IMPRESSIVE | EF_AWARD_EXCELLENT | EF_AWARD_GAUNTLET );
		ent->s.eFlags &= ~(EF_AWARD_IMPRESSIVE | EF_AWARD_EXCELLENT | EF_AWARD_GAUNTLET );
		if(client->headhit)
		{
			client->ps.eFlags |= EF_AWARD_IMPRESSIVE;
			ent->s.eFlags |= EF_AWARD_IMPRESSIVE;
		}
		
		
	}
	else if ( level.time > client->rewardTime ) {
		client->ps.eFlags &= ~(EF_AWARD_IMPRESSIVE | EF_AWARD_EXCELLENT | EF_AWARD_GAUNTLET );
	}

	if ( client->noclip ) {
		client->ps.pm_type = PM_NOCLIP;
	} else if ( client->ps.stats[STAT_HEALTH] <= 0 ) {
		client->ps.pm_type = PM_DEAD;
	} else {
		client->ps.pm_type = PM_NORMAL;
	}

	client->ps.gravity = g_gravity.value;
	if(ucmd->upmove > 0 && client->pers.playerclass >= 0 && client->pers.playerclass < p_classes)
		client->ps.gravity *= p_class[client->pers.playerclass].jumpgravity;

	// set speed
	
	client->ps.stats[STAT_JUMPSCALE] = 270; // to make sure a default gets set
	client->ps.stats[STAT_JUMP_PAUSE] = 250; // to make sure a default gets set
	if(client->ps.stats[STAT_LEG_DAMAGE] > 0 && client->ps.stats[STAT_HEALTH] > 0)
	{		
		client->ps.speed = g_speed.value * client->ps.stats[STAT_HEALTH] / (client->ps.stats[STAT_HEALTH] + client->ps.stats[STAT_LEG_DAMAGE]);
	}
	else
		client->ps.speed = g_speed.value;

	if(client->pers.playerclass >= 0 && client->pers.playerclass < p_classes)
	{
		int p = client->pers.playerclass;
		client->ps.stats[STAT_JUMP_PAUSE] = p_class[p].jumppause * 1000;
		if(client->ps.groundEntityNum == ENTITYNUM_NONE)
		{
			client->ps.speed *= p_class[p].jumpspeed;
		}
		else if(ucmd->upmove < 0)
		{
			client->ps.speed *= p_class[p].crouchspeed;
		}
		else
		{
			client->ps.speed *= p_class[p].speed;
		}
		if(client->ps.stats[STAT_LEG_DAMAGE] > 0 && client->ps.stats[STAT_HEALTH] > 0)
		{			
			client->ps.stats[STAT_JUMPSCALE] = 270 * sqrt(p_class[client->pers.playerclass].jumpfactor) * client->ps.stats[STAT_HEALTH] / (client->ps.stats[STAT_HEALTH] + client->ps.stats[STAT_LEG_DAMAGE] + 0.01);
		}
		else
			client->ps.stats[STAT_JUMPSCALE] = 270 * sqrt(p_class[client->pers.playerclass].jumpfactor);
	}
	else if(g_PlayerClasses.integer == 2) // force classes, and this isn't a valid class.
	{	// turn into a spectator
		StopFollowing(ent); 
		
	}
	
	if(client->ps.stats[STAT_EXTENDED_INFO] & EXT_LIMIT_WEAPONS) 
	{ // one weapon mode gives speed bonus/penalty based on weapon 
		float mass = client->ps.stats[STAT_MASS];
		if(mass != 0)
		{		
			mass += BG_FindItemForWeapon(client->ps.weapon)->mass;
			//if(client->ps.stats[STAT_HOLDABLE_ITEM])
			//	mass += BG_FindItemForHoldable(client->ps.stats[STAT_HOLDABLE_ITEM])->mass;
			client->ps.speed *= client->ps.stats[STAT_MASS] / mass;			
			client->ps.stats[STAT_JUMPSCALE] *= sqrt(client->ps.stats[STAT_MASS] / mass);
		}

		/*switch(client->ps.weapon)
		{
		case WP_NONE:
			//client->ps.speed *= 1.25;
			//client->ps.stats[STAT_JUMPSCALE] *= 1.1;
			break;
		case WP_GAUNTLET:
			client->ps.speed *= mass / (mass + 10.0);
			client->ps.stats[STAT_JUMPSCALE] *= sqrt(mass / (mass + 10.0));
			break;
		case WP_MACHINEGUN:
			client->ps.speed *= mass / (mass + 25.0);
			client->ps.stats[STAT_JUMPSCALE] *= sqrt(mass / (mass + 25.0));
			break;
		case WP_SHOTGUN:
			client->ps.speed *= mass / (mass + 20.0);
			client->ps.stats[STAT_JUMPSCALE] *= sqrt(mass / (mass + 20.0));
			break;
		case WP_GRENADE_LAUNCHER:
			client->ps.speed *= mass / (mass + 15.0);
			client->ps.stats[STAT_JUMPSCALE] *= sqrt(mass / (mass + 15.0));
			break;
		case WP_ROCKET_LAUNCHER:
			client->ps.speed *= mass / (mass + 60.0);
			client->ps.stats[STAT_JUMPSCALE] *= sqrt(mass / (mass + 60.0));
			break;
		case WP_LIGHTNING:
			client->ps.speed *= mass / (mass + 50.0);
			client->ps.stats[STAT_JUMPSCALE] *= sqrt(mass / (mass + 50.0));
			break;
		case WP_RAILGUN:
			client->ps.speed *= mass / (mass + 30.0);
			client->ps.stats[STAT_JUMPSCALE] *= sqrt(mass / (mass + 30.0));
			break;
		case WP_PLASMAGUN:
			client->ps.speed *= mass / (mass + 40.0);
			client->ps.stats[STAT_JUMPSCALE] *= sqrt(mass / (mass + 40.0));
			break;
		case WP_BFG:
			client->ps.speed *= mass / (mass + 100.0);
			client->ps.stats[STAT_JUMPSCALE] *= sqrt(mass / (mass + 100.0));
			break;
		default:
			break;
		}*/
	}

	if ( client->ps.powerups[PW_HASTE] ) {
		client->ps.speed *= 1.3;
	}

	// Let go of the hook if we aren't firing
	if ( client->ps.weapon == WP_GRAPPLING_HOOK &&
		client->hook && !( ucmd->buttons & BUTTON_ATTACK ) ) {
		Weapon_HookFree(client->hook);
	}

	if(client->ps.weapon == WP_NONE && client->ps.stats[STAT_WEAPONS] > 1 && client->ps.weaponstate == WEAPON_READY && client->ps.weaponTime == 0)
	{ // we've got a weapon - use it damnit!
		int i;
		for(i = 1; i < MAX_WEAPONS; i++)
		{
			if(client->ps.stats[STAT_WEAPONS] & (1 << i))
			{
				client->ps.weapon = i;
				client->ps.weaponstate = WEAPON_RAISING;
				break;
			}
		}
	}
	// set up for pmove
	oldEventSequence = client->ps.eventSequence;

	memset (&pm, 0, sizeof(pm));

	// check for the hit-scan gauntlet, don't let the action
	// go through as an attack unless it actually hits something
	if ( (client->ps.weapon == WP_GAUNTLET || client->ps.weapon == WP_NONE ) &&
		( ucmd->buttons & BUTTON_ATTACK ) && client->ps.weaponTime <= 0 ) {
		pm.gauntletHit = CheckMeleeAttack( ent );
	}

	pm.ps = &client->ps;
	pm.cmd = *ucmd;
	if ( pm.ps->pm_type == PM_DEAD ) {
		pm.tracemask = MASK_PLAYERSOLID & ~CONTENTS_BODY;
	}
	else {
		pm.tracemask = MASK_PLAYERSOLID;
	}
	pm.trace = trap_Trace;
	pm.pointcontents = trap_PointContents;
	pm.debugLevel = g_debugMove.integer;
	pm.noFootsteps = ( g_dmflags.integer & DF_NO_FOOTSTEPS ) > 0;

	VectorCopy( client->ps.origin, oldOrigin );

	// perform a pmove
	Pmove (&pm);
	if(g_OneWeaponPerPlayer.integer && ent->client->ps.weapon != WP_NONE && ((ent->client->ps.torsoAnim & ~ANIM_TOGGLEBIT) == TORSO_DROP) )
	{ // ## Hentai - gotta drop da gun
		TossClientWeapon(ent, ent->s.weapon);		
	}
	// save results of pmove
	if ( ent->client->ps.eventSequence != oldEventSequence ) {
		ent->eventTime = level.time;
	}
	BG_PlayerStateToEntityState( &ent->client->ps, &ent->s, qtrue );
	if ( !( ent->client->ps.eFlags & EF_FIRING ) ) {
		client->fireHeld = qfalse;		// for grapple
	}

#if 1
	// use the precise origin for linking
	VectorCopy( ent->client->ps.origin, ent->r.currentOrigin );
#else
	// use the snapped origin for linking so it matches client predicted versions
	VectorCopy( ent->s.pos.trBase, ent->r.currentOrigin );
#endif

	VectorCopy (pm.mins, ent->r.mins);
	VectorCopy (pm.maxs, ent->r.maxs);

	ent->waterlevel = pm.waterlevel;
	ent->watertype = pm.watertype;

	// execute client events
	ClientEvents( ent, oldEventSequence );

	// link entity now, after any personal teleporters have been used
	trap_LinkEntity (ent);
	if ( !ent->client->noclip ) {
		G_TouchTriggers( ent );
	}

	//test for solid areas in the AAS file
	BotTestSolid(ent->r.currentOrigin);

	// touch other objects
	ClientImpacts( ent, &pm );

	// swap and latch button actions
	client->oldbuttons = client->buttons;
	client->buttons = ucmd->buttons;
	client->latched_buttons |= client->buttons & ~client->oldbuttons;

	// check for respawning
	if ( client->ps.stats[STAT_HEALTH] <= 0 ) {
		// wait for the attack button to be pressed
		if ( level.time > client->respawnTime ) {
			// forcerespawn is to prevent users from waiting out powerups
			if ( g_forcerespawn.integer > 0 && 
				( level.time - client->respawnTime ) > g_forcerespawn.integer * 1000 ) {
				respawn( ent );
				return;
			}
		
			// pressing attack or use is the normal respawn method
			if ( ucmd->buttons & ( BUTTON_ATTACK | BUTTON_USE_HOLDABLE ) ) {
				respawn( ent );
			}
		}
		return;
	}	
	if(client->bleeding)
	{		
		//client->ps.powerups[PW_BLEEDING] = level.time + 30;
		if((level.time - client->bleedtime) >= 250)
		{
			client->bleed_damage += client->bleeding * (level.time - client->bleedtime) / 1000.0;
			client->bleedtime = level.time;
		}
		if(client->bleed_damage >= 1)
		{
			//ent->pain_debounce_time = level.time + rand() % 5;
			if(client->bleed_damage > client->ps.stats[STAT_HEALTH] + 5)
				client->bleed_damage = client->ps.stats[STAT_HEALTH] + 5;			
			if(client->ps.stats[STAT_HEALTH] > 0)
				G_Damage(ent, ent, ent->enemy, NULL, NULL, (int) floor(client->bleed_damage), DAMAGE_NO_ARMOR | DAMAGE_NO_BLEEDING, MOD_BLEED);
			else
				G_AddEvent(ent, EV_BLOOD, 16);
			client->bleed_damage -= floor(client->bleed_damage);
			if(client->ps.stats[STAT_HEALTH] > client->pers.maxHealth)
				client->bleeding = 0;
		}
	}
	//else
		//client->ps.powerups[PW_BLEEDING] = 0;
	client->waistheight = G_PlayerAngles(client, client->muzzle, client->headpos);
	// perform once-a-second actions
	ClientTimerActions( ent, msec );
}

/*
==================
ClientThink

A new command has arrived from the client
==================
*/
void ClientThink( int clientNum ) {
	gentity_t *ent;

	ent = g_entities + clientNum;
	trap_GetUsercmd( clientNum, &ent->client->pers.cmd );

	// mark the time we got info, so we can display the
	// phone jack if they don't get any for a while
	ent->client->lastCmdTime = level.time;

	if ( !g_syncronousClients.integer ) {
		ClientThink_real( ent );
	}
}


void G_RunClient( gentity_t *ent ) {
	if ( !g_syncronousClients.integer ) {
		return;
	}
	ent->client->pers.cmd.serverTime = level.time;
	ClientThink_real( ent );
}


/*
==================
SpectatorClientEndFrame

==================
*/
void SpectatorClientEndFrame( gentity_t *ent ) {
	gclient_t	*cl;

	// if we are doing a chase cam or a remote view, grab the latest info
	if ( ent->client->sess.spectatorState == SPECTATOR_FOLLOW ) {
		int		clientNum;

		clientNum = ent->client->sess.spectatorClient;

		// team follow1 and team follow2 go to whatever clients are playing
		if ( clientNum == -1 ) {
			clientNum = level.follow1;
		} else if ( clientNum == -2 ) {
			clientNum = level.follow2;
		}
		if ( clientNum >= 0 ) {
			cl = &level.clients[ clientNum ];
			if ( cl->pers.connected == CON_CONNECTED && cl->sess.sessionTeam != TEAM_SPECTATOR ) {
				ent->client->ps = cl->ps;
				ent->client->ps.pm_flags |= PMF_FOLLOW;
				return;
			} else {
				// drop them to free spectators unless they are dedicated camera followers
				if ( ent->client->sess.spectatorClient >= 0 ) {
					ent->client->sess.spectatorState = SPECTATOR_FREE;
					ClientBegin( ent->client - level.clients );
				}
			}
		}
	}

	if ( ent->client->sess.spectatorState == SPECTATOR_SCOREBOARD ) {
		ent->client->ps.pm_flags |= PMF_SCOREBOARD;
	} else {
		ent->client->ps.pm_flags &= ~PMF_SCOREBOARD;
	}
}

/*
==============
ClientEndFrame

Called at the end of each server frame for each connected client
A fast client will have multiple ClientThink for each ClientEdFrame,
while a slow client may have multiple ClientEndFrame between ClientThink.
==============
*/
void ClientEndFrame( gentity_t *ent ) {
	int			i;
	clientPersistant_t	*pers;

	if ( ent->client->sess.sessionTeam == TEAM_SPECTATOR ) {
		SpectatorClientEndFrame( ent );
		return;
	}

	pers = &ent->client->pers;

	// turn off any expired powerups
	for ( i = 0 ; i < MAX_POWERUPS ; i++ ) {
		if ( ent->client->ps.powerups[ i ] < level.time ) {
			ent->client->ps.powerups[ i ] = 0;
		}
	}

	// save network bandwidth
#if 0
	if ( !g_syncronousClients->integer && ent->client->ps.pm_type == PM_NORMAL ) {
		// FIXME: this must change eventually for non-sync demo recording
		VectorClear( ent->client->ps.viewangles );
	}
#endif

	//
	// If the end of unit layout is displayed, don't give
	// the player any normal movement attributes
	//
	if ( level.intermissiontime ) {
		return;
	}

	// burn from lava, etc
	P_WorldEffects (ent);

	// apply all the damage taken this frame
	P_DamageFeedback (ent);

	// add the EF_CONNECTION flag if we haven't gotten commands recently
	if ( level.time - ent->client->lastCmdTime > 1000 ) {
		ent->s.eFlags |= EF_CONNECTION;
	} else {
		ent->s.eFlags &= ~EF_CONNECTION;
	}

	ent->client->ps.stats[STAT_HEALTH] = ent->health;	// FIXME: get rid of ent->health...

	G_SetClientSound (ent);

	// set the latest infor
	BG_PlayerStateToEntityState( &ent->client->ps, &ent->s, qtrue );
}


