// Copyright (C) 1999-2000 Id Software, Inc.
//
// g_weapon.c 
// perform the server side effects of a weapon firing

#include "g_local.h"

static	float	s_quadFactor;
static	vec3_t	forward, right, up;
static	vec3_t	muzzle;


/*
======================================================================

GAUNTLET

======================================================================
*/
gentity_t *TossClientWeapon(gentity_t *self, gitem_t *weapon);
void Weapon_ThrowBall( gentity_t *ent ) {
gentity_t	*m;

	// extra vertical velocity
	forward[2] += 0.2;
	VectorNormalize( forward );

	m = TossClientWeapon(ent, BG_FindItemForWeapon(WP_BALL));
	m->flags |= EF_BOUNCE;
	m->clipmask = MASK_SOLID;
	m->enemy = ent;
	//m->r.contents = CONTENTS_TRIGGER;
	m->count = 1;
	m->physicsBounce = 0.85;
	AngleVectors(ent->client->ps.viewangles, forward, NULL, NULL);
	VectorMA(muzzle, 8, forward, m->s.pos.trBase);
	VectorMA(m->s.pos.trDelta, 0.9, ent->client->ps.velocity, m->s.pos.trDelta );						
	VectorMA(m->s.pos.trDelta, 900, forward, m->s.pos.trDelta);	
	
		
}

void Weapon_Fist( gentity_t *ent ) {

}

void Weapon_Gauntlet( gentity_t *ent ) {

}

/*
===============
CheckGauntletAttack
===============
*/
qboolean CheckMeleeAttack( gentity_t *ent ) {
	trace_t		tr;
	vec3_t		end;
	gentity_t	*tent;
	gentity_t	*traceEnt;
	int			damage;
	vec3_t		mins, maxs;

	// set aiming directions
	AngleVectors (ent->client->ps.viewangles, forward, right, up);

	CalcMuzzlePoint ( ent, forward, right, up, muzzle, end );

	VectorMA (muzzle, 32, forward, end);
	VectorClear(mins);
	VectorClear(maxs);
	if(bg_itemlist[ent->s.weapon].giType == IT_WEAPON)
	{
		if(bg_itemlist[ent->s.weapon].giTag == WP_REDFLAG || bg_itemlist[ent->s.weapon].giTag == WP_BLUEFLAG || bg_itemlist[ent->s.weapon].giTag == WP_SWORD)
		{
			VectorMA (muzzle, 24, forward, end); // give it more reach
			VectorSet(mins, -24, -24, -24);
			VectorSet(maxs, 24, 24, 24);
		}
	}

	trap_Trace (&tr, muzzle, mins, maxs, end, ent->s.number, MASK_SHOT);
	if ( tr.surfaceFlags & SURF_NOIMPACT ) {
		return qfalse;
	}

	traceEnt = &g_entities[ tr.entityNum ];

	// send blood impact
	
	if(bg_itemlist[ent->s.weapon].giType == IT_WEAPON)
	{
		if ( traceEnt->takedamage && traceEnt->client ) {		
			if(p_class[traceEnt->client->pers.playerclass].bleeds)		
				tent = G_TempEntity( tr.endpos, EV_MISSILE_HIT );
			else
				tent = G_TempEntity( tr.endpos, EV_MISSILE_HIT_ROBOT );		
			tent->s.otherEntityNum = traceEnt->s.number;
			tent->s.eventParm = DirToByte( tr.plane.normal );		
			tent->s.weapon = bg_itemlist[ent->s.weapon].giTag;
		}
		else if(traceEnt->takedamage == -1)
		{ // bloody corpse
			tent = G_TempEntity( tr.endpos, EV_MISSILE_HIT );		
			tent->s.otherEntityNum = traceEnt->s.number;
			tent->s.eventParm = DirToByte( tr.plane.normal );
			tent->s.weapon = bg_itemlist[ent->s.weapon].giTag;
		}
	}
	if ( !traceEnt->takedamage) {
		return qfalse;
	}

	if (ent->client->ps.powerups[PW_QUAD] ) {
		G_AddEvent( ent, EV_POWERUP_QUAD, 0 );
		s_quadFactor = g_quadfactor.value;
	} else {
		s_quadFactor = 1;
	}

	if(bg_itemlist[ent->s.weapon].giType == IT_WEAPON && bg_itemlist[ent->s.weapon].giTag == WP_SWORD)
	{
		damage = (20 + p_class[ent->client->pers.playerclass].strength / 25) * s_quadFactor;
		switch(ent->client->ps.torsoAnim & ~ANIM_TOGGLEBIT)
		{		
		default:			
			damage *= 0.025;
			if(damage < 1)
				damage = 1;
			if(!traceEnt->client)
				damage = 0;
			break;
		case TORSO_ATTACK:
		case TORSO_ATTACK2:
			damage *= 0.5;
			break;
		case TORSO_RAISE:
			damage *= 0.75;
			break;
		case TORSO_DROP:
			damage *= 1;
			break;
		}
		if(damage)
			G_Damage( traceEnt, ent, ent, forward, tr.endpos, damage, 0, MOD_SWORD );
	}
	else if(bg_itemlist[ent->s.weapon].giType == IT_WEAPON && bg_itemlist[ent->s.weapon].giTag == WP_GAUNTLET)
	{
		damage = 25 * s_quadFactor;
		if(ent->client->ps.powerups[PW_BERZERK])
			damage *= 10;
		G_Damage( traceEnt, ent, ent, forward, tr.endpos, damage, 0, MOD_GAUNTLET );
		if(ent->client->ps.ammo[AMMO_CURRENT] > 0)
			G_Damage( traceEnt, ent, ent, forward, tr.endpos, damage, DAMAGE_NO_BLEEDING, MOD_GAUNTLET );
	}
	else if(!ent->s.weapon)
	{
		if(traceEnt->client || s_quadFactor > 1)
			damage = (10 + p_class[ent->client->pers.playerclass].strength / 100) * s_quadFactor;
		else if(traceEnt->s.eType == ET_CORPSE)
			damage = 0;
		else
			damage = 1;
		if(ent->client->ps.powerups[PW_BERZERK])
			damage = 40000; // SPLUTCH!!!
		if(damage)
			G_Damage( traceEnt, ent, ent, forward, tr.endpos, damage, DAMAGE_NO_BLEEDING, MOD_PUNCH );
	}
	else
	{
		int dmg = 10 + p_class[ent->client->pers.playerclass].strength / 100;
		dmg += bg_itemlist[ent->s.weapon].mass;
		if(traceEnt->client || s_quadFactor > 1)
			damage = dmg * s_quadFactor;
		else if(traceEnt->s.eType == ET_CORPSE)
			damage = dmg / 10 + 1;
		else
			damage = 1;
		if(ent->client->ps.powerups[PW_BERZERK])
			damage *= 10;
		if(damage)
			G_Damage( traceEnt, ent, ent, forward, tr.endpos, damage, DAMAGE_NO_BLEEDING, MOD_GAUNTLET );
	}

	return qtrue;
}


/*
======================================================================

MACHINEGUN

======================================================================
*/

/*
======================
SnapVectorTowards

Round a vector to integers for more efficient network
transmission, but make sure that it rounds towards a given point
rather than blindly truncating.  This prevents it from truncating 
into a wall.
======================
*/
void SnapVectorTowards( vec3_t v, vec3_t to ) {
	int		i;

	for ( i = 0 ; i < 3 ; i++ ) {
		if ( to[i] <= v[i] ) {
			v[i] = (int)v[i];
		} else {
			v[i] = (int)v[i] + 1;
		}
	}
}

#define MACHINEGUN_SPREAD	200
#define	MACHINEGUN_DAMAGE	15
//was 7
#define	MACHINEGUN_TEAM_DAMAGE	10
// was 5		// wimpier MG in teamplay

void Bullet_Fire (gentity_t *ent, float spread, int damage ) {
	vec3_t		end, dir;
	float		r;
	float		u;

	damage *= s_quadFactor;
	r = crandom()*spread;
	u = crandom()*spread;
	VectorMA (muzzle, 8192, forward, end);
	VectorMA (end, r, right, end);
	VectorMA (end, u, up, end);

	VectorSubtract(end, muzzle, dir);
	VectorNormalize(dir);
	fire_bullet(ent, muzzle, dir);
	if(g_recoil.value)
	{
		vec3_t kvel;
		float	mass;

		if(ent->client->ps.stats[STAT_MASS])
			mass = ent->client->ps.stats[STAT_MASS];
		
		else
			mass = 200;

		VectorScale (forward, g_recoil.value / mass, kvel);

		// set the timer so that the other client can't cancel
		// out the movement immediately
		if(bg_itemlist[ent->s.weapon].giTag == WP_MACHINEGUN)
		{
			ent->client->ps.delta_angles[PITCH] -= ANGLE2SHORT((2 + random()) * FRAMETIME / (mass + ent->client->ps.stats[STAT_STRENGTH]));
		}
		else if(ent->client->ps.pm_flags & PMF_DUCKED)
		{
			ent->client->ps.delta_angles[PITCH] += ANGLE2SHORT(crandom() * FRAMETIME / (mass + ent->client->ps.stats[STAT_STRENGTH]));
			ent->client->ps.delta_angles[YAW] -= ANGLE2SHORT(2 * crandom() * FRAMETIME / (mass + ent->client->ps.stats[STAT_STRENGTH]));
		}
		else
		{
			ent->client->ps.delta_angles[PITCH] += ANGLE2SHORT(2 * crandom() * FRAMETIME / (mass + ent->client->ps.stats[STAT_STRENGTH]));
			ent->client->ps.delta_angles[YAW] -= ANGLE2SHORT(5 * crandom() * FRAMETIME / (mass + ent->client->ps.stats[STAT_STRENGTH]));
		}
		if ( !ent->client->ps.pm_time ) {
			int		t;

			t = VectorLength(kvel) * 2;
			if(ent->client->ps.groundEntityNum == ENTITYNUM_NONE)
			{
				t *= 2;
				
				if(bg_itemlist[ent->s.weapon].giTag == WP_CHAINGUN)
				{
					vec3_t cross;
					CrossProduct(ent->client->muzzle, ent->client->ps.velocity, cross);
					ent->client->ps.delta_angles[PITCH] += ANGLE2SHORT(FRAMETIME * 0.005 * DotProduct(cross, right) / (mass + ent->client->ps.stats[STAT_STRENGTH]));
					ent->client->ps.delta_angles[YAW] -= ANGLE2SHORT(FRAMETIME  * 0.005 * DotProduct(cross, up) / (mass + ent->client->ps.stats[STAT_STRENGTH]));
				}
			}
			else if(ent->client->ps.velocity[0] == 0 && ent->client->ps.velocity[1] == 0)
			{
				t /= 2;
				if(ent->client->ps.pm_flags & PMF_DUCKED)
					t /= 2;
			}
			else if(bg_itemlist[ent->s.weapon].giTag == WP_CHAINGUN)
			{
				vec3_t cross;
				CrossProduct(ent->client->muzzle, ent->client->ps.velocity, cross);
				ent->client->ps.delta_angles[PITCH] += ANGLE2SHORT(FRAMETIME * 0.005 * DotProduct(cross, right) / (mass + ent->client->ps.stats[STAT_STRENGTH]));
				ent->client->ps.delta_angles[YAW] -= ANGLE2SHORT(FRAMETIME * 0.005 * DotProduct(cross, up) / (mass + ent->client->ps.stats[STAT_STRENGTH]));
			}
			else
			{
				ent->client->ps.delta_angles[PITCH] += ANGLE2SHORT(FRAMETIME * (3 + random()) / (mass + ent->client->ps.stats[STAT_STRENGTH]));
			}
			if ( t < 50 ) {
				t = 50;
			}
			else if ( t > 200 ) {
				t = 200;
			}
			ent->client->ps.pm_time = t;
			ent->client->ps.pm_flags |= PMF_TIME_KNOCKBACK;
		}
		VectorSubtract (ent->client->ps.velocity, kvel, ent->client->ps.velocity);
	}
	
}


/*
======================================================================

SHOTGUN

======================================================================
*/

qboolean ShotgunPellet( vec3_t start, vec3_t end, gentity_t *ent ) {
	trace_t		tr;
	int			damage;
	gentity_t		*traceEnt;

	trap_Trace (&tr, start, NULL, NULL, end, ent->s.number, MASK_SHOT);
	traceEnt = &g_entities[ tr.entityNum ];

	// send bullet impact
	if (  tr.surfaceFlags & SURF_NOIMPACT ) {
		return qfalse;
	}

	if ( traceEnt->takedamage) {
		damage = DEFAULT_SHOTGUN_DAMAGE * s_quadFactor;

		G_Damage( traceEnt, ent, ent, forward, tr.endpos,
			damage, 0, MOD_SHOTGUN);
		
		if(traceEnt->client && (p_class[traceEnt->client->pers.playerclass].bleeds))
		{
			G_AddEvent(traceEnt, EV_BLOOD, damage + 8);// just to be sure
			G_AddPredictableEvent(traceEnt, EV_BLOOD, damage + 8);
			//G_Printf("sent EV_BLOOD(%d)\n", damage);				
		}
		
		if( LogAccuracyHit( traceEnt, ent ) ) 
			return qtrue;
		
	}
	return qfalse;
}

// this should match CG_ShotgunPattern
void ShotgunPattern( vec3_t origin, vec3_t origin2, int seed, gentity_t *ent ) {
	int			i;
	float		r, u;
	vec3_t		end;
	vec3_t		forward, right, up;
	int			oldScore;
	qboolean	hitClient = qfalse;

	// derive the right and up vectors from the forward vector, because
	// the client won't have any other information
	VectorNormalize2( origin2, forward );
	PerpendicularVector( right, forward );
	CrossProduct( forward, right, up );

	oldScore = ent->client->ps.persistant[PERS_SCORE];

	// generate the "random" spread pattern
	for ( i = 0 ; i < DEFAULT_SHOTGUN_COUNT ; i++ ) {
		
		r = Q_crandom(&seed) * DEFAULT_SHOTGUN_SPREAD;
		u = Q_crandom(&seed) * DEFAULT_SHOTGUN_SPREAD;
		
		VectorMA( origin, 8192, forward, end);
		VectorMA (end, r, right, end);
		VectorMA (end, u, up, end);

		if( ShotgunPellet( origin, end, ent ) && !hitClient ) {
			hitClient = qtrue;
			ent->client->ps.persistant[PERS_ACCURACY_HITS]++;
		}
	}
}

void weapon_shotgun_fire (gentity_t *ent) {
	gentity_t		*tent;
	
	// send shotgun blast
	
	tent = G_TempEntity( muzzle, EV_SHOTGUN );
	VectorScale( forward, 4096, tent->s.origin2 );
	SnapVector( tent->s.origin2 );
	tent->s.eventParm = rand() & 255;		// seed for spread pattern
	tent->s.otherEntityNum = ent->s.number;

	ShotgunPattern( muzzle, tent->s.origin2, tent->s.eventParm, ent );

	if(g_recoil.value)
	{
		vec3_t kvel;
		float	mass;

		if(ent->client->ps.stats[STAT_MASS])
			mass = ent->client->ps.stats[STAT_MASS];
		
		else
			mass = 200;

		ent->client->ps.delta_angles[YAW] -= ANGLE2SHORT(5 * (random() + 2) * FRAMETIME / (mass + ent->client->ps.stats[STAT_STRENGTH]));
	
		VectorScale (forward, g_recoil.value * 12.0 / mass, kvel);

		// set the timer so that the other client can't cancel
		// out the movement immediately
		if ( !ent->client->ps.pm_time ) {
			int		t;

			t = VectorLength(kvel) * 2;
			if(ent->client->ps.groundEntityNum == ENTITYNUM_NONE)
			{
				t *= 2;
				ent->client->ps.delta_angles[YAW] -= ANGLE2SHORT(10 * (random() + 2) * FRAMETIME / (mass + ent->client->ps.stats[STAT_STRENGTH]));

			}
			else if(ent->client->ps.velocity[0] == 0 && ent->client->ps.velocity[1] == 0)
			{
				t /= 2;
				VectorScale(kvel, 0.75, kvel);
				if(ent->client->ps.pm_flags & PMF_DUCKED)
					t /= 2;
					VectorScale(kvel, 0.5, kvel);
			}
			else
			{
				ent->client->ps.delta_angles[YAW] -= ANGLE2SHORT(5 * (random() + 2) * FRAMETIME / (mass + ent->client->ps.stats[STAT_STRENGTH]));
			}
			if ( t < 50 ) {
				t = 50;
			}
			else if ( t > 200 ) {
				t = 200;
			}
			ent->client->ps.pm_time = t;
			ent->client->ps.pm_flags |= PMF_TIME_KNOCKBACK;
		}
		VectorSubtract (ent->client->ps.velocity, kvel, ent->client->ps.velocity);
	}
	
}

void weapon_supershotgun_fire (gentity_t *ent) {
	gentity_t		*tent;

	// send shotgun blast
	if(ent->client->ps.ammo[AMMO_CURRENT] & 1) // left barrel
		VectorMA(muzzle, -4, right, muzzle);
	else
		VectorMA(muzzle, 4, right, muzzle);
	tent = G_TempEntity( muzzle, EV_SHOTGUN );
	VectorScale( forward, 4096, tent->s.origin2 );
	SnapVector( tent->s.origin2 );
	tent->s.eventParm = rand() & 255;		// seed for spread pattern
	tent->s.otherEntityNum = ent->s.number;

	ShotgunPattern( tent->s.pos.trBase, tent->s.origin2, tent->s.eventParm, ent );


	if(g_recoil.value)
	{
		vec3_t kvel;
		float	mass;

		if(ent->client->ps.stats[STAT_MASS])
			mass = ent->client->ps.stats[STAT_MASS];
		
		else
			mass = 200;

		ent->client->ps.delta_angles[YAW] -= ANGLE2SHORT((4 + 2 * (ent->client->ps.ammo[AMMO_CURRENT] & 1)) * (random() + 2) * FRAMETIME / (mass + ent->client->ps.stats[STAT_STRENGTH]));
	
		VectorScale (forward, g_recoil.value * 12.0 / mass, kvel);

		// set the timer so that the other client can't cancel
		// out the movement immediately
		if ( !ent->client->ps.pm_time ) {
			int		t;

			t = VectorLength(kvel) * 2;
			if(ent->client->ps.groundEntityNum == ENTITYNUM_NONE)
			{
				t *= 2;
				ent->client->ps.delta_angles[YAW] -= ANGLE2SHORT((7 + 5 * (ent->client->ps.ammo[AMMO_CURRENT] & 1)) * (random() + 2) * FRAMETIME / (mass + ent->client->ps.stats[STAT_STRENGTH]));

			}
			else if(ent->client->ps.velocity[0] == 0 && ent->client->ps.velocity[1] == 0)
			{
				t /= 2;
				VectorScale(kvel, 0.75, kvel);
				if(ent->client->ps.pm_flags & PMF_DUCKED)
					t /= 2;
					VectorScale(kvel, 0.5, kvel);
			}
			else
			{
				ent->client->ps.delta_angles[YAW] -= ANGLE2SHORT((4 + 2 * (ent->client->ps.ammo[AMMO_CURRENT] & 1)) * (random() + 2) * FRAMETIME / (mass + ent->client->ps.stats[STAT_STRENGTH]));
			}
			if ( t < 50 ) {
				t = 50;
			}
			else if ( t > 200 ) {
				t = 200;
			}
			ent->client->ps.pm_time = t;
			ent->client->ps.pm_flags |= PMF_TIME_KNOCKBACK;
		}
		VectorSubtract (ent->client->ps.velocity, kvel, ent->client->ps.velocity);
	}
	
}


/*
======================================================================

GRENADE LAUNCHER

======================================================================
*/

void weapon_grenade_fire (gentity_t *ent) {
	gentity_t	*m;

	// extra vertical velocity
	forward[2] += 0.2;
	VectorNormalize( forward );

	m = toss_grenade (ent, muzzle, forward);
	m->damage *= s_quadFactor;
	m->splashDamage *= s_quadFactor;
	if(s_quadFactor > 1)
		m->s.powerups |= 1 << PW_QUAD;

}

void weapon_grenadelauncher_fire (gentity_t *ent) {
	gentity_t	*m;

	// extra vertical velocity
	forward[2] += 0.2;
	VectorNormalize( forward );

	m = fire_grenade (ent, muzzle, forward);
	m->damage *= s_quadFactor;
	m->splashDamage *= s_quadFactor;
	if(s_quadFactor > 1)
		m->s.powerups |= 1 << PW_QUAD;

	if(g_recoil.value)
	{
		vec3_t kvel;
		float	mass;

		if(ent->client->ps.stats[STAT_MASS])
			mass = ent->client->ps.stats[STAT_MASS];
		
		else
			mass = 200;

		VectorScale (forward, g_recoil.value * 10.0 / mass, kvel);
		VectorSubtract (ent->client->ps.velocity, kvel, ent->client->ps.velocity);

		// set the timer so that the other client can't cancel
		// out the movement immediately
		if ( !ent->client->ps.pm_time ) {
			int		t;

			t = VectorLength(kvel) * 2;
			if(ent->client->ps.groundEntityNum == ENTITYNUM_NONE)
			{
				t *= 2;
			}
			else if(ent->client->ps.velocity[0] == 0 && ent->client->ps.velocity[1] == 0)
			{
				t /= 2;
				if(ent->client->ps.pm_flags & PMF_DUCKED)
					t /= 2;
			}
			if ( t < 50 ) {
				t = 50;
			}
			else if ( t > 200 ) {
				t = 200;
			}
			ent->client->ps.pm_time = t;
			ent->client->ps.pm_flags |= PMF_TIME_KNOCKBACK;
		}
	}

}

/*
======================================================================

ROCKET

======================================================================
*/

void Weapon_RocketLauncher_Fire (gentity_t *ent) {
	gentity_t	*m;

	m = fire_rocket (ent, muzzle, forward);
	m->damage *= s_quadFactor;
	m->splashDamage *= s_quadFactor;
	if(s_quadFactor > 1)
		m->s.powerups |= 1 << PW_QUAD;

	if(g_recoil.value)
	{
		vec3_t kvel;
		float	mass;

		if(ent->client->ps.stats[STAT_MASS])
			mass = ent->client->ps.stats[STAT_MASS];
		
		else
			mass = 200;

		VectorScale (forward, g_recoil.value * 5 / mass, kvel);
		VectorSubtract (ent->client->ps.velocity, kvel, ent->client->ps.velocity);

		// set the timer so that the other client can't cancel
		// out the movement immediately
		if ( !ent->client->ps.pm_time ) {
			int		t;

			t = VectorLength(kvel) * 2;
			if(ent->client->ps.groundEntityNum == ENTITYNUM_NONE)
			{
				t *= 2;
			}
			else if(ent->client->ps.velocity[0] == 0 && ent->client->ps.velocity[1] == 0)
			{
				t /= 2;
				if(ent->client->ps.pm_flags & PMF_DUCKED)
					t /= 2;
			}
			if ( t < 50 ) {
				t = 50;
			}
			else if ( t > 200 ) {
				t = 200;
			}
			ent->client->ps.pm_time = t;
			ent->client->ps.pm_flags |= PMF_TIME_KNOCKBACK;
		}
	}

}


/*
======================================================================

PLASMA GUN

======================================================================
*/

void Weapon_Plasmagun_Fire (gentity_t *ent) {
	gentity_t	*m;

	m = fire_plasma (ent, muzzle, forward);
	m->damage *= s_quadFactor;
	m->splashDamage *= s_quadFactor;
	if(s_quadFactor > 1)
		m->s.powerups |= 1 << PW_QUAD;

}

/*
======================================================================

RAILGUN

======================================================================
*/
#define	MAX_RAIL_HITS	32 // four!? what were they thinking?


/*
=================
weapon_railgun_fire
=================
*/
void weapon_railgun_fire (gentity_t *ent) {
	vec3_t		end;
	trace_t		trace;
	gentity_t	*tent;
	gentity_t	*traceEnt;
	int			damage;
	int			radiusDamage;
	int			i;
	int			hits, hitloc;
	int			unlinked;
	gentity_t	*unlinkedEntities[MAX_RAIL_HITS];

	damage = 100 * s_quadFactor;
	radiusDamage = 30 * s_quadFactor;

	VectorMA (muzzle, 8192, forward, end);

	// trace only against the solids, so the railgun will go through people
	unlinked = 0;
	hits = 0;
	do {
		trap_Trace (&trace, muzzle, NULL, NULL, end, ent->s.number, MASK_SHOT );
		if ( trace.entityNum >= ENTITYNUM_MAX_NORMAL ) {
			break;
		}
		traceEnt = &g_entities[ trace.entityNum ];
		if ( traceEnt->takedamage ) {
			if( LogAccuracyHit( traceEnt, ent ) ) {
				hits++;
			}
			hitloc = G_Damage (traceEnt, ent, ent, forward, trace.endpos, damage, 0, MOD_RAILGUN);			
		}
		if ( trace.contents & CONTENTS_SOLID ) {

			break;		// we hit something solid enough to stop the beam
		}
		// unlink this entity, so the next trace will go past it
		trap_UnlinkEntity( traceEnt );
		unlinkedEntities[unlinked] = traceEnt;
		unlinked++;
	} while ( unlinked < MAX_RAIL_HITS );

	// link back in any entities we unlinked
	for ( i = 0 ; i < unlinked ; i++ ) {
		trap_LinkEntity( unlinkedEntities[i] );
	}

	// the final trace endpos will be the terminal point of the rail trail

	// snap the endpos to integers to save net bandwidth, but nudged towards the line
	SnapVectorTowards( trace.endpos, muzzle );

	// send railgun beam effect
	tent = G_TempEntity( trace.endpos, EV_RAILTRAIL );
	tent->r.svFlags |= SVF_BROADCAST;
	tent->s.powerups = ent->s.powerups;
	if(ent->client->ps.powerups[PW_QUAD])
		tent->s.powerups |= PW_QUAD;
	// set player number for custom colors on the railtrail
	tent->s.clientNum = ent->s.clientNum;

	VectorCopy( muzzle, tent->s.origin2 );
	// move origin a bit to come closer to the drawn gun muzzle
	VectorMA( tent->s.origin2, 4, right, tent->s.origin2 );
	VectorMA( tent->s.origin2, -1, up, tent->s.origin2 );

	// no explosion at end if SURF_NOIMPACT, but still make the trail
	if ( trace.surfaceFlags & SURF_NOIMPACT ) {
		tent->s.eventParm = 255;	// don't make the explosion at the end
	} else {
		tent->s.eventParm = DirToByte( trace.plane.normal );
	}
	tent->s.clientNum = ent->s.clientNum;

	// give the shooter a reward sound if they have made two railgun hits in a row
	if ( hits == 0 ) {
		// complete miss
		ent->client->accurateCount = 0;
	} else {
		// check for "impressive" reward sound
		ent->client->accurateCount += hits;
		if ( ent->client->accurateCount >= 2 ) {
			ent->client->accurateCount -= 2;
			ent->client->ps.persistant[PERS_REWARD_COUNT]++;
			ent->client->ps.persistant[PERS_REWARD] = REWARD_IMPRESSIVE;
			ent->client->ps.persistant[PERS_IMPRESSIVE_COUNT]++;
			
			// add the sprite over the player's head
			ent->client->ps.eFlags &= ~(EF_AWARD_EXCELLENT | EF_AWARD_GAUNTLET );
			ent->client->ps.eFlags |= EF_AWARD_IMPRESSIVE;
			ent->client->rewardTime = level.time + REWARD_SPRITE_TIME;
		}
		ent->client->ps.persistant[PERS_ACCURACY_HITS]++;
	}

	if(g_recoil.value)
	{
		vec3_t kvel;
		float	mass;

		if(ent->client->ps.stats[STAT_MASS])
			mass = ent->client->ps.stats[STAT_MASS];
		
		else
			mass = 200;

		VectorScale (forward, g_recoil.value * 100.0 / mass, kvel);
		
		// set the timer so that the other client can't cancel
		// out the movement immediately
		if ( !ent->client->ps.pm_time ) {
			int		t;

			t = VectorLength(kvel) * 2;
			if(ent->client->ps.groundEntityNum == ENTITYNUM_NONE)
			{
				t *= 2;
			}
			else if(ent->client->ps.velocity[0] == 0 && ent->client->ps.velocity[1] == 0)
			{
				t /= 2;
				VectorScale(kvel, 0.75, kvel);
				if(ent->client->ps.pm_flags & PMF_DUCKED)
					t /= 2;
					VectorScale(kvel, 0.5, kvel);
			}
			if ( t < 50 ) {
				t = 50;
			}
			else if ( t > 200 ) {
				t = 200;
			}
			ent->client->ps.pm_time = t;
			ent->client->ps.pm_flags |= PMF_TIME_KNOCKBACK;
		}
		VectorSubtract (ent->client->ps.velocity, kvel, ent->client->ps.velocity);
	}
}



/*
======================================================================

LIGHTNING GUN

======================================================================
*/

void Weapon_LightningFire( gentity_t *ent ) {
	trace_t		tr;
	vec3_t		end;
	gentity_t	*traceEnt, *tent = NULL;
	int			damage;

	damage = 8 * s_quadFactor;

	VectorMA( muzzle, LIGHTNING_RANGE, forward, end );

	trap_Trace( &tr, muzzle, NULL, NULL, end, ent->s.number, MASK_SHOT );

	if ( tr.entityNum == ENTITYNUM_NONE ) {
		return;
	}

	traceEnt = &g_entities[ tr.entityNum ];

	if ( traceEnt->takedamage) 
	{
		int hitloc = G_Damage( traceEnt, ent, ent, forward, tr.endpos,
			damage, DAMAGE_NO_BLEEDING, MOD_LIGHTNING);
	

		if ( traceEnt->client ) {
			if((p_class[traceEnt->client->pers.playerclass].bleeds) && (traceEnt->client->ps.stats[STAT_ARMOR] <= 0 || hitloc != HIT_TORSO))
				tent = G_TempEntity( tr.endpos, EV_MISSILE_HIT );
			else
				tent = G_TempEntity( tr.endpos, EV_MISSILE_HIT_ROBOT );

			tent->s.otherEntityNum = traceEnt->s.number;
			tent->s.eventParm = DirToByte( tr.plane.normal );
			tent->s.weapon = ent->s.weapon;
			if( LogAccuracyHit( traceEnt, ent ) ) {
				ent->client->ps.persistant[PERS_ACCURACY_HITS]++;
			}
		}
		else if(traceEnt->takedamage == -1) 
		{ // bloody corpse
			tent = G_TempEntity( tr.endpos, EV_MISSILE_HIT );
			tent->s.otherEntityNum = traceEnt->s.number;
			tent->s.weapon = ent->s.weapon;
			tent->s.eventParm = DirToByte( tr.plane.normal );
		}
		else if ( !( tr.surfaceFlags & SURF_NOIMPACT ) ) 
		{
			tent = G_TempEntity( tr.endpos, EV_MISSILE_MISS );
			tent->s.eventParm = DirToByte( tr.plane.normal );
		}

	} 
	else if ( !( tr.surfaceFlags & SURF_NOIMPACT ) ) 
	{
		tent = G_TempEntity( tr.endpos, EV_MISSILE_MISS );
		tent->s.eventParm = DirToByte( tr.plane.normal );
	}
	if(tent)
		tent->s.weapon = WP_LIGHTNING;

}

//======================================================================
void Weapon_FlamethrowerFire( gentity_t *ent ) {
	trace_t		tr;
	vec3_t		end;
	gentity_t	*traceEnt, *tent;
	int			damage;

	damage = 8 * s_quadFactor;

	VectorMA( muzzle, FLAMETHROWER_RANGE, forward, end );

	trap_Trace( &tr, muzzle, NULL, NULL, end, ent->s.number, MASK_SHOT | CONTENTS_WATER);

	if(!(tr.contents & CONTENTS_WATER))
	{
		G_RadiusDamage( tr.endpos, ent, 8, 60, 800, NULL, MOD_FLAMETHROWER );
	}
	if ( tr.entityNum == ENTITYNUM_NONE ) 
		return;

	traceEnt = &g_entities[ tr.entityNum ];

	
	if ( traceEnt->takedamage && traceEnt->client ) 
	{ // all remarked out because it really doesn't matter
		/*if(traceEnt->client->ps.stats[STAT_EXTENDED_INFO] & EXT_BLEEDS)		
			tent = G_TempEntity( tr.endpos, EV_MISSILE_HIT );
		else
			tent = G_TempEntity( tr.endpos, EV_MISSILE_HIT_ROBOT );
		tent->s.otherEntityNum = traceEnt->s.number;
		tent->s.eventParm = DirToByte( tr.plane.normal );
		tent->s.weapon = ent->s.weapon;
		if( LogAccuracyHit( traceEnt, ent ) ) {
			ent->client->ps.persistant[PERS_ACCURACY_HITS]++;
		}*/
	} else if ( !( tr.surfaceFlags & SURF_NOIMPACT ) ) {
		tent = G_TempEntity( tr.endpos, EV_MISSILE_MISS );
		tent->s.eventParm = DirToByte( tr.plane.normal );
		tent->s.weapon = WP_FLAMETHROWER;
	}
}


/*
===============
LogAccuracyHit
===============
*/
qboolean LogAccuracyHit( gentity_t *target, gentity_t *attacker ) {
	if(!attacker || !target)
		return qfalse;
	if( !target->takedamage ) {
		return qfalse;
	}

	if ( target == attacker ) {
		return qfalse;
	}

	if( !target->client ) {
		return qfalse;
	}

	if( !attacker->client ) {
		return qfalse;
	}

	if( target->client->ps.stats[STAT_HEALTH] <= 0 ) {
		return qfalse;
	}

	if ( OnSameTeam( target, attacker ) ) {
		return qfalse;
	}

	return qtrue;
}


/* 
===============
CalcMuzzlePoint

set muzzle location relative to pivoting eye
===============
*/
void CalcMuzzlePoint ( gentity_t *ent, vec3_t forward, vec3_t right, vec3_t up, vec3_t muzzlePoint, vec3_t aimdir ) {
	
	if(!ent->client->pers.customView || ent->client->muzzle[2] == 443556)
	{
		VectorCopy( ent->s.pos.trBase, muzzlePoint );

		muzzlePoint[2] += ent->client->ps.viewheight;
		VectorMA( muzzlePoint, 14, forward, muzzlePoint );
		VectorCopy(forward, aimdir);
	}
	else
	{
		trace_t tr;	
		vec3_t eyepos, paralax;
		
		
		VectorAdd( ent->s.pos.trBase, ent->client->muzzle, muzzlePoint );
		if(ent->client->pers.AutoAim)
		{
			VectorAdd( ent->s.pos.trBase, ent->client->headpos, eyepos );

			VectorMA(eyepos, 8192, forward, paralax);
			trap_Trace( &tr, eyepos, NULL, NULL, paralax, ent->s.number, MASK_SHOT );
			VectorSubtract(tr.endpos, muzzle, aimdir);
			VectorNormalize(aimdir);
		}
		else
		{
			VectorCopy(forward, aimdir);
		}
		//G_Printf("?");
	}
	// snap to integer coordinates for more efficient network bandwidth usage
	SnapVector( muzzlePoint );
	
	
}



/*
===============
FireWeapon
===============
*/
void FireWeapon( gentity_t *ent ) {
	
	vec3_t aimdir;
	if (ent->client->ps.powerups[PW_QUAD] ) {
		s_quadFactor = g_quadfactor.value;
	} else {
		s_quadFactor = 1;
	}

	// track shots taken for accuracy tracking.  gauntet is just not tracked
	if( bg_itemlist[ent->s.weapon].giType == IT_WEAPON && bg_itemlist[ent->s.weapon].giTag != WP_GAUNTLET ) {
		ent->client->ps.persistant[PERS_ACCURACY_SHOTS]++;
	}

	// set aiming directions
	AngleVectors (ent->client->ps.viewangles, forward, right, up);
	CalcMuzzlePoint ( ent, forward, right, up, muzzle, aimdir );
	VectorCopy(aimdir, forward); // paralax corrected
	if(bg_itemlist[ent->s.weapon].giType != IT_WEAPON)
	{ // do special stuff here
		if(bg_itemlist[ent->s.weapon].giType == IT_AMMO && bg_itemlist[ent->s.weapon].giTag == AMMO_GRENADES)
		{ // hand grenade - fire primes, release throws.
			weapon_grenade_fire( ent );
			ent->s.weapon = ent->client->ps.weapon = 0;
			ent->client->ps.ammo[AMMO_CURRENT] = 0;
		}
		return;
	}
	// fire the specific weapon
	switch( bg_itemlist[ent->s.weapon].giTag ) {
	case WP_NONE:
		Weapon_Fist( ent );
	case WP_GAUNTLET:
		Weapon_Gauntlet( ent );
		break;
	case WP_FLAMETHROWER:
		Weapon_FlamethrowerFire( ent );
		break;
	case WP_LIGHTNING:
		Weapon_LightningFire( ent );
		break;
	case WP_SINGLE_SHOTGUN:
		weapon_shotgun_fire( ent );
		break;
	case WP_SHOTGUN:
		weapon_supershotgun_fire( ent );
		break;
	case WP_MACHINEGUN:
	case WP_CHAINGUN:
		Bullet_Fire( ent, MACHINEGUN_SPREAD, MACHINEGUN_DAMAGE );
		break;
	case WP_GRENADE_LAUNCHER:
		weapon_grenadelauncher_fire( ent );
		break;
	case WP_ROCKET_LAUNCHER:
		Weapon_RocketLauncher_Fire( ent );
		break;
	case WP_PLASMAGUN:
		Weapon_Plasmagun_Fire( ent );
		break;
	case WP_RAILGUN:
		weapon_railgun_fire( ent );
		break;
	case WP_BALL:
		Weapon_ThrowBall( ent );
		break;
	default:
// FIXME		G_Error( "Bad ent->s.weapon" );
		break;
	}
}

