// Copyright (C) 1999-2000 Id Software, Inc.
//

#include <limits.h>

#include "g_local.h"

typedef enum _flag_status {
	FLAG_ATBASE = 0,
	FLAG_TAKEN,
	FLAG_DROPPED
} flagStatus_t;

typedef struct teamgame_s
{
	float last_flag_capture;
	int last_capture_team;
	flagStatus_t redStatus;
	flagStatus_t blueStatus;
} teamgame_t;

teamgame_t teamgame;

void Team_SetFlagStatus( int team, flagStatus_t status );

gentity_t *ReturnedPlayer;

void Team_InitGame(void)
{
	memset(&teamgame, 0, sizeof teamgame);
	teamgame.redStatus = teamgame.blueStatus = -1; // Invalid to force update

	Team_SetFlagStatus( TEAM_RED, FLAG_ATBASE );
	Team_SetFlagStatus( TEAM_BLUE, FLAG_ATBASE );
	ReturnedPlayer = NULL;
}

int OtherTeam(int team) {
	if (team==TEAM_RED)
		return TEAM_BLUE;
	else if (team==TEAM_BLUE)
		return TEAM_RED;
	return team;
}

const char *TeamName(int team)  {
	if (team==TEAM_RED)
		return g_RedTeamName.string;
	else if (team==TEAM_BLUE)
		return g_BlueTeamName.string;
	else if (team==TEAM_SPECTATOR)
		return "SPECTATOR";
	return "FREE";
}

const char *OtherTeamName(int team) {
	if (team==TEAM_RED)
		return g_BlueTeamName.string;
	else if (team==TEAM_BLUE)
		return g_RedTeamName.string;
	else if (team==TEAM_SPECTATOR)
		return "SPECTATOR";
	return "FREE";
}

const char *TeamColorString(int team) {
	if (team==TEAM_RED)
		return S_COLOR_RED;
	else if (team==TEAM_BLUE)
		return S_COLOR_BLUE;
	else if (team==TEAM_SPECTATOR)
		return S_COLOR_YELLOW;
	return S_COLOR_WHITE;
}

// NULL for everyone
void QDECL PrintMsg( gentity_t *ent, const char *fmt, ... ) {
	char		msg[1024];
	va_list		argptr;
	char		*p;
	
	va_start (argptr,fmt);
	if (vsprintf (msg, fmt, argptr) > sizeof(msg)) {
		G_Error ( "PrintMsg overrun" );
	}
	va_end (argptr);

	// double quotes are bad
	while ((p = strchr(msg, '"')) != NULL)
		*p = '\'';

	trap_SendServerCommand ( ( (ent == NULL) ? -1 : ent-g_entities ), va("print \"%s\"", msg ));
}

/*
==============
OnSameTeam
==============
*/
qboolean OnSameTeam( gentity_t *ent1, gentity_t *ent2 ) {
	if ( !ent1->client || !ent2->client ) {
		return qfalse;
	}

	if ( g_gametype.integer < GT_TEAM ) {
		return qfalse;
	}

	if ( ent1->client->sess.sessionTeam == ent2->client->sess.sessionTeam ) {
		return qtrue;
	}

	return qfalse;
}

void Team_SetFlagStatus( int team, flagStatus_t status )
{
	qboolean modified = qfalse;

	switch (team) {
	case TEAM_RED :
		if ( teamgame.redStatus != status ) {
			teamgame.redStatus = status;
			modified = qtrue;
		}
		break;
	case TEAM_BLUE :
		if ( teamgame.blueStatus != status ) {
			teamgame.blueStatus = status;
			modified = qtrue;
		}
		break;
	}

	if (modified) {
		char st[4];

		st[0] = '0' + (int)teamgame.redStatus;
		st[1] = '0' + (int)teamgame.blueStatus;
		st[2] = 0;

		trap_SetConfigstring( CS_FLAGSTATUS, st );
	}
}

void Team_CheckDroppedItem( gentity_t *dropped )
{
	if (dropped->item->giType == IT_WEAPON && dropped->item->giTag == WP_REDFLAG)
		Team_SetFlagStatus( TEAM_RED, FLAG_DROPPED );
	else if (dropped->item->giType == IT_WEAPON && dropped->item->giTag == WP_BLUEFLAG)
		Team_SetFlagStatus( TEAM_BLUE, FLAG_DROPPED );
}


/*
================
Team_FragBonuses

Calculate the bonuses for flag defense, flag carrier defense, etc.
Note that bonuses are not cumlative.  You get one, they are in importance
order.
================
*/
void Team_FragBonuses(gentity_t *targ, gentity_t *inflictor, gentity_t *attacker)
{
	int i;
	gentity_t *ent;
	int flag_pw, enemy_flag_pw;
	int otherteam;
	gentity_t *flag, *carrier = NULL;
	char *c;
	vec3_t v1, v2;
	int team;

	// no bonus for fragging yourself
	if (!targ->client || !attacker->client || targ == attacker)
		return;

	team = targ->client->sess.sessionTeam;
	otherteam = OtherTeam(targ->client->sess.sessionTeam);
	if (otherteam < 0)
		return; // whoever died isn't on a team

	// same team, if the flag at base, check to he has the enemy flag
	if (team == TEAM_RED) {
		flag_pw = PW_REDFLAG;
		enemy_flag_pw = PW_BLUEFLAG;
	} else {
		flag_pw = PW_BLUEFLAG;
		enemy_flag_pw = PW_REDFLAG;
	}

	// did the attacker frag the flag carrier?
	if (targ->client->ps.powerups[enemy_flag_pw]) {
		attacker->client->pers.teamState.lastfraggedcarrier = level.time;
		AddScore(attacker, CTF_FRAG_CARRIER_BONUS);
		attacker->client->pers.teamState.fragcarrier++;
		PrintMsg(NULL, "%s" S_COLOR_WHITE " fragged %s's flag carrier!\n",
			attacker->client->pers.netname, TeamName(team));

		// the target had the flag, clear the hurt carrier
		// field on the other team
		for (i = 0; i < g_maxclients.integer; i++) {
			ent = g_entities + i;
			if (ent->inuse && ent->client->sess.sessionTeam == otherteam)
				ent->client->pers.teamState.lasthurtcarrier = 0;
		}
		return;
	}

	if (targ->client->pers.teamState.lasthurtcarrier &&
		level.time - targ->client->pers.teamState.lasthurtcarrier < CTF_CARRIER_DANGER_PROTECT_TIMEOUT &&
		!attacker->client->ps.powerups[flag_pw]) {
		// attacker is on the same team as the flag carrier and
		// fragged a guy who hurt our flag carrier
		AddScore(attacker, CTF_CARRIER_DANGER_PROTECT_BONUS);

		attacker->client->pers.teamState.carrierdefense++;
		targ->client->pers.teamState.lasthurtcarrier = 0;

		team = attacker->client->sess.sessionTeam;
		PrintMsg(NULL, "%s" S_COLOR_WHITE " defends %s's flag carrier against an agressive enemy\n",
			attacker->client->pers.netname, TeamName(team));
		return;
	}

	// flag and flag carrier area defense bonuses

	// we have to find the flag and carrier entities

	// find the flag
	switch (attacker->client->sess.sessionTeam) {
	case TEAM_RED:
		c = "team_CTF_redflag";
		break;
	case TEAM_BLUE:
		c = "team_CTF_blueflag";
		break;		
	default:
		return;
	}

	flag = NULL;
	while ((flag = G_Find (flag, FOFS(classname), c)) != NULL) {
		if (!(flag->flags & FL_DROPPED_ITEM))
			break;
	}

	if (!flag)
		return; // can't find attacker's flag

	// find attacker's team's flag carrier
	for (i = 0; i < g_maxclients.integer; i++) {
		carrier = g_entities + i;
		if (carrier->inuse && carrier->client->ps.powerups[flag_pw])
			break;
		carrier = NULL;
	}

	// ok we have the attackers flag and a pointer to the carrier

	// check to see if we are defending the base's flag
	VectorSubtract(targ->r.currentOrigin, flag->r.currentOrigin, v1);
	VectorSubtract(attacker->r.currentOrigin, flag->r.currentOrigin, v2);

	if ( ( ( VectorLength(v1) < CTF_TARGET_PROTECT_RADIUS &&
		trap_InPVS(flag->r.currentOrigin, targ->r.currentOrigin ) ) ||
		( VectorLength(v2) < CTF_TARGET_PROTECT_RADIUS &&
		trap_InPVS(flag->r.currentOrigin, attacker->r.currentOrigin ) ) ) &&
		attacker->client->sess.sessionTeam != targ->client->sess.sessionTeam) {

		// we defended the base flag
		AddScore(attacker, CTF_FLAG_DEFENSE_BONUS);
		attacker->client->pers.teamState.basedefense++;
		if (flag->r.svFlags & SVF_NOCLIENT) {
			PrintMsg(NULL, "%s" S_COLOR_WHITE " defends the %s base.\n",
				attacker->client->pers.netname, 
				TeamName(attacker->client->sess.sessionTeam));
		} else {
			PrintMsg(NULL, "%s" S_COLOR_WHITE " defends the %s flag.\n",
				attacker->client->pers.netname, 
				TeamName(attacker->client->sess.sessionTeam));
		}
		return;
	}

	if (carrier && carrier != attacker) {
		VectorSubtract(targ->r.currentOrigin, carrier->r.currentOrigin, v1);
		VectorSubtract(attacker->r.currentOrigin, carrier->r.currentOrigin, v1);

		if ( ( ( VectorLength(v1) < CTF_ATTACKER_PROTECT_RADIUS &&
			trap_InPVS(carrier->r.currentOrigin, targ->r.currentOrigin ) ) ||
			( VectorLength(v2) < CTF_ATTACKER_PROTECT_RADIUS &&
				trap_InPVS(carrier->r.currentOrigin, attacker->r.currentOrigin ) ) ) &&
			attacker->client->sess.sessionTeam != targ->client->sess.sessionTeam) {
			AddScore(attacker, CTF_CARRIER_PROTECT_BONUS);
			attacker->client->pers.teamState.carrierdefense++;
			PrintMsg(NULL, "%s" S_COLOR_WHITE " defends the %s's flag carrier.\n",
				attacker->client->pers.netname, 
				TeamName(attacker->client->sess.sessionTeam));
			return;
		}
	}
}

/*
================
Team_CheckHurtCarrier

Check to see if attacker hurt the flag carrier.  Needed when handing out bonuses for assistance to flag
carrier defense.
================
*/
void Team_CheckHurtCarrier(gentity_t *targ, gentity_t *attacker)
{
	int flag_pw;

	if (!targ->client || !attacker->client)
		return;

	if (targ->client->sess.sessionTeam == TEAM_RED)
		flag_pw = PW_BLUEFLAG;
	else
		flag_pw = PW_REDFLAG;

	if (targ->client->ps.powerups[flag_pw] &&
		targ->client->sess.sessionTeam != attacker->client->sess.sessionTeam)
		attacker->client->pers.teamState.lasthurtcarrier = level.time;
}


void Team_ResetItem(gentity_t *self)
{
	gentity_t *ent = NULL;
	if(g_gametype.integer < GT_CTF)
		return;
	
	while((ent = G_Find (ent, FOFS(classname), self->classname)) != NULL) 
	{
		if(ent->s.eType == ET_TELEPORT_TRIGGER)
		{
			G_SetOrigin(self, ent->r.currentOrigin);
			self->s.eFlags &= ~EF_DEAD;			
			self->enemy = NULL;
			self->nextthink = 0;
			self->touch = Touch_Item;
			if(self->item->giTag == WP_REDFLAG)
				Team_SetFlagStatus( TEAM_RED, FLAG_ATBASE );
			else if(self->item->giTag == WP_BLUEFLAG)
				Team_SetFlagStatus( TEAM_BLUE, FLAG_ATBASE );

			break;
		}
	}
	
	
}

void Team_ReturnFlagSound(gentity_t *ent)
{
	// play powerup spawn sound to all clients
	gentity_t	*te;

	if (ent == NULL) {
		G_Printf ("Warning:  NULL passed to Team_ReturnFlagSound\n");
		return;
	}

	te = G_TempEntity( ent->s.pos.trBase, EV_GLOBAL_SOUND );
	te->s.eventParm = G_SoundIndex( ent->item->giTag == WP_REDFLAG ?
		"sound/teamplay/flagret_red.wav" :
		"sound/teamplay/flagret_blu.wav" );
	te->r.svFlags |= SVF_BROADCAST;
}

void Team_FreeEntity(gentity_t *ent)
{
	Team_ResetItem(ent);
}

/*
==============
Team_DroppedFlagThink

Automatically set in Launch_Item if the item is one of the flags

Flags are unique in that if they are dropped, the base flag must be respawned when they time out
==============
*/
void Team_DroppedFlagThink(gentity_t *ent)
{
	Team_ReturnFlagSound(ent);
	Team_ResetItem(ent);
	// Reset Flag will delete this entity
}

qboolean Team_TouchFlag( gentity_t *ent, gentity_t *other ) {
	int i, team, myteam;
	gentity_t *player;
	gentity_t	*te;
	
	if(ent->item->giTag == WP_REDFLAG)
		team = TEAM_RED;
	else
		team = TEAM_BLUE;
	myteam = other->client->ps.persistant[PERS_TEAM];

	if(g_gametype.integer == GT_RUGBY)
	{
		if(bg_itemlist[other->client->ps.weapon].giType != IT_WEAPON || bg_itemlist[other->client->ps.weapon].giTag != WP_BALL)
			return qfalse; // can only score if you have the ball as your weapon.
		else if(team == myteam)
		{
			PrintMsg (NULL, "%s" S_COLOR_WHITE " returned the ball.\n", other->client->pers.netname);
			AddScore(other, CTF_RECOVERY_BONUS);
			other->client->pers.teamState.flagrecovery++;
			other->client->pers.teamState.lastreturnedflag = level.time;
			Team_ReturnFlagSound(ent);			
		}
		else
		{
			PrintMsg( NULL, "%s" S_COLOR_WHITE " scored a goal for the %s team!\n",
			other->client->pers.netname, TeamName(myteam));
			teamgame.last_flag_capture = level.time;
			teamgame.last_capture_team = team;
		}
		te = TossClientWeapon(other, BG_FindItemForWeapon(WP_BALL));
		Team_ResetItem(te);
		CalculateRanks();
		return qfalse;
	}			
	else if(g_gametype.integer == GT_CTF)
	{
		if(team != myteam)
		{
			PrintMsg (NULL, "%s" S_COLOR_WHITE " got the %s flag!\n",
				other->client->pers.netname, TeamName(team));
			AddScore(other, CTF_FLAG_BONUS);

			other->client->pers.teamState.flagsince = level.time;

			Team_SetFlagStatus( team, FLAG_TAKEN );
			CalculateRanks();
			return qtrue; // pick this flag up
		}
		if ( ent->s.eFlags & EF_DEAD ) {
			// hey, its not home.  return it by teleporting it back
			PrintMsg( NULL, "%s" S_COLOR_WHITE " returned the %s flag!\n", 
				other->client->pers.netname, TeamName(team));
			AddScore(other, CTF_RECOVERY_BONUS);
			other->client->pers.teamState.flagrecovery++;
			other->client->pers.teamState.lastreturnedflag = level.time;
			Team_ResetItem(ent);
			Team_ReturnFlagSound(ent);
			CalculateRanks();

			return qfalse; // don't pick up; it just got sent back to base
		}
		if(bg_itemlist[other->s.weapon].giType != IT_WEAPON || bg_itemlist[other->s.weapon].giTag < WP_REDFLAG)
		{
			return qfalse; // can't pick up our own flag
		}
	}
	PrintMsg( NULL, "%s" S_COLOR_WHITE " captured the %s flag!\n",
		other->client->pers.netname, TeamName(OtherTeam(team)));

	teamgame.last_flag_capture = level.time;
	teamgame.last_capture_team = team;

	// Increase the team's score
	level.teamScores[ other->client->sess.sessionTeam ]++;

	other->client->pers.teamState.captures++;

	// other gets another 10 frag bonus
	AddScore(other, CTF_CAPTURE_BONUS);

	te = G_TempEntity( ent->s.pos.trBase, EV_GLOBAL_SOUND );
	te->s.eventParm = G_SoundIndex( myteam == TEAM_RED ?
		"sound/teamplay/flagcap_red.wav" :
		"sound/teamplay/flagcap_blu.wav" );
	te->r.svFlags |= SVF_BROADCAST;

	// Ok, let's do the player loop, hand out the bonuses
	for (i = 0; i < g_maxclients.integer; i++) {
		player = &g_entities[i];
		if (!player->inuse)
			continue;

		if (player->client->ps.persistant[PERS_TEAM] != other->client->ps.persistant[PERS_TEAM]) 
			player->client->pers.teamState.lasthurtcarrier = -5;
		else if (player->client->ps.persistant[PERS_TEAM] == other->client->ps.persistant[PERS_TEAM]) 
		{
			if (player != other)
				AddScore(player, CTF_TEAM_BONUS);
			// award extra points for capture assists
			if (player->client->pers.teamState.lastreturnedflag + CTF_RETURN_FLAG_ASSIST_TIMEOUT > level.time) 
			{
				PrintMsg( NULL, 
						"%s" S_COLOR_WHITE " gets an assist for returning the %s flag!\n", 
						player->client->pers.netname,
						TeamName(team));
				AddScore (player, CTF_RETURN_FLAG_ASSIST_BONUS);
				other->client->pers.teamState.assists++;
			}
			if (player->client->pers.teamState.lastfraggedcarrier + CTF_FRAG_CARRIER_ASSIST_TIMEOUT > level.time) 
			{
				PrintMsg( NULL, "%s" S_COLOR_WHITE " gets an assist for fragging the %s flag carrier!\n", 
						player->client->pers.netname,
						TeamName(OtherTeam(team)));
				AddScore(player, CTF_FRAG_CARRIER_ASSIST_BONUS);
				other->client->pers.teamState.assists++;
			}
		}
	}
	te = TossClientWeapon(other, &bg_itemlist[other->client->ps.weapon]);
	Team_ResetItem(te);

	CalculateRanks();

	return qfalse; // Don't pick up our own flag after the capture
}


/*
===========
Team_GetLocation

Report a location for the player. Uses placed nearby target_location entities
============
*/
gentity_t *Team_GetLocation(gentity_t *ent)
{
	gentity_t		*eloc, *best;
	float			bestlen, len;
	vec3_t			origin;

	best = NULL;
	bestlen = 3*8192.0*8192.0;

	VectorCopy( ent->r.currentOrigin, origin );

	for (eloc = level.locationHead; eloc; eloc = eloc->nextTrain) {
		len = ( origin[0] - eloc->r.currentOrigin[0] ) * ( origin[0] - eloc->r.currentOrigin[0] )
			+ ( origin[1] - eloc->r.currentOrigin[1] ) * ( origin[1] - eloc->r.currentOrigin[1] )
			+ ( origin[2] - eloc->r.currentOrigin[2] ) * ( origin[2] - eloc->r.currentOrigin[2] );

		if ( len > bestlen ) {
			continue;
		}

		if ( !trap_InPVS( origin, eloc->r.currentOrigin ) ) {
			continue;
		}

		bestlen = len;
		best = eloc;
	}

	return best;
}


/*
===========
Team_GetLocation

Report a location for the player. Uses placed nearby target_location entities
============
*/
qboolean Team_GetLocationMsg(gentity_t *ent, char *loc, int loclen)
{
	gentity_t *best;

	best = Team_GetLocation( ent );
	
	if (!best)
		return qfalse;

	if (best->count) {
		if (best->count < 0)
			best->count = 0;
		if (best->count > 7)
			best->count = 7;
		Com_sprintf(loc, loclen, "%c%c%s" S_COLOR_WHITE, Q_COLOR_ESCAPE, best->count + '0', best->message );
	} else
		Com_sprintf(loc, loclen, "%s", best->message);

	return qtrue;
}


/*---------------------------------------------------------------------------*/

/*
================
SelectRandomDeathmatchSpawnPoint

go to a random point that doesn't telefrag
================
*/
#define	MAX_TEAM_SPAWN_POINTS	16
gentity_t *SelectRandomTeamSpawnPoint( int teamstate, team_t team ) {
	gentity_t	*spot;
	int			count;
	int			selection;
	gentity_t	*spots[MAX_TEAM_SPAWN_POINTS];
	char		*classname;

	if (teamstate == TEAM_BEGIN) {
		if (team == TEAM_RED)
			classname = "team_CTF_redplayer";
		else if (team == TEAM_BLUE)
			classname = "team_CTF_blueplayer";
		else
			return NULL;
	} else {
		if (team == TEAM_RED)
			classname = "team_CTF_redspawn";
		else if (team == TEAM_BLUE)
			classname = "team_CTF_bluespawn";
		else
			return NULL;
	}
	count = 0;

	spot = NULL;

	while ((spot = G_Find (spot, FOFS(classname), classname)) != NULL) {
		if ( SpotWouldTelefrag( spot ) ) {
			continue;
		}
		spots[ count ] = spot;
		if (++count == MAX_TEAM_SPAWN_POINTS)
			break;
	}

	if ( !count ) {	// no spots that won't telefrag
		return G_Find( NULL, FOFS(classname), classname);
	}

	selection = rand() % count;
	return spots[ selection ];
}


/*
===========
SelectCTFSpawnPoint

============
*/
gentity_t *SelectCTFSpawnPoint ( team_t team, int teamstate, vec3_t origin, vec3_t angles ) {
	gentity_t	*spot;

	spot = SelectRandomTeamSpawnPoint ( teamstate, team );

	if (!spot) {
		return SelectSpawnPoint( vec3_origin, origin, angles );
	}

	VectorCopy (spot->s.origin, origin);
	origin[2] += 9;
	VectorCopy (spot->s.angles, angles);

	return spot;
}

/*---------------------------------------------------------------------------*/

static int QDECL SortClients( const void *a, const void *b ) {
	return *(int *)a - *(int *)b;
}


/*
==================
TeamplayLocationsMessage

Format:
	clientNum location health armor weapon powerups

==================
*/
void TeamplayInfoMessage( gentity_t *ent ) {
	char		entry[1024];
	char		string[1400];
	int			stringlength;
	int			i, j;
	gentity_t	*player;
	int			cnt;
	int			h, a;
	int			clients[TEAM_MAXOVERLAY];

	if ( ! ent->client->pers.teamInfo )
		return;

	// figure out what client should be on the display
	// we are limited to 8, but we want to use the top eight players
	// but in client order (so they don't keep changing position on the overlay)
	for (i = 0, cnt = 0; i < g_maxclients.integer && cnt < TEAM_MAXOVERLAY; i++) {
		player = g_entities + level.sortedClients[i];
		if (player->inuse && player->client->sess.sessionTeam == 
			ent->client->sess.sessionTeam ) {
			clients[cnt++] = level.sortedClients[i];
		}
	}

	// We have the top eight players, sort them by clientNum
	qsort( clients, cnt, sizeof( clients[0] ), SortClients );

	// send the latest information on all clients
	string[0] = 0;
	stringlength = 0;

	for (i = 0, cnt = 0; i < g_maxclients.integer && cnt < TEAM_MAXOVERLAY; i++) {
		player = g_entities + i;
		if (player->inuse && player->client->sess.sessionTeam == 
			ent->client->sess.sessionTeam ) {

			h = player->client->ps.stats[STAT_HEALTH];
			a = player->client->ps.stats[STAT_ARMOR];
			if (h < 0) h = 0;
			if (a < 0) a = 0;

			Com_sprintf (entry, sizeof(entry),
				" %i %i %i %i %i %i", 
//				level.sortedClients[i], player->client->pers.teamState.location, h, a, 
				i, player->client->pers.teamState.location, h, a, 
				player->client->ps.weapon, player->s.powerups);
			j = strlen(entry);
			if (stringlength + j > sizeof(string))
				break;
			strcpy (string + stringlength, entry);
			stringlength += j;
			cnt++;
		}
	}

	trap_SendServerCommand( ent-g_entities, va("tinfo %i%s", cnt, string) );
}

void CheckTeamStatus(void)
{
	int i;
	gentity_t *loc, *ent;

	if (level.time - level.lastTeamLocationTime > TEAM_LOCATION_UPDATE_TIME) {

		level.lastTeamLocationTime = level.time;

		for (i = 0; i < g_maxclients.integer; i++) {
			ent = g_entities + i;
			if (ent->inuse && 
				(ent->client->sess.sessionTeam == TEAM_RED ||
				ent->client->sess.sessionTeam == TEAM_BLUE)) {
				loc = Team_GetLocation( ent );
				if (loc)
					ent->client->pers.teamState.location = loc->health;
				else
					ent->client->pers.teamState.location = 0;
			}
		}

		for (i = 0; i < g_maxclients.integer; i++) {
			ent = g_entities + i;
			if (ent->inuse && 
				(ent->client->sess.sessionTeam == TEAM_RED ||
				ent->client->sess.sessionTeam == TEAM_BLUE)) {
				TeamplayInfoMessage( ent );
			}
		}
	}
}

/*-----------------------------------------------------------------*/

/*QUAKED team_CTF_redplayer (1 0 0) (-16 -16 -16) (16 16 32)
Only in CTF games.  Red players spawn here at game start.
*/
void SP_team_CTF_redplayer( gentity_t *ent ) {
}


/*QUAKED team_CTF_blueplayer (0 0 1) (-16 -16 -16) (16 16 32)
Only in CTF games.  Blue players spawn here at game start.
*/
void SP_team_CTF_blueplayer( gentity_t *ent ) {
}


/*QUAKED team_CTF_redspawn (1 0 0) (-16 -16 -24) (16 16 32)
potential spawning position for red team in CTF games.
Targets will be fired when someone spawns in on them.
*/
void SP_team_CTF_redspawn(gentity_t *ent) {
}

/*QUAKED team_CTF_bluespawn (0 0 1) (-16 -16 -24) (16 16 32)
potential spawning position for blue team in CTF games.
Targets will be fired when someone spawns in on them.
*/
void SP_team_CTF_bluespawn(gentity_t *ent) {
}



