// Copyright (C) 1999-2000 Id Software, Inc.
//
#include "g_local.h"

#define	MISSILE_PRESTEP_TIME	0;//50

/*
================
G_BounceMissile

================
*/
void G_BounceMissile( gentity_t *ent, trace_t *trace ) {
	vec3_t	velocity;
	float	dot;
	int		hitTime;
	// reflect the velocity on the trace plane
	hitTime = level.previousTime + ( level.time - level.previousTime ) * trace->fraction;
	BG_EvaluateTrajectoryDelta( &ent->s.pos, hitTime, velocity );
	dot = DotProduct( velocity, trace->plane.normal );
	VectorMA( trace->endpos, 2, trace->plane.normal, ent->r.currentOrigin);
	VectorCopy(trace->endpos, ent->s.pos.trBase);
	ent->s.pos.trTime = hitTime;
		
	VectorMA( velocity, -2*dot, trace->plane.normal, ent->s.pos.trDelta );
	if ( ent->s.eFlags & EF_BOUNCE_HALF ) {
		VectorScale( ent->s.pos.trDelta, 0.65, ent->s.pos.trDelta );
		// check for stop
		if ( trace->plane.normal[2] > 0.2 && VectorLength( ent->s.pos.trDelta ) < 40 ) 
		{
			G_SetOrigin( ent, trace->endpos );			
		}
	}

	//VectorAdd( ent->r.currentOrigin, trace->plane.normal, ent->r.currentOrigin);
	
	
}


/*
================
G_MissileImpact

================
*/
void G_MissileImpact( gentity_t *ent, trace_t *trace ) {
	gentity_t		*other;
	qboolean		hitClient = qfalse;
	int hitloc = 0;
	other = &g_entities[trace->entityNum];		
	
	// check for bounce
	if ( ent->s.eFlags & ( EF_BOUNCE | EF_BOUNCE_HALF ) ) 
	{
		G_BounceMissile( ent, trace );
		if(other->takedamage && (other->client || (other->s.eType == ET_CORPSE)))
		{
			vec3_t velocity;
			VectorCopy(ent->s.pos.trDelta, velocity);
			VectorNormalize(velocity);
			VectorCopy(other->s.pos.trDelta, ent->s.pos.trDelta);
			hitloc = G_Damage (other, ent, &g_entities[ent->r.ownerNum], velocity,
				trace->endpos, ent->damage, DAMAGE_NO_BLEEDING, ent->methodOfDeath);
		}
		else
			G_AddEvent( ent, EV_GRENADE_BOUNCE, 0 );
		return;
	}

	// impact damage
	if (other->takedamage) {
		// FIXME: wrong damage direction?
		if ( ent->damage ) {
			vec3_t	velocity;
			int dflags = 0;

			if( LogAccuracyHit( other, &g_entities[ent->r.ownerNum] ) ) {
				g_entities[ent->r.ownerNum].client->ps.persistant[PERS_ACCURACY_HITS]++;
				hitClient = qtrue;
			}
			BG_EvaluateTrajectoryDelta( &ent->s.pos, level.time, velocity );
			if ( VectorLength( velocity ) == 0 ) {
				velocity[2] = 1;	// stepped on a grenade
			}
			if(ent->s.eFlags & (EF_BOUNCE | EF_BOUNCE_HALF))
				dflags = 0; // grenades have shrapnel				
			else if(ent->methodOfDeath == MOD_MACHINEGUN)
				dflags = 0;
			else
				dflags = DAMAGE_NO_BLEEDING; 

				G_Damage (other, ent, &g_entities[ent->r.ownerNum], velocity, 
				trace->endpos, ent->damage, dflags, ent->methodOfDeath);
		}
	}
	
	// is it cheaper in bandwidth to just remove this ent and create a new
	// one, rather than changing the missile into the explosion?

	if ( other->takedamage && other->client ) {
			
		if((p_class[other->client->pers.playerclass].bleeds) && (hitloc != HIT_TORSO || other->client->ps.stats[STAT_ARMOR] <= 0))
			G_AddEvent( ent, EV_MISSILE_HIT, DirToByte( trace->plane.normal ) );
		else
			G_AddEvent( ent, EV_MISSILE_HIT_ROBOT, DirToByte( trace->plane.normal ) );
		
		ent->s.otherEntityNum = other->s.number;
	}
	else if ( other->takedamage == -1 )
	{ // bloody corpses
		G_AddEvent( ent, EV_MISSILE_HIT, DirToByte( trace->plane.normal ) );
	}
	else if ( other->takedamage )
	{
		G_AddEvent( ent, EV_MISSILE_HIT_ROBOT, DirToByte( trace->plane.normal ) );
	}
	else if (!(ent->s.eFlags & EF_DEAD) && ent->s.weapon == WP_MACHINEGUN || ent->s.weapon == WP_CHAINGUN ) {

		vec3_t dir;
		VectorCopy(ent->s.pos.trDelta, dir);
		VectorNormalize(dir);
		ent->damage -= (5 + (rand() & 15)) * fabs(DotProduct(trace->plane.normal, dir));
		if(ent->damage > 0)
		{
			float speed;
			gentity_t *tent = G_TempEntity( trace->endpos, EV_MISSILE_MISS );			
			tent->s.eventParm = DirToByte( trace->plane.normal );
			tent->s.otherEntityNum = ent->s.number;
			tent->s.weapon = ent->s.weapon;	
			tent->s.clientNum = ent->s.clientNum;
			tent->s.powerups = ent->s.powerups;
			VectorCopy(trace->endpos, tent->s.origin);
			if(level.time - ent->s.pos.trTime < FRAMETIME)
			{
				VectorCopy(ent->s.pos.trBase, tent->s.origin2);
			}
			else
			{
				BG_EvaluateTrajectory(&ent->s.pos, level.time - FRAMETIME, tent->s.origin2);
			}
			G_BounceMissile( ent, trace );		
			speed = VectorNormalize(ent->s.pos.trDelta);
			speed *= (random() + 4.0) / 5.0;
			VectorScale(ent->s.pos.trDelta, (speed / 1000.0 + random() * 10), ent->s.pos.trDelta);
			VectorAdd(ent->s.pos.trDelta, trace->plane.normal, ent->s.pos.trDelta);
			VectorAdd(ent->s.pos.trDelta, bytedirs[rand() % NUMVERTEXNORMALS], ent->s.pos.trDelta);
			VectorNormalize(ent->s.pos.trDelta);
			VectorScale(ent->s.pos.trDelta, speed, ent->s.pos.trDelta);
			return;
		}
		
		VectorMA( trace->endpos, 2, trace->plane.normal, ent->s.pos.trBase);
		ent->s.pos.trTime = level.time;
		ent->r.svFlags = SVF_BROADCAST;
		G_AddEvent( ent, EV_MISSILE_MISS, DirToByte( trace->plane.normal ) );

	} else {
		VectorMA( trace->endpos, 2, trace->plane.normal, ent->s.pos.trBase);
		ent->s.pos.trTime = level.time;
		
		G_AddEvent( ent, EV_MISSILE_MISS, DirToByte( trace->plane.normal ) );
	}

	ent->freeAfterEvent = qtrue;

	// change over to a normal entity right at the point of impact
	ent->s.eType = ET_GENERAL;

	SnapVectorTowards( trace->endpos, ent->s.pos.trBase );	// save net bandwidth

	G_SetOrigin( ent, trace->endpos );

	// splash damage (now also applies to person directly hit)
	if ( ent->splashDamage ) {
		if( G_RadiusDamage( trace->endpos, ent->parent, ent->splashDamage, ent->splashRadius, ent->splashDuration,
			NULL, ent->splashMethodOfDeath ) ) {
			if( !hitClient ) {
				g_entities[ent->r.ownerNum].client->ps.persistant[PERS_ACCURACY_HITS]++;
			}
		}
	}

	trap_LinkEntity( ent );
}

/*
================
G_ExplodeMissile

Explode a missile without an impact
================
*/
void G_ExplodeMissile( gentity_t *ent ) {
	vec3_t		dir;
	vec3_t		origin;

	BG_EvaluateTrajectory( &ent->s.pos, level.time, origin );
	SnapVector( origin );
	G_SetOrigin( ent, origin );

	// we don't have a valid direction, so just point straight up
	dir[0] = dir[1] = 0;
	dir[2] = 1;

	ent->s.eType = ET_GENERAL;
	G_AddEvent( ent, EV_MISSILE_MISS, DirToByte( dir ) );

	ent->freeAfterEvent = qtrue;

	// splash damage
	if ( ent->splashDamage ) {
		if( G_RadiusDamage( ent->r.currentOrigin, ent->parent, ent->splashDamage, ent->splashRadius, ent->splashDuration, NULL
			, ent->splashMethodOfDeath ) ) {
			g_entities[ent->r.ownerNum].client->ps.persistant[PERS_ACCURACY_HITS]++;
		}
	}
	
	if(ent->splashMethodOfDeath == MOD_GRENADE_SPLASH && !(ent->s.eFlags & EF_DEAD))
	{
		int i;
		for(i = 0; i < 8; i++)
		{
			gentity_t *bolt = fire_bullet(ent->parent, ent->r.currentOrigin, bytedirs[rand() % NUMVERTEXNORMALS]);			
			if(i & 1) // only show half of them client-side, to save bandwidth - we'll be making a LOT of client-side ones to hide the 'real' ones.
				bolt->r.svFlags |= SVF_NOCLIENT;
			VectorScale(bolt->s.pos.trDelta, 0.1, bolt->s.pos.trDelta);
			bolt->s.powerups = (ent->s.powerups & PW_QUAD);
			bolt->methodOfDeath = MOD_GRENADE_SPLASH;
		}
	}
	trap_LinkEntity( ent );
}


/*
================
G_RunMissile

================
*/
void G_RunMissile( gentity_t *ent ) {
	vec3_t		origin, velocity;
	trace_t		tr;
	float dt = (level.time - ent->s.pos.trTime) * 0.001;
	vec3_t forward;
	
	if((ent->methodOfDeath == MOD_ROCKET) )
	{
		ent->s.angles[0] += crandom() * ROCKET_VEER * dt;
		ent->s.angles[1] += crandom() * ROCKET_VEER * dt;
		ent->s.angles[2] += crandom() * ROCKET_VEER * dt;
		AngleVectors(ent->s.angles, forward, NULL, NULL);		
		VectorMA(ent->s.pos.trDelta, ROCKET_ACCEL * dt * 0.5, forward, ent->s.pos.trDelta);
	}
	// get current position

	BG_EvaluateTrajectory( &ent->s.pos, level.time, origin );
	BG_EvaluateTrajectoryDelta( &ent->s.pos, level.time, velocity );
	if(!VectorLength(origin))
	{
		return;
	}
	// trace a line from the previous position to the current position,
	// ignoring interactions with the missile owner
	if(ent->s.eFlags & EF_DEAD)
		trap_Trace( &tr, ent->r.currentOrigin, ent->r.mins, ent->r.maxs, origin, 0, ent->clipmask );
	else
		trap_Trace( &tr, ent->r.currentOrigin, ent->r.mins, ent->r.maxs, origin, ent->r.ownerNum, ent->clipmask );



	if ( tr.startsolid ) {
		//G_Printf("???");
		tr.fraction = 0;
	}

	trap_LinkEntity( ent );
	VectorCopy(tr.endpos, ent->r.currentOrigin);
	if ( tr.fraction != 1.0 ) {
		// never explode or bounce on sky
		if ( tr.surfaceFlags & SURF_NOIMPACT ) {
			G_FreeEntity( ent );
			return;
		}
		if((ent->methodOfDeath == MOD_ROCKET) && (tr.plane.normal[0] == -1 || tr.plane.normal[1] == -1 || tr.plane.normal[2] == -1))
		{ // for some reason these specific angles don't show up
			tr.plane.normal[0] += crandom() * 0.125;
			tr.plane.normal[1] += crandom() * 0.125;
			tr.plane.normal[2] += crandom() * 0.125;
			VectorNormalize(tr.plane.normal);
		}
	
		G_MissileImpact( ent, &tr );
		if ( ent->s.eType != ET_MISSILE ) {
			return;		// exploded
		}
	}

	else 
	{
		
		if((ent->methodOfDeath == MOD_ROCKET))
		{
			VectorMA(ent->s.pos.trBase, dt, ent->s.pos.trDelta, ent->s.pos.trBase);
			VectorMA(ent->s.pos.trDelta, ROCKET_ACCEL * dt * 0.5, forward, ent->s.pos.trDelta);
			ent->s.pos.trType = TR_INTERPOLATE;
			ent->s.pos.trTime = level.time;
		}
	}
	// check think function after bouncing
	G_RunThink( ent );
}


//=============================================================================

gentity_t *fire_bullet (gentity_t *self, vec3_t start, vec3_t dir) {
	gentity_t	*bolt;
	float		*vel;
	if(self->client)
		vel = self->client->ps.velocity;
	else
		vel = self->s.pos.trDelta;

	VectorNormalize (dir);
	
	bolt = G_Spawn();
	bolt->classname = "bullet";
	bolt->nextthink = level.time + 10000;
	bolt->think = G_FreeEntity;	
	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_MACHINEGUN;
	bolt->s.powerups = (self->s.powerups & PW_QUAD);
	bolt->r.ownerNum = self->s.number;
	bolt->parent = self;
	bolt->damage = 10;
	bolt->splashDamage = 5;
	bolt->splashRadius = 1;
	bolt->methodOfDeath = MOD_MACHINEGUN;
	bolt->splashMethodOfDeath = MOD_MACHINEGUN;
	bolt->clipmask = MASK_SHOT;
	bolt->r.mins[0] = 0;
	bolt->r.mins[1] = 0;
	bolt->r.mins[2] = 0;
	bolt->r.maxs[0] = 0;
	bolt->r.maxs[1] = 0;
	bolt->r.maxs[2] = 0;
	bolt->r.svFlags = 0;//SVF_NOCLIENT; // no tracer
	bolt->s.pos.trType = TR_GRAVITY;//TR_LINEAR;
	bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;		// move a bit on the very first frame
	
	VectorCopy( start, bolt->s.pos.trBase );
	VectorScale( dir, 8000, bolt->s.pos.trDelta );
	VectorMA(bolt->s.pos.trDelta, 0.9, vel, bolt->s.pos.trDelta );						
	
	SnapVector( bolt->s.pos.trDelta );			// save net bandwidth

	VectorCopy (start, bolt->r.currentOrigin);

	return bolt;
}	

/*
=================
fire_plasma

=================
*/
gentity_t *fire_plasma (gentity_t *self, vec3_t start, vec3_t dir) {
	gentity_t	*bolt;
	float		*vel;
	if(self->client)
		vel = self->client->ps.velocity;
	else
		vel = self->s.pos.trDelta;

	VectorNormalize (dir);

	bolt = G_Spawn();
	bolt->classname = "plasma";
	bolt->nextthink = level.time + 10000;
	bolt->think = G_ExplodeMissile;
	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_PLASMAGUN;
	bolt->r.ownerNum = self->s.number;
	bolt->parent = self;
	bolt->damage = 20;
	bolt->splashDamage = 15;
	bolt->splashRadius = 20;
	bolt->methodOfDeath = MOD_PLASMA;
	bolt->splashMethodOfDeath = MOD_PLASMA_SPLASH;
	bolt->clipmask = MASK_SHOT;
	bolt->r.mins[0] = -4;
	bolt->r.mins[1] = -4;
	bolt->r.mins[2] = -4;
	bolt->r.maxs[0] = 4;
	bolt->r.maxs[1] = 4;
	bolt->r.maxs[2] = 4;
	bolt->s.pos.trType = TR_LINEAR;
	bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;		// move a bit on the very first frame
	VectorCopy( start, bolt->s.pos.trBase );
	VectorScale( dir, 2000, bolt->s.pos.trDelta );
	VectorMA(bolt->s.pos.trDelta, 0.9, vel, bolt->s.pos.trDelta );						
	
	SnapVector( bolt->s.pos.trDelta );			// save net bandwidth

	VectorCopy (start, bolt->r.currentOrigin);

	return bolt;
}	

//=============================================================================

/*
=================
fire_grenade
=================
*/
gentity_t *toss_grenade (gentity_t *self, vec3_t start, vec3_t dir) {
	gentity_t	*bolt;
	float		*vel;
	if(self->client)
		vel = self->client->ps.velocity;
	else
		vel = self->s.pos.trDelta;

	VectorNormalize (dir);

	bolt = G_Spawn();
	bolt->classname = "grenade";
	bolt->nextthink = level.time + 2500;
	bolt->think = G_ExplodeMissile;
	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_GRENADE_LAUNCHER;
	bolt->s.eFlags = EF_BOUNCE_HALF;
	bolt->r.ownerNum = self->s.number;
	bolt->parent = self;
	bolt->damage = 1;
	bolt->splashDamage = DEFAULT_GRENADE_DAMAGE;
	bolt->splashRadius = DEFAULT_GRENADE_RADIUS;
	bolt->splashDuration = DEFAULT_GRENADE_DURATION;
	bolt->methodOfDeath = MOD_GRENADE;
	
	bolt->splashMethodOfDeath = MOD_GRENADE_SPLASH;
	bolt->clipmask = MASK_SHOT;
	bolt->r.mins[0] = -1;
	bolt->r.mins[1] = -1;
	bolt->r.mins[2] = -1;
	bolt->r.maxs[0] = 1;
	bolt->r.maxs[1] = 1;
	bolt->r.maxs[2] = 1;
	
	bolt->s.pos.trType = TR_GRAVITY;
	bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;		// move a bit on the very first frame
	VectorCopy( start, bolt->s.pos.trBase );
	VectorScale( dir, 300, bolt->s.pos.trDelta );
	VectorMA(bolt->s.pos.trDelta, 0.9, vel, bolt->s.pos.trDelta );						
	
	SnapVector( bolt->s.pos.trDelta );			// save net bandwidth

	VectorCopy (start, bolt->r.currentOrigin);

	return bolt;
}
/*
=================
fire_grenade
=================
*/
gentity_t *fire_grenade (gentity_t *self, vec3_t start, vec3_t dir) {
	gentity_t	*bolt;
	float		*vel;
	if(self->client)
		vel = self->client->ps.velocity;
	else
		vel = self->s.pos.trDelta;
	VectorNormalize (dir);
	bolt = G_Spawn();
	bolt->classname = "grenade";
	bolt->nextthink = level.time + 2500;
	bolt->think = G_ExplodeMissile;
	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_GRENADE_LAUNCHER;
	bolt->s.eFlags = EF_BOUNCE_HALF;
	bolt->r.ownerNum = self->s.number;
	bolt->parent = self;
	bolt->damage = 2;
	bolt->splashDamage = DEFAULT_GRENADE_DAMAGE;
	bolt->splashRadius = DEFAULT_GRENADE_RADIUS;
	bolt->splashDuration = DEFAULT_GRENADE_DURATION;
	bolt->methodOfDeath = MOD_GRENADE;
	
	bolt->splashMethodOfDeath = MOD_GRENADE_SPLASH;
	bolt->clipmask = MASK_SHOT;
	bolt->r.mins[0] = -1;
	bolt->r.mins[1] = -1;
	bolt->r.mins[2] = -1;
	bolt->r.maxs[0] = 1;
	bolt->r.maxs[1] = 1;
	bolt->r.maxs[2] = 1;
	
	bolt->s.pos.trType = TR_GRAVITY;
	bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;		// move a bit on the very first frame
	VectorCopy( start, bolt->s.pos.trBase );
	VectorScale( dir, 700, bolt->s.pos.trDelta );
	VectorMA(bolt->s.pos.trDelta, 0.9, vel, bolt->s.pos.trDelta );						
	
	SnapVector( bolt->s.pos.trDelta );			// save net bandwidth

	VectorCopy (start, bolt->r.currentOrigin);

	return bolt;
}

//=============================================================================


//=============================================================================


/*void RocketThink( gentity_t *self )
{
	vec3_t forward;
	
	self->nextthink +=10;
	
	self->count++;
	if(self->count >= 100)
	{
		G_ExplodeMissile(self);
		return;
	}
	if(g_RocketAccel.value || g_RocketVeer.value)
	{
		float dt = (level.time - self->s.pos.trTime) * 0.001;
		self->s.angles[0] += crandom() * g_RocketVeer.value * dt;
		self->s.angles[1] += crandom() * g_RocketVeer.value * dt;
		self->s.angles[2] += crandom() * g_RocketVeer.value * dt;
		AngleVectors(self->s.angles, forward, NULL, NULL);		
		//VectorScale(self->s.pos.trDelta, 0.9, self->s.pos.trDelta );
		VectorMA(self->s.pos.trDelta, g_RocketAccel.value * dt * 0.5, forward, self->s.pos.trDelta);
		VectorMA(self->s.pos.trBase, dt, self->s.pos.trDelta, self->s.pos.trBase);
		VectorMA(self->s.pos.trDelta, g_RocketAccel.value * dt * 0.5, forward, self->s.pos.trDelta);
		self->s.pos.trType = TR_INTERPOLATE;
		self->s.pos.trTime = level.time;
	}
	
}*/
/*
=================
fire_rocket
=================
*/
gentity_t *fire_rocket (gentity_t *self, vec3_t start, vec3_t dir) {
	gentity_t	*bolt;
	float		*vel;
	if(self->client)
		vel = self->client->ps.velocity;
	else
		vel = self->s.pos.trDelta;
	if(!VectorLength(start))
	{
		return NULL;
	}
	VectorNormalize (dir);

	bolt = G_Spawn();
	bolt->classname = "rocket";
	bolt->nextthink = level.time + 10000;
	bolt->think = G_ExplodeMissile;
	bolt->s.eType = ET_MISSILE;
	bolt->r.svFlags = SVF_USE_CURRENT_ORIGIN;
	bolt->s.weapon = WP_ROCKET_LAUNCHER;
	bolt->r.ownerNum = self->s.number;
	bolt->parent = self;
	bolt->damage = 50;
	bolt->splashDamage = DEFAULT_ROCKET_DAMAGE;
	bolt->splashRadius = DEFAULT_ROCKET_RADIUS;
	bolt->splashDuration = DEFAULT_ROCKET_DURATION;
	bolt->methodOfDeath = MOD_ROCKET;
	bolt->splashMethodOfDeath = MOD_ROCKET_SPLASH;
	bolt->clipmask = MASK_SHOT;
	bolt->count = 0;
	bolt->s.pos.trType = TR_LINEAR;
	bolt->s.pos.trTime = level.time - MISSILE_PRESTEP_TIME;		// move a bit on the very first frame
	VectorCopy( start, bolt->s.pos.trBase );
	bolt->splashDuration = 600;
	bolt->r.mins[0] = -2;
	bolt->r.mins[1] = -2;
	bolt->r.mins[2] = -2;
	bolt->r.maxs[0] = 2;
	bolt->r.maxs[1] = 2;
	bolt->r.maxs[2] = 2;
		
	vectoangles(dir, bolt->s.angles);
	VectorScale( dir, ROCKET_SPEED, bolt->s.pos.trDelta );	
	VectorMA(bolt->s.pos.trDelta, 0.9, vel, bolt->s.pos.trDelta );						
	SnapVector( bolt->s.pos.trDelta );			// save net bandwidth
	VectorCopy (start, bolt->r.currentOrigin);
	return bolt;
}

