// Copyright (C) 1999-2000 Id Software, Inc.
//
#include "g_local.h"

/*

  Items are any object that a player can touch to gain some effect.

  Pickup will return the number of seconds until they should respawn.

  all items should pop when dropped in lava or slime

  Respawnable items don't actually go away when picked up, they are
  just made invisible and untouchable.  This allows them to ride
  movers and respawn apropriately.
*/


#define	RESPAWN_ARMOR		25
#define	RESPAWN_TEAM_WEAPON	30
#define	RESPAWN_HEALTH		35
#define	RESPAWN_AMMO		40
#define	RESPAWN_WEAPON		10
#define	RESPAWN_HOLDABLE	60
#define	RESPAWN_MEGAHEALTH	120
#define	RESPAWN_POWERUP		120

static void G_SetBounds(gentity_t *ent)
{
	gitem_t *item = ent->item;
	if(item->giType == IT_AMMO && item->giTag == AMMO_GRENADES)
	{ // very very tiny
		VectorSet (ent->r.mins, -1, -1, -1);
		VectorSet (ent->r.maxs, 1, 1, 1);
	}
	else if(item->giType == IT_POWERUP)
	{ // they are filled with largeness
			VectorSet (ent->r.mins, -ITEM_RADIUS, -ITEM_RADIUS, -ITEM_RADIUS);
			VectorSet (ent->r.maxs, ITEM_RADIUS, ITEM_RADIUS, ITEM_RADIUS);
	}
	else if(item->giType == IT_WEAPON)
	{ // large
			VectorSet (ent->r.mins, -8, -8, -5);
			VectorSet (ent->r.maxs, 8, 8, 5);
	}
	else if(item->giType == IT_HEALTH)
	{ // small
		VectorSet (ent->r.mins, -2, -2, -1);
		VectorSet (ent->r.maxs, 2, 2, 1);
	}
	else
	{ // medium
		VectorSet (ent->r.mins, -5, -5, -3);
		VectorSet (ent->r.maxs, 5, 5, 3);
	}

}

//======================================================================

int Pickup_Powerup( gentity_t *ent, gentity_t *other ) {
	int			quantity;
	int			i;
	gclient_t	*client;

	if ( !other->client->ps.powerups[ent->item->giTag] ) {
		// round timing to seconds to make multiple powerup timers
		// count in sync
		other->client->ps.powerups[ent->item->giTag] = 
			level.time - ( level.time % 1000 );
	}

	if ( ent->count ) {
		quantity = ent->count;
	} else {
		quantity = ent->item->quantity;
		if(ent->item->giType == PW_FLIGHT)
			quantity = g_FlightTime.integer;
	}
	other->client->ps.powerups[ent->item->giTag] += quantity * 1000;

	// give any nearby players a "denied" anti-reward
	for ( i = 0 ; i < level.maxclients ; i++ ) {
		vec3_t		delta;
		float		len;
		vec3_t		forward;
		trace_t		tr;

		client = &level.clients[i];
		if ( client == other->client ) {
			continue;
		}
		if ( client->pers.connected == CON_DISCONNECTED ) {
			continue;
		}
		if ( client->ps.stats[STAT_HEALTH] <= 0 ) {
			continue;
		}

    // if same team in team game, no sound
    // cannot use OnSameTeam as it expects to g_entities, not clients
  	if ( g_gametype.integer >= GT_TEAM && other->client->sess.sessionTeam == client->sess.sessionTeam  ) {
      continue;
    }

		// if too far away, no sound
		VectorSubtract( ent->s.pos.trBase, client->ps.origin, delta );
		len = VectorNormalize( delta );
		if ( len > 192 ) {
			continue;
		}

		// if not facing, no sound
		AngleVectors( client->ps.viewangles, forward, NULL, NULL );
		if ( DotProduct( delta, forward ) < 0.4 ) {
			continue;
		}

		// if not line of sight, no sound
		trap_Trace( &tr, client->ps.origin, NULL, NULL, ent->s.pos.trBase, ENTITYNUM_NONE, CONTENTS_SOLID );
		if ( tr.fraction != 1.0 ) {
			continue;
		}

		// anti-reward
		client->ps.persistant[PERS_REWARD_COUNT]++;
		client->ps.persistant[PERS_REWARD] = REWARD_DENIED;
	}

	return RESPAWN_POWERUP;
}

//======================================================================

int Pickup_Holdable (gentity_t *ent, gentity_t *other) {
	int		quantity;

	quantity = ent->count;
	if(ent->item->giType == IT_AMMO && ent->item->giTag == AMMO_GRENADES)
	{
		ent->s.powerups = --ent->count;
		other->client->ps.weapon = ent->item - bg_itemlist;
		other->client->ps.ammo[AMMO_CURRENT] = 1;
		other->client->ps.weaponstate = WEAPON_RAISING;
		other->client->ps.weaponTime += 250;
		other->client->ps.torsoAnim = ( ( other->client->ps.torsoAnim & ANIM_TOGGLEBIT ) ^ ANIM_TOGGLEBIT ) | TORSO_RAISE;
		if(ent->count)
			return -32768;
		else
			return RESPAWN_WEAPON;
	}
	if(ent->item->giType == IT_WEAPON && (ent->item->giTag == WP_REDFLAG || ent->item->giTag == WP_BLUEFLAG))
	{
		if (!Team_TouchFlag(ent, other))
			return -32768;

	}
	// add the weapon and switch
	other->client->ps.weapon = ent->item - bg_itemlist;
	other->client->ps.ammo[AMMO_CURRENT] = quantity;
	other->client->ps.weaponstate = WEAPON_RAISING;
	other->client->ps.weaponTime += 250;
	other->client->ps.torsoAnim = ( ( other->client->ps.torsoAnim & ANIM_TOGGLEBIT ) ^ ANIM_TOGGLEBIT ) | TORSO_RAISE;
	if(ent->item->giTag >= WP_REDFLAG)
		other->client->ps.ammo[AMMO_CURRENT] = -1;
	if(g_BallTimeLimit.value > 0 && ent->item->giTag == WP_BALL)
	{
		other->client->ps.powerups[PW_BALL] = 	level.time + g_BallTimeLimit.value * 1000 - ( level.time % 1000 );	
	}
	return RESPAWN_WEAPON;
}


//======================================================================
/*
int Pickup_Health (gentity_t *ent, gentity_t *other) {
	int			max;
	int			quantity;

	other->client->ps.stats[STAT_LEG_DAMAGE] /= (1 + sqrt(ent->item->quantity));
	other->client->ps.stats[STAT_HEAD_DAMAGE] /= (1 + sqrt(ent->item->quantity));
	// small and mega healths will go over the max
	if ( ent->item->quantity != 5 && ent->item->quantity != 100  ) {
		max = MAX_HEALTH;
	} else {
		max = MAX_HEALTH * 2;
	}
	other->client->ps.stats[STAT_BLEEDING] = 0;
	other->client->bleed_damage = 0;
	other->client->bleedtime = level.time;
	if ( ent->count ) {
		quantity = ent->count;
	} else {
		quantity = ent->item->quantity;
	}

	other->health += quantity;	

	if (other->health > max ) {
		other->health = max;
	}
	other->client->ps.stats[STAT_HEALTH] = other->health;
	if(other->health >= MAX_HEALTH)
	{
		other->client->ps.stats[STAT_LEG_DAMAGE] = 0;
		other->client->ps.stats[STAT_HEAD_DAMAGE] = 0;
	}
	if ( ent->item->giTag == 100 ) {		// mega health respawns slow
		return RESPAWN_MEGAHEALTH;
	}

	return RESPAWN_HEALTH;
}
*/
//======================================================================

int Pickup_Armor( gentity_t *ent, gentity_t *other ) {
	other->client->ps.stats[STAT_ARMOR] = ent->item->quantity;
	return RESPAWN_ARMOR;
}

//======================================================================

/*
===============
RespawnItem
===============
*/
void RespawnItem( gentity_t *ent ) {
	// randomly select from teamed entities
	if (ent->team) {
		gentity_t	*master;
		int	count;
		int choice;

		if ( !ent->teammaster ) {
			G_Error( "RespawnItem: bad teammaster");
		}
		master = ent->teammaster;

		for (count = 0, ent = master; ent; ent = ent->teamchain, count++)
			;

		choice = rand() % count;

		for (count = 0, ent = master; count < choice; ent = ent->teamchain, count++)
			;
	}
	if ( ent->item->giType == IT_WEAPON && ent->count == 0) 
	{
		ent->count = ent->item->quantity;
		{
			switch(ent->item->giTag)
			{
			case WP_SHOTGUN:
			case WP_SINGLE_SHOTGUN:
				ent->count = 12;
				break;
			case WP_GRENADE_LAUNCHER:
			case WP_ROCKET_LAUNCHER:
				ent->count = 5;
				break;
			default:
				break;
			}
		}
		ent->s.powerups = ent->count;
	
	}
	
	ent->r.contents = CONTENTS_TRIGGER;
	ent->s.eFlags &= ~EF_NODRAW;
	ent->r.svFlags &= ~SVF_NOCLIENT;
	ent->s.powerups = ent->count;
	trap_LinkEntity (ent);

	if ( ent->item->giType == IT_POWERUP ) {
		// play powerup spawn sound to all clients
		gentity_t	*te;

		te = G_TempEntity( ent->s.pos.trBase, EV_GLOBAL_SOUND );
		te->s.eventParm = G_SoundIndex( "sound/items/poweruprespawn.wav" );
		te->r.svFlags |= SVF_BROADCAST;
	}

	// play the normal respawn sound only to nearby clients
	G_AddEvent( ent, EV_ITEM_RESPAWN, 0 );

	ent->nextthink = 0;
}

/*
===============
Touch_Item
===============
*/
void Touch_Item (gentity_t *ent, gentity_t *other, trace_t *trace) {
	int			respawn;

	if (!other->client)
	{
		//G_Printf("non client attempted pickup\n");
		return;
	}
	else if (other->client->ps.stats[STAT_HEALTH] < 1)
	{
		//G_Printf("dead client attempted pickup\n");
		return;		// dead people can't pickup
	}
	else if(ent->onfire > level.time)
	{
		//G_Printf("attempted pickup of burning item\n");
		G_Damage(other, ent, ent, vec3_origin, ent->s.origin, 4 + (rand() & 7), DAMAGE_NO_KNOCKBACK | DAMAGE_NO_BLEEDING, MOD_FLAMETHROWER);
		return;
	}
	// the same pickup rules are used for client side and server side
	if ( !BG_CanItemBeGrabbed( &ent->s, &other->client->ps ) ) {		
		return;
	}

	G_LogPrintf( "Item: %i %s\n", other->s.number, ent->item->classname );

	// call the item-specific pickup function
	if( ent->item->giType == IT_POWERUP)
		respawn = Pickup_Powerup(ent, other);
	else
		respawn = Pickup_Holdable(ent, other);
	
	if(respawn == -32768)
		return;		
	else if(ent->item->giType != IT_POWERUP)
		ent->freeAfterEvent = qtrue; // if it respawns, it won't be in this lifetime

	// play the normal pickup sound
	if ( other->client->pers.predictItemPickup ) {
		G_AddPredictableEvent( other, EV_ITEM_PICKUP, ent->s.modelindex );
	} else {
		G_AddEvent( other, EV_ITEM_PICKUP, ent->s.modelindex );
	}

	// powerup pickups are global broadcasts
	if ( ent->item->giType == IT_POWERUP || (ent->item->giType == IT_WEAPON && (ent->item->giTag == WP_REDFLAG || ent->item->giTag == WP_BLUEFLAG || ent->item->giTag == WP_BALL))) {
		gentity_t	*te;

		te = G_TempEntity( ent->s.pos.trBase, EV_GLOBAL_ITEM_PICKUP );
		te->s.eventParm = ent->s.modelindex;
		te->r.svFlags |= SVF_BROADCAST;
	}

	// fire item targets
	G_UseTargets (ent, other);

	// wait of -1 will not respawn
	if ( ent->wait == -1 ) {
		ent->r.svFlags |= SVF_NOCLIENT;
		ent->s.eFlags |= EF_NODRAW;
		ent->r.contents = 0;
		ent->unlinkAfterEvent = qtrue;
		return;
	}

	// non zero wait overrides respawn time
	if ( ent->wait ) {
		respawn = ent->wait;
	}

	// random can be used to vary the respawn time
	if ( ent->random ) {
		respawn += crandom() * ent->random;
		if ( respawn < 1 ) {
			respawn = 1;
		}
	}

	// dropped items will not respawn
	if ( ent->flags & FL_DROPPED_ITEM ) {
		ent->freeAfterEvent = qtrue;
	}

	// picked up items still stay around, they just don't
	// draw anything.  This allows respawnable items
	// to be placed on movers.
	ent->r.svFlags |= SVF_NOCLIENT;
	ent->s.eFlags |= EF_NODRAW;
	ent->r.contents = 0;

	// ZOID
	// A negative respawn times means to never respawn this item (but don't 
	// delete it).  This is used by items that are respawned by third party 
	// events such as ctf flags
	if(ent->item->giType == IT_AMMO && ent->count > 0)
	{
		ent->nextthink = level.time + FRAMETIME;
		ent->think = RespawnItem;
	}
	else if(ent->item->giType == IT_WEAPON || ent->item->giType == IT_AMMO || ent->item->giType == IT_ARMOR)
	{
		ent->freeAfterEvent = qtrue;
		return;
	}
	else if ( respawn <= 0 ) {
		ent->nextthink = 0;
		ent->think = 0;
	} else {
		ent->nextthink = level.time + respawn * 1000;
		ent->think = RespawnItem;
	}
	trap_LinkEntity( ent );
}


//======================================================================

/*
================
LaunchItem

Spawns an item and tosses it forward
================
*/
gentity_t *LaunchItem( gitem_t *item, vec3_t origin, vec3_t velocity ) {
	gentity_t	*dropped;

	dropped = G_Spawn();

	dropped->s.eType = ET_ITEM;
	dropped->s.modelindex = item - bg_itemlist;	// store item number in modelindex
	dropped->s.modelindex2 = 1; // This is non-zero is it's a dropped item
	vectoangles(velocity, dropped->s.angles); //## Hentai ##
	dropped->classname = item->classname;
	dropped->item = item;
	G_SetBounds(dropped);
	dropped->r.contents = CONTENTS_TRIGGER;
	dropped->mass = item->mass;
	dropped->touch = Touch_Item;

	G_SetOrigin( dropped, origin );
	dropped->s.pos.trType = TR_GRAVITY;
	dropped->s.pos.trTime = level.time;
	//dropped->physicsBounce = 0.50;		// items are bouncy	
	VectorCopy( velocity, dropped->s.pos.trDelta );

	dropped->s.eFlags |= EF_BOUNCE_HALF;

	if (item->giType == IT_WEAPON && item->giTag >= WP_REDFLAG) { // Special case for CTF flags
		dropped->think = Team_DroppedFlagThink;
		dropped->nextthink = level.time + 30000;
		Team_CheckDroppedItem( dropped );
	}
	else
	{ // weapons should NEVER go away in this mod
		dropped->think = G_FreeEntity;
		dropped->nextthink = level.time + 9999999;
	}
	
	dropped->flags = FL_DROPPED_ITEM;
	dropped->s.eFlags |= EF_DEAD;
	if(dropped->item->giType == IT_AMMO)
	{
		dropped->health = 100;
		dropped->takedamage = -1;
	}
	else if(dropped->item->giType != IT_POWERUP)
	{
		dropped->health = 2500;
		dropped->takedamage = -1;
	}

	trap_LinkEntity (dropped);

	return dropped;
}

/*
================
Drop_Item

Spawns an item and tosses it forward
================
*/
gentity_t *Drop_Item( gentity_t *ent, gitem_t *item, float angle ) {
	vec3_t	velocity;
	vec3_t	angles;

	VectorCopy( ent->s.apos.trBase, angles );
	angles[YAW] += angle;
	angles[PITCH] = 0;	// always forward

	AngleVectors( angles, velocity, NULL, NULL );
	VectorScale( velocity, 150, velocity );
	velocity[2] += 200 + crandom() * 50;
	
	return LaunchItem( item, ent->s.pos.trBase, velocity );
}

/*
================
Use_Item

Respawn the item
================
*/
void Use_Item( gentity_t *ent, gentity_t *other, gentity_t *activator ) {
	RespawnItem( ent );
}

//======================================================================

/*
================
FinishSpawningItem

Traces down to find where an item should rest, instead of letting them
free fall from their spawn points
================
*/
void FinishSpawningItem( gentity_t *ent ) {
	trace_t		tr;
	vec3_t		dest;

	G_SetBounds(ent);

	ent->s.eType = ET_ITEM;
	ent->s.modelindex = ent->item - bg_itemlist;		// store item number in modelindex
	ent->s.modelindex2 = 0; // zero indicates this isn't a dropped item
	ent->s.angles[YAW] = rand() % 360;
	ent->r.contents = CONTENTS_TRIGGER;
	ent->touch = Touch_Item;
	// useing an item causes it to respawn
	ent->use = Use_Item;

	if ( ent->spawnflags & 1 ) {
		// suspended
		VectorSet( dest, ent->s.origin[0], ent->s.origin[1], ent->s.origin[2] - 32 );
		trap_Trace( &tr, ent->s.origin, ent->r.mins, ent->r.maxs, dest, ent->s.number, MASK_SOLID );
		if(tr.fraction < 1)
		{ // avoid making it hover if we possibly can, since it looks gay.
			G_SetOrigin( ent, tr.endpos );
		}
		else
		{
			G_SetOrigin( ent, ent->s.origin );
			ent->s.eFlags |= EF_TELEPORT_BIT;
		}
	} else {
		// drop to floor
		VectorSet( dest, ent->s.origin[0], ent->s.origin[1], ent->s.origin[2] - 4096 );
		trap_Trace( &tr, ent->s.origin, ent->r.mins, ent->r.maxs, dest, ent->s.number, MASK_SOLID );
		if ( tr.startsolid ) {
			G_Printf ("FinishSpawningItem: %s startsolid at %s\n", ent->classname, vtos(ent->s.origin));
			G_FreeEntity( ent );
			return;
		}

		// allow to ride movers
		ent->s.groundEntityNum = tr.entityNum;
		ent->s.eFlags &= ~EF_TELEPORT_BIT;
		G_SetOrigin( ent, tr.endpos );
	}

	// team slaves and targeted items aren't present at start
	if ( ( ent->flags & FL_TEAMSLAVE ) || ent->targetname ) {
		ent->s.eFlags |= EF_NODRAW;
		ent->r.contents = 0;
		return;
	}
	// ## Hentai ##
	if(ent->item->giType == IT_WEAPON && ent->item->giTag >= WP_REDFLAG)
	{
		gentity_t *spawnent = G_Spawn();
		spawnent->classname = ent->classname;
		spawnent->r.svFlags |= SVF_NOCLIENT;
		spawnent->s.eType = ET_TELEPORT_TRIGGER;
		G_SetOrigin(spawnent, ent->s.origin);
		spawnent->neverFree = qtrue;
	}
	else if ( ent->item->giType == IT_WEAPON && g_allowWeapons.integer == 0) 
	{
		G_FreeEntity( ent );
		return;
	
	}	
	else if(ent->item->giType == IT_AMMO && (g_allowAmmo.integer == 0 || g_allowWeapons.integer == 0))
	{
		G_FreeEntity( ent );
		return;
	
	}	
	else if ( ent->item->giType == IT_HEALTH && g_allowHealth.integer == 0) 
	{
		G_FreeEntity( ent );
		return;
	}
	else if ( ent->item->giType == IT_ARMOR && g_allowArmor.integer == 0) 
	{
		G_FreeEntity( ent );
		return;
	}
	else if ( ( ent->item->giType == IT_WEAPON || ent->item->giType == IT_AMMO || ent->item->giType == IT_ARMOR || ent->item->giType == IT_HEALTH ) && !(ent->spawnflags & 32768)) 	
	{//32768 means we explicitly set count somewhere
		ent->count = ent->item->quantity;
		ent->s.powerups = ent->count;
	
	}else if ( ent->item->giType == IT_POWERUP ) {
		float	respawn;
		
		if(g_allowPowerups.integer == 0) // don't spawn
		{
			G_FreeEntity( ent );
			return; 
		} // ## Hentai ##
		
		// powerups don't spawn in for a while	
		respawn = 45 + crandom() * 15;
		ent->s.eFlags |= EF_NODRAW;
		ent->r.contents = 0;
		ent->nextthink = level.time + respawn * 1000;
		ent->think = RespawnItem;
		return;
	}
	
	if(ent->item->giType == IT_AMMO)
	{
		ent->health = 100;
		ent->takedamage = -1;
	}
	else if(ent->item->giType != IT_POWERUP)
	{
		ent->health = 2500;
		ent->takedamage = -1;
	}

	trap_LinkEntity (ent);
	if(ent->item->giType == IT_WEAPON && (ent->item->giTag == WP_REDFLAG || ent->item->giTag == WP_BLUEFLAG || ent->item->giTag == WP_BALL) )
		ent->nextthink = level.time + 1000;
		ent->think = Team_ResetItem;
}


qboolean	itemRegistered[MAX_ITEMS];

/*
==================
G_CheckTeamItems
==================
*/
void G_CheckTeamItems( void ) {

	// Set up team stuff
	Team_InitGame();

	if ( g_gametype.integer == GT_CTF ) {
		gitem_t	*item;

		// make sure we actually have two flags...
		item = BG_FindItem( "Red Flag" );
		if ( !item || !itemRegistered[ item - bg_itemlist ] ) {
			G_Printf( "^1WARNING: No team_CTF_redflag in map" );
		}
		item = BG_FindItem( "Blue Flag" );
		if ( !item || !itemRegistered[ item - bg_itemlist ] ) {
			G_Printf( "^1WARNING: No team_CTF_blueflag in map" );
		}
	}
}

/*
==============
ClearRegisteredItems
==============
*/
void ClearRegisteredItems( void ) {
	memset( itemRegistered, 0, sizeof( itemRegistered ) );

	// players always start with the base weapon
	//RegisterItem( BG_FindItemForWeapon( WP_MACHINEGUN ) );
	//RegisterItem( BG_FindItemForWeapon( WP_GAUNTLET ) );
}

/*
===============
RegisterItem

The item will be added to the precache list
===============
*/
void RegisterItem( gitem_t *item ) {
	if ( !item ) {
		G_Error( "RegisterItem: NULL" );
	}
	itemRegistered[ item - bg_itemlist ] = qtrue;
}


/*
===============
SaveRegisteredItems

Write the needed items to a config string
so the client will know which ones to precache
===============
*/
void SaveRegisteredItems( void ) {
	char	string[MAX_ITEMS+1];
	int		i;
	int		count;

	count = 0;
	for ( i = 0 ; i < bg_numItems ; i++ ) {
		if ( itemRegistered[i] ) {
			count++;
			string[i] = '1';
		} else {
			string[i] = '0';
		}
	}
	string[ bg_numItems ] = 0;

	G_Printf( "%i items registered\n", count );
	trap_SetConfigstring(CS_ITEMS, string);
}


/*
============
G_SpawnItem

Sets the clipping size and plants the object on the floor.

Items can't be immediately dropped to floor, because they might
be on an entity that hasn't spawned yet.
============
*/
void G_SpawnItem (gentity_t *ent, gitem_t *item) {
	G_SpawnFloat( "random", "0", &ent->random );
	G_SpawnFloat( "wait", "0", &ent->wait );
	RegisterItem( item );
	ent->item = item;
	// some movers spawn on the second frame, so delay item
	// spawns until the third frame so they can ride trains
	ent->nextthink = level.time + FRAMETIME * 2;
	ent->think = FinishSpawningItem;
	ent->physicsBounce = 0.50;		// items are bouncy
	ent->enemy = NULL;
	if ( item->giType == IT_POWERUP ) {
		G_SoundIndex( "sound/items/poweruprespawn.wav" );
	}
}


/*
================
G_BounceItem

================
*/
void G_BounceItem( gentity_t *ent, trace_t *trace ) {
	vec3_t	velocity;
	float	dot;
	int		hitTime;

	// reflect the velocity on the trace plane
	//G_Printf("Bounce?");
	hitTime = level.previousTime + ( level.time - level.previousTime ) * trace->fraction;
	BG_EvaluateTrajectoryDelta( &ent->s.pos, hitTime, velocity );
	dot = DotProduct( velocity, trace->plane.normal );
	VectorMA( velocity, -2*dot, trace->plane.normal, ent->s.pos.trDelta );

	// cut the velocity to keep from bouncing forever
	VectorScale( ent->s.pos.trDelta, ent->physicsBounce, ent->s.pos.trDelta );
	if(VectorLength(ent->s.pos.trDelta) > 10)
	{
		vec3_t dir;
		VectorCopy(ent->s.pos.trDelta, dir);
		VectorNormalize(dir);

		VectorMA(ent->s.pos.trDelta, -10, dir, ent->s.pos.trDelta);
	}
	ent->s.clientNum = 0;
	if(!ent->item)
		return; // this can happen while dying
	// check for stop
	if(ent->item->giType == IT_WEAPON && ent->item->giTag == WP_BALL)
	{
		float speed = VectorLength(ent->s.pos.trDelta);
		
		//G_Printf("%f ", speed);
		if(speed > 0 && speed < 60)
		{
			ent->nextthink = level.time + 5000;
			ent->think = Team_ResetItem;
			G_SetOrigin(ent, trace->endpos);
			
		}
		else if(speed > 34)
		{
			G_AddEvent( ent, EV_GRENADE_BOUNCE, (int) (speed) );
			ent->nextthink = level.time + 90000;
			ent->think = Team_ResetItem;
			
		}
		
	}
	else if ( trace->plane.normal[2] > 0 && ent->s.pos.trDelta[2] < 40 ) {
		trace->endpos[2] += 1.0;	// make sure it is off ground
		SnapVector( trace->endpos );
		G_SetOrigin( ent, trace->endpos );
		ent->s.groundEntityNum = trace->entityNum;
	}
	else if(ent->item->giType == IT_AMMO && ent->item->giTag == AMMO_GRENADES)
	{
		G_AddEvent( ent, EV_GRENADE_BOUNCE, 0);
	}
	VectorAdd( ent->r.currentOrigin, trace->plane.normal, ent->r.currentOrigin);
	VectorCopy( ent->r.currentOrigin, ent->s.pos.trBase );
	ent->s.pos.trTime = level.time;
}


/*
================
G_RunItem

================
*/
void G_RunItem( gentity_t *ent ) {
	vec3_t		origin;
	trace_t		tr;
	int			contents;
	int			mask;
	
	if(ent->item)
	{		
		if(ent->item->giType == IT_WEAPON && ent->item->giTag >= WP_REDFLAG) 
			ent->takedamage = 0; // don't want flags dying
		
		if(ent->count >= -1)
			ent->s.powerups = ent->count;
		else if(ent->item->giType == IT_AMMO || ent->item->giType == IT_ARMOR )
		{
			G_FreeEntity(ent);
			return;
		}
		else
			ent->s.powerups = 0;
		if(ent->health < 0 && ent->pain_debounce_time < level.time)
		{
			int ammotype;
			ent->pain_debounce_time = level.time + 50;
			switch(ent->item->giType)
			{
			case IT_WEAPON:
				ammotype = ent->item->giUses;
				break;
			case IT_AMMO:
				ammotype = ent->item->giTag;
				break;
			default:
				G_FreeEntity(ent);
				return;
			}
			if(ent->count <= 0)
				ammotype = 0;
			switch(ammotype)
			{
			case AMMO_BULLETS:
				ent->s.pos.trType = TR_GRAVITY;
				BG_EvaluateTrajectory(&ent->s.pos, level.time, ent->s.pos.trBase);
				BG_EvaluateTrajectoryDelta(&ent->s.pos, level.time, ent->s.pos.trDelta);
				ent->s.pos.trTime = level.time;
				ent->splashMethodOfDeath = MOD_GRENADE_SPLASH; // only needed for shrapnel
				ent->splashDamage = ent->count;
				ent->splashRadius = 64;
				ent->splashDuration = 0;
				ent->s.weapon = WP_GRENADE_LAUNCHER; // for a fast explosoin				
				ent->parent = &g_entities[ENTITYNUM_WORLD];
				G_ExplodeMissile(ent);
				return;
			case AMMO_ROCKETS:
				ent->pain_debounce_time = level.time + 250;
				BG_EvaluateTrajectory(&ent->s.pos, level.time, ent->s.pos.trBase);
				BG_EvaluateTrajectoryDelta(&ent->s.pos, level.time, ent->s.pos.trDelta);
				ent->s.pos.trType = TR_GRAVITY;				
				ent->s.pos.trTime = level.time;
				ent->splashMethodOfDeath = MOD_ROCKET_SPLASH;
				ent->splashDamage = DEFAULT_ROCKET_DAMAGE * ent->count;
				ent->splashRadius = DEFAULT_ROCKET_RADIUS;
				ent->splashDuration = DEFAULT_ROCKET_DURATION;
				ent->s.weapon = WP_ROCKET_LAUNCHER;
				G_ExplodeMissile(ent);
				return;
			case AMMO_GRENADES:
				BG_EvaluateTrajectory(&ent->s.pos, level.time, ent->s.pos.trBase);
				BG_EvaluateTrajectoryDelta(&ent->s.pos, level.time, ent->s.pos.trDelta);
				ent->s.pos.trType = TR_GRAVITY;
				ent->s.pos.trTime = level.time;
				ent->splashMethodOfDeath = MOD_GRENADE_SPLASH;
				ent->splashDamage = DEFAULT_GRENADE_DAMAGE * ent->count;
				ent->splashRadius = DEFAULT_GRENADE_RADIUS;
				ent->splashDuration = DEFAULT_GRENADE_DURATION;
				ent->s.weapon = WP_GRENADE_LAUNCHER;
				ent->parent = &g_entities[ENTITYNUM_WORLD];
				G_ExplodeMissile(ent);
				return;
			default:
				G_FreeEntity(ent);
				return;
			}
			return;
		}
		if((ent->s.eFlags & EF_DEAD) && ent->s.clientNum > -1 && ent->s.time2 < level.time)
		{
			ent->s.clientNum = -1;
		}
	}
	// if groundentity has been set to -1, it may have been pushed off an edge
	if ( ent->s.groundEntityNum == -1 ) {
		if ( ent->s.pos.trType != TR_GRAVITY ) {
			ent->s.pos.trType = TR_GRAVITY;
			ent->s.pos.trTime = level.time;
		}
	}

	if(ent->onfire > level.time)
	{		
		if(trap_PointContents( ent->r.currentOrigin, -1 ) & CONTENTS_WATER)
			ent->onfire = 0;
		else if(level.time > ent->burntime)
		{
			G_AddEvent(ent, EV_ONFIRE, 1);
			if(ent->item && ent->item->giType == IT_WEAPON && ent->item->giTag >= WP_REDFLAG) // don't want flags dying
				ent->pain_debounce_time = level.time + 1000;
			else
				G_Damage(ent, NULL, ent, vec3_origin, vec3_origin, (rand() % 3), DAMAGE_NO_KNOCKBACK, MOD_FLAMETHROWER);
			if(ent->item && ent->count > 0 && ent->pain_debounce_time < level.time)
			{
				int test;
				gentity_t *bolt;
				float mass;
				switch(ent->item->giType)
				{
				case IT_AMMO:
					test = ent->item->giTag;
					break;
				case IT_WEAPON:
					test = ent->item->giUses;
					break;
				case IT_ARMOR:
					test = 0;
					ent->count = ent->health / 50;
					break;
				default:
					test = 0;
					break;
				}
				switch(test)
				{
				case AMMO_BULLETS:
					ent->pain_debounce_time = level.time + 50;
					BG_EvaluateTrajectory(&ent->s.pos, level.time, ent->s.pos.trBase);
					BG_EvaluateTrajectoryDelta(&ent->s.pos, level.time, ent->s.pos.trDelta);
					ent->s.pos.trType = TR_GRAVITY;
					ent->s.pos.trTime = level.time;
					bolt = fire_bullet(&g_entities[ENTITYNUM_WORLD], ent->s.pos.trBase, bytedirs[rand() & 0xff]);
					mass = G_CalcMass(ent->item, ent->count);
					if(mass <= 2)
						break;

					VectorMA(ent->s.pos.trDelta, 10.0 / mass, bytedirs[rand() & 0xff], ent->s.pos.trDelta);
					ent->count--;
					break;
				case AMMO_ROCKETS:
					ent->pain_debounce_time = level.time + 250;
					BG_EvaluateTrajectory(&ent->s.pos, level.time, ent->s.pos.trBase);
					BG_EvaluateTrajectoryDelta(&ent->s.pos, level.time, ent->s.pos.trDelta);
					ent->s.pos.trType = TR_GRAVITY;
					ent->s.pos.trTime = level.time;
					bolt = fire_rocket(&g_entities[ENTITYNUM_WORLD], ent->s.pos.trBase, bytedirs[rand() & 0xff]);
					ent->count--;
					break;
				case AMMO_GRENADES:
					if(!(rand() & 7))
					{ // BOOM
						BG_EvaluateTrajectory(&ent->s.pos, level.time, ent->s.pos.trBase);
						BG_EvaluateTrajectoryDelta(&ent->s.pos, level.time, ent->s.pos.trDelta);
						ent->s.pos.trType = TR_GRAVITY;
						ent->s.pos.trTime = level.time;
						ent->splashMethodOfDeath = MOD_GRENADE_SPLASH;
						ent->splashDamage = DEFAULT_GRENADE_DAMAGE * ent->count;
						ent->splashRadius = DEFAULT_GRENADE_RADIUS;
						ent->splashDuration = DEFAULT_GRENADE_DURATION;
						ent->s.weapon = WP_GRENADE_LAUNCHER;
						ent->parent = &g_entities[ENTITYNUM_WORLD];
						G_ExplodeMissile(ent);
						return; // we're DONE.
					}
				default:
					break;
				}
			}
			ent->burntime = level.time + 100;
			

		}
	}

	if ( ent->s.pos.trType == TR_STATIONARY ) {
		// check think function
		G_RunThink( ent );
		VectorCopy(ent->r.currentOrigin, ent->s.origin);
		VectorCopy(ent->r.currentAngles, ent->s.angles);
		return;
	}
	// get current position
	BG_EvaluateTrajectory( &ent->s.apos, level.time, ent->r.currentAngles );
	BG_EvaluateTrajectory( &ent->s.pos, level.time, origin );

	// trace a line from the previous position to the current position
	if ( ent->clipmask ) {
		mask = ent->clipmask;
	} else {
		mask = MASK_PLAYERSOLID & ~CONTENTS_BODY;//MASK_SOLID;
	}
	if(ent->item)
	{
		G_SetBounds(ent);
	}
	trap_Trace( &tr, ent->r.currentOrigin, ent->r.mins, ent->r.maxs, origin, 
		ent->r.ownerNum, mask );
	
	VectorCopy( tr.endpos, ent->r.currentOrigin );
	VectorCopy(tr.endpos, ent->s.origin);
	if ( tr.startsolid ) {
		tr.fraction = 0;
	}

	trap_LinkEntity( ent );	// FIXME: avoid this for stationary?

	// check think function
	G_RunThink( ent );

	if ( tr.fraction == 1 ) {
		ent->s.groundEntityNum = -1;
		return;
	}

	// if it is in a nodrop volume, remove it
	contents = trap_PointContents( ent->r.currentOrigin, -1 );
	if(contents & CONTENTS_LAVA)
	{
		ent->onfire = level.time + 500;
	}
	else if ( contents & CONTENTS_NODROP ) {
		if (ent->item && ent->item->giType == IT_WEAPON && (ent->item->giTag == WP_REDFLAG || ent->item->giTag == WP_BLUEFLAG || ent->item->giTag == WP_BALL)) {
			Team_FreeEntity(ent);
		} else {
			G_FreeEntity( ent );
		}
		return;
	}

	G_BounceItem( ent, &tr );
}

