// Copyright (C) 1999-2000 Id Software, Inc.
//
// g_combat.c

#include "g_local.h"


/*
============
AddScore

Adds score to both the client and his team
============
*/
void AddScore( gentity_t *ent, int score ) {
	if ( !ent->client ) {
		return;
	}
	// no scoring during pre-match warmup
	if ( level.warmupTime ) {
		return;
	}
	ent->client->ps.persistant[PERS_SCORE] += score;
	if (g_gametype.integer == GT_TEAM)
		level.teamScores[ ent->client->ps.persistant[PERS_TEAM] ] += score;
	CalculateRanks();
}

/*
=================
TossClientItems

Toss the weapon and powerups for the killed player
=================
*/
// ## Hentai ## - drop a weapon
void Team_ResetItem(gentity_t *self);
void Team_SpawnFalg(int w);

float G_CalcMass(gitem_t *item, int count);
gentity_t *TossClientWeapon(gentity_t *self, gitem_t *item)
{	// drop the player's weapon and ammo

	if(!item || !item->giType)
	{
		return NULL;
	}
	else
	{
		gentity_t	*drop;
		int angle = 0;//crandom() * 10;
		if(self->health <= 0)
			angle = crandom() * crandom() * 90;
		// spawn the item
		drop = Drop_Item( self, item, angle );
		if(self->onfire > 0)
			drop->onfire = self->onfire;
		else
			drop->onfire = 0;
		drop->count = self->client->ps.ammo[AMMO_CURRENT];
		drop->mass = G_CalcMass(item, drop->count);
		if(self->health > 0)
		{
			vec3_t right;
			AngleVectors(self->s.apos.trBase, NULL, right, NULL);
			VectorMA(drop->r.currentOrigin, 16, right, drop->r.currentOrigin);
			VectorMA(drop->s.pos.trBase, 16, right, drop->s.pos.trBase);
			VectorMA(self->client->ps.velocity, 0.05 * (p_class[self->client->pers.playerclass].strength) / (drop->mass + 50), drop->s.pos.trDelta, drop->s.pos.trDelta);
		}
		else
		{
			VectorMA(self->client->ps.velocity, 0.05 * (-self->health) / (drop->mass + 50), drop->s.pos.trDelta, drop->s.pos.trDelta);
		}
		if(VectorLength(drop->s.pos.trDelta) > 200)
		{
			VectorNormalize(drop->s.pos.trDelta);
			VectorMA(self->client->ps.velocity, 200, drop->s.pos.trDelta, drop->s.pos.trDelta);
		}
		
		

		drop->s.groundEntityNum = -1;
		drop->s.powerups = drop->count;
		drop->s.eFlags |= EF_DEAD; // client-dropped gun
		if(item->giType == IT_WEAPON && item->giTag == WP_BALL)
		{
			vec3_t forward, right, up, aimdir;
			AngleVectors(self->client->ps.viewangles, forward, right, up);
			CalcMuzzlePoint ( self, forward, right, up, drop->s.pos.trBase, aimdir);			
			drop->nextthink = level.time + 30000;
			drop->think = Team_ResetItem;
			drop->physicsBounce = 0.85;			
			drop->r.mins[0] = -6;
			drop->r.mins[1] = -6;
			drop->r.mins[2] = -3;
			drop->r.maxs[0] = 6;
			drop->r.maxs[1] = 6;
			drop->r.maxs[2] = 9;
			drop->r.contents = CONTENTS_BODY;			
		}
		else if(item->giType == IT_WEAPON && (item->giTag == WP_REDFLAG || item->giTag == WP_BLUEFLAG)	)
		{
			drop->nextthink = level.time + 30000;
			drop->think = Team_ResetItem;			
			drop->r.mins[0] = -15;
			drop->r.mins[1] = -15;
			drop->r.mins[2] = -15;
			drop->r.maxs[0] = 15;
			drop->r.maxs[1] = 15;
			drop->r.maxs[2] = 15;
		}
		drop->s.clientNum = self->s.clientNum;
		drop->s.time2 = level.time + 2500;
		drop->enemy = self;
		self->client->ps.weapon = 0;
		self->client->ps.ammo[AMMO_CURRENT] = 0;
		return drop;
		}

}
// ## Hentai
void TossClientItems( gentity_t *self ) {
	gitem_t		*item;
	float		angle;
	int			i;
	gentity_t	*drop;

	// ## Hentai ##
	TossClientWeapon(self, &bg_itemlist[self->s.weapon]);	
	self->s.weapon = 0;
	for(i = AMMO_ITEM1; i < AMMO_ITEMS; i+= 2)
	{ // get rid of all our inventory
		if(self->client->ps.ammo[i])
		{
			drop = Drop_Item( self, &bg_itemlist[self->client->ps.ammo[i]], crandom() * 30);
			self->client->ps.ammo[i] = 0;
			drop->s.powerups = drop->count = self->client->ps.ammo[i + 1];
			self->client->ps.ammo[i + 1] = 0;
			drop->onfire = self->onfire;
		}
	}
	// drop all the powerups if not in teamplay
	if ( g_gametype.integer != GT_TEAM ) {
		angle = 45;
		for ( i = 1 ; i < PW_NUM_POWERUPS ; i++ ) {
			if ( self->client->ps.powerups[ i ] > level.time ) {
				item = BG_FindItemForPowerup( i );
				if ( i == PW_BERZERK || !item ) {
					continue;
				}
				
				drop = Drop_Item( self, item, angle );
				// decide how many seconds it has left
				drop->count = ( self->client->ps.powerups[ i ] - level.time ) / 1000;
				if ( drop->count < 1 ) {
					drop->count = 1;
				}
				angle += 45;
			}
		}
	}
}


/*
==================
LookAtKiller
==================
*/
void LookAtKiller( gentity_t *self, gentity_t *inflictor, gentity_t *attacker ) {
	vec3_t		dir;
	vec3_t		angles;

	if ( attacker && attacker != self ) {
		VectorSubtract (attacker->s.pos.trBase, self->s.pos.trBase, dir);
	} else if ( inflictor && inflictor != self ) {
		VectorSubtract (inflictor->s.pos.trBase, self->s.pos.trBase, dir);
	} else {
		self->client->ps.stats[STAT_DEAD_YAW] = self->s.angles[YAW];
		return;
	}

	self->client->ps.stats[STAT_DEAD_YAW] = vectoyaw ( dir );

	angles[YAW] = vectoyaw ( dir );
	angles[PITCH] = 0; 
	angles[ROLL] = 0;
}

/*
==================
GibEntity
==================
*/
void GibEntity( gentity_t *self, int killer ) {
	
	char *model;
	
	if(!self->client)
	{
		if(self->takedamage == -1)
			G_AddEvent( self, EV_GIB_PLAYER, -40 - self->health );
		else
			G_AddEvent( self, EV_GIB_NOBLOOD, -40 - self->health );
		model = BG_GetCorpseModel(self->s.modelindex);
	}
	else 
	{
		self->client->ps.stats[STAT_HEAD_DAMAGE] = 0;
		self->client->ps.stats[STAT_LEG_DAMAGE] = 0;
		if(p_class[self->client->pers.playerclass].bleeds)
		{
			G_AddEvent( self, EV_GIB_PLAYER, -40 - self->health );
			model = p_class[self->client->pers.playerclass].name;
		}
		else
		{
			G_AddEvent( self, EV_GIB_NOBLOOD, -40 - self->health );
			model = p_class[self->client->pers.playerclass].name;
		}
	}
	self->takedamage = qfalse;
	self->s.eType = ET_INVISIBLE;
	self->r.contents = 0;
	
	if(!G_ModelInUse(model)	)
		G_RemoveModel(model);

}

// these are just for logging, the client prints its own messages
char	*modNames[] = {
	"MOD_UNKNOWN",
	"MOD_SHOTGUN",
	"MOD_GAUNTLET",
	"MOD_MACHINEGUN",
	"MOD_GRENADE",
	"MOD_GRENADE_SPLASH",
	"MOD_ROCKET",
	"MOD_ROCKET_SPLASH",
	"MOD_PLASMA",
	"MOD_PLASMA_SPLASH",
	"MOD_RAILGUN",
	"MOD_LIGHTNING",
	"MOD_FLAMETHROWER",
	"MOD_CHAINGUN",
	"MOD_WATER",
	"MOD_SLIME",
	"MOD_LAVA",
	"MOD_CRUSH",
	"MOD_TELEFRAG",
	"MOD_FALLING",
	"MOD_SUICIDE",
	"MOD_TARGET_LASER",
	"MOD_TRIGGER_HURT",
	"MOD_PUNCH",
	"MOD_BLEED",
	"MOD_BERZERK",
	"MOD_SWORD",
	"MOD_PENALTY"
	
};


/*
==================
player_die
==================
*/
void player_die( gentity_t *self, gentity_t *inflictor, gentity_t *attacker, int damage, int meansOfDeath ) {
	gentity_t	*ent;
	int			anim;
	int			contents;
	int			killer;
	int			i;
	char		*killerName, *obit;
	
	if ( self->client->ps.pm_type == PM_DEAD ) {
		if(self->s.eType == ET_INVISIBLE)
			return; // we're already gone
		contents = trap_PointContents( self->r.currentOrigin, -1 );
		
		if ( self->health <= GIB_HEALTH ) 
		{// gib death			
			GibEntity( self, attacker->s.number);
			trap_LinkEntity (self);
			return;
		} 
		else		
		{// normal death
			
			static int i;

			switch ( i ) {
			case 0:
				anim = BOTH_DEATH1;
				break;
			case 1:
				anim = BOTH_DEATH2;
				break;
			case 2:
			default:
				anim = BOTH_DEATH3;
				break;
			}
			
			self->client->ps.legsAnim = 
				( ( self->client->ps.legsAnim & ANIM_TOGGLEBIT ) ^ ANIM_TOGGLEBIT ) | anim;
			self->client->ps.torsoAnim = 
				( ( self->client->ps.torsoAnim & ANIM_TOGGLEBIT ) ^ ANIM_TOGGLEBIT ) | anim;

//			if(self->client->ps.stats[STAT_HEAD_DAMAGE >= 100])
//				G_AddEvent( self, EV_HEAD_EXPLODEY1 + i, attacker->s.number );
//			else
//				G_AddEvent( self, EV_DEATH1 + i, attacker->s.number );

			// globally cycle through the different death animations
			i = ( i + 1 ) % 3;
		}

		trap_LinkEntity (self);

		return;
	}

	if ( level.intermissiontime ) {
		return;
	}

//	if (self->client && self->client->hook)
//		Weapon_HookFree(self->client->hook);

	self->client->ps.pm_type = PM_DEAD;

	if ( attacker ) {
		killer = attacker->s.number;
		if ( attacker->client ) {
			killerName = attacker->client->pers.netname;
		} else {
			killerName = "<non-client>";
		}
	} else {
		killer = ENTITYNUM_WORLD;
		killerName = "<world>";
	}

	if ( killer < 0 || killer >= MAX_CLIENTS ) {
		killer = ENTITYNUM_WORLD;
		killerName = "<world>";
	}

	if ( meansOfDeath < 0 || meansOfDeath >= sizeof( modNames ) / sizeof( modNames[0] ) ) {
		obit = "<bad obituary>";
	} else {
		obit = modNames[ meansOfDeath ];
	}

	G_LogPrintf("Kill: %i %i %i: %s killed %s by %s\n", 
		killer, self->s.number, meansOfDeath, killerName, 
		self->client->pers.netname, obit );

	// broadcast the death event to everyone
	ent = G_TempEntity( self->r.currentOrigin, EV_OBITUARY );
	ent->s.eventParm = meansOfDeath;
	ent->s.otherEntityNum = self->s.number;
	ent->s.otherEntityNum2 = killer;
	ent->r.svFlags = SVF_BROADCAST;	// send to everyone

	self->enemy = attacker;

	self->client->ps.persistant[PERS_KILLED]++;

	if (attacker && attacker->client) {
		if ( attacker == self || OnSameTeam (self, attacker ) ) {
			AddScore( attacker, -1 );
		} else {
			AddScore( attacker, 1 );

			if( meansOfDeath == MOD_GAUNTLET || meansOfDeath == MOD_PUNCH ) {
				attacker->client->ps.persistant[PERS_GAUNTLET_FRAG_COUNT]++;
				attacker->client->ps.persistant[PERS_REWARD] = REWARD_GAUNTLET;
				attacker->client->ps.persistant[PERS_REWARD_COUNT]++;

				// add the sprite over the player's head
				attacker->client->ps.eFlags &= ~(EF_AWARD_EXCELLENT | EF_AWARD_GAUNTLET );
				attacker->client->ps.eFlags |= EF_AWARD_GAUNTLET;
				attacker->client->rewardTime = level.time + REWARD_SPRITE_TIME;

				// also play humiliation on target
				self->client->ps.persistant[PERS_REWARD] = REWARD_GAUNTLET;
				self->client->ps.persistant[PERS_REWARD_COUNT]++;
			}

			// check for two kills in a short amount of time
			// if this is close enough to the last kill, give a reward sound
			if ( level.time - attacker->client->lastKillTime < CARNAGE_REWARD_TIME ) {
				attacker->client->ps.persistant[PERS_REWARD_COUNT]++;
				attacker->client->ps.persistant[PERS_REWARD] = REWARD_EXCELLENT;
				attacker->client->ps.persistant[PERS_EXCELLENT_COUNT]++;

				// add the sprite over the player's head
				attacker->client->ps.eFlags &= ~(EF_AWARD_EXCELLENT | EF_AWARD_GAUNTLET );
				attacker->client->ps.eFlags |= EF_AWARD_EXCELLENT;
				attacker->client->rewardTime = level.time + REWARD_SPRITE_TIME;
			}
			attacker->client->lastKillTime = level.time;

		}
	} else {
		AddScore( self, -1 );
	}
	
	
	// Add team bonuses
	Team_FragBonuses(self, inflictor, attacker);

	// if client is in a nodrop area, don't make a corpse (but return CTF flags!)
	contents = trap_PointContents( self->r.currentOrigin, -1 );
	TossClientItems( self );

	Cmd_Score_f( self );		// show scores
	// send updated scores to any clients that are following this one,
	// or they would get stale scoreboards
	for ( i = 0 ; i < level.maxclients ; i++ ) {
		gclient_t	*client;

		client = &level.clients[i];
		if ( client->pers.connected != CON_CONNECTED ) {
			continue;
		}
		if ( client->sess.sessionTeam != TEAM_SPECTATOR ) {
			continue;
		}
		if ( client->sess.spectatorClient == self->s.number ) {
			Cmd_Score_f( g_entities + i );
		}
	}

	self->takedamage = qtrue;	// can still be gibbed

	self->s.weapon = WP_NONE;
	self->s.powerups = 0;
	self->r.contents = CONTENTS_CORPSE;

	self->s.angles[0] = 0;
	self->s.angles[2] = 0;
	LookAtKiller (self, inflictor, attacker);

	VectorCopy( self->s.angles, self->client->ps.viewangles );

	self->s.loopSound = 0;

	self->r.maxs[2] = -8;

	// don't allow respawn until the death anim is done
	// g_forcerespawn may force spawning at some later time
	self->client->respawnTime = level.time + 1700;

	// remove powerups
	memset( self->client->ps.powerups, 0, sizeof(self->client->ps.powerups) );

	// never gib in a nodrop
	if ( self->health <= GIB_HEALTH ) 
	{// gib death		
		GibEntity( self, killer);
		trap_LinkEntity (self);
		return;
	} 
	else
	{// normal death
		static int i;

		switch ( i ) {
		case 0:
			anim = BOTH_DEATH1;
			break;
		case 1:
			anim = BOTH_DEATH2;
			break;
		case 2:
		default:
			anim = BOTH_DEATH3;
			break;
		}

		self->client->ps.legsAnim = 
			( ( self->client->ps.legsAnim & ANIM_TOGGLEBIT ) ^ ANIM_TOGGLEBIT ) | anim;
		self->client->ps.torsoAnim = 
			( ( self->client->ps.torsoAnim & ANIM_TOGGLEBIT ) ^ ANIM_TOGGLEBIT ) | anim;

		if(self->client->ps.stats[STAT_HEAD_DAMAGE] >= 100)
			G_AddEvent( self, EV_HEAD_EXPLODEY1 + i, attacker->s.number );
		else
			G_AddEvent( self, EV_DEATH1 + i, attacker->s.number );

		// globally cycle through the different death animations
		i = ( i + 1 ) % 3;
	}

	trap_LinkEntity (self);
}


/*
================
CheckArmor
================
*/
int CheckArmor (gentity_t *ent, int damage, int dflags, int mod)
{
	gclient_t	*client;
	int			save;
	int			count;

	if (!damage)
		return 0;

	client = ent->client;

	if (!client)
		return 0;

	if (dflags & DAMAGE_NO_ARMOR)
		return 0;

	
	// armor
	count = client->ps.stats[STAT_ARMOR];
	
	if(mod == MOD_LIGHTNING) // you have ARMOR on? you poor, poor fool.
		save = -(damage * random() + 1);
	else if(mod == MOD_PUNCH)
		save = damage;	
	else
		save = damage * count * 0.01;
	if (save > damage)
		save = damage;

	if (!save)
		return 0;

	if((dflags & DAMAGE_RADIUS) || (mod == MOD_LIGHTNING || mod == MOD_PLASMA || mod == MOD_ROCKET_SPLASH || mod == MOD_GRENADE_SPLASH || mod == MOD_PLASMA_SPLASH || mod == MOD_SWORD || mod == MOD_FLAMETHROWER || mod == MOD_GAUNTLET))
		client->ps.stats[STAT_ARMOR] -= fabs(save) * 1.5; // explosions chew up armor good
	else if(mod != MOD_PUNCH)
		client->ps.stats[STAT_ARMOR] -= save * 0.1; // make armor a bit tougher vs. bullets
	if(client->ps.stats[STAT_ARMOR] < 0)
		client->ps.stats[STAT_ARMOR] = 0;
	return save;
}


/*
============
T_Damage

targ		entity that is being damaged
inflictor	entity that is causing the damage
attacker	entity that caused the inflictor to damage targ
	example: targ=monster, inflictor=rocket, attacker=player

dir			direction of the attack for knockback
point		point at which the damage is being inflicted, used for headshots
damage		amount of damage being inflicted
knockback	force to be applied against targ as a result of the damage

inflictor, attacker, dir, and point can be NULL for environmental effects

dflags		these flags are used to control how T_Damage works
	DAMAGE_RADIUS			damage was indirect (from a nearby explosion)
	DAMAGE_NO_ARMOR			armor does not protect from this damage
	DAMAGE_NO_KNOCKBACK		do not affect velocity, just view angles
	DAMAGE_NO_PROTECTION	kills godmode, armor, everything
============
*/

int G_Damage( gentity_t *targ, gentity_t *inflictor, gentity_t *attacker,
			   vec3_t dir, vec3_t point, int damage, int dflags, int mod ) {
	gclient_t	*client;
	int			bleed;
	int			take;
	int			save;
	int			asave;
	int			knockback;
	int			hit_area;
	float prob = 0;

	if (!targ->takedamage) {
		return HIT_MISSED;
	}

	// the intermission has allready been qualified for, so don't
	// allow any extra scoring
	if ( level.intermissionQueued ) {
		return HIT_MISSED;
	}

	if ( !inflictor ) {
		inflictor = &g_entities[ENTITYNUM_WORLD];
	}
	if ( !attacker ) {
		attacker = &g_entities[ENTITYNUM_WORLD];
	}

	client = targ->client;

	if ( !dir ) {
		dflags |= DAMAGE_NO_KNOCKBACK;
	} else {
		VectorNormalize(dir);
	}

	if ( targ->flags & FL_NO_KNOCKBACK ) 
		knockback = 0;
	else if ( dflags & DAMAGE_NO_KNOCKBACK ) 
		knockback = 0;
	else 
	{
		knockback = damage;
		if ( knockback > 200 ) 
			knockback = 200;
	}

	if( !client ) 
	{
		
			// shootable doors / buttons don't actually have any health
		switch(targ->s.eType)
		{
		case ET_CORPSE:
				
			if(knockback)
			{
				vec3_t	kvel;
				VectorScale (dir, g_knockback.value * (float)knockback / 200.0, kvel);
				//VectorCopy(targ->s.origin, targ->s.pos.trBase);
				VectorAdd (targ->s.pos.trDelta, kvel, targ->s.pos.trDelta);		
					targ->s.pos.trTime = level.time;
			}
			targ->health -= damage;
			if ( targ->health > GIB_HEALTH )
				return HIT_AREA;
			
			GibEntity( targ, 0 );
			return HIT_AREA;
		case ET_MOVER:
			
			if ( targ->use && targ->moverState == MOVER_POS1 ) 
				targ->use( targ, inflictor, attacker );
		
			return HIT_MISSED;
		case ET_ITEM:

			if(knockback)
			{
				BG_EvaluateTrajectory(&targ->s.pos, level.time, targ->s.pos.trBase);
				BG_EvaluateTrajectoryDelta(&targ->s.pos, level.time, targ->s.pos.trDelta);
				targ->s.pos.trType = TR_GRAVITY;
				targ->s.pos.trTime = level.time;
				VectorMA (targ->s.pos.trDelta, g_knockback.value * (float)knockback / (G_CalcMass(targ->item, targ->count) + 1), dir, targ->s.pos.trDelta);
			}
			if((((mod == MOD_GRENADE_SPLASH || mod == MOD_ROCKET_SPLASH) && (damage * random() > 50)) || mod == MOD_FLAMETHROWER || mod == MOD_LAVA) && (inflictor != targ))
			{
				if(level.time > targ->onfire) 
					targ->onfire = level.time;		
				targ->onfire += damage * 100;
				targ->enemy = attacker;
				if(mod == MOD_FLAMETHROWER)
				{
					damage *= 2;
				}
			}
			targ->health -= damage;
			return HIT_AREA;
		default:
			return 0;
		}
		
		return 0; // can't go past here, since all further code assumes a client
	}
	

	if ( client->noclip || targ->flags & FL_GODMODE ) {
		return HIT_MISSED;
	}

	// figure momentum add, even if the damage won't be taken
	if ( knockback ) {
		vec3_t	kvel;
		float	mass;

		if(client->ps.stats[STAT_MASS])
			mass = client->ps.stats[STAT_MASS];
		
		else
			mass = 200;

		VectorScale (dir, g_knockback.value * (float)knockback / mass, kvel);
		VectorAdd (client->ps.velocity, kvel, client->ps.velocity);

		// set the timer so that the other client can't cancel
		// out the movement immediately
		if ( !client->ps.pm_time ) {
			int		t;

			t = knockback * 2;
			if ( t < 50 ) {
				t = 50;
			}
			if ( t > 200 ) {
				t = 200;
			}
			client->ps.pm_time = t;
			client->ps.pm_flags |= PMF_TIME_KNOCKBACK;
		}
	
	}

	// check for completely getting out of the damage
	
	// battlesuit protects from all radius damage (but takes knockback)
	// and protects 50% against all damage
	if ( client->ps.powerups[PW_BATTLESUIT] ) {
		G_AddEvent( targ, EV_POWERUP_BATTLESUIT, 0 );
		if ( dflags & DAMAGE_RADIUS ) {
			return HIT_MISSED;
		}
		damage *= g_SuitFactor.value;
	}
	
	// add to the attacker's hit counter
	if ( attacker->client && targ != attacker && targ->health > 0 && mod != MOD_BLEED) {
		if ( OnSameTeam( targ, attacker ) ) {
			attacker->client->ps.persistant[PERS_HITS] -= damage;
		} else {
			attacker->client->ps.persistant[PERS_HITS] += damage;
		}
	}
	client->ps.persistant[PERS_ATTACKER] = attacker->s.number;

	if ( damage < 1 ) {
		damage = 1;
	}
	take = damage;
	save = 0; asave = 0;
	
	
	hit_area = 0;
	client->area_hit = -1;
	if((((mod == MOD_GRENADE_SPLASH || mod == MOD_ROCKET_SPLASH) && (take * random() > 50)) || mod == MOD_FLAMETHROWER || mod == MOD_LAVA) && (inflictor != targ))
	{
		if(level.time > targ->onfire) 
			targ->onfire = level.time;		
		targ->onfire += take * 100;
		targ->enemy = attacker;
		
	}
	if(mod == MOD_PUNCH && bg_itemlist[targ->s.weapon].giType == IT_WEAPON && bg_itemlist[targ->s.weapon].giTag == WP_BALL)
	{ // punching someone with the ball makes them drop it.
		TossClientWeapon(targ, BG_FindItemForWeapon(WP_BALL));
	}
		
	if(mod == MOD_FALLING)
		hit_area = HIT_LEGS;
	else if(dflags & DAMAGE_RADIUS)
	{
		hit_area = HIT_AREA;
	}
	else if(mod == MOD_BLEED)
	{
		if(p_class[client->pers.playerclass].bleeds)
			G_AddPredictableEvent(targ, EV_BLOOD, take);
		else // what are we doing here anyway?
			return HIT_MISSED;
	}
	else if(point && dir)
	{
		vec3_t start, end, vdir;
		float xd, yd, zd, dis, headsize;
		if(mod == MOD_MACHINEGUN || mod == MOD_CHAINGUN || mod == MOD_SHOTGUN || mod == MOD_GRENADE_SPLASH || mod == MOD_RAILGUN)
		{
			gentity_t *tent = G_TempEntity( point, EV_BLOOD_SPURT );			
			tent->s.eventParm = DirToByte( dir );
			tent->s.otherEntityNum = attacker->s.number;
			tent->s.clientNum = targ->s.clientNum;
			tent->s.powerups = (damage / 10) + 1;
			VectorCopy(point, tent->s.origin);
		}
		if(client->pers.playerclass >= 0 && client->pers.playerclass < MAX_PCLASSES)
			headsize = p_class[targ->client->pers.playerclass].headsize;
		else
			headsize = 8;
		
		VectorSubtract(point, targ->s.pos.trBase, start);
		VectorMA(start, -16, dir, start); // in case head sticks out of bounding box
		VectorSubtract(targ->client->headpos, start, vdir);
		dis = VectorNormalize(vdir);
		VectorMA(start, dis * DotProduct(vdir, dir), dir, end);
		xd = end[0] - targ->client->headpos[0];
		yd = end[1] - targ->client->headpos[1];
		zd = end[2] - targ->client->headpos[2];
		dis = sqrt(xd*xd + yd*yd + zd*zd);
				
		if(dis < headsize)
		{			
			hit_area = HIT_HEAD;
		}
		else if(start[2] < targ->client->waistheight)
			hit_area = HIT_LEGS;
		else
			hit_area = HIT_TORSO;			
	}
	else
	{
		hit_area = HIT_AREA;
	}
	
	switch(mod)
	{
	default:
		if(hit_area == HIT_TORSO)
			prob = damage / (damage + 500.0);
		else
			prob = 0;
		break;
	case MOD_GRENADE_SPLASH:
		prob = damage / (damage + 100.0);
		break;
	case MOD_ROCKET_SPLASH:
		prob = damage / (damage + 100.0);
		break;
	case MOD_FLAMETHROWER:
		prob = damage / (damage + 25.0);
		break;
	}
	if(prob)
	{
		int i;
			if(client->ps.ammo[AMMO_CURRENT] && (bg_itemlist[client->ps.weapon].giType == IT_AMMO || bg_itemlist[client->ps.weapon].giType == IT_WEAPON))
		{ // this might have an explodable ammo
			int ammotype;
			gentity_t *bolt = NULL;
			vec3_t org;
			VectorMA(targ->r.currentOrigin, 34, bytedirs[rand() & 0xff], org);
			if(bg_itemlist[client->ps.weapon].giType == IT_AMMO)
				ammotype = bg_itemlist[client->ps.weapon].giTag;
			else
				ammotype = bg_itemlist[client->ps.weapon].giUses;
			switch(ammotype)
			{
			case AMMO_BULLETS:
				bolt = fire_bullet(&g_entities[ENTITYNUM_WORLD], org, bytedirs[rand() & 0xff]);
				G_AddEvent(bolt, EV_FIRE_WEAPON_FROMITEM, 0);
				client->ps.delta_angles[PITCH] += ANGLE2SHORT(10 * crandom() * FRAMETIME / (client->ps.stats[STAT_MASS] + client->ps.stats[STAT_STRENGTH]));
				client->ps.delta_angles[YAW] += ANGLE2SHORT(40 * crandom() * FRAMETIME / (client->ps.stats[STAT_MASS] + client->ps.stats[STAT_STRENGTH]));
				client->ps.ammo[AMMO_CURRENT]--;
				break;
			case AMMO_ROCKETS:
				if(random() < prob * prob)
				{
					bolt = fire_rocket(&g_entities[ENTITYNUM_WORLD], org, bytedirs[rand() & 0xff]);
					G_AddEvent(bolt, EV_FIRE_WEAPON_FROMITEM, 0);
					bolt->nextthink = random() * 1000;
					client->ps.delta_angles[PITCH] += ANGLE2SHORT(15 * crandom() * FRAMETIME / (client->ps.stats[STAT_MASS] + client->ps.stats[STAT_STRENGTH]));
					client->ps.delta_angles[YAW] += ANGLE2SHORT(60 * crandom() * FRAMETIME / (client->ps.stats[STAT_MASS] + client->ps.stats[STAT_STRENGTH]));
					client->ps.ammo[AMMO_CURRENT]--;
					break;
				}
			case AMMO_GRENADES:
				if(random() < prob * (prob / (prob + 1)))
				{
					bolt = toss_grenade(&g_entities[ENTITYNUM_WORLD], org, bytedirs[rand() & 0xff]);
					bolt->nextthink = random() * 200;
					client->ps.delta_angles[PITCH] += ANGLE2SHORT(25 * crandom() * FRAMETIME / (client->ps.stats[STAT_MASS] + client->ps.stats[STAT_STRENGTH]));
					client->ps.delta_angles[YAW] += ANGLE2SHORT(100 * crandom() * FRAMETIME / (client->ps.stats[STAT_MASS] + client->ps.stats[STAT_STRENGTH]));
					client->ps.ammo[AMMO_CURRENT]--;
				break;
				}
			default:
				break;
			}
			if(bg_itemlist[client->ps.weapon].giType == IT_WEAPON && !client->ps.ammo[AMMO_CURRENT])
				client->ps.weapon = 0;
			if(bolt)
				bolt->s.eFlags |= EF_DEAD;
		}
		for(i = 2; i < AMMO_ITEMS; i += 2)
		{
			if(client->ps.ammo[i + 1] && (bg_itemlist[client->ps.ammo[i]].giType == IT_AMMO || bg_itemlist[client->ps.ammo[i]].giType == IT_WEAPON))
			{ // this might have an explodable ammo
				int ammotype;
				gentity_t *bolt = NULL;
				vec3_t org;
				VectorMA(targ->r.currentOrigin, 34, bytedirs[rand() & 0xff], org);
				if(bg_itemlist[client->ps.ammo[i]].giType == IT_AMMO)
					ammotype = bg_itemlist[client->ps.ammo[i]].giTag;
				else
					ammotype = bg_itemlist[client->ps.ammo[i]].giUses;
				switch(ammotype)
				{
				case AMMO_BULLETS:
					bolt = fire_bullet(targ, org, bytedirs[rand() & 0xff]);
					G_AddEvent(bolt, EV_FIRE_WEAPON, 0);
					client->ps.delta_angles[PITCH] += ANGLE2SHORT(10 * crandom() * FRAMETIME / (client->ps.stats[STAT_MASS] + client->ps.stats[STAT_STRENGTH]));
					client->ps.delta_angles[YAW] += ANGLE2SHORT(40 * crandom() * FRAMETIME / (client->ps.stats[STAT_MASS] + client->ps.stats[STAT_STRENGTH]));
					client->ps.ammo[i + 1]--;
					break;
				case AMMO_ROCKETS:
					if(random() < prob * prob)
					{
						bolt = fire_rocket(targ, org, bytedirs[rand() & 0xff]);
						G_AddEvent(bolt, EV_FIRE_WEAPON, 0);
						bolt->nextthink = random() * 1000;
						client->ps.delta_angles[PITCH] += ANGLE2SHORT(15 * crandom() * FRAMETIME / (client->ps.stats[STAT_MASS] + client->ps.stats[STAT_STRENGTH]));
						client->ps.delta_angles[YAW] += ANGLE2SHORT(60 * crandom() * FRAMETIME / (client->ps.stats[STAT_MASS] + client->ps.stats[STAT_STRENGTH]));
						client->ps.ammo[i + 1]--;
						break;
					}
				case AMMO_GRENADES:
					if(random() < prob * (prob / (prob + 1)))
					{
						bolt = toss_grenade(targ, org, bytedirs[rand() & 0xff]);
						bolt->nextthink = random() * 200;
						client->ps.ammo[i + 1]--;
						break;
					}
				default:
					break;
				}
				if(bg_itemlist[client->ps.ammo[i]].giType == IT_WEAPON && !client->ps.ammo[i + 1])
					client->ps.ammo[i] = 0;
				if(bolt)
				{
					break;
				}
			}
		}
	}
	client->area_hit = hit_area;
	switch(hit_area)
	{
	case HIT_HEAD:
		if(client->pers.playerclass >= 0 && client->pers.playerclass < MAX_PCLASSES)
			take *= p_class[client->pers.playerclass].head_dmgfactor;	
			client->ps.stats[STAT_HEAD_DAMAGE] += take;
		break;
	case HIT_AREA:
		asave = CheckArmor(targ, take / 2, dflags, mod);
		take -= asave;
		if(targ->client->pers.playerclass >= 0 && targ->client->pers.playerclass < MAX_PCLASSES)		
			take *= (p_class[targ->client->pers.playerclass].torso_dmgfactor + p_class[targ->client->pers.playerclass].legs_dmgfactor) / 2.0;
		if(dflags & DAMAGE_RADIUS)
			targ->client->ps.stats[STAT_HEAD_DAMAGE] += take * 0.4;
		break;	
		
	case HIT_TORSO:
		asave = CheckArmor(targ, take, dflags, mod);
		take -= asave;
		if(targ->client->pers.playerclass >= 0 && targ->client->pers.playerclass < MAX_PCLASSES)		
			take *= p_class[targ->client->pers.playerclass].torso_dmgfactor;
		if(dflags & DAMAGE_RADIUS)
			targ->client->ps.stats[STAT_HEAD_DAMAGE] += take * 0.25;
		break;
	case HIT_LEGS:
		if(targ->client->pers.playerclass >= 0 && targ->client->pers.playerclass < MAX_PCLASSES)		
			take *= p_class[targ->client->pers.playerclass].legs_dmgfactor;
		
		targ->client->ps.stats[STAT_LEG_DAMAGE] += take;		
		if(dflags & DAMAGE_RADIUS)
			targ->client->ps.stats[STAT_HEAD_DAMAGE] += take * 0.1;
		break;
	}
	// add to the damage inflicted on a player this frame
	// the total will be turned into screen blends and view angle kicks
	// at the end of the frame
	bleed = 0;
	if((!(dflags & DAMAGE_NO_BLEEDING)) && (p_class[client->pers.playerclass].bleeds))
	{
		if(hit_area == HIT_HEAD)
			bleed = take * 25;
		else if(hit_area == HIT_LEGS)
			bleed = take * 5;
		else if(hit_area == HIT_TORSO)
		{
			if(!asave)						
				bleed = take * 15;
		}
		else if(asave)
			bleed = take * 5;
		else
			bleed = take * 10;
			client->ps.stats[STAT_BLEEDING] += bleed;
		
		
		client->bleedtime = level.time;
		G_AddEvent(targ, EV_BLOOD, take);
		G_AddPredictableEvent(targ, EV_BLOOD, take); // just to be sure
		
	}
	else if((dflags & DAMAGE_NO_BLEEDING) && !(p_class[client->pers.playerclass].bleeds))
	{ // non-bleeders take more damage from non-bleeding hits
		{
			if(!asave)
				take *= 2;
		}
	}
	if ( g_debugDamage.integer) {
		G_Printf( "client:%i health:%i bleed:%i damage:%i armor:%i location:%i\n", targ->s.number,
		targ->health, bleed, take, asave, hit_area );
	}
	
	if(mod != MOD_BLEED)
	{
		client->damage_armor += asave;
		client->damage_blood += take;
		client->damage_knockback += knockback;			
	}
	if ( dir ) 
	{
		VectorCopy ( dir, client->damage_from );
		client->damage_fromWorld = qfalse;
	}
	else 
	{
		VectorCopy ( targ->r.currentOrigin, client->damage_from );
		client->damage_fromWorld = qtrue;
	}

	// See if it's the player hurting the emeny flag carrier
	Team_CheckHurtCarrier(targ, attacker);

	// set the last client who damaged the target
	client->lasthurt_client = attacker->s.number;
	client->lasthurt_mod = mod;
	
	// do the damage
	if(mod == MOD_BLEED && targ->health < take)
		take = 0;
	if (take) {
		client->ps.stats[STAT_HEALTH] -= take;
		targ->health = client->ps.stats[STAT_HEALTH];		
		targ->enemy = attacker;
		client->area_hit = hit_area;
			
		if(targ->client->ps.stats[STAT_HEAD_DAMAGE] >= 100)
			targ->health = client->ps.stats[STAT_HEALTH] = targ->health = 0;

		if ( targ->health <= 0 ) {
			targ->flags |= FL_NO_KNOCKBACK;
			if(hit_area == HIT_HEAD)
			{
				client->ps.stats[STAT_HEALTH] += take; // give it back
				if(client->ps.stats[STAT_HEALTH] > 0) // but not enough to revive us (headless zombies are gross)
					client->ps.stats[STAT_HEALTH] = 0;
				targ->health = client->ps.stats[STAT_HEALTH];
			}
			if (targ->health < -999)
				targ->health = -999;
			targ->enemy = attacker;
			player_die(targ, inflictor, attacker, take, mod);
			return hit_area;
		} 
		else if(mod == MOD_BLEED)
		{
		}
		else if ( targ->pain ) {
			targ->pain (targ, attacker, take);
		}
	}
	return hit_area;
}


/*
============
CanDamage

Returns qtrue if the inflictor can directly damage the target.  Used for
explosions and melee attacks.
============
*/
qboolean CanDamage (gentity_t *targ, vec3_t origin) {
	vec3_t	dest;
	trace_t	tr;
	vec3_t	midpoint;

	// use the midpoint of the bounds instead of the origin, because
	// bmodels may have their origin is 0,0,0
	VectorAdd (targ->r.absmin, targ->r.absmax, midpoint);
	VectorScale (midpoint, 0.5, midpoint);

	VectorCopy (midpoint, dest);
	trap_Trace ( &tr, origin, vec3_origin, vec3_origin, dest, ENTITYNUM_NONE, MASK_SOLID);
	if (tr.fraction == 1.0)
		return qtrue;

	// this should probably check in the plane of projection, 
	// rather than in world coordinate, and also include Z
	VectorCopy (midpoint, dest);
	dest[0] += 15.0;
	dest[1] += 15.0;
	trap_Trace ( &tr, origin, vec3_origin, vec3_origin, dest, ENTITYNUM_NONE, MASK_SOLID);
	if (tr.fraction == 1.0)
		return qtrue;

	VectorCopy (midpoint, dest);
	dest[0] += 15.0;
	dest[1] -= 15.0;
	trap_Trace ( &tr, origin, vec3_origin, vec3_origin, dest, ENTITYNUM_NONE, MASK_SOLID);
	if (tr.fraction == 1.0)
		return qtrue;

	VectorCopy (midpoint, dest);
	dest[0] -= 15.0;
	dest[1] += 15.0;
	trap_Trace ( &tr, origin, vec3_origin, vec3_origin, dest, ENTITYNUM_NONE, MASK_SOLID);
	if (tr.fraction == 1.0)
		return qtrue;

	VectorCopy (midpoint, dest);
	dest[0] -= 15.0;
	dest[1] -= 15.0;
	trap_Trace ( &tr, origin, vec3_origin, vec3_origin, dest, ENTITYNUM_NONE, MASK_SOLID);
	if (tr.fraction == 1.0)
		return qtrue;


	return qfalse;
}


/*
============
G_RadiusDamage
============
*/
void ContinueRadiusDamage(gentity_t *self)
{
	int damage, radius;
	self->nextthink = level.time + 50;	
	self->count++;
	damage = self->splashDamage * (self->count + 1) / self->count;
	if(damage < 1)
		damage = 1;
	if(self->methodOfDeath == MOD_FLAMETHROWER)
	{
		radius = self->splashRadius;
	}
	else
		radius = self->splashRadius * self->count;
	G_RadiusDamage(self->s.origin, self->enemy, damage, radius, 0, NULL, self->methodOfDeath);
	if(level.time > self->damage)
		self->think = G_FreeEntity;
}
qboolean G_RadiusDamage ( vec3_t origin, gentity_t *attacker, float damage, float radius, float duration,
					 gentity_t *ignore, int mod) {
	float		points, dist;
	gentity_t	*ent;
	int			entityList[MAX_GENTITIES];
	int			numListedEntities;
	vec3_t		mins, maxs;
	vec3_t		v;
	vec3_t		dir;
	int			i, e;
	qboolean	hitClient = qfalse;

	if(duration > 0)
	{
		gentity_t	*linger = G_Spawn();
		G_SetOrigin(linger, origin);
		linger->s.origin[0] = origin[0];
		linger->s.origin[1] = origin[1];
		linger->s.origin[2] = origin[2];
		linger->s.eType = ET_INVISIBLE;
		linger->enemy = attacker;
		linger->splashDamage = damage * 50 / duration; // will do total damage over its duration
		linger->splashRadius = radius * 50 / duration; // will expand outward to its final radius
		linger->count = 0; // number of times called (increases once every 50ms)
		linger->damage = level.time + duration; // this stores how long it stays
		linger->think = ContinueRadiusDamage;
		linger->nextthink = level.time + 50; // will cause damage every 50 ms
		linger->methodOfDeath = mod;
		linger->inuse = qtrue;
		damage /= 2; // initial 'flash' only does half damage
	}
	if ( radius < 1 ) {
		radius = 1;
	}

	for ( i = 0 ; i < 3 ; i++ ) {
		mins[i] = origin[i] - radius;
		maxs[i] = origin[i] + radius;
	}

	numListedEntities = trap_EntitiesInBox( mins, maxs, entityList, MAX_GENTITIES );

	for ( e = 0 ; e < numListedEntities ; e++ ) {
		ent = &g_entities[entityList[ e ]];

		if (ent == ignore)
			continue;
		if (!ent->takedamage)
			continue;

		// find the distance from the edge of the bounding box
		for ( i = 0 ; i < 3 ; i++ ) {
			if ( origin[i] < ent->r.absmin[i] ) {
				v[i] = ent->r.absmin[i] - origin[i];
			} else if ( origin[i] > ent->r.absmax[i] ) {
				v[i] = origin[i] - ent->r.absmax[i];
			} else {
				v[i] = 0;
			}
		}

		dist = VectorLength( v );
		if ( dist >= radius ) {
			continue;
		}

		points = damage * ( 1.0 - dist / radius );

		if( CanDamage (ent, origin) ) {
			if( LogAccuracyHit( ent, attacker ) ) {
				hitClient = qtrue;
			}
			VectorSubtract (ent->r.currentOrigin, origin, dir);
			if(mod == MOD_GRENADE_SPLASH || mod == MOD_MACHINEGUN)
				G_Damage (ent, NULL, attacker, dir, origin, (int)points, DAMAGE_RADIUS, mod);
			else if(mod == MOD_FLAMETHROWER)
				G_Damage (ent, NULL, attacker, dir, origin, (int)points, DAMAGE_RADIUS | DAMAGE_NO_KNOCKBACK | DAMAGE_NO_BLEEDING, mod);
			else
				G_Damage (ent, NULL, attacker, dir, origin, (int)points, DAMAGE_RADIUS | DAMAGE_NO_BLEEDING, mod);
		}
	}

	return hitClient;
}
